#!/usr/bin/perl

#------------------------------------------------------------------------------#
# Script for extracting data from benchmark logs into table
#
# Original by V. Schuppan
#------------------------------------------------------------------------------#

# Include program location in module search path. From perlfaq8.
BEGIN {
    use Cwd qw(abs_path);
    use File::Basename qw(dirname);

    my $path   = abs_path( $0 );
    our $directory = dirname( $path );
}
use lib $directory;

use strict;
use warnings;

#------------------------------------------------------------------------------#
# parameters
#

# @examples
use Config_examples;

# @switchess
my @switchess = (
    "all",
    "noprunemain",
    "nopruneloopits",
    "aug2",
    "bfsloopitinitcg",
    "bfsloopitinitce",
    "bfsloopconclusion2e"
    );

# @sharings
my @sharings = (
    "none",
    "same"
    );

# Original delimiter style
# Beginning of each line: name of example
my $txt_line_header_fmt = "%-34s |";
# sizenonnf experiment: size input
my $txt_sizenonnf_fmt = " %5s |";
# Column header for sizenonnf
my $txt_sizenonnf_tableheader_fmt = " %-5s |";
# none experiment: time, memory, size input
my $txt_none_fmt = " %5s/%7s/%5s/%5s/%5s/%5s |";
# Column header for none
my $txt_none_tableheader_fmt = " %-37s |";
# simple_proof experiment: time, memory, size core, size rg
my $txt_simple_proof_fmt = " %5s/%7s/%4s/%4s/%7s/%7s/%8s |";
# Column header for simple_proof
my $txt_simple_proof_tableheader_fmt = " %-48s |";
# simple_proofdeletion experiment: time, memory, size core, its
my $txt_simple_proofdeletion_fmt = " %5s/%7s/%4s/%4s/%4s/%3s/%3s/%3s |";
# Column header for simple_proofdeletion
my $txt_simple_proofdeletion_tableheader_fmt = " %-40s |";
# simple_deletion experiment: time, memory, size core, its
my $txt_simple_deletion_fmt = " %5s/%7s/%4s/%3s/%3s/%3s |";
# Column header for simple_deletion
my $txt_simple_deletion_tableheader_fmt = " %-30s |";
# partitioned_proof experiment: time, memory, partitions
my $txt_partitioned_proof_fmt = " %6s/%8s/%5s/%5s/%5s/%5s |";
# Column header for partitioned_proof
my $txt_partitioned_proof_tableheader_fmt = " %-39s |";
# pltlmup experiment: time, memory, conjuncts, conjuncts core
my $txt_pltlmup_fmt = " %6s/%8s/%4s/%4s |";
# Column header for pltlmup
my $txt_pltlmup_tableheader_fmt = " %-25s |";
# procmine experiment: time, memory, conjuncts, conjuncts core
my $txt_procmine_fmt = " %6s/%8s/%4s |";
# Column header for procmine
my $txt_procmine_tableheader_fmt = " %-20s |";

# Spreadsheet delimiter style
# Beginning of each line: name of example
my $csv_line_header_fmt = "%-34s ;";
# sizenonnf experiment: size input
my $csv_sizenonnf_fmt = " %5s ;";
# Column header for sizenonnf
my $csv_sizenonnf_tableheader_fmt = " %-5s ;";
# none experiment: time, memory, size input
my $csv_none_fmt = " %5s;%7s;%5s;%5s;%5s;%5s ;";
# Column header for none
my $csv_none_tableheader_fmt = " %-32s ;;;;;;";
# simple_proof experiment: time, memory, sat/unsat, etc
my $csv_simple_proof_fmt = " %5s;%7s;%4s;%4s;%7s;%7s;%8s ;";
# Column header for simple_proof
my $csv_simple_proof_tableheader_fmt = " %-42s ;;;;;;;";
# simple_proofdeletion experiment:
my $csv_simple_proofdeletion_fmt = "%5s;%7s;%4s;%4s;%4s;%3s;%3s;%3s ;";
# Column header for simple_proofdeletion
my $csv_simple_proofdeletion_tableheader_fmt = " %-32s ;;;;;;;;";
# simple_deletion experiment:
my $csv_simple_deletion_fmt = "%5s;%7s;%4s;%3s;%3s;%3s ;";
# Column header for simple_deletion
my $csv_simple_deletion_tableheader_fmt = " %-24s ;;;;;;";
# partitioned_proof experiment: time, memory, partitions
my $csv_partitioned_proof_fmt = " %5s;%7s;%4s;%4s;%4s;%4s ;";
# Column header for partitioned_proof
my $csv_partitioned_proof_tableheader_fmt = " %-28s ;;;;;;";
# pltlmup experiment: time, memory, conjuncts, conjuncts core
my $csv_pltlmup_fmt = " %5s;%7s;%4s;%4s ;";
# Column header for pltlmup
my $csv_pltlmup_tableheader_fmt = " %-20s ;;;;";
# procmine experiment: time, memory, conjuncts, conjuncts core
my $csv_procmine_fmt = " %5s;%7s;%4s ;";
# Column header for procmine
my $csv_procmine_tableheader_fmt = " %-16s ;;;";

# End of each line.
my $line_footer_fmt = "\n";

#------------------------------------------------------------------------------#
# global variables
#

# These are set either to txt_... or to csv_...
my $line_header_fmt;
my $sizenonnf_fmt;
my $sizenonnf_tableheader_fmt;
my $none_fmt;
my $none_tableheader_fmt;
my $simple_proof_fmt;
my $simple_proof_tableheader_fmt;
my $simple_proofdeletion_fmt;
my $simple_proofdeletion_tableheader_fmt;
my $simple_deletion_fmt;
my $simple_deletion_tableheader_fmt;
my $partitioned_proof_fmt;
my $partitioned_proof_tableheader_fmt;
my $pltlmup_fmt;
my $pltlmup_tableheader_fmt;
my $procmine_fmt;
my $procmine_tableheader_fmt;

#------------------------------------------------------------------------------#
# usage
#
sub usage() {
    printf "usage: generatetab.pl [-h|--help] [-txt|-csv]\n";
    exit 1;
}

#------------------------------------------------------------------------------#
# print_header
#
# Prints the first line of the table.
sub print_header() {
    printf $line_header_fmt, "example";
    printf $sizenonnf_tableheader_fmt, "";
    foreach my $sharing (@sharings) {
        printf $none_tableheader_fmt, "no uc ${sharing}";
    }
    foreach my $sharing (@sharings) {
        printf $simple_proof_tableheader_fmt, "uc proof ${sharing}";
    }
    foreach my $sharing (@sharings) {
        printf $simple_proofdeletion_tableheader_fmt, "uc proofdeletion ${sharing}";
    }
    foreach my $sharing (@sharings) {
        printf $simple_deletion_tableheader_fmt, "uc deletion ${sharing}";
    }
    foreach my $switches (@switchess) {
        foreach my $sharing (@sharings) {
            printf $simple_proof_tableheader_fmt, "uc proof ${switches} ${sharing}";
        }
    }
    foreach my $sharing (@sharings) {
        printf $partitioned_proof_tableheader_fmt, "partitioned uc proof ${sharing}";
    }
    printf $pltlmup_tableheader_fmt, "pltlmup";
    printf $procmine_tableheader_fmt, "procmine";
    printf $procmine_tableheader_fmt, "procmine nomus";
    printf "\n";
    printf $line_header_fmt, "";
    printf $sizenonnf_fmt, "sti";
    foreach my $sharing (@sharings) {
        printf $none_fmt, "t", "m", "stj", "sts", "clj", "cls";
    }
    foreach my $sharing (@sharings) {
        printf $simple_proof_fmt, "t", "m", "stc", "clc", "pv", "pe", "pve";
    }
    foreach my $sharing (@sharings) {
        printf $simple_proofdeletion_fmt, "t", "m", "stp", "clc", "stpd", "it", "itu", "its";
    }
    foreach my $sharing (@sharings) {
        printf $simple_deletion_fmt, "t", "m", "std", "it", "itu", "its";
    }
    foreach my $switches (@switchess) {
        foreach my $sharing (@sharings) {
            printf $simple_proof_fmt, "t", "m", "stc", "clc", "pv", "pe", "pve";
        }
    }
    foreach my $sharing (@sharings) {
        printf $partitioned_proof_fmt, "t", "m", "p", "po", "ppp", ">=2";
    }
    printf $pltlmup_fmt, "t", "m", "c", "cc";
    printf $procmine_fmt, "t", "m", "cc";
    printf $procmine_fmt, "t", "m", "cc";
    printf "\n";
}

#------------------------------------------------------------------------------#
# print_line_header
#
# Prints the first part of a line.
sub print_line_header($) {
    my $example = shift;

    printf $line_header_fmt, $example;
}

#------------------------------------------------------------------------------#
# print_line_footer
#
# Prints the last part of a line.
sub print_line_footer() {
    printf $line_footer_fmt;
}

#------------------------------------------------------------------------------#
# print_sizenonnf
#
sub print_sizenonnf($) {
    my $no_nodes_syntaxtree_before_simplification = shift;

    printf $sizenonnf_fmt,
    $no_nodes_syntaxtree_before_simplification;
}

#------------------------------------------------------------------------------#
# print_none
#
sub print_none($$$$$$$) {
    my $time = shift;
    my $mem = shift;
    my $res = shift;
    my $no_nodes_syntaxtree_before_simplification = shift;
    my $no_nodes_syntaxtree_after_simplification = shift;
    my $no_clauses_input_problem_before_simplification = shift;
    my $no_clauses_input_problem_after_simplification = shift;

    printf $none_fmt,
    $time,
    $mem,
    $no_nodes_syntaxtree_before_simplification,
    $no_nodes_syntaxtree_after_simplification,
    $no_clauses_input_problem_before_simplification,
    $no_clauses_input_problem_after_simplification;
}

#------------------------------------------------------------------------------#
# print_simple_proof
#
sub print_simple_proof($$$$$$$$$$$$) {
    my $time = shift;
    my $mem = shift;
    my $res = shift;
    my $no_nodes_syntaxtree_before_simplification = shift;
    my $no_nodes_syntaxtree_after_simplification = shift;
    my $no_clauses_input_problem_before_simplification = shift;
    my $no_clauses_input_problem_after_simplificatio = shift;
    my $no_nodes_syntaxtree_proof_unsat_core = shift;
    my $no_clauses_proof_unsat_core = shift;
    my $no_vertices_peak = shift;
    my $no_edges_peak = shift;
    my $no_vertices_plus_edges_peak = shift;

    printf $simple_proof_fmt,
    $time,
    $mem,
    $no_nodes_syntaxtree_proof_unsat_core,
    $no_clauses_proof_unsat_core,
    $no_vertices_peak,
    $no_edges_peak,
    $no_vertices_plus_edges_peak;
}

#------------------------------------------------------------------------------#
# print_simple_proofdeletion
#
sub print_simple_proofdeletion($$$$$$$$$$$$$$$$) {
    my $time = shift;
    my $mem = shift;
    my $res = shift;
    my $no_nodes_syntaxtree_before_simplification = shift;
    my $no_nodes_syntaxtree_after_simplification = shift;
    my $no_clauses_input_problem_before_simplification = shift;
    my $no_clauses_input_problem_after_simplification = shift;
    my $no_nodes_syntaxtree_proof_unsat_core = shift;
    my $no_clauses_proof_unsat_core = shift;
    my $no_vertices_peak = shift;
    my $no_edges_peak = shift;
    my $no_vertices_plus_edges_peak = shift;
    my $no_nodes_syntaxtree_proofdeletion_unsat_core = shift;
    my $no_deletion_its = shift;
    my $no_deletion_its_unsat = shift;
    my $no_deletion_its_sat = shift;

    printf $simple_proofdeletion_fmt,
    $time,
    $mem,
    $no_nodes_syntaxtree_proof_unsat_core,
    $no_clauses_proof_unsat_core,
    $no_nodes_syntaxtree_proofdeletion_unsat_core,
    $no_deletion_its,
    $no_deletion_its_unsat,
    $no_deletion_its_sat;
}

#------------------------------------------------------------------------------#
# print_simple_deletion
#
sub print_simple_deletion($$$$$$$$$$$) {
    my $time = shift;
    my $mem = shift;
    my $res = shift;
    my $no_nodes_syntaxtree_before_simplification = shift;
    my $no_nodes_syntaxtree_after_simplification = shift;
    my $no_clauses_input_problem_before_simplification = shift;
    my $no_clauses_input_problem_after_simplification = shift;
    my $no_nodes_syntaxtree_deletion_unsat_core = shift;
    my $no_deletion_its = shift;
    my $no_deletion_its_unsat = shift;
    my $no_deletion_its_sat = shift;

    printf $simple_deletion_fmt,
    $time,
    $mem,
    $no_nodes_syntaxtree_deletion_unsat_core,
    $no_deletion_its,
    $no_deletion_its_unsat,
    $no_deletion_its_sat;
}

#------------------------------------------------------------------------------#
# print_partitioned_proof
#
sub print_partitioned_proof($$$$$$$$$$$$$$$$) {
    my $time = shift;
    my $mem = shift;
    my $res = shift;
    my $no_nodes_syntaxtree_before_simplification = shift;
    my $no_nodes_syntaxtree_after_simplification = shift;
    my $no_clauses_input_problem_before_simplification = shift;
    my $no_clauses_input_problem_after_simplificatio = shift;
    my $no_nodes_syntaxtree_proof_unsat_core = shift;
    my $no_clauses_proof_unsat_core = shift;
    my $no_vertices_peak = shift;
    my $no_edges_peak = shift;
    my $no_vertices_plus_edges_peak = shift;
    my $no_propositions_ltl_core = shift;
    my $no_proposition_occurrences_ltl_core = shift;
    my $no_partitions_per_proposition_peak_ltl_core = shift;
    my $no_propositions_with_2_or_more_partitions_ltl_core = shift;
    
    printf $partitioned_proof_fmt,
    $time,
    $mem,
    $no_propositions_ltl_core,
    $no_proposition_occurrences_ltl_core,
    $no_partitions_per_proposition_peak_ltl_core,
    $no_propositions_with_2_or_more_partitions_ltl_core;
}

#------------------------------------------------------------------------------#
# print_pltlmup
#
sub print_pltlmup($$$$$) {
    my $time = shift;
    my $mem = shift;
    my $res = shift;
    my $no_conjuncts = shift;
    my $no_conjuncts_core = shift;
    
    printf $pltlmup_fmt,
    $time,
    $mem,
    $no_conjuncts,
    $no_conjuncts_core;
}

#------------------------------------------------------------------------------#
# print_procmine
#
sub print_procmine($$$$) {
    my $time = shift;
    my $mem = shift;
    my $res = shift;
    my $no_conjuncts_core = shift;
    
    printf $procmine_fmt,
    $time,
    $mem,
    $no_conjuncts_core;
}

#------------------------------------------------------------------------------#
# parse_sat_unsat
#
sub parse_sat_unsat($$) {
    my $line = shift;
    my $rres = shift;

    if ($line =~ /^Satisfiable/) {
        $$rres = "s";
        return 1;
    } elsif ($line =~ /^Unsatisfiable/) {
        $$rres = "u";
        return 1;
    }
    return 0;
}

#------------------------------------------------------------------------------#
# parse_sat_unsat_procmine
#
sub parse_sat_unsat_procmine($$) {
    my $line = shift;
    my $rres = shift;

    if ($line =~ /^Result: Formula is satisfiable./) {
        $$rres = "s";
        return 1;
    } elsif ($line =~ /^Result: Formula is not satisfiable./) {
        $$rres = "u";
        return 1;
    }
    return 0;
}

#------------------------------------------------------------------------------#
# parse_time_memory_errors
#
sub parse_time_memory_errors($$$$) {
    my $line = shift;
    my $rres = shift;
    my $rtime = shift;
    my $rmem = shift;

    if ($line =~ /^\[run\] time:\s+([0-9\.]+) seconds/) {
        my $time_unrounded = $1;
        $$rtime = sprintf("%.1f", $time_unrounded);
        return 1;
    } elsif ($line =~ /^\[run\] space:\s+([\-0-9\.]+) MB/) {
        $$rmem = $1;
        return 1;
    } elsif ($line =~ /^\[run\] status:\s+out of time/) {
        $$rres = $$rtime = $$rmem = "to";
        return 1;
    } elsif ($line =~ /^\[run\] status:\s+out of memory/) {
        $$rres = $$rtime = $$rmem = "mo";
        return 1;
    } elsif ($line =~ /^\[run\] status:\s+segmentation fault/) {
        $$rres = $$rtime = $$rmem = "er";
        return 1;
    } elsif ($line =~ /^\[run\] status:\s+signal\(6\)/) {
        $$rres = $$rtime = $$rmem = "er";
        return 1;
    } elsif ($line =~ /^\[run\] status:\s+signal\(9\)/) {
        $$rres = $$rtime = $$rmem = "to";
        return 1;
    } elsif ($line =~ /system: assertion/) {
        $$rres = $$rtime = $$rmem = "er";
        return 1;
    } elsif ($line =~ /syntax error/) {
        $$rres = $$rtime = $$rmem = "er";
        return 1;
    } elsif ($line =~ /Segmentation fault/) {
        $$rres = $$rtime = $$rmem = "er";
        return 1;
    } elsif ($line =~ /^Fatal error: exception Stack_overflow/) {
        $$rres = $$rtime = $$rmem = "er";
        return 1;
    }
    return 0;
}

#------------------------------------------------------------------------------#
# parse_simplification
#
sub parse_simplification($$$$) {
    my $line = shift;
    my $rres = shift;
    my $rtime = shift;
    my $rmem = shift;

    if ($line =~ /Input formula is or has been simplified to False./) {
        $$rres = $$rtime = $$rmem = "si";
        return 1;
    }
    return 0;
}

#------------------------------------------------------------------------------#
# parse_statistics_all
#
sub parse_statistics_all($$$$$) {
    my $line = shift;
    my $rno_nodes_syntaxtree_before_simplification = shift;
    my $rno_nodes_syntaxtree_after_simplification = shift;
    my $rno_clauses_input_problem_before_simplification = shift;
    my $rno_clauses_input_problem_after_simplification = shift;

    if ($line =~ /^MainAux: number of nodes in syntax tree before LTL simplification:\s+([0-9\.]+)/) {
        $$rno_nodes_syntaxtree_before_simplification = $1;
        return 1;
    } elsif ($line =~ /^MainAux: number of nodes in syntax tree after LTL simplification:\s+([0-9\.]+)/) {
        $$rno_nodes_syntaxtree_after_simplification = $1;
        return 1;
    } elsif ($line =~ /^BuilderLtl: number of clauses before pure literal simplification:\s+([0-9\.]+)/) {
        $$rno_clauses_input_problem_before_simplification = $1;
        return 1;
    } elsif ($line =~ /^BuilderLtl: number of clauses after pure literal simplification:\s+([0-9\.]+)/) {
        $$rno_clauses_input_problem_after_simplification = $1;
        return 1;
    }
    return 0;
}

#------------------------------------------------------------------------------#
# parse_statistics_simple_proof
#
sub parse_statistics_simple_proof($$$$$$) {
    my $line = shift;
    my $rno_nodes_syntaxtree_proof_unsat_core = shift;
    my $rno_clauses_proof_unsat_core = shift;
    my $rno_vertices_peak = shift;
    my $rno_edges_peak = shift;
    my $rno_vertices_plus_edges_peak = shift;

    if ($line =~ /^MainAux: number of nodes in syntax tree of proof core:\s+([0-9\.]+)/) {
        $$rno_nodes_syntaxtree_proof_unsat_core = $1;
        return 1;
    } elsif ($line =~ /^ResolutionGraph: number of clauses in unsatisfiable core:\s+([0-9\.]+)/) {
        $$rno_clauses_proof_unsat_core = $1;
        return 1;
    } elsif ($line =~ /esolutionGraph: peak number of vertices:\s+([0-9\.]+)/) {
        $$rno_vertices_peak = $1;
        return 1;
    } elsif ($line =~ /esolutionGraph: peak number of edges:\s+([0-9\.]+)/) {
        $$rno_edges_peak = $1;
        return 1;
    } elsif ($line =~ /esolutionGraph: peak number of vertices \+ edges:\s+([0-9\.]+)/) {
        $$rno_vertices_plus_edges_peak = $1;
        return 1;
    }
    return 0;
}

#------------------------------------------------------------------------------#
# parse_statistics_simple_deletion_proofdeletion
#
sub parse_statistics_simple_deletion_proofdeletion($$$$$$) {
    my $line = shift;
    my $rno_nodes_syntaxtree_deletion_proofdeletion_unsat_core = shift;
    my $rno_deletion_its = shift;
    my $rno_deletion_its_unsat = shift;
    my $rno_deletion_its_sat = shift;
    my $algorithm = shift;

    if ($algorithm eq "proofdeletion" && $line =~ /^MainAux: number of nodes in syntax tree of proofdeletion core:\s+([0-9\.]+)/) {
        $$rno_nodes_syntaxtree_deletion_proofdeletion_unsat_core = $1;
        return 1;
    } elsif ($algorithm eq "deletion" && $line =~ /^MainAux: number of nodes in syntax tree of deletion core:\s+([0-9\.]+)/) {
        $$rno_nodes_syntaxtree_deletion_proofdeletion_unsat_core = $1;
        return 1;
    } elsif ($line =~ /^MainAux: number of iterations in unsat core algorithm deletion:\s+([0-9\.]+)/) {
        $$rno_deletion_its = $1;
        return 1;
    } elsif ($line =~ /^MainAux: number of iterations in unsat core algorithm deletion with unsat result:\s+([0-9\.]+)/) {
        $$rno_deletion_its_unsat = $1;
        return 1;
    } elsif ($line =~ /^MainAux: number of iterations in unsat core algorithm deletion with sat result:\s+([0-9\.]+)/) {
        $$rno_deletion_its_sat = $1;
        return 1;
    }
    return 0;
}

#------------------------------------------------------------------------------#
# parse_statistics_partitioned_proof
#
sub parse_statistics_partitioned_proof($$$$$) {
    my $line = shift;
    my $rno_propositions_ltl_core = shift;
    my $rno_proposition_occurrences_ltl_core = shift;
    my $rno_partitions_per_proposition_peak_ltl_core = shift;
    my $rno_propositions_with_2_or_more_partitions_ltl_core = shift;

    if ($line =~ /^MapperSnfToLtl: number of propositions in LTL core:\s+([0-9\.]+)/) {
        $$rno_propositions_ltl_core = $1;
        return 1;
    } elsif ($line =~ /^MapperSnfToLtl: number of proposition occurrences in LTL core:\s+([0-9\.]+)/) {
        $$rno_proposition_occurrences_ltl_core = $1;
        return 1;
    } elsif ($line =~ /^MapperSnfToLtl: peak number of partitions per proposition in LTL core:\s+([0-9\.]+)/) {
        $$rno_partitions_per_proposition_peak_ltl_core = $1;
        return 1;
    } elsif ($line =~ /^MapperSnfToLtl: number of propositions with 2 or more partitions in LTL core:\s+([0-9\.]+)/) {
        $$rno_propositions_with_2_or_more_partitions_ltl_core = $1;
        return 1;
    }
    return 0;
}

#------------------------------------------------------------------------------#
# parse_statistics_pltlmup
#
sub parse_statistics_pltlmup($$$) {
    my $line = shift;
    my $rno_conjuncts = shift;
    my $rno_conjuncts_core = shift;

    if ($line =~ /^Formula count:\s+([0-9\.]+)/) {
        $$rno_conjuncts = $1;
        return 1;
    } elsif ($line =~ /^MUS Size:\s+([0-9\.]+)/) {
        $$rno_conjuncts_core = $1;
        return 1;
    }
    return 0;
}

#------------------------------------------------------------------------------#
# parse_statistics_procmine
#
sub parse_statistics_procmine($$) {
    my $line = shift;
    my $rno_conjuncts_core = shift;

    # count number of occurrences of ; in $line
    #
    # http://stackoverflow.com/questions/9538542/counting-number-of-occurrences-of-a-string-inside-another-perl
    $$rno_conjuncts_core = () = $line =~ /;/g;
    return 1;
}

#------------------------------------------------------------------------------#
# evaluate_sizenonnf
#
sub evaluate_sizenonnf($) {
    my $example = shift;
    my $logfile = "${example}.sizenonnf.log";
    my $line;
    my $no_nodes_syntaxtree_before_simplification;

    if (-f "${logfile}") {
        open(LOG, $logfile) or die "error: can't open $logfile for reading: $!";

        $no_nodes_syntaxtree_before_simplification = "na";

        while (($no_nodes_syntaxtree_before_simplification eq "na") && ($line = <LOG>)) {
            if ($line =~ /^BuilderLtl: number of nodes in syntax tree before LTL simplification:\s+([0-9\.]+)/) {
                $no_nodes_syntaxtree_before_simplification = $1;
            }
        }

        close(LOG);
    } else {
        $no_nodes_syntaxtree_before_simplification = "---";
    }

    return ($no_nodes_syntaxtree_before_simplification);
}

#------------------------------------------------------------------------------#
# evaluate_none
#
sub evaluate_none($$) {
    my $example = shift;
    my $sharing = shift;
    my $logfile = "${example}.none.${sharing}.log";
    my $line;
    my $time;
    my $mem;
    my $res;
    my $no_nodes_syntaxtree_before_simplification;
    my $no_nodes_syntaxtree_after_simplification;
    my $no_clauses_input_problem_before_simplification;
    my $no_clauses_input_problem_after_simplification;

    if (-f "${logfile}") {
        open(LOG, $logfile) or die "error: can't open $logfile for reading: $!";

        $res = $time = $mem = "na";
        $no_nodes_syntaxtree_before_simplification = "na";
        $no_nodes_syntaxtree_after_simplification = "na";
        $no_clauses_input_problem_before_simplification = "na";
        $no_clauses_input_problem_after_simplification = "na";

        while (!($res eq "to" || $res eq "mo" || $res eq "er" || $res eq "si") && ($line = <LOG>)) {
            my $tmp;

            # sat/unsat results
            $tmp = parse_sat_unsat($line, \$res);

            # time/memory/errors
            unless ($tmp) { $tmp = parse_time_memory_errors($line, \$res, \$time, \$mem); }

            # simplification
            unless ($tmp) { $tmp = parse_simplification($line, \$res, \$time, \$mem); }

            # statistics
            unless ($tmp) {
                $tmp = parse_statistics_all($line,
                                            \$no_nodes_syntaxtree_before_simplification,
                                            \$no_nodes_syntaxtree_after_simplification,
                                            \$no_clauses_input_problem_before_simplification,
                                            \$no_clauses_input_problem_after_simplification);
            }
        }

        close(LOG);
    } else {
        $res = $time = $mem = "---";
        $no_nodes_syntaxtree_before_simplification = "---";
        $no_nodes_syntaxtree_after_simplification = "---";
        $no_clauses_input_problem_before_simplification = "---";
        $no_clauses_input_problem_after_simplification = "---";
    }

    return ($time,
            $mem,
            $res,
            $no_nodes_syntaxtree_before_simplification,
            $no_nodes_syntaxtree_after_simplification,
            $no_clauses_input_problem_before_simplification,
            $no_clauses_input_problem_after_simplification);
}

#------------------------------------------------------------------------------#
# evaluate_simple_proof
#
sub evaluate_simple_proof($$$) {
    my $example = shift;
    my $switches = shift;
    my $sharing = shift;
    my $logfile = "${example}.proof${switches}.${sharing}.log";
    my $line;
    my $time;
    my $mem;
    my $res;
    my $no_nodes_syntaxtree_before_simplification;
    my $no_nodes_syntaxtree_after_simplification;
    my $no_clauses_input_problem_before_simplification;
    my $no_clauses_input_problem_after_simplification;
    my $no_nodes_syntaxtree_proof_unsat_core;
    my $no_clauses_proof_unsat_core;
    my $no_vertices_peak;
    my $no_edges_peak;
    my $no_vertices_plus_edges_peak;

    if (-f "${logfile}") {
        open(LOG, $logfile) or die "error: can't open $logfile for reading: $!";

        $res = $time = $mem = "na";
        $no_nodes_syntaxtree_before_simplification = "na";
        $no_nodes_syntaxtree_after_simplification = "na";
        $no_clauses_input_problem_before_simplification = "na";
        $no_clauses_input_problem_after_simplification = "na";
        $no_nodes_syntaxtree_proof_unsat_core = "na";
        $no_clauses_proof_unsat_core = "na";
        $no_vertices_peak = "na";
        $no_edges_peak = "na";
        $no_vertices_plus_edges_peak = "na";

        while (!($res eq "to" || $res eq "mo" || $res eq "er" || $res eq "si") && ($line = <LOG>)) {
            my $tmp;

            # sat/unsat results
            $tmp = parse_sat_unsat($line, \$res);

            # time/memory/errors
            unless ($tmp) { $tmp = parse_time_memory_errors($line, \$res, \$time, \$mem); }

            # simplification
            unless ($tmp) { $tmp = parse_simplification($line, \$res, \$time, \$mem); }

            # statistics
            unless ($tmp) {
                $tmp = parse_statistics_all($line,
                                            \$no_nodes_syntaxtree_before_simplification,
                                            \$no_nodes_syntaxtree_after_simplification,
                                            \$no_clauses_input_problem_before_simplification,
                                            \$no_clauses_input_problem_after_simplification);
            }
            unless ($tmp) {
                $tmp = parse_statistics_simple_proof($line,
                                                     \$no_nodes_syntaxtree_proof_unsat_core,
                                                     \$no_clauses_proof_unsat_core,
                                                     \$no_vertices_peak,
                                                     \$no_edges_peak,
                                                     \$no_vertices_plus_edges_peak);
            }
        }

        close(LOG);
    } else {
        $res = $time = $mem = "---";
        $no_nodes_syntaxtree_before_simplification = "---";
        $no_nodes_syntaxtree_after_simplification = "---";
        $no_clauses_input_problem_before_simplification = "---";
        $no_clauses_input_problem_after_simplification = "---";
        $no_nodes_syntaxtree_proof_unsat_core = "---";
        $no_clauses_proof_unsat_core = "---";
        $no_vertices_peak = "---";
        $no_edges_peak = "---";
        $no_vertices_plus_edges_peak = "---";
    }

    return ($time,
            $mem,
            $res,
            $no_nodes_syntaxtree_before_simplification,
            $no_nodes_syntaxtree_after_simplification,
            $no_clauses_input_problem_before_simplification,
            $no_clauses_input_problem_after_simplification,
            $no_nodes_syntaxtree_proof_unsat_core,
            $no_clauses_proof_unsat_core,
            $no_vertices_peak,
            $no_edges_peak,
            $no_vertices_plus_edges_peak);
}

#------------------------------------------------------------------------------#
# evaluate_simple_proofdeletion
#
sub evaluate_simple_proofdeletion($$) {
    my $example = shift;
    my $sharing = shift;
    my $logfile = "${example}.proofdeletion.${sharing}.log";
    my $line;
    my $time;
    my $mem;
    my $res;
    my $no_nodes_syntaxtree_before_simplification;
    my $no_nodes_syntaxtree_after_simplification;
    my $no_clauses_input_problem_before_simplification;
    my $no_clauses_input_problem_after_simplification;
    my $no_nodes_syntaxtree_proof_unsat_core;
    my $no_clauses_proof_unsat_core;
    my $no_vertices_peak;
    my $no_edges_peak;
    my $no_vertices_plus_edges_peak;
    my $no_nodes_syntaxtree_proofdeletion_unsat_core;
    my $no_deletion_its;
    my $no_deletion_its_unsat;
    my $no_deletion_its_sat;

    if (-f "${logfile}") {
        open(LOG, $logfile) or die "error: can't open $logfile for reading: $!";

        $res = $time = $mem = "na";
        $no_nodes_syntaxtree_before_simplification = "na";
        $no_nodes_syntaxtree_after_simplification = "na";
        $no_clauses_input_problem_before_simplification = "na";
        $no_clauses_input_problem_after_simplification = "na";
        $no_nodes_syntaxtree_proof_unsat_core = "na";
        $no_clauses_proof_unsat_core = "na";
        $no_vertices_peak = "na";
        $no_edges_peak = "na";
        $no_vertices_plus_edges_peak = "na";
        $no_nodes_syntaxtree_proofdeletion_unsat_core = "na";
        $no_deletion_its = "na";
        $no_deletion_its_unsat = "na";
        $no_deletion_its_sat = "na";

        while (!($res eq "to" || $res eq "mo" || $res eq "er" || $res eq "si") && ($line = <LOG>)) {
            my $tmp;

            # sat/unsat results
            $tmp = parse_sat_unsat($line, \$res);

            # time/memory/errors
            unless ($tmp) { $tmp = parse_time_memory_errors($line, \$res, \$time, \$mem); }

            # simplification
            unless ($tmp) { $tmp = parse_simplification($line, \$res, \$time, \$mem); }

            # statistics
            unless ($tmp) {
                $tmp = parse_statistics_all($line,
                                            \$no_nodes_syntaxtree_before_simplification,
                                            \$no_nodes_syntaxtree_after_simplification,
                                            \$no_clauses_input_problem_before_simplification,
                                            \$no_clauses_input_problem_after_simplification);
            }
            unless ($tmp) {
                $tmp = parse_statistics_simple_proof($line,
                                                     \$no_nodes_syntaxtree_proof_unsat_core,
                                                     \$no_clauses_proof_unsat_core,
                                                     \$no_vertices_peak,
                                                     \$no_edges_peak,
                                                     \$no_vertices_plus_edges_peak);
            }
            unless ($tmp) {
                $tmp = parse_statistics_simple_deletion_proofdeletion($line,
                                                                      \$no_nodes_syntaxtree_proofdeletion_unsat_core,
                                                                      \$no_deletion_its,
                                                                      \$no_deletion_its_unsat,
                                                                      \$no_deletion_its_sat,
                                                                      "proofdeletion");
            }
        }

        close(LOG);
    } else {
        $res = $time = $mem = "---";
        $no_nodes_syntaxtree_before_simplification = "---";
        $no_nodes_syntaxtree_after_simplification = "---";
        $no_clauses_input_problem_before_simplification = "---";
        $no_clauses_input_problem_after_simplification = "---";
        $no_nodes_syntaxtree_proof_unsat_core = "---";
        $no_clauses_proof_unsat_core = "---";
        $no_vertices_peak = "---";
        $no_edges_peak = "---";
        $no_vertices_plus_edges_peak = "---";
        $no_nodes_syntaxtree_proofdeletion_unsat_core = "---";
        $no_deletion_its = "---";
        $no_deletion_its_unsat = "---";
        $no_deletion_its_sat = "---";
    }

    return ($time,
            $mem,
            $res,
            $no_nodes_syntaxtree_before_simplification,
            $no_nodes_syntaxtree_after_simplification,
            $no_clauses_input_problem_before_simplification,
            $no_clauses_input_problem_after_simplification,
            $no_nodes_syntaxtree_proof_unsat_core,
            $no_clauses_proof_unsat_core,
            $no_vertices_peak,
            $no_edges_peak,
            $no_vertices_plus_edges_peak,
            $no_nodes_syntaxtree_proofdeletion_unsat_core,
            $no_deletion_its,
            $no_deletion_its_unsat,
            $no_deletion_its_sat);
}

#------------------------------------------------------------------------------#
# evaluate_simple_deletion
#
sub evaluate_simple_deletion($$) {
    my $example = shift;
    my $sharing = shift;
    my $logfile = "${example}.deletion.${sharing}.log";
    my $line;
    my $time;
    my $mem;
    my $res;
    my $no_nodes_syntaxtree_before_simplification;
    my $no_nodes_syntaxtree_after_simplification;
    my $no_clauses_input_problem_before_simplification;
    my $no_clauses_input_problem_after_simplification;
    my $no_nodes_syntaxtree_deletion_unsat_core;
    my $no_deletion_its;
    my $no_deletion_its_unsat;
    my $no_deletion_its_sat;

    if (-f "${logfile}") {
        open(LOG, $logfile) or die "error: can't open $logfile for reading: $!";

        $res = $time = $mem = "na";
        $no_nodes_syntaxtree_before_simplification = "na";
        $no_nodes_syntaxtree_after_simplification = "na";
        $no_clauses_input_problem_before_simplification = "na";
        $no_clauses_input_problem_after_simplification = "na";
        $no_nodes_syntaxtree_deletion_unsat_core = "na";
        $no_deletion_its = "na";
        $no_deletion_its_unsat = "na";
        $no_deletion_its_sat = "na";

        while (!($res eq "to" || $res eq "mo" || $res eq "er" || $res eq "si") && ($line = <LOG>)) {
            my $tmp;

            # sat/unsat results
            $tmp = parse_sat_unsat($line, \$res);

            # time/memory/errors
            unless ($tmp) { $tmp = parse_time_memory_errors($line, \$res, \$time, \$mem); }

            # simplification
            unless ($tmp) { $tmp = parse_simplification($line, \$res, \$time, \$mem); }

            # statistics
            unless ($tmp) {
                $tmp = parse_statistics_all($line,
                                            \$no_nodes_syntaxtree_before_simplification,
                                            \$no_nodes_syntaxtree_after_simplification,
                                            \$no_clauses_input_problem_before_simplification,
                                            \$no_clauses_input_problem_after_simplification);
            }
            unless ($tmp) {
                $tmp = parse_statistics_simple_deletion_proofdeletion($line,
                                                                      \$no_nodes_syntaxtree_deletion_unsat_core,
                                                                      \$no_deletion_its,
                                                                      \$no_deletion_its_unsat,
                                                                      \$no_deletion_its_sat,
                                                                      "deletion");
            }
        }

        close(LOG);
    } else {
        $res = $time = $mem = "---";
        $no_nodes_syntaxtree_before_simplification = "---";
        $no_nodes_syntaxtree_after_simplification = "---";
        $no_clauses_input_problem_before_simplification = "---";
        $no_clauses_input_problem_after_simplification = "---";
        $no_nodes_syntaxtree_deletion_unsat_core = "---";
        $no_deletion_its = "---";
        $no_deletion_its_unsat = "---";
        $no_deletion_its_sat = "---";
    }

    return ($time,
            $mem,
            $res,
            $no_nodes_syntaxtree_before_simplification,
            $no_nodes_syntaxtree_after_simplification,
            $no_clauses_input_problem_before_simplification,
            $no_clauses_input_problem_after_simplification,
            $no_nodes_syntaxtree_deletion_unsat_core,
            $no_deletion_its,
            $no_deletion_its_unsat,
            $no_deletion_its_sat);
}

#------------------------------------------------------------------------------#
# evaluate_partitioned_proof
#
sub evaluate_partitioned_proof($$) {
    my $example = shift;
    my $sharing = shift;
    my $logfile = "${example}.proof_partitionedpropositions.${sharing}.log";
    my $line;
    my $time;
    my $mem;
    my $res;
    my $no_nodes_syntaxtree_before_simplification;
    my $no_nodes_syntaxtree_after_simplification;
    my $no_clauses_input_problem_before_simplification;
    my $no_clauses_input_problem_after_simplification;
    my $no_nodes_syntaxtree_proof_unsat_core;
    my $no_clauses_proof_unsat_core;
    my $no_vertices_peak;
    my $no_edges_peak;
    my $no_vertices_plus_edges_peak;
    my $no_propositions_ltl_core;
    my $no_proposition_occurrences_ltl_core;
    my $no_partitions_per_proposition_peak_ltl_core;
    my $no_propositions_with_2_or_more_partitions_ltl_core;

    if (-f "${logfile}") {
        open(LOG, $logfile) or die "error: can't open $logfile for reading: $!";

        $res = $time = $mem = "na";
        $no_nodes_syntaxtree_before_simplification = "na";
        $no_nodes_syntaxtree_after_simplification = "na";
        $no_clauses_input_problem_before_simplification = "na";
        $no_clauses_input_problem_after_simplification = "na";
        $no_nodes_syntaxtree_proof_unsat_core = "na";
        $no_clauses_proof_unsat_core = "na";
        $no_vertices_peak = "na";
        $no_edges_peak = "na";
        $no_vertices_plus_edges_peak = "na";
        $no_propositions_ltl_core = "na";
        $no_proposition_occurrences_ltl_core = "na";
        $no_partitions_per_proposition_peak_ltl_core = "na";
        $no_propositions_with_2_or_more_partitions_ltl_core = "na";

        while (!($res eq "to" || $res eq "mo" || $res eq "er" || $res eq "si") && ($line = <LOG>)) {
            my $tmp;

            # sat/unsat results
            $tmp = parse_sat_unsat($line, \$res);

            # time/memory/errors
            unless ($tmp) { $tmp = parse_time_memory_errors($line, \$res, \$time, \$mem); }

            # simplification
            unless ($tmp) { $tmp = parse_simplification($line, \$res, \$time, \$mem); }

            # statistics
            unless ($tmp) {
                $tmp = parse_statistics_all($line,
                                            \$no_nodes_syntaxtree_before_simplification,
                                            \$no_nodes_syntaxtree_after_simplification,
                                            \$no_clauses_input_problem_before_simplification,
                                            \$no_clauses_input_problem_after_simplification);
            }
            unless ($tmp) {
                $tmp = parse_statistics_simple_proof($line,
                                                     \$no_nodes_syntaxtree_proof_unsat_core,
                                                     \$no_clauses_proof_unsat_core,
                                                     \$no_vertices_peak,
                                                     \$no_edges_peak,
                                                     \$no_vertices_plus_edges_peak);
            }
            unless ($tmp) {
                $tmp = parse_statistics_partitioned_proof($line,
                                                          \$no_propositions_ltl_core,
                                                          \$no_proposition_occurrences_ltl_core,
                                                          \$no_partitions_per_proposition_peak_ltl_core,
                                                          \$no_propositions_with_2_or_more_partitions_ltl_core);
            }
        }

        close(LOG);
    } else {
        $res = $time = $mem = "---";
        $no_nodes_syntaxtree_before_simplification = "---";
        $no_nodes_syntaxtree_after_simplification = "---";
        $no_clauses_input_problem_before_simplification = "---";
        $no_clauses_input_problem_after_simplification = "---";
        $no_nodes_syntaxtree_proof_unsat_core = "---";
        $no_clauses_proof_unsat_core = "---";
        $no_vertices_peak = "---";
        $no_edges_peak = "---";
        $no_vertices_plus_edges_peak = "---";
        $no_propositions_ltl_core = "---";
        $no_proposition_occurrences_ltl_core = "---";
        $no_partitions_per_proposition_peak_ltl_core = "---";
        $no_propositions_with_2_or_more_partitions_ltl_core = "---";
    }

    return ($time,
            $mem,
            $res,
            $no_nodes_syntaxtree_before_simplification,
            $no_nodes_syntaxtree_after_simplification,
            $no_clauses_input_problem_before_simplification,
            $no_clauses_input_problem_after_simplification,
            $no_nodes_syntaxtree_proof_unsat_core,
            $no_clauses_proof_unsat_core,
            $no_vertices_peak,
            $no_edges_peak,
            $no_vertices_plus_edges_peak,
            $no_propositions_ltl_core,
            $no_proposition_occurrences_ltl_core,
            $no_partitions_per_proposition_peak_ltl_core,
            $no_propositions_with_2_or_more_partitions_ltl_core);
}

#------------------------------------------------------------------------------#
# evaluate_pltlmup
#
sub evaluate_pltlmup($) {
    my $example = shift;
    my $logfile = "${example}.pltlmup.log";
    my $line;
    my $time;
    my $mem;
    my $res;
    my $no_conjuncts;
    my $no_conjuncts_core;

    if (-f "${logfile}") {
        open(LOG, $logfile) or die "error: can't open $logfile for reading: $!";

        $res = $time = $mem = "na";
        $no_conjuncts = "na";
        $no_conjuncts_core = "na";

        while (!($res eq "to" || $res eq "mo" || $res eq "er" || $res eq "si") && ($line = <LOG>)) {
            my $tmp;

            # sat/unsat results
            $tmp = parse_sat_unsat($line, \$res);

            # time/memory/errors
            unless ($tmp) { $tmp = parse_time_memory_errors($line, \$res, \$time, \$mem); }

            # statistics
            unless ($tmp) {
                $tmp = parse_statistics_pltlmup($line,
                                                \$no_conjuncts,
                                                \$no_conjuncts_core);
            }
        }

        close(LOG);
    } else {
        $res = $time = $mem = "---";
        $no_conjuncts = "---";
        $no_conjuncts_core = "---";
    }

    return ($time,
            $mem,
            $res,
            $no_conjuncts,
            $no_conjuncts_core);
}

#------------------------------------------------------------------------------#
# evaluate_procmine
#
sub evaluate_procmine($$) {
    my $example = shift;
    my $mus = shift;
    my $logfile;
    my $line;
    my $time;
    my $mem;
    my $res;
    my $no_conjuncts_core;

    if ($mus eq "yes") {
        $logfile = "${example}.procmine.log";
    } else {
        $logfile = "${example}.procminenomus.log";
    }
    if (-f "${logfile}") {
        open(LOG, $logfile) or die "error: can't open $logfile for reading: $!";

        $res = $time = $mem = "na";
        $no_conjuncts_core = "na";

        while (!($res eq "to" || $res eq "mo" || $res eq "er" || $res eq "si") && ($line = <LOG>)) {
            my $tmp;

            # sat/unsat results
            $tmp = parse_sat_unsat_procmine($line, \$res);

            # time/memory/errors
            unless ($tmp) { $tmp = parse_time_memory_errors($line, \$res, \$time, \$mem); }

            # statistics
            unless ($tmp) {
                if (($line =~ /^Root cause of inconsistency is:/) && ($line = <LOG>)) {
                    $tmp = parse_statistics_procmine($line,
                                                     \$no_conjuncts_core);
                }
            }
        }

        close(LOG);
    } else {
        $res = $time = $mem = "---";
        $no_conjuncts_core = "---";
    }

    return ($time,
            $mem,
            $res,
            $no_conjuncts_core);
}

#------------------------------------------------------------------------------#
# evaluate_all_exps
#
sub evaluate_all_exps {
    my $example;
    my $sharing;
    
    my $time;
    my $mem;
    my $res;
    my $no_nodes_syntaxtree_before_simplification;
    my $no_nodes_syntaxtree_after_simplification;
    my $no_clauses_input_problem_before_simplification;
    my $no_clauses_input_problem_after_simplification;
    my $no_nodes_syntaxtree_proof_unsat_core;
    my $no_clauses_proof_unsat_core;
    my $no_vertices_peak;
    my $no_edges_peak;
    my $no_vertices_plus_edges_peak;
    my $no_nodes_syntaxtree_proofdeletion_unsat_core;
    my $no_nodes_syntaxtree_deletion_unsat_core;
    my $no_deletion_its;
    my $no_deletion_its_unsat;
    my $no_deletion_its_sat;
    my $no_propositions_ltl_core;
    my $no_proposition_occurrences_ltl_core;
    my $no_partitions_per_proposition_peak_ltl_core;
    my $no_propositions_with_2_or_more_partitions_ltl_core;
    my $no_conjuncts;
    my $no_conjuncts_core;

    print_header();
    foreach $example (@examples) {
        print_line_header($example);
        ($no_nodes_syntaxtree_before_simplification
        ) = evaluate_sizenonnf($example);
        print_sizenonnf($no_nodes_syntaxtree_before_simplification);
        foreach $sharing (@sharings) {
            ($time,
             $mem,
             $res,
             $no_nodes_syntaxtree_before_simplification,
             $no_nodes_syntaxtree_after_simplification,
             $no_clauses_input_problem_before_simplification,
             $no_clauses_input_problem_after_simplification
            ) = evaluate_none($example, $sharing);
            print_none($time,
                       $mem,
                       $res,
                       $no_nodes_syntaxtree_before_simplification,
                       $no_nodes_syntaxtree_after_simplification,
                       $no_clauses_input_problem_before_simplification,
                       $no_clauses_input_problem_after_simplification);
        }
        foreach $sharing (@sharings) {
            ($time,
             $mem,
             $res,
             $no_nodes_syntaxtree_before_simplification,
             $no_nodes_syntaxtree_after_simplification,
             $no_clauses_input_problem_before_simplification,
             $no_clauses_input_problem_after_simplification,
             $no_nodes_syntaxtree_proof_unsat_core,
             $no_clauses_proof_unsat_core,
             $no_vertices_peak,
             $no_edges_peak,
             $no_vertices_plus_edges_peak
            ) = evaluate_simple_proof($example, "", $sharing);
            print_simple_proof($time,
                               $mem,
                               $res,
                               $no_nodes_syntaxtree_before_simplification,
                               $no_nodes_syntaxtree_after_simplification,
                               $no_clauses_input_problem_before_simplification,
                               $no_clauses_input_problem_after_simplification,
                               $no_nodes_syntaxtree_proof_unsat_core,
                               $no_clauses_proof_unsat_core,
                               $no_vertices_peak,
                               $no_edges_peak,
                               $no_vertices_plus_edges_peak);
        }
        foreach $sharing (@sharings) {
            ($time,
             $mem,
             $res,
             $no_nodes_syntaxtree_before_simplification,
             $no_nodes_syntaxtree_after_simplification,
             $no_clauses_input_problem_before_simplification,
             $no_clauses_input_problem_after_simplification,
             $no_nodes_syntaxtree_proof_unsat_core,
             $no_clauses_proof_unsat_core,
             $no_vertices_peak,
             $no_edges_peak,
             $no_vertices_plus_edges_peak,
             $no_nodes_syntaxtree_proofdeletion_unsat_core,
             $no_deletion_its,
             $no_deletion_its_unsat,
             $no_deletion_its_sat
            ) = evaluate_simple_proofdeletion($example, $sharing);
            print_simple_proofdeletion($time,
                                       $mem,
                                       $res,
                                       $no_nodes_syntaxtree_before_simplification,
                                       $no_nodes_syntaxtree_after_simplification,
                                       $no_clauses_input_problem_before_simplification,
                                       $no_clauses_input_problem_after_simplification,
                                       $no_nodes_syntaxtree_proof_unsat_core,
                                       $no_clauses_proof_unsat_core,
                                       $no_vertices_peak,
                                       $no_edges_peak,
                                       $no_vertices_plus_edges_peak,
                                       $no_nodes_syntaxtree_proofdeletion_unsat_core,
                                       $no_deletion_its,
                                       $no_deletion_its_unsat,
                                       $no_deletion_its_sat);
        }
        foreach $sharing (@sharings) {
            ($time,
             $mem,
             $res,
             $no_nodes_syntaxtree_before_simplification,
             $no_nodes_syntaxtree_after_simplification,
             $no_clauses_input_problem_before_simplification,
             $no_clauses_input_problem_after_simplification,
             $no_nodes_syntaxtree_deletion_unsat_core,
             $no_deletion_its,
             $no_deletion_its_unsat,
             $no_deletion_its_sat
            ) = evaluate_simple_deletion($example, $sharing);
            print_simple_deletion($time,
                                  $mem,
                                  $res,
                                  $no_nodes_syntaxtree_before_simplification,
                                  $no_nodes_syntaxtree_after_simplification,
                                  $no_clauses_input_problem_before_simplification,
                                  $no_clauses_input_problem_after_simplification,
                                  $no_nodes_syntaxtree_deletion_unsat_core,
                                  $no_deletion_its,
                                  $no_deletion_its_unsat,
                                  $no_deletion_its_sat);
        }
        foreach my $switches (@switchess) {
            foreach $sharing (@sharings) {
                ($time,
                 $mem,
                 $res,
                 $no_nodes_syntaxtree_before_simplification,
                 $no_nodes_syntaxtree_after_simplification,
                 $no_clauses_input_problem_before_simplification,
                 $no_clauses_input_problem_after_simplification,
                 $no_nodes_syntaxtree_proof_unsat_core,
                 $no_clauses_proof_unsat_core,
                 $no_vertices_peak,
                 $no_edges_peak,
                 $no_vertices_plus_edges_peak
                ) = evaluate_simple_proof($example, "_$switches", $sharing);
                print_simple_proof($time,
                                   $mem,
                                   $res,
                                   $no_nodes_syntaxtree_before_simplification,
                                   $no_nodes_syntaxtree_after_simplification,
                                   $no_clauses_input_problem_before_simplification,
                                   $no_clauses_input_problem_after_simplification,
                                   $no_nodes_syntaxtree_proof_unsat_core,
                                   $no_clauses_proof_unsat_core,
                                   $no_vertices_peak,
                                   $no_edges_peak,
                                   $no_vertices_plus_edges_peak);
            }
        }
        foreach $sharing (@sharings) {
            ($time,
             $mem,
             $res,
             $no_nodes_syntaxtree_before_simplification,
             $no_nodes_syntaxtree_after_simplification,
             $no_clauses_input_problem_before_simplification,
             $no_clauses_input_problem_after_simplification,
             $no_nodes_syntaxtree_proof_unsat_core,
             $no_clauses_proof_unsat_core,
             $no_vertices_peak,
             $no_edges_peak,
             $no_vertices_plus_edges_peak,
             $no_propositions_ltl_core,
             $no_proposition_occurrences_ltl_core,
             $no_partitions_per_proposition_peak_ltl_core,
             $no_propositions_with_2_or_more_partitions_ltl_core
            ) = evaluate_partitioned_proof($example, $sharing);
            print_partitioned_proof($time,
                                    $mem,
                                    $res,
                                    $no_nodes_syntaxtree_before_simplification,
                                    $no_nodes_syntaxtree_after_simplification,
                                    $no_clauses_input_problem_before_simplification,
                                    $no_clauses_input_problem_after_simplification,
                                    $no_nodes_syntaxtree_proof_unsat_core,
                                    $no_clauses_proof_unsat_core,
                                    $no_vertices_peak,
                                    $no_edges_peak,
                                    $no_vertices_plus_edges_peak,
                                    $no_propositions_ltl_core,
                                    $no_proposition_occurrences_ltl_core,
                                    $no_partitions_per_proposition_peak_ltl_core,
                                    $no_propositions_with_2_or_more_partitions_ltl_core);
        }
        ($time,
         $mem,
         $res,
         $no_conjuncts,
         $no_conjuncts_core
        ) = evaluate_pltlmup($example);
        print_pltlmup($time,
                      $mem,
                      $res,
                      $no_conjuncts,
                      $no_conjuncts_core);
        ($time,
         $mem,
         $res,
         $no_conjuncts_core
        ) = evaluate_procmine($example, "yes");
        print_procmine($time,
                      $mem,
                      $res,
                      $no_conjuncts_core);
        ($time,
         $mem,
         $res,
         $no_conjuncts_core
        ) = evaluate_procmine($example, "no");
        print_procmine($time,
                      $mem,
                      $res,
                      $no_conjuncts_core);

        print_line_footer();
    }
}

#------------------------------------------------------------------------------#
# main
#
{
    my $format = "";

#------------------------------------------------------------------------------#
# read parameters
    {
	my $argno = 0;

	while (defined $ARGV[$argno]) {
	    if ($ARGV[$argno] eq "-h" || $ARGV[$argno] eq "--help") {
		$argno++;
		usage;
            } elsif (($ARGV[$argno] eq "-txt") && ($format eq "")) {
                $format = "txt";
		$argno++;
            } elsif (($ARGV[$argno] eq "-csv") && ($format eq "")) {
                $format = "csv";
		$argno++;
	    } else {
		usage;
            }
	}
        if ($format eq "") {
            usage;
        }
    }
    
#------------------------------------------------------------------------------#
# do work
#
    {
        if ($format eq "txt") {
            ($line_header_fmt,
             $sizenonnf_fmt,
             $sizenonnf_tableheader_fmt,
             $none_fmt,
             $none_tableheader_fmt,
             $simple_proof_fmt,
             $simple_proof_tableheader_fmt,
             $simple_proofdeletion_fmt,
             $simple_proofdeletion_tableheader_fmt,
             $simple_deletion_fmt,
             $simple_deletion_tableheader_fmt,
             $partitioned_proof_fmt,
             $partitioned_proof_tableheader_fmt,
             $pltlmup_fmt,
             $pltlmup_tableheader_fmt,
             $procmine_fmt,
             $procmine_tableheader_fmt) =
                 ($txt_line_header_fmt,
                  $txt_sizenonnf_fmt,
                  $txt_sizenonnf_tableheader_fmt,
                  $txt_none_fmt,
                  $txt_none_tableheader_fmt,
                  $txt_simple_proof_fmt,
                  $txt_simple_proof_tableheader_fmt,
                  $txt_simple_proofdeletion_fmt,
                  $txt_simple_proofdeletion_tableheader_fmt,
                  $txt_simple_deletion_fmt,
                  $txt_simple_deletion_tableheader_fmt,
                  $txt_partitioned_proof_fmt,
                  $txt_partitioned_proof_tableheader_fmt,
                  $txt_pltlmup_fmt,
                  $txt_pltlmup_tableheader_fmt,
                  $txt_procmine_fmt,
                  $txt_procmine_tableheader_fmt);
        } elsif ($format eq "csv") {
            ($line_header_fmt,
             $sizenonnf_fmt,
             $sizenonnf_tableheader_fmt,
             $none_fmt,
             $none_tableheader_fmt,
             $simple_proof_fmt,
             $simple_proof_tableheader_fmt,
             $simple_proofdeletion_fmt,
             $simple_proofdeletion_tableheader_fmt,
             $simple_deletion_fmt,
             $simple_deletion_tableheader_fmt,
             $partitioned_proof_fmt,
             $partitioned_proof_tableheader_fmt,
             $pltlmup_fmt,
             $pltlmup_tableheader_fmt,
             $procmine_fmt,
             $procmine_tableheader_fmt) =
                 ($csv_line_header_fmt,
                  $csv_sizenonnf_fmt,
                  $csv_sizenonnf_tableheader_fmt,
                  $csv_none_fmt,
                  $csv_none_tableheader_fmt,
                  $csv_simple_proof_fmt,
                  $csv_simple_proof_tableheader_fmt,
                  $csv_simple_proofdeletion_fmt,
                  $csv_simple_proofdeletion_tableheader_fmt,
                  $csv_simple_deletion_fmt,
                  $csv_simple_deletion_tableheader_fmt,
                  $csv_partitioned_proof_fmt,
                  $csv_partitioned_proof_tableheader_fmt,
                  $csv_pltlmup_fmt,
                  $csv_pltlmup_tableheader_fmt,
                  $csv_procmine_fmt,
                  $csv_procmine_tableheader_fmt);
        }

        evaluate_all_exps();
    }

#------------------------------------------------------------------------------#
# clean up and exit
#
    exit 0;
}
