#!/usr/bin/perl

#------------------------------------------------------------------------------#
# Script for extracting data from benchmark logs into table
#
# Original by V. Schuppan
#------------------------------------------------------------------------------#

# Include program location in module search path. From perlfaq8.
BEGIN {
    use Cwd qw(abs_path);
    use File::Basename qw(dirname);

    my $path   = abs_path( $0 );
    our $directory = dirname( $path );
}
use lib $directory;

use strict;
use warnings;

#------------------------------------------------------------------------------#
# parameters
#

# @examples
use Config_examples;

# Original delimiter style
# Beginning of each line: name of example
my $txt_line_header_fmt = "%-34s |";
# none experiment: time, memory, sat/unsat
my $txt_none_fmt = " %5s/%7s |";
# Column header for none
my $txt_none_tableheader_fmt = " %-13s |";
# simple experiment: time, memory, sat/unsat, etc
my $txt_simple_fmt = " %5s/%7s/%7s/%7s/%7s/%7s |";
# Column header for simple
my $txt_simple_tableheader_fmt = " %-45s |";
# positionsetsg experiment: time, memory, sat/unsat, etc
my $txt_positionsetsg_fmt = " %5s/%7s/%5s/%5s/%5s/%7s |";
# Column header for positionsetsg
my $txt_positionsetsg_tableheader_fmt = " %-39s |";
# positionsetss experiment: time, memory, sat/unsat, etc
my $txt_positionsetss_fmt = " %5s/%7s/%5s/%5s/%5s/%7s |";
# Column header for positionsetss
my $txt_positionsetss_tableheader_fmt = " %-39s |";

# Spreadsheet delimiter style
# Beginning of each line: name of example
my $csv_line_header_fmt = "%-34s ;";
# none experiment: time, memory, sat/unsat
my $csv_none_fmt = " %5s;%7s ;";
# Column header for none
my $csv_none_tableheader_fmt = " %-5s;        ;";
# simple experiment: time, memory, sat/unsat, etc
my $csv_simple_fmt = " %5s;%7s;%5s;%5s;%4s;%4s ;";
# Column header for simple
my $csv_simple_tableheader_fmt = " %-6s ;     ;     ;     ;    ;     ;";
# positionsetsg experiment: time, memory, sat/unsat, etc
my $csv_positionsetsg_fmt = " %5s;%7s;%5s;%5s;%5s;%5s ;";
# Column header for positionsets
my $csv_positionsetsg_tableheader_fmt = " %-13s; ;   ;     ;     ;      ;";
# positionsetss experiment: time, memory, sat/unsat, etc
my $csv_positionsetss_fmt = " %5s;%7s;%5s;%5s;%5s;%5s ;";
# Column header for positionsets
my $csv_positionsetss_tableheader_fmt = " %-13s; ;   ;     ;     ;      ;";

# End of each line.
my $line_footer_fmt = "\n";

#------------------------------------------------------------------------------#
# global variables
#

# These are set either to txt_... or to csv_...
my $line_header_fmt;
my $none_fmt;
my $none_tableheader_fmt;
my $simple_fmt;
my $simple_tableheader_fmt;
my $positionsetsg_fmt;
my $positionsetsg_tableheader_fmt;
my $positionsetss_fmt;
my $positionsetss_tableheader_fmt;

#------------------------------------------------------------------------------#
# usage
#
sub usage() {
    printf "usage: generatetab.pl [-h|--help] [-txt|-csv]\n";
    exit 1;
}

#------------------------------------------------------------------------------#
# print_header
#
# Prints the first line of the table.
sub print_header($) {
    printf $line_header_fmt, "example";
    printf $none_tableheader_fmt, "no uc";
    printf $simple_tableheader_fmt, "uc w/o sets of time points";
    printf $positionsetsg_tableheader_fmt, "uc w/ sets of time points Gawrychowski";
    printf $positionsetss_tableheader_fmt, "uc w/ sets of time points Sawa";
    printf "\n";
    printf $line_header_fmt, "";
    printf $none_fmt, "t", "m";
    printf $simple_fmt, "t", "m", "sti", "stc", "cli", "clc";
    printf $positionsetsg_fmt, "t", "m", "all", "nt", "dif", "bwr";
    printf $positionsetss_fmt, "t", "m", "all", "nt", "dif", "mbwr";
    printf "\n";
}

#------------------------------------------------------------------------------#
# print_line_header
#
# Prints the first part of a line.
sub print_line_header($) {
    my $example = shift;

    printf $line_header_fmt, $example;
}

#------------------------------------------------------------------------------#
# print_line_footer
#
# Prints the last part of a line.
sub print_line_footer() {
    printf $line_footer_fmt;
}

#------------------------------------------------------------------------------#
# print_none
#
sub print_none($$$) {
    my $time = shift;
    my $mem = shift;
    my $res = shift;

    printf $none_fmt,
    $time,
    $mem;
}

#------------------------------------------------------------------------------#
# print_simple
#
sub print_simple($$$$$$$) {
    my $time = shift;
    my $mem = shift;
    my $res = shift;
    my $no_nodes_syntaxtree_before_simplification = shift;
    my $no_nodes_syntaxtree_unsat_core = shift;
    my $no_clauses_input_problem = shift;
    my $no_clauses_unsat_core = shift;

    printf $simple_fmt,
    $time,
    $mem,
    $no_nodes_syntaxtree_before_simplification,
    $no_nodes_syntaxtree_unsat_core,
    $no_clauses_input_problem,
    $no_clauses_unsat_core;
}

#------------------------------------------------------------------------------#
# print_positionsetsg
#
sub print_positionsetsg($$$$$$$) {
    my $time = shift;
    my $mem = shift;
    my $res = shift;
    my $no_max_backward_reachable_vertices = shift;
    my $no_sls = shift;
    my $no_sls_different_from_trivial = shift;
    my $no_different_sls = shift;

    printf $positionsetsg_fmt,
    $time,
    $mem,
    $no_sls,
    $no_sls_different_from_trivial,
    $no_different_sls,
    $no_max_backward_reachable_vertices;
}

#------------------------------------------------------------------------------#
# print_positionsetss
#
sub print_positionsetss($$$$$$$) {
    my $time = shift;
    my $mem = shift;
    my $res = shift;
    my $no_max_backward_reachable_vertices = shift;
    my $no_sls = shift;
    my $no_sls_different_from_trivial = shift;
    my $no_different_sls = shift;

    printf $positionsetss_fmt,
    $time,
    $mem,
    $no_sls,
    $no_sls_different_from_trivial,
    $no_different_sls,
    $no_max_backward_reachable_vertices;
}

#------------------------------------------------------------------------------#
# evaluate_none
#
sub evaluate_none($$) {
    my $example = shift;
    my $nnf = shift;
    my $logfile = "${example}.none.${nnf}.log";
    my $line;
    my $time;
    my $mem;
    my $res;

    if (-f "${logfile}") {
        open(LOG, $logfile) or die "error: can't open $logfile for reading: $!";

        $res = $time = $mem = "na";

        while (!($res eq "to" || $res eq "mo" || $res eq "er" || $res eq "si") && ($line = <LOG>)) {
            # sat/unsat results
            if ($line =~ /^Satisfiable/) {
                $res = "s";
            } elsif ($line =~ /^Unsatisfiable/) {
                $res = "u";
            }

            # time/memory/errors
            elsif ($line =~ /^\[run\] time:\s+([0-9\.]+) seconds/) {
                my $time_unrounded = $1;
                $time = sprintf("%.1f", $time_unrounded);
            } elsif ($line =~ /^\[run\] space:\s+([\-0-9\.]+) MB/) {
                $mem = $1;
            } elsif ($line =~ /^\[run\] status:\s+out of time/) {
                $res = $time = $mem = "to";
            } elsif ($line =~ /^\[run\] status:\s+out of memory/) {
                $res = $time = $mem = "mo";
            } elsif ($line =~ /^\[run\] status:\s+segmentation fault/) {
                $res = $time = $mem = "er";
            } elsif ($line =~ /^\[run\] status:\s+signal\(6\)/) {
                $res = $time = $mem = "er";
            } elsif ($line =~ /^\[run\] status:\s+signal\(9\)/) {
                $res = $time = $mem = "to";
            } elsif ($line =~ /system: assertion/) {
                $res = $time = $mem = "er";
            } elsif ($line =~ /syntax error/) {
                $res = $time = $mem = "er";
            } elsif ($line =~ /Segmentation fault/) {
                $res = $time = $mem = "er";
            }

            # simplification
            elsif ($line =~ /Input formula is or has been simplified to False./) {
                $res = $time = $mem = "si";
            }
        }

        close(LOG);
    } else {
        $res = $time = $mem = "---";
    }

    return ($time,
            $mem,
            $res);
}

#------------------------------------------------------------------------------#
# evaluate_simple
#
sub evaluate_simple($$) {
    my $example = shift;
    my $nnf = shift;
    my $logfile = "${example}.simple.${nnf}.log";
    my $line;
    my $time;
    my $mem;
    my $res;
    my $no_nodes_syntaxtree_before_simplification;
    my $no_nodes_syntaxtree_unsat_core;
    my $no_clauses_input_problem;
    my $no_clauses_unsat_core;

    if (-f "${logfile}") {
        open(LOG, $logfile) or die "error: can't open $logfile for reading: $!";

        $res = $time = $mem = "na";
        $no_nodes_syntaxtree_before_simplification = "na";
        $no_nodes_syntaxtree_unsat_core = "na";
        $no_clauses_input_problem = "na";
        $no_clauses_unsat_core = "na";

        while (!($res eq "to" || $res eq "mo" || $res eq "er" || $res eq "si") && ($line = <LOG>)) {
            # sat/unsat results
            if ($line =~ /^Satisfiable/) {
                $res = "s";
            } elsif ($line =~ /^Unsatisfiable/) {
                $res = "u";
            }

            # time/memory/errors
            elsif ($line =~ /^\[run\] time:\s+([0-9\.]+) seconds/) {
                my $time_unrounded = $1;
                $time = sprintf("%.1f", $time_unrounded);
            } elsif ($line =~ /^\[run\] space:\s+([\-0-9\.]+) MB/) {
                $mem = $1;
            } elsif ($line =~ /^\[run\] status:\s+out of time/) {
                $res = $time = $mem = "to";
            } elsif ($line =~ /^\[run\] status:\s+out of memory/) {
                $res = $time = $mem = "mo";
            } elsif ($line =~ /^\[run\] status:\s+segmentation fault/) {
                $res = $time = $mem = "er";
            } elsif ($line =~ /^\[run\] status:\s+signal\(6\)/) {
                $res = $time = $mem = "er";
            } elsif ($line =~ /^\[run\] status:\s+signal\(9\)/) {
                $res = $time = $mem = "to";
            } elsif ($line =~ /system: assertion/) {
                $res = $time = $mem = "er";
            } elsif ($line =~ /syntax error/) {
                $res = $time = $mem = "er";
            } elsif ($line =~ /Segmentation fault/) {
                $res = $time = $mem = "er";
            }

            # simplification
            elsif ($line =~ /Input formula is or has been simplified to False./) {
                $res = $time = $mem = "si";
            }

            # statistics
            elsif ($line =~ /^TrpLtlParser: number of nodes in syntax tree before LTL simplification:\s+([0-9\.]+)/) {
                $no_nodes_syntaxtree_before_simplification = $1;
            } elsif ($line =~ /^ResolutionGraph: number of nodes in syntax tree of unsatisfiable core:\s+([0-9\.]+)/) {
                $no_nodes_syntaxtree_unsat_core = $1;
            } elsif ($line =~ /^ResolutionGraph: number of clauses in input problem:\s+([0-9\.]+)/) {
                $no_clauses_input_problem = $1;
            } elsif ($line =~ /^ResolutionGraph: number of clauses in unsatisfiable core:\s+([0-9\.]+)/) {
                $no_clauses_unsat_core = $1;
            }

        }

        close(LOG);
    } else {
        $res = $time = $mem = "---";
        $no_nodes_syntaxtree_before_simplification = "---";
        $no_nodes_syntaxtree_unsat_core = "---";
        $no_clauses_input_problem = "---";
        $no_clauses_unsat_core = "---";
    }

    return ($time,
            $mem,
            $res,
            $no_nodes_syntaxtree_before_simplification,
            $no_nodes_syntaxtree_unsat_core,
            $no_clauses_input_problem,
            $no_clauses_unsat_core);
}

#------------------------------------------------------------------------------#
# evaluate_positionsetsg
#
sub evaluate_positionsetsg($$) {
    my $example = shift;
    my $nnf = shift;
    my $logfile = "${example}.positionsetsg.${nnf}.log";
    my $corefile = "${example}.trp.positionsetsg.${nnf}.ltl";
    my $line;
    my $time;
    my $mem;
    my $res;
    my $no_backward_reachable_vertices;
    my $no_sls;
    my $no_sls_different_from_trivial;
    my $no_different_sls;

    if (-f "${logfile}") {
        open(LOG, $logfile) or die "error: can't open $logfile for reading: $!";

        $res = $time = $mem = "na";
        $no_backward_reachable_vertices = "na";
        $no_sls = "na";
        $no_sls_different_from_trivial = "na";
        $no_different_sls = "na";

        while (!($res eq "to" || $res eq "mo" || $res eq "er" || $res eq "si") && ($line = <LOG>)) {
            # sat/unsat results
            if ($line =~ /^Satisfiable/) {
                $res = "s";
            } elsif ($line =~ /^Unsatisfiable/) {
                $res = "u";
            }

            # time/memory/errors
            elsif ($line =~ /^\[run\] time:\s+([0-9\.]+) seconds/) {
                my $time_unrounded = $1;
                $time = sprintf("%.1f", $time_unrounded);
            } elsif ($line =~ /^\[run\] space:\s+([\-0-9\.]+) MB/) {
                $mem = $1;
            } elsif ($line =~ /^\[run\] status:\s+out of time/) {
                $res = $time = $mem = "to";
            } elsif ($line =~ /^\[run\] status:\s+out of memory/) {
                $res = $time = $mem = "mo";
            } elsif ($line =~ /^\[run\] status:\s+segmentation fault/) {
                $res = $time = $mem = "er";
            } elsif ($line =~ /^\[run\] status:\s+signal\(6\)/) {
                $res = $time = $mem = "er";
            } elsif ($line =~ /^\[run\] status:\s+signal\(9\)/) {
                $res = $time = $mem = "to";
            } elsif ($line =~ /system: assertion/) {
                $res = $time = $mem = "er";
            } elsif ($line =~ /syntax error/) {
                $res = $time = $mem = "er";
            } elsif ($line =~ /Segmentation fault/) {
                $res = $time = $mem = "er";
            }

            # simplification
            elsif ($line =~ /Input formula is or has been simplified to False./) {
                $res = $time = $mem = "si";
            }

            # statistics
            elsif ($line =~ /^ResolutionGraph: number of vertices after pruning to reachable vertices:\s+([0-9\.]+)/) {
                $no_backward_reachable_vertices = $1;
            }
        }

        close(LOG);

        if (-f "${corefile}") {
            $no_sls =
                `cat ${corefile} |sed -n -e "s/[^{}]*\\({[()0-9N, ]*}\\)[^{}]*/\\1\\n/pg"|grep -v "^ *\$"|sort|wc -l|tr -d "\\n"`;
            $no_sls_different_from_trivial =
                `cat ${corefile} |sed -n -e "s/[^{}]*\\({[()0-9N, ]*}\\)[^{}]*/\\1\\n/pg"|grep -v "^ *\$"|sort|grep -v "{([01] N, 0)}"|wc -l|tr -d "\\n"`;
            $no_different_sls =
                `cat ${corefile} |sed -n -e "s/[^{}]*\\({[()0-9N, ]*}\\)[^{}]*/\\1\\n/pg"|grep -v "^ *\$"|sort|uniq|wc -l|tr -d "\\n"`;
        }
    } else {
        $res = $time = $mem = "---";
        $no_backward_reachable_vertices = "---";
        $no_sls = "---";
        $no_sls_different_from_trivial = "---";
        $no_different_sls = "---";
    }

    return ($time,
            $mem,
            $res,
            $no_backward_reachable_vertices,
            $no_sls,
            $no_sls_different_from_trivial,
            $no_different_sls);
}

#------------------------------------------------------------------------------#
# evaluate_positionsetss
#
sub evaluate_positionsetss($$) {
    my $example = shift;
    my $nnf = shift;
    my $logfile = "${example}.positionsetss.${nnf}.log";
    my $corefile = "${example}.trp.positionsetss.${nnf}.ltl";
    my $line;
    my $time;
    my $mem;
    my $res;
    my $no_max_backward_reachable_vertices;
    my $no_sls;
    my $no_sls_different_from_trivial;
    my $no_different_sls;

    if (-f "${logfile}") {
        open(LOG, $logfile) or die "error: can't open $logfile for reading: $!";

        $res = $time = $mem = "na";
        $no_max_backward_reachable_vertices = "na";
        $no_sls = "na";
        $no_sls_different_from_trivial = "na";
        $no_different_sls = "na";

        while (!($res eq "to" || $res eq "mo" || $res eq "er" || $res eq "si") && ($line = <LOG>)) {
            # sat/unsat results
            if ($line =~ /^Satisfiable/) {
                $res = "s";
            } elsif ($line =~ /^Unsatisfiable/) {
                $res = "u";
            }

            # time/memory/errors
            elsif ($line =~ /^\[run\] time:\s+([0-9\.]+) seconds/) {
                my $time_unrounded = $1;
                $time = sprintf("%.1f", $time_unrounded);
            } elsif ($line =~ /^\[run\] space:\s+([\-0-9\.]+) MB/) {
                $mem = $1;
            } elsif ($line =~ /^\[run\] status:\s+out of time/) {
                $res = $time = $mem = "to";
            } elsif ($line =~ /^\[run\] status:\s+out of memory/) {
                $res = $time = $mem = "mo";
            } elsif ($line =~ /^\[run\] status:\s+segmentation fault/) {
                $res = $time = $mem = "er";
            } elsif ($line =~ /^\[run\] status:\s+signal\(6\)/) {
                $res = $time = $mem = "er";
            } elsif ($line =~ /^\[run\] status:\s+signal\(9\)/) {
                $res = $time = $mem = "to";
            } elsif ($line =~ /system: assertion/) {
                $res = $time = $mem = "er";
            } elsif ($line =~ /syntax error/) {
                $res = $time = $mem = "er";
            } elsif ($line =~ /Segmentation fault/) {
                $res = $time = $mem = "er";
            }

            # simplification
            elsif ($line =~ /Input formula is or has been simplified to False./) {
                $res = $time = $mem = "si";
            }

            # statistics
            elsif ($line =~ /^ResolutionGraph: maximal number of backward reachable vertices:\s+([0-9\.]+)/) {
                $no_max_backward_reachable_vertices = $1;
            }
        }

        close(LOG);

        if (-f "${corefile}") {
            $no_sls =
                `cat ${corefile} |sed -n -e "s/[^{}]*\\({[()0-9N, ]*}\\)[^{}]*/\\1\\n/pg"|grep -v "^ *\$"|sort|wc -l|tr -d "\\n"`;
            $no_sls_different_from_trivial =
                `cat ${corefile} |sed -n -e "s/[^{}]*\\({[()0-9N, ]*}\\)[^{}]*/\\1\\n/pg"|grep -v "^ *\$"|sort|grep -v "{([01] N, 0)}"|wc -l|tr -d "\\n"`;
            $no_different_sls =
                `cat ${corefile} |sed -n -e "s/[^{}]*\\({[()0-9N, ]*}\\)[^{}]*/\\1\\n/pg"|grep -v "^ *\$"|sort|uniq|wc -l|tr -d "\\n"`;
        }
    } else {
        $res = $time = $mem = "---";
        $no_max_backward_reachable_vertices = "---";
        $no_sls = "---";
        $no_sls_different_from_trivial = "---";
        $no_different_sls = "---";
    }

    return ($time,
            $mem,
            $res,
            $no_max_backward_reachable_vertices,
            $no_sls,
            $no_sls_different_from_trivial,
            $no_different_sls);
}

#------------------------------------------------------------------------------#
# evaluate_all_exps
#
sub evaluate_all_exps {
    my $example;
    my $nnf;
    
    print_header("f");
    foreach $example (@examples) {
        if ($example =~ /^genbuf_c_l[1-3]_[0-9]+$/) {
            $nnf = "nnf";
        } else {
            $nnf = "nonnf";
        }
        print_line_header($example);
        {
            my $time;
            my $mem;
            my $res;
            
            ($time,
             $mem,
             $res)
                = evaluate_none($example, $nnf);
            print_none($time,
                       $mem,
                       $res);
        }
        {
            my $time;
            my $mem;
            my $res;
            my $no_nodes_syntaxtree_before_simplification;
            my $no_nodes_syntaxtree_unsat_core;
            my $no_clauses_input_problem;
            my $no_clauses_unsat_core;
            
            ($time,
             $mem,
             $res,
             $no_nodes_syntaxtree_before_simplification,
             $no_nodes_syntaxtree_unsat_core,
             $no_clauses_input_problem,
             $no_clauses_unsat_core)
                = evaluate_simple($example, $nnf);
            print_simple($time,
                         $mem,
                         $res,
                         $no_nodes_syntaxtree_before_simplification,
                         $no_nodes_syntaxtree_unsat_core,
                         $no_clauses_input_problem,
                         $no_clauses_unsat_core);
        }
        {
            my $time;
            my $mem;
            my $res;
            my $no_max_backward_reachable_vertices;
            my $no_sls;
            my $no_sls_different_from_trivial;
            my $no_different_sls;
            
            ($time,
             $mem,
             $res,
             $no_max_backward_reachable_vertices,
             $no_sls,
             $no_sls_different_from_trivial,
             $no_different_sls)
                = evaluate_positionsetsg($example, $nnf);
            print_positionsetsg($time,
                                $mem,
                                $res,
                                $no_max_backward_reachable_vertices,
                                $no_sls,
                                $no_sls_different_from_trivial,
                                $no_different_sls);
        }
        {
            my $time;
            my $mem;
            my $res;
            my $no_max_backward_reachable_vertices;
            my $no_sls;
            my $no_sls_different_from_trivial;
            my $no_different_sls;
            
            ($time,
             $mem,
             $res,
             $no_max_backward_reachable_vertices,
             $no_sls,
             $no_sls_different_from_trivial,
             $no_different_sls)
                = evaluate_positionsetss($example, $nnf);
            print_positionsetss($time,
                                $mem,
                                $res,
                                $no_max_backward_reachable_vertices,
                                $no_sls,
                                $no_sls_different_from_trivial,
                                $no_different_sls);
        }
        print_line_footer();
    }
}

#------------------------------------------------------------------------------#
# main
#
{
    my $format = "";

#------------------------------------------------------------------------------#
# read parameters
    {
	my $argno = 0;

	while (defined $ARGV[$argno]) {
	    if ($ARGV[$argno] eq "-h" || $ARGV[$argno] eq "--help") {
		$argno++;
		usage;
            } elsif (($ARGV[$argno] eq "-txt") && ($format eq "")) {
                $format = "txt";
		$argno++;
            } elsif (($ARGV[$argno] eq "-csv") && ($format eq "")) {
                $format = "csv";
		$argno++;
	    } else {
		usage;
            }
	}
        if ($format eq "") {
            usage;
        }
    }
    
#------------------------------------------------------------------------------#
# do work
#
    {
        if ($format eq "txt") {
            ($line_header_fmt,
             $none_fmt,
             $none_tableheader_fmt,
             $simple_fmt,
             $simple_tableheader_fmt,
             $positionsetsg_fmt,
             $positionsetsg_tableheader_fmt,
             $positionsetss_fmt,
             $positionsetss_tableheader_fmt) =
                 ($txt_line_header_fmt,
                  $txt_none_fmt,
                  $txt_none_tableheader_fmt,
                  $txt_simple_fmt,
                  $txt_simple_tableheader_fmt,
                  $txt_positionsetsg_fmt,
                  $txt_positionsetsg_tableheader_fmt,
                  $txt_positionsetss_fmt,
                  $txt_positionsetss_tableheader_fmt);
        } elsif ($format eq "csv") {
            ($line_header_fmt,
             $none_fmt,
             $none_tableheader_fmt,
             $simple_fmt,
             $simple_tableheader_fmt,
             $positionsetsg_fmt,
             $positionsetsg_tableheader_fmt,
             $positionsetss_fmt,
             $positionsetss_tableheader_fmt) =
                ($csv_line_header_fmt,
                 $csv_none_fmt,
                 $csv_none_tableheader_fmt,
                 $csv_simple_fmt,
                 $csv_simple_tableheader_fmt,
                 $csv_positionsetss_fmt,
                 $csv_positionsetss_tableheader_fmt,
                 $csv_positionsetss_fmt,
                 $csv_positionsetss_tableheader_fmt);
        }

        evaluate_all_exps();
    }

#------------------------------------------------------------------------------#
# clean up and exit
#
    exit 0;
}
