#!/usr/bin/perl

#------------------------------------------------------------------------------#
# Script for extracting data from benchmark logs into table
#
# Original by V. Schuppan
#------------------------------------------------------------------------------#

# Include program location in module search path. From perlfaq8.
BEGIN {
    use Cwd qw(abs_path);
    use File::Basename qw(dirname);

    my $path   = abs_path( $0 );
    our $directory = dirname( $path );
}
use lib $directory;

use strict;
use warnings;

#------------------------------------------------------------------------------#
# parameters
#

# @examples
use Config_examples;

# @switchess
my @switchess = (
    "all",
    "noprunemain",
    "nopruneloopits",
    "aug2",
    "bfsloopitinitcg",
    "bfsloopitinitce",
    "bfsloopconclusion2e"
    );

# Original delimiter style
# Beginning of each line: name of example
my $txt_line_header_fmt = "%-34s |";
# sizenonnf experiment: size input
my $txt_sizenonnf_fmt = " %5s |";
# Column header for sizenonnf
my $txt_sizenonnf_tableheader_fmt = " %-5s |";
# none experiment: time, memory, size input
my $txt_none_fmt = " %5s/%7s/%5s/%5s/%5s |";
# Column header for none
my $txt_none_tableheader_fmt = " %-31s |";
# simple_proof experiment: time, memory, size core, size rg
my $txt_simple_proof_fmt = " %5s/%7s/%4s/%4s |";
# Column header for simple_proof
my $txt_simple_proof_tableheader_fmt = " %-23s |";
# simple_proofdeletion experiment: time, memory, size core, its
my $txt_simple_proofdeletion_fmt = " %5s/%7s/%4s/%4s/%4s/%3s/%3s/%3s |";
# Column header for simple_proofdeletion
my $txt_simple_proofdeletion_tableheader_fmt = " %-40s |";
# setsoftimepointsg_proof experiment:
my $txt_setsoftimepointsg_proof_fmt = " %5s/%7s/%5s/%5s/%5s/%7s |";
# Column header for setsoftimepointsg_proof
my $txt_setsoftimepointsg_proof_tableheader_fmt = " %-39s |";
# setsoftimepointss_proof experiment:
my $txt_setsoftimepointss_proof_fmt = " %5s/%7s/%5s/%5s/%5s/%7s |";
# Column header for setsoftimepointss_proof
my $txt_setsoftimepointss_proof_tableheader_fmt = " %-39s |";

# Spreadsheet delimiter style
# Beginning of each line: name of example
my $csv_line_header_fmt = "%-34s ;";
# sizenonnf experiment: size input
my $csv_sizenonnf_fmt = " %5s ;";
# Column header for sizenonnf
my $csv_sizenonnf_tableheader_fmt = " %-5s ;";
# none experiment: time, memory, size input
my $csv_none_fmt = " %5s;%7s;%5s;%5s;%5s ;";
# Column header for none
my $csv_none_tableheader_fmt = " %-27s ;;;;;";
# simple_proof experiment: time, memory, sat/unsat, etc
my $csv_simple_proof_fmt = " %5s;%7s;%4s;%4s ;";
# Column header for simple_proof
my $csv_simple_proof_tableheader_fmt = " %-20s ;;;;";
# simple_proofdeletion experiment:
my $csv_simple_proofdeletion_fmt = "%5s;%7s;%4s;%4s;%4s;%3s;%3s;%3s; ;";
# Column header for simple_proofdeletion
my $csv_simple_proofdeletion_tableheader_fmt = " %-32s ;;;;;;;;;";
# setsoftimepointsg_proof experiment:
my $csv_setsoftimepointsg_proof_fmt = " %5s;%7s;%5s;%5s;%5s;%5s ;";
# Column header for setsoftimepointsg_proof
my $csv_setsoftimepointsg_proof_tableheader_fmt = " %-32s ;;;;;;";
# setsoftimepointss_proof experiment:
my $csv_setsoftimepointss_proof_fmt = " %5s;%7s;%5s;%5s;%5s;%5s ;";
# Column header for setsoftimepointss_proof
my $csv_setsoftimepointss_proof_tableheader_fmt = " %-32s ;;;;;;";

# End of each line.
my $line_footer_fmt = "\n";

#------------------------------------------------------------------------------#
# global variables
#

# These are set either to txt_... or to csv_...
my $line_header_fmt;
my $sizenonnf_fmt;
my $sizenonnf_tableheader_fmt;
my $none_fmt;
my $none_tableheader_fmt;
my $simple_proof_fmt;
my $simple_proof_tableheader_fmt;
my $simple_proofdeletion_fmt;
my $simple_proofdeletion_tableheader_fmt;
my $setsoftimepointsg_proof_fmt;
my $setsoftimepointsg_proof_tableheader_fmt;
my $setsoftimepointss_proof_fmt;
my $setsoftimepointss_proof_tableheader_fmt;

#------------------------------------------------------------------------------#
# usage
#
sub usage() {
    printf "usage: generatetab.pl [-h|--help] [-txt|-csv]\n";
    exit 1;
}

#------------------------------------------------------------------------------#
# print_header
#
# Prints the first line of the table.
sub print_header() {
    printf $line_header_fmt, "example";
    printf $sizenonnf_tableheader_fmt, "";
    printf $none_tableheader_fmt, "no uc";
    printf $simple_proof_tableheader_fmt, "uc w/o s.o.t.p.";
    printf $setsoftimepointsg_proof_tableheader_fmt, "uc w/ s.o.t.p. Gawrychowski";
    printf $setsoftimepointss_proof_tableheader_fmt, "uc w/ s.o.t.p. Sawa";
    printf "\n";
    printf $line_header_fmt, "";
    printf $sizenonnf_fmt, "sti";
    printf $none_fmt, "t", "m", "sts", "cli", "cls";
    printf $simple_proof_fmt, "t", "m", "stc", "clc";
    printf $setsoftimepointsg_proof_fmt, "t", "m", "all", "nt", "dif", "bwr";
    printf $setsoftimepointss_proof_fmt, "t", "m", "all", "nt", "dif", "mbwr";
    printf "\n";
}

#------------------------------------------------------------------------------#
# print_line_header
#
# Prints the first part of a line.
sub print_line_header($) {
    my $example = shift;

    printf $line_header_fmt, $example;
}

#------------------------------------------------------------------------------#
# print_line_footer
#
# Prints the last part of a line.
sub print_line_footer() {
    printf $line_footer_fmt;
}

#------------------------------------------------------------------------------#
# print_sizenonnf
#
sub print_sizenonnf($) {
    my $no_nodes_syntaxtree_before_simplification = shift;

    printf $sizenonnf_fmt,
    $no_nodes_syntaxtree_before_simplification;
}

#------------------------------------------------------------------------------#
# print_none
#
sub print_none($$$$$$$) {
    my $time = shift;
    my $mem = shift;
    my $res = shift;
    my $no_nodes_syntaxtree_before_simplification = shift;
    my $no_nodes_syntaxtree_after_simplification = shift;
    my $no_clauses_input_problem_before_simplification = shift;
    my $no_clauses_input_problem_after_simplification = shift;

    printf $none_fmt,
    $time,
    $mem,
    $no_nodes_syntaxtree_after_simplification,
    $no_clauses_input_problem_before_simplification,
    $no_clauses_input_problem_after_simplification;
}

#------------------------------------------------------------------------------#
# print_simple_proof
#
sub print_simple_proof($$$$$$$$$$$$) {
    my $time = shift;
    my $mem = shift;
    my $res = shift;
    my $no_nodes_syntaxtree_before_simplification = shift;
    my $no_nodes_syntaxtree_after_simplification = shift;
    my $no_clauses_input_problem_before_simplification = shift;
    my $no_clauses_input_problem_after_simplificatio = shift;
    my $no_nodes_syntaxtree_proof_unsat_core = shift;
    my $no_clauses_proof_unsat_core = shift;
    my $no_vertices_peak = shift;
    my $no_edges_peak = shift;
    my $no_vertices_plus_edges_peak = shift;

    printf $simple_proof_fmt,
    $time,
    $mem,
    $no_nodes_syntaxtree_proof_unsat_core,
    $no_clauses_proof_unsat_core;
}

#------------------------------------------------------------------------------#
# print_simple_proofdeletion
#
sub print_simple_proofdeletion($$$$$$$$$$$$$$$$) {
    my $time = shift;
    my $mem = shift;
    my $res = shift;
    my $no_nodes_syntaxtree_before_simplification = shift;
    my $no_nodes_syntaxtree_after_simplification = shift;
    my $no_clauses_input_problem_before_simplification = shift;
    my $no_clauses_input_problem_after_simplificatio = shift;
    my $no_nodes_syntaxtree_proof_unsat_core = shift;
    my $no_clauses_proof_unsat_core = shift;
    my $no_vertices_peak = shift;
    my $no_edges_peak = shift;
    my $no_vertices_plus_edges_peak = shift;
    my $no_nodes_syntaxtree_proofdeletion_unsat_core = shift;
    my $no_deletion_its = shift;
    my $no_deletion_its_unsat = shift;
    my $no_deletion_its_sat = shift;

    printf $simple_proofdeletion_fmt,
    $time,
    $mem,
    $no_nodes_syntaxtree_proof_unsat_core,
    $no_clauses_proof_unsat_core,
    $no_nodes_syntaxtree_proofdeletion_unsat_core,
    $no_deletion_its,
    $no_deletion_its_unsat,
    $no_deletion_its_sat;
}

#------------------------------------------------------------------------------#
# print_setsoftimepointsg_proof
#
sub print_setsoftimepointsg_proof($$$$$$$$$$$$$$$$) {
    my $time = shift;
    my $mem = shift;
    my $res = shift;
    my $no_nodes_syntaxtree_before_simplification = shift;
    my $no_nodes_syntaxtree_after_simplification = shift;
    my $no_clauses_input_problem_before_simplification = shift;
    my $no_clauses_input_problem_after_simplificatio = shift;
    my $no_nodes_syntaxtree_proof_unsat_core = shift;
    my $no_clauses_proof_unsat_core = shift;
    my $no_vertices_peak = shift;
    my $no_edges_peak = shift;
    my $no_vertices_plus_edges_peak = shift;
    my $no_vertices_max_backward_reachable = shift;
    my $no_sls = shift;
    my $no_sls_different_from_trivial = shift;
    my $no_sls_different = shift;

    printf $setsoftimepointsg_proof_fmt,
    $time,
    $mem,
    $no_sls,
    $no_sls_different_from_trivial,
    $no_sls_different,
    $no_vertices_max_backward_reachable;
}

#------------------------------------------------------------------------------#
# print_setsoftimepointss_proof
#
sub print_setsoftimepointss_proof($$$$$$$$$$$$$$$$) {
    my $time = shift;
    my $mem = shift;
    my $res = shift;
    my $no_nodes_syntaxtree_before_simplification = shift;
    my $no_nodes_syntaxtree_after_simplification = shift;
    my $no_clauses_input_problem_before_simplification = shift;
    my $no_clauses_input_problem_after_simplificatio = shift;
    my $no_nodes_syntaxtree_proof_unsat_core = shift;
    my $no_clauses_proof_unsat_core = shift;
    my $no_vertices_peak = shift;
    my $no_edges_peak = shift;
    my $no_vertices_plus_edges_peak = shift;
    my $no_vertices_max_backward_reachable = shift;
    my $no_sls = shift;
    my $no_sls_different_from_trivial = shift;
    my $no_sls_different = shift;

    printf $setsoftimepointss_proof_fmt,
    $time,
    $mem,
    $no_sls,
    $no_sls_different_from_trivial,
    $no_sls_different,
    $no_vertices_max_backward_reachable;
}

#------------------------------------------------------------------------------#
# parse_sat_unsat
#
sub parse_sat_unsat($$) {
    my $line = shift;
    my $rres = shift;

    if ($line =~ /^Satisfiable/) {
        $$rres = "s";
        return 1;
    } elsif ($line =~ /^Unsatisfiable/) {
        $$rres = "u";
        return 1;
    }
    return 0;
}

#------------------------------------------------------------------------------#
# parse_time_memory_errors
#
sub parse_time_memory_errors($$$$) {
    my $line = shift;
    my $rres = shift;
    my $rtime = shift;
    my $rmem = shift;

    if ($line =~ /^\[run\] time:\s+([0-9\.]+) seconds/) {
        my $time_unrounded = $1;
        $$rtime = sprintf("%.1f", $time_unrounded);
        return 1;
    } elsif ($line =~ /^\[run\] space:\s+([\-0-9\.]+) MB/) {
        $$rmem = $1;
        return 1;
    } elsif ($line =~ /^\[run\] status:\s+out of time/) {
        $$rres = $$rtime = $$rmem = "to";
        return 1;
    } elsif ($line =~ /^\[run\] status:\s+out of memory/) {
        $$rres = $$rtime = $$rmem = "mo";
        return 1;
    } elsif ($line =~ /^\[run\] status:\s+segmentation fault/) {
        $$rres = $$rtime = $$rmem = "er";
        return 1;
    } elsif ($line =~ /^\[run\] status:\s+signal\(6\)/) {
        $$rres = $$rtime = $$rmem = "er";
        return 1;
    } elsif ($line =~ /^\[run\] status:\s+signal\(9\)/) {
        $$rres = $$rtime = $$rmem = "to";
        return 1;
    } elsif ($line =~ /system: assertion/) {
        $$rres = $$rtime = $$rmem = "er";
        return 1;
    } elsif ($line =~ /syntax error/) {
        $$rres = $$rtime = $$rmem = "er";
        return 1;
    } elsif ($line =~ /Segmentation fault/) {
        $$rres = $$rtime = $$rmem = "er";
        return 1;
    }
    return 0;
}

#------------------------------------------------------------------------------#
# parse_simplification
#
sub parse_simplification($$$$) {
    my $line = shift;
    my $rres = shift;
    my $rtime = shift;
    my $rmem = shift;

    if ($line =~ /Input formula is or has been simplified to False./) {
        $$rres = $$rtime = $$rmem = "si";
        return 1;
    }
    return 0;
}

#------------------------------------------------------------------------------#
# parse_statistics_all
#
sub parse_statistics_all($$$$$) {
    my $line = shift;
    my $rno_nodes_syntaxtree_before_simplification = shift;
    my $rno_nodes_syntaxtree_after_simplification = shift;
    my $rno_clauses_input_problem_before_simplification = shift;
    my $rno_clauses_input_problem_after_simplification = shift;

    if ($line =~ /^MainAux: number of nodes in syntax tree before LTL simplification:\s+([0-9\.]+)/) {
        $$rno_nodes_syntaxtree_before_simplification = $1;
        return 1;
    } elsif ($line =~ /^MainAux: number of nodes in syntax tree after LTL simplification:\s+([0-9\.]+)/) {
        $$rno_nodes_syntaxtree_after_simplification = $1;
        return 1;
    } elsif ($line =~ /^BuilderLtl: number of clauses before pure literal simplification:\s+([0-9\.]+)/) {
        $$rno_clauses_input_problem_before_simplification = $1;
        return 1;
    } elsif ($line =~ /^BuilderLtl: number of clauses after pure literal simplification:\s+([0-9\.]+)/) {
        $$rno_clauses_input_problem_after_simplification = $1;
        return 1;
    }
    return 0;
}

#------------------------------------------------------------------------------#
# parse_statistics_simple_proof
#
sub parse_statistics_simple_proof($$$$$$) {
    my $line = shift;
    my $rno_nodes_syntaxtree_proof_unsat_core = shift;
    my $rno_clauses_proof_unsat_core = shift;
    my $rno_vertices_peak = shift;
    my $rno_edges_peak = shift;
    my $rno_vertices_plus_edges_peak = shift;

    if ($line =~ /^MainAux: number of nodes in syntax tree of proof core:\s+([0-9\.]+)/) {
        $$rno_nodes_syntaxtree_proof_unsat_core = $1;
        return 1;
    } elsif ($line =~ /^ResolutionGraph: number of clauses in unsatisfiable core:\s+([0-9\.]+)/) {
        $$rno_clauses_proof_unsat_core = $1;
        return 1;
    } elsif ($line =~ /esolutionGraph: peak number of vertices:\s+([0-9\.]+)/) {
        $$rno_vertices_peak = $1;
        return 1;
    } elsif ($line =~ /esolutionGraph: peak number of edges:\s+([0-9\.]+)/) {
        $$rno_edges_peak = $1;
        return 1;
    } elsif ($line =~ /esolutionGraph: peak number of vertices \+ edges:\s+([0-9\.]+)/) {
        $$rno_vertices_plus_edges_peak = $1;
        return 1;
    }
    return 0;
}

#------------------------------------------------------------------------------#
# parse_statistics_simple_proofdeletion
#
sub parse_statistics_simple_proofdeletion($$$$$) {
    my $line = shift;
    my $rno_nodes_syntaxtree_proofdeletion_unsat_core = shift;
    my $rno_deletion_its = shift;
    my $rno_deletion_its_unsat = shift;
    my $rno_deletion_its_sat = shift;

    if ($line =~ /^MainAux: number of nodes in syntax tree of proofdeletion core:\s+([0-9\.]+)/) {
        $$rno_nodes_syntaxtree_proofdeletion_unsat_core = $1;
        return 1;
    } elsif ($line =~ /^MainAux: number of iterations in unsat core algorithm deletion:\s+([0-9\.]+)/) {
        $$rno_deletion_its = $1;
        return 1;
    } elsif ($line =~ /^MainAux: number of iterations in unsat core algorithm deletion with unsat result:\s+([0-9\.]+)/) {
        $$rno_deletion_its_unsat = $1;
        return 1;
    } elsif ($line =~ /^MainAux: number of iterations in unsat core algorithm deletion with sat result:\s+([0-9\.]+)/) {
        $$rno_deletion_its_sat = $1;
        return 1;
    }
    return 0;
}

#------------------------------------------------------------------------------#
# parse_statistics_setsoftimepoints_proof
#
sub parse_statistics_setsoftimepoints_proof($$) {
    my $line = shift;
    my $rno_vertices_max_backward_reachable = shift;

    if ($line =~ /^ResolutionGraph: number of vertices after pruning to reachable vertices:\s+([0-9\.]+)/) {
        $$rno_vertices_max_backward_reachable = $1;
        return 1;
    }
    return 0;
}

#------------------------------------------------------------------------------#
# evaluate_sizenonnf
#
sub evaluate_sizenonnf($) {
    my $example = shift;
    my $logfile = "${example}.sizenonnf.log";
    my $line;
    my $no_nodes_syntaxtree_before_simplification;

    if (-f "${logfile}") {
        open(LOG, $logfile) or die "error: can't open $logfile for reading: $!";

        $no_nodes_syntaxtree_before_simplification = "na";

        while (($no_nodes_syntaxtree_before_simplification eq "na") && ($line = <LOG>)) {
            if ($line =~ /^BuilderLtl: number of nodes in syntax tree before LTL simplification:\s+([0-9\.]+)/) {
                $no_nodes_syntaxtree_before_simplification = $1;
            }
        }

        close(LOG);
    } else {
        $no_nodes_syntaxtree_before_simplification = "---";
    }

    return ($no_nodes_syntaxtree_before_simplification);
}

#------------------------------------------------------------------------------#
# evaluate_none
#
sub evaluate_none($) {
    my $example = shift;
    my $logfile = "${example}.none.log";
    my $line;
    my $time;
    my $mem;
    my $res;
    my $no_nodes_syntaxtree_before_simplification;
    my $no_nodes_syntaxtree_after_simplification;
    my $no_clauses_input_problem_before_simplification;
    my $no_clauses_input_problem_after_simplification;

    if (-f "${logfile}") {
        open(LOG, $logfile) or die "error: can't open $logfile for reading: $!";

        $res = $time = $mem = "na";
        $no_nodes_syntaxtree_before_simplification = "na";
        $no_nodes_syntaxtree_after_simplification = "na";
        $no_clauses_input_problem_before_simplification = "na";
        $no_clauses_input_problem_after_simplification = "na";

        while (!($res eq "to" || $res eq "mo" || $res eq "er" || $res eq "si") && ($line = <LOG>)) {
            my $tmp;

            # sat/unsat results
            $tmp = parse_sat_unsat($line, \$res);

            # time/memory/errors
            unless ($tmp) { $tmp = parse_time_memory_errors($line, \$res, \$time, \$mem); }

            # simplification
            unless ($tmp) { $tmp = parse_simplification($line, \$res, \$time, \$mem); }

            # statistics
            unless ($tmp) {
                $tmp = parse_statistics_all($line,
                                            \$no_nodes_syntaxtree_before_simplification,
                                            \$no_nodes_syntaxtree_after_simplification,
                                            \$no_clauses_input_problem_before_simplification,
                                            \$no_clauses_input_problem_after_simplification);
            }
        }

        close(LOG);
    } else {
        $res = $time = $mem = "---";
        $no_nodes_syntaxtree_before_simplification = "---";
        $no_nodes_syntaxtree_after_simplification = "---";
        $no_clauses_input_problem_before_simplification = "---";
        $no_clauses_input_problem_after_simplification = "---";
    }

    return ($time,
            $mem,
            $res,
            $no_nodes_syntaxtree_before_simplification,
            $no_nodes_syntaxtree_after_simplification,
            $no_clauses_input_problem_before_simplification,
            $no_clauses_input_problem_after_simplification);
}

#------------------------------------------------------------------------------#
# evaluate_simple_proof
#
sub evaluate_simple_proof($) {
    my $example = shift;
    my $logfile = "${example}.simple.log";
    my $line;
    my $time;
    my $mem;
    my $res;
    my $no_nodes_syntaxtree_before_simplification;
    my $no_nodes_syntaxtree_after_simplification;
    my $no_clauses_input_problem_before_simplification;
    my $no_clauses_input_problem_after_simplification;
    my $no_nodes_syntaxtree_proof_unsat_core;
    my $no_clauses_proof_unsat_core;
    my $no_vertices_peak;
    my $no_edges_peak;
    my $no_vertices_plus_edges_peak;

    if (-f "${logfile}") {
        open(LOG, $logfile) or die "error: can't open $logfile for reading: $!";

        $res = $time = $mem = "na";
        $no_nodes_syntaxtree_before_simplification = "na";
        $no_nodes_syntaxtree_after_simplification = "na";
        $no_clauses_input_problem_before_simplification = "na";
        $no_clauses_input_problem_after_simplification = "na";
        $no_nodes_syntaxtree_proof_unsat_core = "na";
        $no_clauses_proof_unsat_core = "na";
        $no_vertices_peak = "na";
        $no_edges_peak = "na";
        $no_vertices_plus_edges_peak = "na";

        while (!($res eq "to" || $res eq "mo" || $res eq "er" || $res eq "si") && ($line = <LOG>)) {
            my $tmp;

            # sat/unsat results
            $tmp = parse_sat_unsat($line, \$res);

            # time/memory/errors
            unless ($tmp) { $tmp = parse_time_memory_errors($line, \$res, \$time, \$mem); }

            # simplification
            unless ($tmp) { $tmp = parse_simplification($line, \$res, \$time, \$mem); }

            # statistics
            unless ($tmp) {
                $tmp = parse_statistics_all($line,
                                            \$no_nodes_syntaxtree_before_simplification,
                                            \$no_nodes_syntaxtree_after_simplification,
                                            \$no_clauses_input_problem_before_simplification,
                                            \$no_clauses_input_problem_after_simplification);
            }
            unless ($tmp) {
                $tmp = parse_statistics_simple_proof($line,
                                                     \$no_nodes_syntaxtree_proof_unsat_core,
                                                     \$no_clauses_proof_unsat_core,
                                                     \$no_vertices_peak,
                                                     \$no_edges_peak,
                                                     \$no_vertices_plus_edges_peak);
            }
        }

        close(LOG);
    } else {
        $res = $time = $mem = "---";
        $no_nodes_syntaxtree_before_simplification = "---";
        $no_nodes_syntaxtree_after_simplification = "---";
        $no_clauses_input_problem_before_simplification = "---";
        $no_clauses_input_problem_after_simplification = "---";
        $no_nodes_syntaxtree_proof_unsat_core = "---";
        $no_clauses_proof_unsat_core = "---";
        $no_vertices_peak = "---";
        $no_edges_peak = "---";
        $no_vertices_plus_edges_peak = "---";
    }

    return ($time,
            $mem,
            $res,
            $no_nodes_syntaxtree_before_simplification,
            $no_nodes_syntaxtree_after_simplification,
            $no_clauses_input_problem_before_simplification,
            $no_clauses_input_problem_after_simplification,
            $no_nodes_syntaxtree_proof_unsat_core,
            $no_clauses_proof_unsat_core,
            $no_vertices_peak,
            $no_edges_peak,
            $no_vertices_plus_edges_peak);
}

#------------------------------------------------------------------------------#
# evaluate_simple_proofdeletion
#
sub evaluate_simple_proofdeletion($) {
    my $example = shift;
    my $logfile = "${example}.proofdeletion.log";
    my $line;
    my $time;
    my $mem;
    my $res;
    my $no_nodes_syntaxtree_before_simplification;
    my $no_nodes_syntaxtree_after_simplification;
    my $no_clauses_input_problem_before_simplification;
    my $no_clauses_input_problem_after_simplification;
    my $no_nodes_syntaxtree_proof_unsat_core;
    my $no_clauses_proof_unsat_core;
    my $no_vertices_peak;
    my $no_edges_peak;
    my $no_vertices_plus_edges_peak;
    my $no_nodes_syntaxtree_proofdeletion_unsat_core;
    my $no_deletion_its;
    my $no_deletion_its_unsat;
    my $no_deletion_its_sat;

    if (-f "${logfile}") {
        open(LOG, $logfile) or die "error: can't open $logfile for reading: $!";

        $res = $time = $mem = "na";
        $no_nodes_syntaxtree_before_simplification = "na";
        $no_nodes_syntaxtree_after_simplification = "na";
        $no_clauses_input_problem_before_simplification = "na";
        $no_clauses_input_problem_after_simplification = "na";
        $no_nodes_syntaxtree_proof_unsat_core = "na";
        $no_clauses_proof_unsat_core = "na";
        $no_vertices_peak = "na";
        $no_edges_peak = "na";
        $no_vertices_plus_edges_peak = "na";
        $no_nodes_syntaxtree_proofdeletion_unsat_core = "na";
        $no_deletion_its = "na";
        $no_deletion_its_unsat = "na";
        $no_deletion_its_sat = "na";

        while (!($res eq "to" || $res eq "mo" || $res eq "er" || $res eq "si") && ($line = <LOG>)) {
            my $tmp;

            # sat/unsat results
            $tmp = parse_sat_unsat($line, \$res);

            # time/memory/errors
            unless ($tmp) { $tmp = parse_time_memory_errors($line, \$res, \$time, \$mem); }

            # simplification
            unless ($tmp) { $tmp = parse_simplification($line, \$res, \$time, \$mem); }

            # statistics
            unless ($tmp) {
                $tmp = parse_statistics_all($line,
                                            \$no_nodes_syntaxtree_before_simplification,
                                            \$no_nodes_syntaxtree_after_simplification,
                                            \$no_clauses_input_problem_before_simplification,
                                            \$no_clauses_input_problem_after_simplification);
            }
            unless ($tmp) {
                $tmp = parse_statistics_simple_proof($line,
                                                     \$no_nodes_syntaxtree_proof_unsat_core,
                                                     \$no_clauses_proof_unsat_core,
                                                     \$no_vertices_peak,
                                                     \$no_edges_peak,
                                                     \$no_vertices_plus_edges_peak);
            }
            unless ($tmp) {
                $tmp = parse_statistics_simple_proofdeletion($line,
                                                             \$no_nodes_syntaxtree_proofdeletion_unsat_core,
                                                             \$no_deletion_its,
                                                             \$no_deletion_its_unsat,
                                                             \$no_deletion_its_sat);
            }
        }

        close(LOG);
    } else {
        $res = $time = $mem = "---";
        $no_nodes_syntaxtree_before_simplification = "---";
        $no_nodes_syntaxtree_after_simplification = "---";
        $no_clauses_input_problem_before_simplification = "---";
        $no_clauses_input_problem_after_simplification = "---";
        $no_nodes_syntaxtree_proof_unsat_core = "---";
        $no_clauses_proof_unsat_core = "---";
        $no_vertices_peak = "---";
        $no_edges_peak = "---";
        $no_vertices_plus_edges_peak = "---";
        $no_nodes_syntaxtree_proofdeletion_unsat_core = "---";
        $no_deletion_its = "---";
        $no_deletion_its_unsat = "---";
        $no_deletion_its_sat = "---";
    }

    return ($time,
            $mem,
            $res,
            $no_nodes_syntaxtree_before_simplification,
            $no_nodes_syntaxtree_after_simplification,
            $no_clauses_input_problem_before_simplification,
            $no_clauses_input_problem_after_simplification,
            $no_nodes_syntaxtree_proof_unsat_core,
            $no_clauses_proof_unsat_core,
            $no_vertices_peak,
            $no_edges_peak,
            $no_vertices_plus_edges_peak,
            $no_nodes_syntaxtree_proofdeletion_unsat_core,
            $no_deletion_its,
            $no_deletion_its_unsat,
            $no_deletion_its_sat);
}

#------------------------------------------------------------------------------#
# evaluate_setsoftimepointsg_proof
#
sub evaluate_setsoftimepointsg_proof($) {
    my $example = shift;
    my $logfile = "${example}.setsoftimepointsg.log";
    my $corefile = "${example}.trp.setsoftimepointsg.ltl";
    my $line;
    my $time;
    my $mem;
    my $res;
    my $no_nodes_syntaxtree_before_simplification;
    my $no_nodes_syntaxtree_after_simplification;
    my $no_clauses_input_problem_before_simplification;
    my $no_clauses_input_problem_after_simplification;
    my $no_nodes_syntaxtree_proof_unsat_core;
    my $no_clauses_proof_unsat_core;
    my $no_vertices_peak;
    my $no_edges_peak;
    my $no_vertices_plus_edges_peak;
    my $no_vertices_max_backward_reachable;
    my $no_sls;
    my $no_sls_different_from_trivial;
    my $no_sls_different;

    if (-f "${logfile}") {
        open(LOG, $logfile) or die "error: can't open $logfile for reading: $!";

        $res = $time = $mem = "na";
        $no_nodes_syntaxtree_before_simplification = "na";
        $no_nodes_syntaxtree_after_simplification = "na";
        $no_clauses_input_problem_before_simplification = "na";
        $no_clauses_input_problem_after_simplification = "na";
        $no_nodes_syntaxtree_proof_unsat_core = "na";
        $no_clauses_proof_unsat_core = "na";
        $no_vertices_peak = "na";
        $no_edges_peak = "na";
        $no_vertices_plus_edges_peak = "na";
        $no_vertices_max_backward_reachable = "na";
        $no_sls = "na";
        $no_sls_different_from_trivial = "na";
        $no_sls_different = "na";

        while (!($res eq "to" || $res eq "mo" || $res eq "er" || $res eq "si") && ($line = <LOG>)) {
            my $tmp;

            # sat/unsat results
            $tmp = parse_sat_unsat($line, \$res);

            # time/memory/errors
            unless ($tmp) { $tmp = parse_time_memory_errors($line, \$res, \$time, \$mem); }

            # simplification
            unless ($tmp) { $tmp = parse_simplification($line, \$res, \$time, \$mem); }

            # statistics
            unless ($tmp) {
                $tmp = parse_statistics_all($line,
                                            \$no_nodes_syntaxtree_before_simplification,
                                            \$no_nodes_syntaxtree_after_simplification,
                                            \$no_clauses_input_problem_before_simplification,
                                            \$no_clauses_input_problem_after_simplification);
            }
            unless ($tmp) {
                $tmp = parse_statistics_simple_proof($line,
                                                     \$no_nodes_syntaxtree_proof_unsat_core,
                                                     \$no_clauses_proof_unsat_core,
                                                     \$no_vertices_peak,
                                                     \$no_edges_peak,
                                                     \$no_vertices_plus_edges_peak);
            }
            unless ($tmp) {
                $tmp = parse_statistics_setsoftimepoints_proof($line,
                                                               \$no_vertices_max_backward_reachable);
            }
        }

        close(LOG);

        if (-f "${corefile}") {
            $no_sls =
                `cat ${corefile} |sed -n -e "s/[^{}]*\\({[()0-9N, ]*}\\)[^{}]*/\\1\\n/pg"|grep -v "^ *\$"|sort|wc -l|tr -d "\\n"`;
            $no_sls_different_from_trivial =
                `cat ${corefile} |sed -n -e "s/[^{}]*\\({[()0-9N, ]*}\\)[^{}]*/\\1\\n/pg"|grep -v "^ *\$"|sort|grep -v "{([01] N, 0)}"|wc -l|tr -d "\\n"`;
            $no_sls_different =
                `cat ${corefile} |sed -n -e "s/[^{}]*\\({[()0-9N, ]*}\\)[^{}]*/\\1\\n/pg"|grep -v "^ *\$"|sort|uniq|wc -l|tr -d "\\n"`;
        }
    } else {
        $res = $time = $mem = "---";
        $no_nodes_syntaxtree_before_simplification = "---";
        $no_nodes_syntaxtree_after_simplification = "---";
        $no_clauses_input_problem_before_simplification = "---";
        $no_clauses_input_problem_after_simplification = "---";
        $no_nodes_syntaxtree_proof_unsat_core = "---";
        $no_clauses_proof_unsat_core = "---";
        $no_vertices_peak = "---";
        $no_edges_peak = "---";
        $no_vertices_plus_edges_peak = "---";
        $no_vertices_max_backward_reachable = "---";
        $no_sls = "---";
        $no_sls_different_from_trivial = "---";
        $no_sls_different = "---";
    }

    return ($time,
            $mem,
            $res,
            $no_nodes_syntaxtree_before_simplification,
            $no_nodes_syntaxtree_after_simplification,
            $no_clauses_input_problem_before_simplification,
            $no_clauses_input_problem_after_simplification,
            $no_nodes_syntaxtree_proof_unsat_core,
            $no_clauses_proof_unsat_core,
            $no_vertices_peak,
            $no_edges_peak,
            $no_vertices_plus_edges_peak,
            $no_vertices_max_backward_reachable,
            $no_sls,
            $no_sls_different_from_trivial,
            $no_sls_different);
}

#------------------------------------------------------------------------------#
# evaluate_setsoftimepointss_proof
#
sub evaluate_setsoftimepointss_proof($) {
    my $example = shift;
    my $logfile = "${example}.setsoftimepointss.log";
    my $corefile = "${example}.trp.setsoftimepointss.ltl";
    my $line;
    my $time;
    my $mem;
    my $res;
    my $no_nodes_syntaxtree_before_simplification;
    my $no_nodes_syntaxtree_after_simplification;
    my $no_clauses_input_problem_before_simplification;
    my $no_clauses_input_problem_after_simplification;
    my $no_nodes_syntaxtree_proof_unsat_core;
    my $no_clauses_proof_unsat_core;
    my $no_vertices_peak;
    my $no_edges_peak;
    my $no_vertices_plus_edges_peak;
    my $no_vertices_max_backward_reachable;
    my $no_sls;
    my $no_sls_different_from_trivial;
    my $no_sls_different;

    if (-f "${logfile}") {
        open(LOG, $logfile) or die "error: can't open $logfile for reading: $!";

        $res = $time = $mem = "na";
        $no_nodes_syntaxtree_before_simplification = "na";
        $no_nodes_syntaxtree_after_simplification = "na";
        $no_clauses_input_problem_before_simplification = "na";
        $no_clauses_input_problem_after_simplification = "na";
        $no_nodes_syntaxtree_proof_unsat_core = "na";
        $no_clauses_proof_unsat_core = "na";
        $no_vertices_peak = "na";
        $no_edges_peak = "na";
        $no_vertices_plus_edges_peak = "na";
        $no_vertices_max_backward_reachable = "na";
        $no_sls = "na";
        $no_sls_different_from_trivial = "na";
        $no_sls_different = "na";

        while (!($res eq "to" || $res eq "mo" || $res eq "er" || $res eq "si") && ($line = <LOG>)) {
            my $tmp;

            # sat/unsat results
            $tmp = parse_sat_unsat($line, \$res);

            # time/memory/errors
            unless ($tmp) { $tmp = parse_time_memory_errors($line, \$res, \$time, \$mem); }

            # simplification
            unless ($tmp) { $tmp = parse_simplification($line, \$res, \$time, \$mem); }

            # statistics
            unless ($tmp) {
                $tmp = parse_statistics_all($line,
                                            \$no_nodes_syntaxtree_before_simplification,
                                            \$no_nodes_syntaxtree_after_simplification,
                                            \$no_clauses_input_problem_before_simplification,
                                            \$no_clauses_input_problem_after_simplification);
            }
            unless ($tmp) {
                $tmp = parse_statistics_simple_proof($line,
                                                     \$no_nodes_syntaxtree_proof_unsat_core,
                                                     \$no_clauses_proof_unsat_core,
                                                     \$no_vertices_peak,
                                                     \$no_edges_peak,
                                                     \$no_vertices_plus_edges_peak);
            }
            unless ($tmp) {
                $tmp = parse_statistics_setsoftimepoints_proof($line,
                                                               \$no_vertices_max_backward_reachable);
            }
        }

        close(LOG);

        if (-f "${corefile}") {
            $no_sls =
                `cat ${corefile} |sed -n -e "s/[^{}]*\\({[()0-9N, ]*}\\)[^{}]*/\\1\\n/pg"|grep -v "^ *\$"|sort|wc -l|tr -d "\\n"`;
            $no_sls_different_from_trivial =
                `cat ${corefile} |sed -n -e "s/[^{}]*\\({[()0-9N, ]*}\\)[^{}]*/\\1\\n/pg"|grep -v "^ *\$"|sort|grep -v "{([01] N, 0)}"|wc -l|tr -d "\\n"`;
            $no_sls_different =
                `cat ${corefile} |sed -n -e "s/[^{}]*\\({[()0-9N, ]*}\\)[^{}]*/\\1\\n/pg"|grep -v "^ *\$"|sort|uniq|wc -l|tr -d "\\n"`;
        }
    } else {
        $res = $time = $mem = "---";
        $no_nodes_syntaxtree_before_simplification = "---";
        $no_nodes_syntaxtree_after_simplification = "---";
        $no_clauses_input_problem_before_simplification = "---";
        $no_clauses_input_problem_after_simplification = "---";
        $no_nodes_syntaxtree_proof_unsat_core = "---";
        $no_clauses_proof_unsat_core = "---";
        $no_vertices_peak = "---";
        $no_edges_peak = "---";
        $no_vertices_plus_edges_peak = "---";
        $no_vertices_max_backward_reachable = "---";
        $no_sls = "---";
        $no_sls_different_from_trivial = "---";
        $no_sls_different = "---";
    }

    return ($time,
            $mem,
            $res,
            $no_nodes_syntaxtree_before_simplification,
            $no_nodes_syntaxtree_after_simplification,
            $no_clauses_input_problem_before_simplification,
            $no_clauses_input_problem_after_simplification,
            $no_nodes_syntaxtree_proof_unsat_core,
            $no_clauses_proof_unsat_core,
            $no_vertices_peak,
            $no_edges_peak,
            $no_vertices_plus_edges_peak,
            $no_vertices_max_backward_reachable,
            $no_sls,
            $no_sls_different_from_trivial,
            $no_sls_different);
}

#------------------------------------------------------------------------------#
# evaluate_all_exps
#
sub evaluate_all_exps {
    my $example;
    
    my $time;
    my $mem;
    my $res;
    my $no_nodes_syntaxtree_before_simplification;
    my $no_nodes_syntaxtree_after_simplification;
    my $no_clauses_input_problem_before_simplification;
    my $no_clauses_input_problem_after_simplification;
    my $no_nodes_syntaxtree_proof_unsat_core;
    my $no_clauses_proof_unsat_core;
    my $no_vertices_peak;
    my $no_edges_peak;
    my $no_vertices_plus_edges_peak;
    my $no_nodes_syntaxtree_proofdeletion_unsat_core;
    my $no_deletion_its;
    my $no_deletion_its_unsat;
    my $no_deletion_its_sat;
    my $no_vertices_max_backward_reachable;
    my $no_sls;
    my $no_sls_different_from_trivial;
    my $no_sls_different;

    print_header();
    foreach $example (@examples) {
        print_line_header($example);
        ($no_nodes_syntaxtree_before_simplification
        ) = evaluate_sizenonnf($example);
        print_sizenonnf($no_nodes_syntaxtree_before_simplification);
        ($time,
         $mem,
         $res,
         $no_nodes_syntaxtree_before_simplification,
         $no_nodes_syntaxtree_after_simplification,
         $no_clauses_input_problem_before_simplification,
         $no_clauses_input_problem_after_simplification
        ) = evaluate_none($example);
        print_none($time,
                   $mem,
                   $res,
                   $no_nodes_syntaxtree_before_simplification,
                   $no_nodes_syntaxtree_after_simplification,
                   $no_clauses_input_problem_before_simplification,
                   $no_clauses_input_problem_after_simplification);
        ($time,
         $mem,
         $res,
         $no_nodes_syntaxtree_before_simplification,
         $no_nodes_syntaxtree_after_simplification,
         $no_clauses_input_problem_before_simplification,
         $no_clauses_input_problem_after_simplification,
         $no_nodes_syntaxtree_proof_unsat_core,
         $no_clauses_proof_unsat_core,
         $no_vertices_peak,
         $no_edges_peak,
         $no_vertices_plus_edges_peak
        ) = evaluate_simple_proof($example);
        print_simple_proof($time,
                           $mem,
                           $res,
                           $no_nodes_syntaxtree_before_simplification,
                           $no_nodes_syntaxtree_after_simplification,
                           $no_clauses_input_problem_before_simplification,
                           $no_clauses_input_problem_after_simplification,
                           $no_nodes_syntaxtree_proof_unsat_core,
                           $no_clauses_proof_unsat_core,
                           $no_vertices_peak,
                           $no_edges_peak,
                           $no_vertices_plus_edges_peak);
        ($time,
         $mem,
         $res,
         $no_nodes_syntaxtree_before_simplification,
         $no_nodes_syntaxtree_after_simplification,
         $no_clauses_input_problem_before_simplification,
         $no_clauses_input_problem_after_simplification,
         $no_nodes_syntaxtree_proof_unsat_core,
         $no_clauses_proof_unsat_core,
         $no_vertices_peak,
         $no_edges_peak,
         $no_vertices_plus_edges_peak,
         $no_vertices_max_backward_reachable,
         $no_sls,
         $no_sls_different_from_trivial,
         $no_sls_different
        ) = evaluate_setsoftimepointsg_proof($example);
        print_setsoftimepointsg_proof($time,
                           $mem,
                           $res,
                           $no_nodes_syntaxtree_before_simplification,
                           $no_nodes_syntaxtree_after_simplification,
                           $no_clauses_input_problem_before_simplification,
                           $no_clauses_input_problem_after_simplification,
                           $no_nodes_syntaxtree_proof_unsat_core,
                           $no_clauses_proof_unsat_core,
                           $no_vertices_peak,
                           $no_edges_peak,
                           $no_vertices_plus_edges_peak,
                           $no_vertices_max_backward_reachable,
                           $no_sls,
                           $no_sls_different_from_trivial,
                           $no_sls_different);
        ($time,
         $mem,
         $res,
         $no_nodes_syntaxtree_before_simplification,
         $no_nodes_syntaxtree_after_simplification,
         $no_clauses_input_problem_before_simplification,
         $no_clauses_input_problem_after_simplification,
         $no_nodes_syntaxtree_proof_unsat_core,
         $no_clauses_proof_unsat_core,
         $no_vertices_peak,
         $no_edges_peak,
         $no_vertices_plus_edges_peak,
         $no_vertices_max_backward_reachable,
         $no_sls,
         $no_sls_different_from_trivial,
         $no_sls_different
        ) = evaluate_setsoftimepointss_proof($example);
        print_setsoftimepointss_proof($time,
                           $mem,
                           $res,
                           $no_nodes_syntaxtree_before_simplification,
                           $no_nodes_syntaxtree_after_simplification,
                           $no_clauses_input_problem_before_simplification,
                           $no_clauses_input_problem_after_simplification,
                           $no_nodes_syntaxtree_proof_unsat_core,
                           $no_clauses_proof_unsat_core,
                           $no_vertices_peak,
                           $no_edges_peak,
                           $no_vertices_plus_edges_peak,
                           $no_vertices_max_backward_reachable,
                           $no_sls,
                           $no_sls_different_from_trivial,
                           $no_sls_different);
        print_line_footer();
    }
}

#------------------------------------------------------------------------------#
# main
#
{
    my $format = "";

#------------------------------------------------------------------------------#
# read parameters
    {
	my $argno = 0;

	while (defined $ARGV[$argno]) {
	    if ($ARGV[$argno] eq "-h" || $ARGV[$argno] eq "--help") {
		$argno++;
		usage;
            } elsif (($ARGV[$argno] eq "-txt") && ($format eq "")) {
                $format = "txt";
		$argno++;
            } elsif (($ARGV[$argno] eq "-csv") && ($format eq "")) {
                $format = "csv";
		$argno++;
	    } else {
		usage;
            }
	}
        if ($format eq "") {
            usage;
        }
    }
    
#------------------------------------------------------------------------------#
# do work
#
    {
        if ($format eq "txt") {
            ($line_header_fmt,
             $sizenonnf_fmt,
             $sizenonnf_tableheader_fmt,
             $none_fmt,
             $none_tableheader_fmt,
             $simple_proof_fmt,
             $simple_proof_tableheader_fmt,
             $simple_proofdeletion_fmt,
             $simple_proofdeletion_tableheader_fmt,
             $setsoftimepointsg_proof_fmt,
             $setsoftimepointsg_proof_tableheader_fmt,
             $setsoftimepointss_proof_fmt,
             $setsoftimepointss_proof_tableheader_fmt) =
                 ($txt_line_header_fmt,
                  $txt_sizenonnf_fmt,
                  $txt_sizenonnf_tableheader_fmt,
                  $txt_none_fmt,
                  $txt_none_tableheader_fmt,
                  $txt_simple_proof_fmt,
                  $txt_simple_proof_tableheader_fmt,
                  $txt_simple_proofdeletion_fmt,
                  $txt_simple_proofdeletion_tableheader_fmt,
                  $txt_setsoftimepointsg_proof_fmt,
                  $txt_setsoftimepointsg_proof_tableheader_fmt,
                  $txt_setsoftimepointss_proof_fmt,
                  $txt_setsoftimepointss_proof_tableheader_fmt);
        } elsif ($format eq "csv") {
            ($line_header_fmt,
             $sizenonnf_fmt,
             $sizenonnf_tableheader_fmt,
             $none_fmt,
             $none_tableheader_fmt,
             $simple_proof_fmt,
             $simple_proof_tableheader_fmt,
             $simple_proofdeletion_fmt,
             $simple_proofdeletion_tableheader_fmt,
             $setsoftimepointsg_proof_fmt,
             $setsoftimepointsg_proof_tableheader_fmt,
             $setsoftimepointss_proof_fmt,
             $setsoftimepointss_proof_tableheader_fmt) =
                 ($csv_line_header_fmt,
                  $csv_sizenonnf_fmt,
                  $csv_sizenonnf_tableheader_fmt,
                  $csv_none_fmt,
                  $csv_none_tableheader_fmt,
                  $csv_simple_proof_fmt,
                  $csv_simple_proof_tableheader_fmt,
                  $csv_simple_proofdeletion_fmt,
                  $csv_simple_proofdeletion_tableheader_fmt,
                  $csv_setsoftimepointsg_proof_fmt,
                  $csv_setsoftimepointsg_proof_tableheader_fmt,
                  $csv_setsoftimepointss_proof_fmt,
                  $csv_setsoftimepointss_proof_tableheader_fmt);
        }

        evaluate_all_exps();
    }

#------------------------------------------------------------------------------#
# clean up and exit
#
    exit 0;
}
