/************************************************************
 *    Copyright (C) 2008-2009                               *
 *    Michel Ludwig (michel.ludwig@liverpool.ac.uk)         *
 *    University of Liverpool                               *
 *                                                          *
 *    Copyright (C) 2012                                    *
 *    Viktor Schuppan (Viktor.Schuppan@gmx.de)              *
 *                                                          *
 *    This program is free software; you can redistribute   *
 *    it and/or modify it under the terms of the GNU        *
 *    General Public License as published by the Free       *
 *    Software Foundation; either version 3 of the License, *
 *    or (at your option) any later version.                *
 *                                                          *
 *    This program is distributed in the hope that it will  *
 *    be useful, but WITHOUT ANY WARRANTY; without even     *
 *    the implied warranty of MERCHANTABILITY or FITNESS    *
 *    FOR A PARTICULAR PURPOSE.  See the GNU General Public *
 *    License for more details.                             *
 *                                                          *
 *    You should have received a copy of the GNU General    *
 *    Public License along with this program; if not, see   *
 *    <http://www.gnu.org/licenses/>.                       *
 ************************************************************/

#include "formula.h"

#include <assert.h>

#include <map>

#include "misc.h"
#include "scannerparser/parser.h"

TreeNode::TreeNode(const TreeNodeList& theChildren)
: m_parent(NULL), m_children(theChildren)
{
}

TreeNode::TreeNode(const TreeNode& n)
: m_parent(NULL)
{
	TreeNodeList nChildren = n.m_children;
	for(TreeNodeList::const_iterator i = nChildren.begin(); i != nChildren.end(); ++i) {
		addChild((*i)->clone());
	}
}

TreeNode::~TreeNode()
{
	if(m_parent) {
		m_parent->removeChild(this);
	}
	TreeNodeList childrenList = m_children;
	for(TreeNodeList::iterator i = childrenList.begin(); i != childrenList.end(); ++i) {
		delete(*i);
	}
}

void TreeNode::addChild(TreeNode *n)
{
	assert(n);
	if(n->m_parent) {
		n->m_parent->removeChild(n);
	}
	m_children.push_back(n);
	n->m_parent = this;
}

const TreeNodeList& TreeNode::children() const
{
	return m_children;
}

void TreeNode::setChildren(const std::list<TreeNode*>& theChildren)
{
	m_children = theChildren;
	for(TreeNodeList::iterator i = m_children.begin(); i != m_children.end(); ++i) {
		assert(*i);
		(*i)->m_parent = this;
	}
}

void TreeNode::clearChildren()
{
	for(TreeNodeList::iterator i = m_children.begin(); i != m_children.end(); ++i) {
		(*i)->m_parent = NULL;
	}
	m_children.clear();
}

unsigned int TreeNode::childrenCount() const
{
	return m_children.size();
}

TreeNode* TreeNode::firstChild()
{
	return *(m_children.begin());
}

TreeNode* TreeNode::secondChild()
{
	return *(++m_children.begin());
}

TreeNode* TreeNode::getParent()
{
	return m_parent;
}

void TreeNode::replaceChild(TreeNode *n1, TreeNode *n2)
{
	for(std::list<TreeNode*>::iterator i = m_children.begin(); i != m_children.end(); ++i) {
		if(*i == n1) {
			(*i)->m_parent = NULL;
			*i = n2;
			n2->m_parent = this;
		}
	}
}

void TreeNode::removeChild(TreeNode *n)
{
	m_children.remove(n);
	n->m_parent = NULL;
}

void TreeNode::output(std::string &theOutput)
{
	// does nothing
}

bool TreeNode::childrenEqual(const TreeNode* n2)
{
	const TreeNodeList& children1 = m_children;
	TreeNodeList children2 = n2->children();

	if(m_children.size() != children2.size()) {
		return false;
	}
	TreeNodeList::const_iterator it = children1.begin();
	TreeNodeList::const_iterator it2 = children2.begin();
	while(it != children1.end() && it2 != children2.end()) {
                if(!(*(*it) == *(*it2))) {
			return false;
		}
		++it;
		++it2;
	}

	return true;
}

bool TreeNode::childrenLess(const TreeNode* n2)
{
	const TreeNodeList& children1 = m_children;
	TreeNodeList children2 = n2->children();

	if(m_children.size() < children2.size()) {
		return true;
	} else if(m_children.size() > children2.size()) {
		return false;
	}
	TreeNodeList::const_iterator it = children1.begin();
	TreeNodeList::const_iterator it2 = children2.begin();
	while(it != children1.end() && it2 != children2.end()) {
                if(*(*it) < *(*it2)) {
			return true;
		} else if(*(*it2) < *(*it)) {
			return false;
		}
		++it;
		++it2;
	}

	return false;
}

unsigned long TreeNode::size() const
{
  unsigned long res = 1; // 1 for itself

  for (TreeNodeList::const_iterator it = children().begin();
       it != children().end();
       ++it)
    {
      res += (*it)->size();
    }

  return res;
}

Identifier::Identifier(bool trueFalseConstant)
: TreeNode(), m_trueFalseConstant(trueFalseConstant)
{
}

Identifier::Identifier(const std::string& id, const std::list<TreeNode*>& theChildren)
: TreeNode(theChildren), m_id(id)
{
}

Identifier::~Identifier()
{
}

void Identifier::output(std::string &theOutput)
{
	if(m_id.length() == 0) {
		if(m_trueFalseConstant) {
			theOutput += "TRUE";
		}
		else {
			theOutput += "FALSE";
		}
		return;
	}
	theOutput += m_id;
	if(m_children.size() > 0) {
		theOutput += "(";
	}
	for(std::list<TreeNode*>::iterator i = m_children.begin(); i != m_children.end(); ++i) {
		if(i != m_children.begin()) {
			theOutput += ",";
		}
		(*i)->output(theOutput);
	}
	if(m_children.size() > 0) {
		theOutput += ")";
	}
}

TreeNode* Identifier::clone()
{
	return new Identifier(*this);
}

bool Identifier::isTrue() const
{
	return (m_id.length() == 0 && m_trueFalseConstant);
}

bool Identifier::isFalse() const
{
	return (m_id.length() == 0 && !m_trueFalseConstant);
}

const std::string& Identifier::getIdentifier() const
{
	return m_id;
}

bool Identifier::operator==(const TreeNode& n2)
{
	const Identifier *identifier2 = dynamic_cast<const Identifier*>(&n2);
	if(!identifier2) {
		return false;
	}

	if(m_id.length() != identifier2->m_id.length()) {
		return false;
	}

	if(m_id.length() == 0) {
		if(m_trueFalseConstant != identifier2->m_trueFalseConstant) {
			return false;
		}
		else {
			return true;
		}
	}

	if(m_id.compare(identifier2->m_id) != 0) {
		return false;
	}
	return childrenEqual(&n2);
}

bool Identifier::operator<(const TreeNode& n2)
{
        const Identifier *identifier2 = dynamic_cast<const Identifier*>(&n2);
        if(!identifier2) {
                return true;
        }

        if(m_id.length() != identifier2->m_id.length()) {
                return m_id.length() < identifier2->m_id.length();
        } else if(m_id.length() == 0) {
                return m_trueFalseConstant < identifier2->m_trueFalseConstant;
        }
        return (m_id.compare(identifier2->m_id) < 0);
}

Operator::Operator(int type, const std::list<TreeNode*>& theChildren)
: TreeNode(theChildren), m_type(type)
{
}

Operator::~Operator()
{
}

TreeNode* Operator::clone()
{
	return new Operator(*this);
}

bool Operator::operator==(const TreeNode& n2)
{
	const Operator *operator2 = dynamic_cast<const Operator*>(&n2);
	if(!operator2) {
		return false;
	}

	if(m_type != operator2->m_type) {
		return false;
	}
	return childrenEqual(&n2);
}

bool Operator::operator<(const TreeNode& n2)
{
	const Operator *operator2 = dynamic_cast<const Operator*>(&n2);
	if(!operator2) {
		return false;
	}

	if(m_type != operator2->m_type) {
		return m_type < operator2->m_type;
	}
	return childrenLess(&n2);
}

TreeNode* Not(TreeNode* n)
{
	TreeNode *toReturn = new Operator(NOT);
	toReturn->addChild(n);
	return toReturn;
}

TreeNode* And(TreeNode* n1, TreeNode *n2)
{
	TreeNode *toReturn = new Operator(AND);
	toReturn->addChild(n1);
	toReturn->addChild(n2);
	return toReturn;
}

TreeNode* And(const TreeNodeList& list)
{
	if(list.empty()) {
		return NULL;
	}
	TreeNode *toReturn = NULL;
	TreeNode *previous = *(list.begin());
	if(list.size() == 1) {
		return previous;
	}
	for(TreeNodeList::const_iterator it = ++list.begin(); it != list.end(); ++it) {
		toReturn = And(*it, previous);
		previous = toReturn;
	}
	return toReturn;
}

TreeNode* Or(TreeNode* n1, TreeNode *n2)
{
	TreeNode *toReturn = new Operator(OR);
	toReturn->addChild(n1);
	toReturn->addChild(n2);
	return toReturn;
}

TreeNode* Or(const TreeNodeList& list)
{
	if(list.empty()) {
		return NULL;
	}
	TreeNode *toReturn = NULL;
	TreeNode *previous = *(list.begin());
	if(list.size() == 1) {
		return previous;
	}
	for(TreeNodeList::const_iterator it = ++list.begin(); it != list.end(); ++it) {
		toReturn = Or(*it, previous);
		previous = toReturn;
	}
	return toReturn;
}

TreeNode* Implication(TreeNode* n1, TreeNode *n2)
{
	TreeNode *toReturn = new Operator(IMPLICATION);
	toReturn->addChild(n1);
	toReturn->addChild(n2);
	return toReturn;
}

TreeNode* Always(TreeNode *n)
{
	TreeNode *toReturn = new Operator(ALWAYS);
	toReturn->addChild(n);
	return toReturn;
}

TreeNode* Next(TreeNode *n)
{
	TreeNode *toReturn = new Operator(NEXT);
	toReturn->addChild(n);
	return toReturn;
}

TreeNode* Sometime(TreeNode *n)
{
	TreeNode *toReturn = new Operator(SOMETIME);
	toReturn->addChild(n);
	return toReturn;
}

TreeNode* Until(TreeNode* n1, TreeNode *n2)
{
	TreeNode *toReturn = new Operator(UNTIL);
	toReturn->addChild(n1);
	toReturn->addChild(n2);
	return toReturn;
}

TreeNode* Unless(TreeNode* n1, TreeNode *n2)
{
	TreeNode *toReturn = new Operator(UNLESS);
	toReturn->addChild(n1);
	toReturn->addChild(n2);
	return toReturn;
}

std::string operatorToString(int op)
{
	switch(op) {
		case AND:
			return "AND";
		break;
		case OR:
			return "OR";
		break;
		case NOT:
			return "NOT";
		break;
		case ALWAYS:
			return "ALWAYS";
		break;
		case NEXT:
			return "NEXT";
		break;
		case SOMETIME:
			return "SOMETIME";
		break;
		case UNTIL:
			return "UNTIL";
		break;
		case UNLESS:
			return "UNLESS";
		break;
		case IMPLICATION:
			return "IMPLICATION";
		break;
		case EQUIVALENCE:
			return "EQUIVALENCE";
		break;
	}
	return "";
}

void Operator::output(std::string &theOutput)
{
	theOutput += operatorToString(m_type);
	theOutput += "(";
	for(std::list<TreeNode*>::iterator i = m_children.begin(); i != m_children.end(); ++i) {
		if(i != m_children.begin()) {
			theOutput += ",";
		}
		(*i)->output(theOutput);
	}
	theOutput += ")";
}

int Operator::getOperator() const
{
	return m_type;
}

void Operator::setOperator(int op)
{
	m_type = op;
}

std::ostream& operator<<(std::ostream &os, TreeNode* n)
{
	if(!n) {
		os << "(null)";
	}
	else {
		std::string output;
		n->output(output);
		os << output;
	}
	return os;
}

TreeNode* transformToTree(tree *t)
{
        TreeNode *toReturn = NULL;
        list *children = tree_Children(t);

        if(tree_Op(t)) {
                int op = tree_Op(t);
                if(op == TRUE || op == FALSE) {
                        return new Identifier(op == TRUE);
                }
                else {
                        toReturn = new Operator(tree_Op(t));
                        for(list *it = children; !list_IsEmpty(it); it = list_Tail(it)) {
                                toReturn->addChild(transformToTree(static_cast<tree*>(list_Element(it))));
                        }
                }
        }
        else {
                std::string id = tree_Id(t);
                toReturn = new Identifier(id);
                for(list *it = children; !list_IsEmpty(it); it = list_Tail(it)) {
                        toReturn->addChild(transformToTree(static_cast<tree*>(list_Element(it))));
                }
        }
        return toReturn;
}

TreeNode* copy(TreeNode *n)
{
	TreeNode *toReturn = n->clone();
	toReturn->m_parent = NULL;

	return toReturn;
}

void removeDuplicateTreeNode(TreeNodeList& list, TreeNodeList::iterator& start)
{
	TreeNodeList::iterator it = start;
	++it;
	for(; it != list.end(); ) {
                if(*(*it) == *(*start)) {
			delete(*it);
			it = list.erase(it);
		}
		else {
			++it;
		}
	}
}

void removeDuplicates(TreeNodeList& list)
{
	if(list.empty()) {
		return;
	}
	for(TreeNodeList::iterator it = list.begin(); it != list.end(); ++it) {
		removeDuplicateTreeNode(list, it);
	}
}

unsigned long sizeTreeNodeList(TreeNodeList const & list)
{
        unsigned long result = 0;
	for(TreeNodeList::const_iterator it = list.begin();
            it != list.end();
            ++it) {
                result += (*it)->size();
	}
        return result;
}
