/************************************************************
 *    Copyright (C) 2012-2014                               *
 *    Viktor Schuppan (Viktor.Schuppan@gmx.de)              *
 *                                                          *
 *    This program is free software; you can redistribute   *
 *    it and/or modify it under the terms of the GNU        *
 *    General Public License as published by the Free       *
 *    Software Foundation, either version 3 of the License, *
 *    or (at your option) any later version.                *
 *                                                          *
 *    This program is distributed in the hope that it will  *
 *    be useful, but WITHOUT ANY WARRANTY; without even     *
 *    the implied warranty of MERCHANTABILITY or FITNESS    *
 *    FOR A PARTICULAR PURPOSE.  See the GNU General Public *
 *    License for more details.                             *
 *                                                          *
 *    You should have received a copy of the GNU General    *
 *    Public License along with this program.  If not, see  *
 *    <http://www.gnu.org/licenses/>.                       *
 ************************************************************/
#ifndef __SNF_TRANSLATOR_H__
#define __SNF_TRANSLATOR_H__

#include <map>
#include <utility>
#include "adaptor/iclause.h"
#include "adaptor/sclause.h"
#include "adaptor/uclause.h"
#include "getopt/options.h"
#include "misc/assert.h"
#include "stl_tim/attribute.h"
#include "stl_tim/literallist.h"
#include "trim/trim.h"
#include "unsat_core/clause_handle.h"
#include "unsat_core/literal_clauseh_occurrence.h"
#include "formula.h"
#include "snf_translator.h"

namespace BuilderLtl
{
  class SnfTranslator
  {
  public:
    SnfTranslator(TreeNodeList & formulaList,
                  UnsatCore::ClauseHandleSet & clausehSet,
                  std::map<TreeNode*, UnsatCore::LiteralClausehOccurrenceSet> &
                  mapSubformulaToUseLCHOsNow,
                  std::map<TreeNode*, UnsatCore::LiteralClausehOccurrenceSet> &
                  mapSubformulaToUseLCHOsNext,
                  std::map<TreeNode*, UnsatCore::LiteralClausehOccurrenceSet> &
                  mapSubformulaToUseLCHOsSometime,
                  UnsatCore::LiteralClausehOccurrenceSet &
                  setSubformulaToUseLCHOsInvalidPos,
                  TRPPPOptions const & options);
    void translate();
  private:
    bool is_initial_clause(TreeNode * n);
    bool is_universal_clause(TreeNode * n);
    bool is_step_clause(TreeNode * n);
    bool is_eventuality_clause(TreeNode * n);
    bool is_propositional_next_conjunction(TreeNode * n,
                                           bool allowNext);
    bool is_propositional_next_disjunction(TreeNode * n,
                                           bool allowNext);
    bool is_literal_eventuality(TreeNode * n);
    void make_and_add_iclause(TreeNode * n);
    void make_and_add_uclause(TreeNode * n);
    void make_and_add_sclause(TreeNode * n);
    void make_and_add_eclause(TreeNode * n);
    void make_and_add_eclause_aux(TreeNode * n_clause,
                                  PropositionalProver::LiteralList & ll,
                                  TreeNode * n_evlit,
                                  bool isTrue);
    void collect_literals_clause(TreeNode * n,
                                 bool isPositivePolarity,
                                 bool isUnderNext,
                                 PropositionalProver::Attribute a,
                                 PropositionalProver::LiteralList & ll,
                                 PropositionalProver::LiteralList & ll_next,
                                 bool & isTrue);
    void translate_to_snf_msftoulcho_clause(TreeNode * n,
                                            UnsatCore::ClauseHandle ch,
                                            bool isUnderNext,
                                            bool isUnderSometime,
                                            bool isPositivePolarity);
    void translate_to_snf_msftoulcho_clause_invalid_pos(TreeNode * n,
                                                     UnsatCore::ClauseHandle ch,
                                                        bool isUnderNext,
                                                        bool isUnderSometime);
    Identifier translate_to_snf_rec(TreeNode * n,
                                    bool isPositivePolarity,
                                    bool isInitial);
    void translate_to_snf_rec_aux(TreeNode * n,
                                  bool isPositivePolarity,
                                  bool isInitial,
                                  Identifier result);
    void translate_to_snf_copy_msftolcho(TreeNode * copy, TreeNode * orig);
    void translate_to_snf_rec_pbl_prop
      (PropositionalProver::Proposition const & p,
       bool pol,
       PropositionalProver::Attribute const & a,
       PropositionalProver::LiteralList & ll);
    void translate_to_snf_rec_pbl_id
      (Identifier const & id,
       bool pol,
       PropositionalProver::Attribute const & a,
       PropositionalProver::LiteralList & ll,
       bool & skip);
    PropositionalProver::Literal translate_to_snf_rec_l_id
      (Identifier const & id,
       bool pol,
       bool & empty,
       bool & skip);
    UnsatCore::ClauseHandle translate_to_snf_rec_make_uclause
      (PropositionalProver::LiteralList const & ll,
       UnsatCore::ClauseHandleSet & myClausehSet,
       bool isInitial);
    UnsatCore::ClauseHandle translate_to_snf_rec_make_sclause
      (PropositionalProver::LiteralList const & ll,
       PropositionalProver::LiteralList const & ll_next,
       UnsatCore::ClauseHandleSet & myClausehSet);
      std::pair<UnsatCore::ClauseHandle, UnsatCore::ClauseHandle>
        translate_to_snf_rec_make_eclause_cached
      (PropositionalProver::LiteralList ll,
       PropositionalProver::Literal const & l,
       UnsatCore::ClauseHandleSet & myClausehSet);
    UnsatCore::ClauseHandle translate_to_snf_rec_make_eclause
      (PropositionalProver::LiteralList const & ll,
       PropositionalProver::Literal const & l,
       UnsatCore::ClauseHandleSet & myClausehSet);
    void translate_to_snf_add_to_msftoulcho_now(TreeNode * n,
                                                UnsatCore::ClauseHandle ch,
                                                PropositionalProver::Literal l);
    void translate_to_snf_add_to_msftoulcho_now_invalid_pos(TreeNode * n,
                                                    UnsatCore::ClauseHandle ch);
    void translate_to_snf_add_to_msftoulcho_now(TreeNode * n,
                                                UnsatCore::ClauseHandle ch,
                                                Identifier id,
                                                bool isPositivePolarity,
                                              PropositionalProver::Attribute a);
    void translate_to_snf_add_to_msftoulcho_next(TreeNode * n,
                                                 UnsatCore::ClauseHandle ch,
                                                PropositionalProver::Literal l);
    void translate_to_snf_add_to_msftoulcho_next_invalid_pos(TreeNode * n,
                                                    UnsatCore::ClauseHandle ch);
    void translate_to_snf_add_to_msftoulcho_next(TreeNode * n,
                                                 UnsatCore::ClauseHandle ch,
                                                 Identifier id,
                                                 bool isPositivePolarity);
    void translate_to_snf_add_to_msftoulcho_sometime(TreeNode * n,
                                                    UnsatCore::ClauseHandle ch);
    void translate_to_snf_add_to_msftoulcho_sometime_invalid_pos(TreeNode * n,
                                                    UnsatCore::ClauseHandle ch);
    Identifier get_fresh_identifier();
    struct less_tree_node: std::binary_function
      <TreeNode * const, TreeNode * const, bool>
    {
      bool operator() (TreeNode * const a,
                       TreeNode * const b) const
      {
        std::map<std::pair<TreeNode * const, TreeNode * const>, bool> cache;
        return less_tree_node_rec(a, b, cache);
      }
    };
    static bool less_tree_node_rec(TreeNode * const a,
                                   TreeNode * const b,
                                   std::map<std::pair<TreeNode * const,
                                                      TreeNode * const>,
                                            bool> & cache)
    {
      std::pair<TreeNode * const, TreeNode * const> pab = std::make_pair(a, b);

      if (cache.find(pab) != cache.end()) {
        return cache[pab];
      } else {
        Identifier * const pid_a = dynamic_cast<Identifier * const>(a);
        Identifier * const pid_b = dynamic_cast<Identifier * const>(b);
        if (pid_a != NULL && pid_b == NULL) {
          // a is an identifier and b is not
          cache[pab] = true;
          return true;
        } else if (pid_a == NULL && pid_b != NULL) {
          // a isn't an identifier but b is
          cache[pab] = false;
          return false;
        } else if (pid_a != NULL && pid_b != NULL) {
          // a and b are identifiers
          if (pid_a->getIdentifier().length() == 0 &&
              pid_b->getIdentifier().length() != 0) {
            // a is bool, b is not
            cache[pab] = true;
            return true;
          } else if (pid_a->getIdentifier().length() != 0 &&
                     pid_b->getIdentifier().length() == 0) {
            // a isn't bool, b is
            cache[pab] = false;
            return false;
          } else if (pid_a->getIdentifier().length() == 0 &&
                     pid_b->getIdentifier().length() == 0) {
            // a and b are bool
            cache[pab] = (pid_a->isFalse() && pid_b->isTrue());
            return (pid_a->isFalse() && pid_b->isTrue());
          } else {
            cache[pab] = (pid_a->getIdentifier().compare(pid_b->getIdentifier()) < 0);
            return (pid_a->getIdentifier().compare(pid_b->getIdentifier()) < 0);
          }
        } else {
          // a and b should be operators
          Operator * const pop_a = dynamic_cast<Operator * const>(a);
          Operator * const pop_b = dynamic_cast<Operator * const>(b);
          if (pop_a == NULL && pop_b != NULL) {
            cache[pab] = true;
            return true;
          } else if (pop_a != NULL && pop_b == NULL) {
            cache[pab] = false;
            return false;
          } else if (pop_a != NULL && pop_b != NULL) {
            if (pop_a->getOperator() != pop_b->getOperator()) {
              cache[pab] = (pop_a->getOperator() < pop_b->getOperator());
              return (pop_a->getOperator() < pop_b->getOperator());
            } else {
              TreeNodeList::const_iterator it_a = pop_a->children().begin();
              TreeNodeList::const_iterator it_b = pop_b->children().begin();
              for (;
                   it_a != pop_a->children().end() &&
                   it_b != pop_b->children().end();
                   ++it_a,
                   ++it_b)
                {
                  if (less_tree_node_rec(*it_a, *it_b, cache)) {
                    cache[pab] = true;
                    return true;
                  } else if (less_tree_node_rec(*it_b, *it_a, cache)) {
                    cache[pab] = false;
                    return false;
                  }
                }
              if (it_a == pop_a->children().end() &&
                  it_b != pop_b->children().end()) {
                cache[pab] = true;
                return true;
              } else if (it_a != pop_a->children().end() &&
                         it_b == pop_b->children().end()) {
                cache[pab] = false;
                return false;
              } else {
                cache[pab] = false;
                return false;
              }
            }
          } else {
            Assert(false, "a and b are neither Identifier nor Operator.");
          }
        }
      }
      // Silence compiler
      Assert(false, "This point should not be reached.");
      return false;
    }
  private:
    TreeNodeList & myFormulaList;
    UnsatCore::ClauseHandleSet & myClausehSet;
    std::map<TreeNode*, UnsatCore::LiteralClausehOccurrenceSet> &
      myMapSubformulaToUseLCHOsNow;
    std::map<TreeNode*, UnsatCore::LiteralClausehOccurrenceSet> &
      myMapSubformulaToUseLCHOsNext;
    std::map<TreeNode*, UnsatCore::LiteralClausehOccurrenceSet> &
      myMapSubformulaToUseLCHOsSometime;
    UnsatCore::LiteralClausehOccurrenceSet &
      mySetSubformulaToUseLCHOsInvalidPos;
    TRPPPOptions const & myOptions;
    std::map<PropositionalProver::Literal, PropositionalProver::Literal>
      myMapEventualityLiteralToLiteral;
    std::map<PropositionalProver::Literal, UnsatCore::ClauseHandle>
      myMapEventualityLiteralToClauseh;
    std::map<TreeNode *, std::pair<Identifier, TreeNode *>, less_tree_node>
      myFormulaToSnfCachePosInit;
    std::map<TreeNode *, std::pair<Identifier, TreeNode *>, less_tree_node>
      myFormulaToSnfCacheNegInit;
    std::map<TreeNode *, std::pair<Identifier, TreeNode *>, less_tree_node>
      myFormulaToSnfCachePosNoInit;
    std::map<TreeNode *, std::pair<Identifier, TreeNode *>, less_tree_node>
      myFormulaToSnfCacheNegNoInit;
    unsigned long myNvvCounter;
  public:
    unsigned long myNoTopLevelSubformulaClauses;
    unsigned long myNoFormulaToSnfCacheHits;
    unsigned long myNoFormulaToSnfCacheMisses;
  }; // class SnfTranslator
} // namespace BuilderLtl
#endif // __SNF_TRANSLATOR_H__
