/**************************************************************/
/* ********************************************************** */
/* *                                                        * */
/* *  Copyright (C) 2001-2011                               * */
/* *  Boris Konev                                           * */
/* *  The University of Liverpool                           * */
/* *                                                        * */
/* *  Copyright (C) 2012-2014                               * */
/* *  Viktor Schuppan                                       * */
/* *                                                        * */
/* *  This program is free software; you can redistribute   * */
/* *  it and/or modify it under the terms of the GNU        * */
/* *  General Public License as published by the Free       * */
/* *  Software Foundation; either version 2 of the License, * */
/* *  or (at your option) any later version.                * */
/* *                                                        * */
/* *  This program is distributed in the hope that it will  * */
/* *  be useful, but WITHOUT ANY WARRANTY; without even     * */
/* *  the implied warranty of MERCHANTABILITY or FITNESS    * */
/* *  FOR A PARTICULAR PURPOSE.  See the GNU General Public * */
/* *  License for more details.                             * */
/* *                                                        * */
/* *  You should have received a copy of the GNU General    * */
/* *  Public License along with this program; if not, write * */
/* *  to the Free Software Foundation, Inc., 59 Temple      * */
/* *  Place, Suite 330, Boston, MA  02111-1307  USA         * */
/* *                                                        * */
/* *                                                        * */
/* ********************************************************** */
/**************************************************************/
#include <iostream>
#include <map>
#include <queue>
#include <set>
#include <utility>
#include <boost/tuple/tuple.hpp>
#include "builder/builder.h"
#include "builder/parse.h"
#include "builder_ltl/builder_ltl.h"
#include "builder_ltl/formula.h"
#include "getopt/options.h"
#include "loop_search/loopsearch.h"
#include "misc/assert.h"
#include "misc/tracer.h"
#include "modules.h"
#include "prover/search.h"
#include "temporal_resolution/classicalrule.h"
#include "unsat_core/clause_handle.h"
#include "unsat_core/lcho_partition.h"
#include "unsat_core/literal_clauseh_occurrence.h"
#include "unsat_core/mapper_snf_to_ltl.h"
#include "unsat_core/resolution_graph.h"
#include "unsat_core/semi_linear_set.h"
#include "unsat_core/unsat_core_private.h"
#include "unsat_core/unsat_core_writer.h"
#include "main_aux.h"

#define BENCHMARKING

MainAux::MainAux(TRPPPOptions const & options)
  : myFormulaList(),
    myMapSubformulaToUseLCHOsNow(),
    myMapSubformulaToUseLCHOsNext(),
    myMapSubformulaToUseLCHOsSometime(),
    mySetSubformulaToUseLCHOsInvalidPos(),
    myMapSubformulaToSemiLinearSet(),
    myClausehSet(),
    myCoreClausehSet(),
    myMapClausehToSemiLinearSet(),
    myMapLCHOToLCHOPartition(),
    myMapLCHOPartitionToIndex(),
    myUnsat(),
    myOptions(options),
    mySubformulasPQ(),
    myWasInSubformulasPQ(),
    myCoreSimplifiedSubformulas(),
    myNoVerticesSyntaxTreeBeforeLtlSimplification(0),
    myNoVerticesSyntaxTreeAfterLtlSimplification(0),
    myNoVerticesSyntaxTreeProofCore(0),
    myNoVerticesSyntaxTreeDeletionCore(0),
    myNoUCDeletionIterations(0),
    myNoUCDeletionIterationsUnsat(0),
    myNoUCDeletionIterationsSat(0)
{ }

MainAux::~MainAux()
{
  if (myOptions.getInputFormat() == INPUT_FORMAT_SNF) {
  } else {
    Assert(myOptions.getInputFormat() == INPUT_FORMAT_LTL,
           "Unexpected input format.");
    TreeNode * top = NULL;
    while (!myFormulaList.empty()) {
      TreeNode * tmp = myFormulaList.front();
      myFormulaList.pop_front();
      if (!isTopLevelFormula(tmp)) {
        Assert(top == NULL || top == tmp->getParent(),
               "Different parents in myFormulaList.");
        top = tmp->getParent();
      }
      delete(tmp);
    }
    if (top != NULL) {
      top->setChildren(TreeNodeList());
      delete(top);
    }
  }
} // MainAux::~MainAux

int MainAux::main()
{
  if (myOptions.getInputFormat() == INPUT_FORMAT_SNF) {
    int res = parse_snf();
    if (res == 1) {
      return res;
    }
  } else {
    Assert(myOptions.getInputFormat() == INPUT_FORMAT_LTL,
           "Unexpected input format.");
    parse_ltl();
    myNoVerticesSyntaxTreeBeforeLtlSimplification =
      sizeTreeNodeList(myFormulaList) + myFormulaList.size() - 1;
  }
  if (!myOptions.isExtractUnsatCore()) {
    if (myOptions.getInputFormat() == INPUT_FORMAT_LTL) {
      if (!myOptions.isNoLTLSimplification()) {
        simplify_ltl(myFormulaList);
        myNoVerticesSyntaxTreeAfterLtlSimplification =
          sizeTreeNodeList(myFormulaList) + myFormulaList.size() - 1;
      }
      BuilderLtl::BuilderLtl::translateLtlToSnf(myFormulaList,
                                                myClausehSet,
                                                myMapSubformulaToUseLCHOsNow,
                                                myMapSubformulaToUseLCHOsNext,
                                              myMapSubformulaToUseLCHOsSometime,
                                            mySetSubformulaToUseLCHOsInvalidPos,
                                                myOptions);
    }
    if (!myOptions.isNoSNFSimplification()) {
      simplify_snf(myClausehSet);
    }
    check_snf_for_empty_clause(myClausehSet);
    myUnsat = prove(myClausehSet);
    if (!myOptions.isQuiet()) {
      std::cout << "MainAux: number of nodes in syntax tree before LTL "
                << "simplification: "
                << myNoVerticesSyntaxTreeBeforeLtlSimplification
                << std::endl;
      std::cout << "MainAux: number of nodes in syntax tree after LTL "
                << "simplification: "
                << myNoVerticesSyntaxTreeAfterLtlSimplification
                << std::endl;
    }
    dumpUnsat();
  } else if (myOptions.getExtractUnsatCoreAlgorithm() ==
             EXTRACT_UNSAT_CORE_ALGORITHM_PROOF) {
    if (myOptions.getInputFormat() == INPUT_FORMAT_LTL) {
      if (!myOptions.isNoLTLSimplification()) {
        simplify_ltl(myFormulaList);
        myNoVerticesSyntaxTreeAfterLtlSimplification =
          sizeTreeNodeList(myFormulaList) + myFormulaList.size() - 1;
      }
      BuilderLtl::BuilderLtl::translateLtlToSnf(myFormulaList,
                                                myClausehSet,
                                                myMapSubformulaToUseLCHOsNow,
                                                myMapSubformulaToUseLCHOsNext,
                                              myMapSubformulaToUseLCHOsSometime,
                                            mySetSubformulaToUseLCHOsInvalidPos,
                                                myOptions);
    }
    if (!myOptions.isNoSNFSimplification()) {
      simplify_snf(myClausehSet);
    }
    check_snf_for_empty_clause(myClausehSet);
    myUnsat = prove_core_extraction(myOptions);
    if (myUnsat) {
#ifndef BENCHMARKING
      if (myOptions.getExtractUnsatCoreFormat() ==
          EXTRACT_UNSAT_CORE_FORMAT_LTL) {
#endif
        UnsatCore::MapperSnfToLtl::computeCoreLtl(myCoreClausehSet,
                                                  myFormulaList,
                                                  myMapSubformulaToUseLCHOsNow,
                                                  myMapSubformulaToUseLCHOsNext,
                                              myMapSubformulaToUseLCHOsSometime,
                                            mySetSubformulaToUseLCHOsInvalidPos,
                                                  myMapLCHOToLCHOPartition,
                                                  myMapLCHOPartitionToIndex,
                                                  myMapClausehToSemiLinearSet,
                                                 myMapSubformulaToSemiLinearSet,
                                                  myOptions);
        myNoVerticesSyntaxTreeProofCore =
          sizeTreeNodeList(myFormulaList) + myFormulaList.size() - 1;
#ifndef BENCHMARKING
      }
#endif
    }
    if (!myOptions.isQuiet()) {
      std::cout << "MainAux: number of nodes in syntax tree before LTL "
                << "simplification: "
                << myNoVerticesSyntaxTreeBeforeLtlSimplification
                << std::endl;
      std::cout << "MainAux: number of nodes in syntax tree after LTL "
                << "simplification: "
                << myNoVerticesSyntaxTreeAfterLtlSimplification
                << std::endl;
      if (myUnsat
#ifndef BENCHMARKING
          &&
          myOptions.getExtractUnsatCoreFormat() ==
          EXTRACT_UNSAT_CORE_FORMAT_LTL
#endif
          ) {
        std::cout << "MainAux: number of nodes in syntax tree of proof core: "
                  << myNoVerticesSyntaxTreeProofCore
                  << std::endl;
      }
    }
    dumpUnsat();
    if (myUnsat) {
#ifndef BENCHMARKING
      UnsatCore::UnsatCoreWriter::dumpCore(myCoreClausehSet,
                                           myMapClausehToSemiLinearSet,
                                           myMapLCHOToLCHOPartition,
                                           myMapLCHOPartitionToIndex,
                                           myFormulaList,
                                           myMapSubformulaToSemiLinearSet,
                                           myOptions);
#else
      UnsatCore::UnsatCoreWriter::dumpCoreBenchmarking(myCoreClausehSet,
                                                    myMapClausehToSemiLinearSet,
                                                       myMapLCHOToLCHOPartition,
                                                      myMapLCHOPartitionToIndex,
                                                       myFormulaList,
                                                 myMapSubformulaToSemiLinearSet,
                                                       myOptions);
#endif
    }
  } else {
    Assert(myOptions.getExtractUnsatCoreAlgorithm() ==
           EXTRACT_UNSAT_CORE_ALGORITHM_DELETION ||
           myOptions.getExtractUnsatCoreAlgorithm() ==
           EXTRACT_UNSAT_CORE_ALGORITHM_PROOFDELETION,
           "Unexpected unsat core algorithm.");

    // WARNING
    //
    // BE CAREFUL WHEN REMOVING THIS - INPUT AND OUTPUT FORMAT BEING
    // LTL MATTERS AT SEVERAL PLACES IN THIS ELSE BRANCH
    Assert(myOptions.getInputFormat() ==
           INPUT_FORMAT_LTL,
           "Not implemented.");
#ifndef BENCHMARKING
    Assert(myOptions.getExtractUnsatCoreFormat() ==
           EXTRACT_UNSAT_CORE_FORMAT_LTL,
           "Not implemented.");
#endif
    // END WARNING
    if (myOptions.getInputFormat() !=
        INPUT_FORMAT_LTL
#ifndef BENCHMARKING
        || myOptions.getExtractUnsatCoreFormat() !=
        EXTRACT_UNSAT_CORE_FORMAT_LTL
#endif
        ) {
      std::cout << "Not implemented."
                << std::endl;
      return 1;
    }

    Assert(myOptions.getSharing() == SHARING_NONE ||
           myOptions.getSharing() == SHARING_SAMEPOLARITY,
           "Sharing is not none or samepolarity.");

    // Initially determine whether input LTL formula is satisfiable
    // (and, when using proofdeletion algorithm, find unsat core)
    if (!myOptions.isQuiet()) {
        std::cout << "MainAux: (proof-)deletion: initially determining "
                  << "(un)satisfiability begin"
                  << std::endl;
    }
    if (!myOptions.isNoLTLSimplification()) {
      simplify_ltl(myFormulaList);
      myNoVerticesSyntaxTreeAfterLtlSimplification =
        sizeTreeNodeList(myFormulaList) + myFormulaList.size() - 1;
    }
    BuilderLtl::BuilderLtl::translateLtlToSnf(myFormulaList,
                                              myClausehSet,
                                              myMapSubformulaToUseLCHOsNow,
                                              myMapSubformulaToUseLCHOsNext,
                                              myMapSubformulaToUseLCHOsSometime,
                                            mySetSubformulaToUseLCHOsInvalidPos,
                                              myOptions);
    if (!myOptions.isNoSNFSimplification()) {
      simplify_snf(myClausehSet);
    }
    check_snf_for_empty_clause(myClausehSet);
    if (myOptions.getExtractUnsatCoreAlgorithm() ==
        EXTRACT_UNSAT_CORE_ALGORITHM_DELETION) {
      myUnsat = prove(myClausehSet);
    } else {
      Assert(myOptions.getExtractUnsatCoreAlgorithm() ==
             EXTRACT_UNSAT_CORE_ALGORITHM_PROOFDELETION,
             "Unexpected unsat core algorithm.");
      TRPPPOptions options = myOptions;
      options.setExtractUnsatCoreSetsOfTimePoints(false);
      Assert(!options.isExtractUnsatCorePartitionedPropositions(),
             "Partitioned propositions unsat core extraction enabled.");
      myUnsat = prove_core_extraction(options);
      if (myUnsat) {
        UnsatCore::MapperSnfToLtl::computeCoreLtl(myCoreClausehSet,
                                                  myFormulaList,
                                                  myMapSubformulaToUseLCHOsNow,
                                                  myMapSubformulaToUseLCHOsNext,
                                              myMapSubformulaToUseLCHOsSometime,
                                            mySetSubformulaToUseLCHOsInvalidPos,
                                                  myMapLCHOToLCHOPartition,
                                                  myMapLCHOPartitionToIndex,
                                                  myMapClausehToSemiLinearSet,
                                                 myMapSubformulaToSemiLinearSet,
                                                  options);
        myCoreClausehSet.clear();
        myNoVerticesSyntaxTreeProofCore =
          sizeTreeNodeList(myFormulaList) + myFormulaList.size() - 1;
      }
    }
    if (!myOptions.isQuiet()) {
        std::cout << "MainAux: (proof-)deletion: initially determining "
                  << "(un)satisfiability end"
                  << std::endl;
    }

    if (myUnsat) {
      TreeNodeList currFormulaList = TreeNodeList();
      for (TreeNodeList::const_iterator it = myFormulaList.begin();
           it != myFormulaList.end();
           ++it) {
        currFormulaList.push_back((*it)->clone());
      }
      TreeNode * currTop = And(currFormulaList);

      Assert(myWasInSubformulasPQ.empty(), "myWasInSubformulas not empty.");
      enqueue_children(currTop, true);

      while (!mySubformulasPQ.empty()) {
        if (!myOptions.isQuiet()) {
          std::cout << "MainAux: (proof-)deletion: deletion iteration begin"
                    << std::endl;
        }
        myNoUCDeletionIterations++;
#ifdef USE_TRACING
        if (checkTrace(unsatcoreModule)) {
          std::cout << "MainAux: contents of mySubformulasPQ begin"
                    << std::endl;
          std::priority_queue<std::pair<TreeNode*, bool> > tmpPQ;
          while (!mySubformulasPQ.empty()) {
            TreeNode * tmpSubformula;
            bool tmpPolarity;
            boost::tie(tmpSubformula, tmpPolarity) = mySubformulasPQ.top();
            mySubformulasPQ.pop();
            std::cout << "MainAux: tmpPolarity = "
                      << tmpPolarity
                      << ", tmpSubformula = "
                      << tmpSubformula
                      << std::endl;
            tmpPQ.push(std::make_pair(tmpSubformula, tmpPolarity));
          }
          mySubformulasPQ = tmpPQ;
          std::cout << "MainAux: contents of mySubformulasPQ end"
                    << std::endl;
        }
#endif
        TreeNode * currSubformula;
        bool currPolarity;
        boost::tie(currSubformula, currPolarity) = mySubformulasPQ.top();
        mySubformulasPQ.pop();
        TRACE(unsatcoreModule, {
            std::cout << "MainAux: currSubformula = "
                      << currSubformula
                      << std::endl;
            std::cout << "MainAux: currPolarity = "
                      << currPolarity
                      << std::endl;
          });
        Assert(!isTopLevelFormula(currSubformula),
               "currSubformula is top level.");
        TreeNode * currParent = NULL;
        TreeNode * currReplacement = new Identifier(currPolarity);
        std::map<TreeNode *, TreeNode *> mapORToSf;
        if (myOptions.getSharing() == SHARING_NONE) {
          currParent = currSubformula->getParent();
          currParent->replaceChild(currSubformula, currReplacement);
        } else {
          Assert(myOptions.getSharing() == SHARING_SAMEPOLARITY,
                 "Sharing is not samepolarity.");
          TRACE(unsatcoreModule, {
              std::cout << "MainAux: before replace_occurrences:"
                        << std::endl
                        << "MainAux: currTop = "
                        << currTop
                        << std::endl;
            });
          replace_occurrences(currTop,
                              true,
                              currSubformula,
                              currPolarity,
                              currReplacement,
                              mapORToSf);
          TRACE(unsatcoreModule, {
              std::cout << "MainAux: after replace_occurrences:"
                        << std::endl
                        << "MainAux: currTop = "
                        << currTop
                        << std::endl;
            });
        }
        TreeNode * candidateTop = currTop->clone();
        TreeNodeList candidateFormulaList;
        if (isAnd(candidateTop)) {
          candidateFormulaList = candidateTop->children();
          // hack to set children of candidateTop to empty list and at
          // the same time clear m_parent in candidateFormulaList
          candidateTop->clearChildren();
          delete(candidateTop);
        } else {
          candidateFormulaList = TreeNodeList();
          candidateFormulaList.push_back(candidateTop);
        }
        if (myOptions.isNoLTLSimplification()) {
          TRACE(unsatcoreModule, {
              candidateTop = new Operator(AND, candidateFormulaList);
              std::cout << "MainAux: Unsat core candidate:"
                        << std::endl
                        << candidateTop
                        << std::endl;
              candidateTop->setChildren(TreeNodeList());
              delete(candidateTop);
            });
        } else {
          TRACE(unsatcoreModule, {
              candidateTop = new Operator(AND, candidateFormulaList);
              std::cout << "MainAux: Unsat core candidate unsimplified:"
                        << std::endl
                        << candidateTop
                        << std::endl;
              candidateTop->setChildren(TreeNodeList());
              delete(candidateTop);
            });
          simplify_ltl(candidateFormulaList);
          TRACE(unsatcoreModule, {
              candidateTop = new Operator(AND, candidateFormulaList);
              std::cout << "MainAux: Unsat core candidate simplified:"
                        << std::endl
                        << candidateTop
                        << std::endl;
              candidateTop->setChildren(TreeNodeList());
              delete(candidateTop);
            });
        }
        UnsatCore::ClauseHandleSet candidateClausehSet;
        std::map<TreeNode *, UnsatCore::LiteralClausehOccurrenceSet>
          dummy_mSfToULCHOsNow;
        std::map<TreeNode *, UnsatCore::LiteralClausehOccurrenceSet>
          dummy_mSfToULCHOsNext;
        std::map<TreeNode *, UnsatCore::LiteralClausehOccurrenceSet>
          dummy_mSfToULCHOsSometime;
        UnsatCore::LiteralClausehOccurrenceSet dummy_mSSfToULCHOsIP;
        BuilderLtl::BuilderLtl::translateLtlToSnf(candidateFormulaList,
                                                  candidateClausehSet,
                                                  dummy_mSfToULCHOsNow,
                                                  dummy_mSfToULCHOsNext,
                                                  dummy_mSfToULCHOsSometime,
                                                  dummy_mSSfToULCHOsIP,
                                                  myOptions);
        while (!candidateFormulaList.empty()) {
          TreeNode * tmp = candidateFormulaList.front();
          candidateFormulaList.pop_front();
          delete(tmp);
        }
        if (!myOptions.isNoSNFSimplification()) {
          simplify_snf(candidateClausehSet);
        }
        bool unsat;
        try {
          check_snf_for_empty_clause(candidateClausehSet); 
          unsat = prove(candidateClausehSet);
        } catch (Exceptions::EmptyClauseInInput &e) {
          unsat = true;
        }
        TRACE(unsatcoreModule, {
            std::cout << "MainAux: Unsat core candidate "
                      << (unsat ? "unsatisfiable" : "satisfiable")
                      << std::endl;
          });
        if (myOptions.getSharing() == SHARING_NONE) {
          if (unsat) {
            delete(currSubformula);
            (void) myCoreSimplifiedSubformulas.insert(currReplacement);
            TRACE(unsatcoreModule, {
                std::cout << "MainAux: Adding "
                          << currReplacement
                          << " ("
                          << (void*) currReplacement
                          << ") to myCoreSimplifiedSubformulas"
                          << std::endl;
              });
            myNoUCDeletionIterationsUnsat++;
          } else {
            currParent->replaceChild(currReplacement, currSubformula);
            enqueue_children(currSubformula, currPolarity);
            delete(currReplacement);
            myNoUCDeletionIterationsSat++;
          }
        } else {
          Assert(myOptions.getSharing() == SHARING_SAMEPOLARITY,
                 "Sharing is not samepolarity.");
          if (unsat) {
            std::set<TreeNode *> toBeDeleted;
            for (std::map<TreeNode *, TreeNode *>::iterator it =
                   mapORToSf.begin();
                 it != mapORToSf.end();
                 ) {
              std::map<TreeNode *, TreeNode *>::iterator itcur = it;
              ++it;
              Assert(!isTopLevelFormula(itcur->first),
                     "itcur->first is top level.");
              (void) myCoreSimplifiedSubformulas.insert(itcur->first);
              TRACE(unsatcoreModule, {
                  std::cout << "MainAux: Adding "
                            << itcur->first
                            << " ("
                            << (void*) itcur->first
                            << ") to myCoreSimplifiedSubformulas"
                            << std::endl;
                });
              toBeDeleted.insert(itcur->second);
            }
            {
              // remove elements of toBeDeleted from mySubformulasPQ
              std::priority_queue<std::pair<TreeNode*, bool> > tmpPQ;
              while (!mySubformulasPQ.empty()) {
                TreeNode * tmpSubformula;
                bool tmpPolarity;
                boost::tie(tmpSubformula, tmpPolarity) = mySubformulasPQ.top();
                mySubformulasPQ.pop();
                if (toBeDeleted.find(tmpSubformula) == toBeDeleted.end()) {
                  tmpPQ.push(std::make_pair(tmpSubformula, tmpPolarity));
                }
              }
              mySubformulasPQ = tmpPQ;
            }
            {
              for (std::set<TreeNode *>::iterator it = toBeDeleted.begin();
                   it != toBeDeleted.end();
                   ) {
                std::set<TreeNode *>::iterator itcur = it;
                ++it;
                delete(*itcur);
              }
            }
            delete(currReplacement);
            myNoUCDeletionIterationsUnsat++;
          } else {
            TRACE(unsatcoreModule, {
                std::cout << "MainAux: before restoring replaced occurrences:"
                          << std::endl
                          << "MainAux: currTop = "
                          << currTop
                          << std::endl;
              });
            for (std::map<TreeNode *, TreeNode *>::iterator it =
                   mapORToSf.begin();
                 it != mapORToSf.end();
                 ) {
              std::map<TreeNode *, TreeNode *>::iterator itcur = it;
              ++it;
              Assert(!isTopLevelFormula(itcur->first),
                     "itcur->first is top level.");
              itcur->first->getParent()->replaceChild(itcur->first,
                                                      itcur->second);
              TRACE(unsatcoreModule, {
                  std::cout << "MainAux: Replaced "
                            << itcur->first
                            << " with "
                            << itcur->second
                            << std::endl;
                });
              delete(itcur->first);
            }
            TRACE(unsatcoreModule, {
                std::cout << "MainAux: after restoring replaced occurrences:"
                          << std::endl
                          << "MainAux: currTop = "
                          << currTop
                          << std::endl;
              });
            enqueue_children(currSubformula, currPolarity);
            delete(currReplacement);
            myNoUCDeletionIterationsSat++;
          }
        }
        if (!myOptions.isQuiet()) {
          std::cout << "MainAux: (proof-)deletion: deletion iteration end"
                    << std::endl;
        }
      }
      if (!myOptions.isQuiet()) {
        std::cout << "MainAux: (proof-)deletion: finalizing begin"
                  << std::endl;
      }
      for (std::set<std::pair<TreeNode *, bool> >::iterator it =
             myWasInSubformulasPQ.begin();
           it != myWasInSubformulasPQ.end();
           ) {
        std::set<std::pair<TreeNode *, bool> >::iterator itcur = it;
        ++it;
        delete(itcur->first);
      }
      while (!myFormulaList.empty()) {
        TreeNode * tmp = myFormulaList.front();
        myFormulaList.pop_front();
        delete(tmp);
      }
      if (isAnd(currTop)) {
        myFormulaList = currTop->children();
        // hack to set children of currTop to empty list and at
        // the same time clear m_parent in currFormulaList
        currTop->clearChildren();
        delete(currTop);
      } else {
        myFormulaList.push_back(currTop);
      }
      TRACE(unsatcoreModule, {
          currTop = new Operator(AND, myFormulaList);
          std::cout << "MainAux: Unsat core before simplification:"
                    << std::endl
                    << currTop
                    << std::endl;
          currTop->setChildren(TreeNodeList());
          delete(currTop);
        });
      if (!myOptions.isNoUnsatCoreSimplification()) {
        TRPPPOptions options = myOptions;
        options.setExtractUnsatCoreSetsOfTimePoints(false);
        Assert(!options.isExtractUnsatCorePartitionedPropositions(),
               "Partitioned propositions unsat core extraction enabled.");
        UnsatCore::MapperSnfToLtl::simplifyCoreLtl(myFormulaList,
                                                   myCoreSimplifiedSubformulas,
                                                   options);
      }
      myNoVerticesSyntaxTreeDeletionCore =
        sizeTreeNodeList(myFormulaList) + myFormulaList.size() - 1;
      TRACE(unsatcoreModule, {
          currTop = new Operator(AND, myFormulaList);
          std::cout << "MainAux: Unsat core after simplification:"
                    << std::endl
                    << currTop
                    << std::endl;
          currTop->setChildren(TreeNodeList());
          delete(currTop);
        });
      if (myOptions.isExtractUnsatCoreSetsOfTimePoints()) {
        Assert(!myOptions.isExtractUnsatCorePartitionedPropositions(),
               "Partitioned propositions unsat core extraction enabled.");
        Assert(myOptions.getSharing() == SHARING_NONE ||
               myOptions.getSharing() == SHARING_SAMEPOLARITY,
               "Sharing is not none or samepolarity.");
        if (!myOptions.isQuiet()) {
          std::cout << "MainAux: (proof-)deletion: computing core with sets of "
                    << "time points begin"
                    << std::endl;
        }
#if 0
        if (!myOptions.isNoLTLSimplification()) {
          simplify_ltl(myFormulaList);
        }
#endif
        myClausehSet.clear();
        myMapSubformulaToUseLCHOsNow.clear();
        myMapSubformulaToUseLCHOsNext.clear();
        myMapSubformulaToUseLCHOsSometime.clear();
        mySetSubformulaToUseLCHOsInvalidPos.clear();
        BuilderLtl::BuilderLtl::translateLtlToSnf(myFormulaList,
                                                  myClausehSet,
                                                  myMapSubformulaToUseLCHOsNow,
                                                  myMapSubformulaToUseLCHOsNext,
                                              myMapSubformulaToUseLCHOsSometime,
                                            mySetSubformulaToUseLCHOsInvalidPos,
                                                  myOptions);
        if (!myOptions.isNoSNFSimplification()) {
          simplify_snf(myClausehSet);
        }
        try {
          check_snf_for_empty_clause(myClausehSet);
        } catch (Exceptions::EmptyClauseInInput &e) {
          std::cout << "Input formula is or has been simplified to the point "
                    << "where its SNF contains the empty clause."
                    << std::endl
                    << "With the options used computation of an unsatisfiable "
                    << "core with sets of time points is not possible."
                    << std::endl;
          return 1;
        }
        Assert(myCoreClausehSet.empty(), "myCoreClausehSet not empty.");
        Assert(myMapClausehToSemiLinearSet.empty(),
               "myMapClausehToSemiLinearSet not empty.");
#ifndef DEBUG
        (void) prove_core_extraction(myOptions);
#else
        bool unsat = prove_core_extraction(myOptions);
#endif
        Assert(unsat, "Unsat unexpectedly false.");
        Assert(myMapSubformulaToSemiLinearSet.empty(),
               "myMapSubformulaToSemiLinearSet not empty.");
        UnsatCore::MapperSnfToLtl::computeCoreLtl(myCoreClausehSet,
                                                  myFormulaList,
                                                  myMapSubformulaToUseLCHOsNow,
                                                  myMapSubformulaToUseLCHOsNext,
                                              myMapSubformulaToUseLCHOsSometime,
                                            mySetSubformulaToUseLCHOsInvalidPos,
                                                  myMapLCHOToLCHOPartition,
                                                  myMapLCHOPartitionToIndex,
                                                  myMapClausehToSemiLinearSet,
                                                 myMapSubformulaToSemiLinearSet,
                                                  myOptions);
        if (!myOptions.isQuiet()) {
          std::cout << "MainAux: (proof-)deletion: computing core with sets of "
                    << "time points end"
                    << std::endl;
        }
      }
      if (!myOptions.isQuiet()) {
        std::cout << "MainAux: (proof-)deletion: finalizing end"
                  << std::endl;
      }
    }
    if (!myOptions.isQuiet()) {
      if (myUnsat) {
        std::cout << "MainAux: number of iterations in unsat core algorithm "
                  << "deletion: "
                  << myNoUCDeletionIterations
                  << std::endl;
        std::cout << "MainAux: number of iterations in unsat core algorithm "
                  << "deletion with unsat result: "
                  << myNoUCDeletionIterationsUnsat
                  << std::endl;
        std::cout << "MainAux: number of iterations in unsat core algorithm "
                  << "deletion with sat result: "
                  << myNoUCDeletionIterationsSat
                  << std::endl;
      }
      std::cout << "MainAux: number of nodes in syntax tree before LTL "
                << "simplification: "
                << myNoVerticesSyntaxTreeBeforeLtlSimplification
                << std::endl;
      std::cout << "MainAux: number of nodes in syntax tree after LTL "
                << "simplification: "
                << myNoVerticesSyntaxTreeAfterLtlSimplification
                << std::endl;
      if (myUnsat) {
        if (myOptions.getExtractUnsatCoreAlgorithm() ==
            EXTRACT_UNSAT_CORE_ALGORITHM_PROOF ||
            myOptions.getExtractUnsatCoreAlgorithm() ==
            EXTRACT_UNSAT_CORE_ALGORITHM_PROOFDELETION) {
          std::cout << "MainAux: number of nodes in syntax tree of proof core: "
                    << myNoVerticesSyntaxTreeProofCore
                    << std::endl;
        }
        std::cout << "MainAux: number of nodes in syntax tree of "
                  << (myOptions.getExtractUnsatCoreAlgorithm() ==
                      EXTRACT_UNSAT_CORE_ALGORITHM_DELETION ?
                      "deletion" :
                      "proofdeletion")
                  << " core: "
                  << myNoVerticesSyntaxTreeDeletionCore
                  << std::endl;
      }
    }
    dumpUnsat();
    if (myUnsat) {
#ifndef BENCHMARKING
      UnsatCore::UnsatCoreWriter::dumpCore(myCoreClausehSet,
                                           myMapClausehToSemiLinearSet,
                                           myMapLCHOToLCHOPartition,
                                           myMapLCHOPartitionToIndex,
                                           myFormulaList,
                                           myMapSubformulaToSemiLinearSet,
                                           myOptions);
#else
      UnsatCore::UnsatCoreWriter::dumpCoreBenchmarking(myCoreClausehSet,
                                                    myMapClausehToSemiLinearSet,
                                                       myMapLCHOToLCHOPartition,
                                                      myMapLCHOPartitionToIndex,
                                                       myFormulaList,
                                                 myMapSubformulaToSemiLinearSet,
                                                       myOptions);
#endif
    }
  }
  return 0;
} // MainAux::main

int MainAux::parse_snf()
{
  Assert(myOptions.getInputFormat() == INPUT_FORMAT_SNF,
         "Input format not SNF.");
  Assert(myClausehSet.empty(), "myClausehSet not empty.");

  Parsers::TRPParser<Builders::Builder<ClauseSet> >
    parser(myOptions.getFileName(),
           myOptions.getOrderFileName());
  if (parser.parseOk()) {
    myClausehSet = clause_set_2_clause_handle_set(parser.getClauseSet(), UnsatCore::mainPartition);
  } else {
    return 1;
  }
  return 0;
} // MainAux::parse_snf

void MainAux::parse_ltl()
{
  Assert(myOptions.getInputFormat() == INPUT_FORMAT_LTL,
         "Input format not LTL.");
  Assert(myFormulaList.empty(), "myFormulaList not empty.");

  myFormulaList = BuilderLtl::BuilderLtl::parseLtl(myOptions);
  BuilderLtl::BuilderLtl::replaceEquivalencesLtl(myFormulaList);
  if (!myOptions.isNoNnf()) {
    BuilderLtl::BuilderLtl::nnfLtl(myFormulaList);
  }
} // MainAux::parse_ltl

void MainAux::simplify_snf(UnsatCore::ClauseHandleSet & clauseHandleSet)
{
  BuilderLtl::BuilderLtl::simplifySnf(clauseHandleSet);
} // MainAux::simplify_snf

void MainAux::simplify_ltl(TreeNodeList & formulaList)
{
  BuilderLtl::BuilderLtl::simplifyLtl(formulaList, myOptions);
} // MainAux::simplify_ltl

void MainAux::check_snf_for_empty_clause(UnsatCore::ClauseHandleSet const &
                                         clauseHandleSet)
{
  BuilderLtl::BuilderLtl::checkSnfForEmptyClause(clauseHandleSet);
} // MainAux::check_snf_for_empty_clause

bool MainAux::prove(UnsatCore::ClauseHandleSet const & clausehSet)
{
  bool res;

  TRPPPOptions options = myOptions;
  options.setExtractUnsatCore(false);

  ClauseSet * pClauseSet =
    new ClauseSet(clause_handle_set_2_clause_set(clausehSet));

  UnsatCore::ClauseHandleSet dummy_chs;
  std::map<UnsatCore::ClauseHandle, UnsatCore::SemiLinearSet> dummy_map;
  UnsatCore::MapLCHOToLCHOPartition dummy_map2;
  std::map<UnsatCore::LCHOPartition, unsigned long> dummy_map3;
  UnsatCore::ResolutionGraph resolutionGraph(dummy_chs,
                                             dummy_map,
                                             dummy_map2,
                                             dummy_map3,
                                             options);

  typedef TemporalProver::Prover<ClauseSet,
    TemporalProver::LoopList<ClauseSet>,
    TemporalProver::ClassicalRule<ClauseSet> > 
    _Prover;
  _Prover prover(*pClauseSet);

  // dumping ClauseSet (if required)
  if(options.isShowInput()) {
    std::cout << "At the begining.." << std::endl;
    prover.getClauseSet().dump(std::cout);
    std::cout << std::endl;
  }
  // end dumping
        
  res = prover.search(options, resolutionGraph);

  // dumping ClauseSet (if needed)
  if(options.isShowResult()) {
    std::cout << "At the end.." << std::endl;
    prover.getClauseSet().dump(std::cout);
    std::cout << std::endl;
  }
  // end dumping

#ifdef COLLECT_STAT
  prover.outputStatistics(std::cout);
#endif // COLLECT_STAT

  delete pClauseSet;

  return res;
} // MainAux::prove

bool MainAux::prove_core_extraction(TRPPPOptions const & options)
{
  Assert(options.isExtractUnsatCore(), "Unsat core extraction not enabled.");

  bool res;

  ClauseSet * pClauseSet =
    new ClauseSet(clause_handle_set_2_clause_set(myClausehSet));

  UnsatCore::ResolutionGraph resolutionGraph(myCoreClausehSet,
                                             myMapClausehToSemiLinearSet,
                                             myMapLCHOToLCHOPartition,
                                             myMapLCHOPartitionToIndex,
                                             options);
  for (ClauseSet::const_i_iterator ii = pClauseSet->initial_begin();
       ii != pClauseSet->initial_end();
       ii++)
    resolutionGraph.addStartingClause(*ii);
  for (ClauseSet::const_u_iterator iu = pClauseSet->universal_begin();
       iu != pClauseSet->universal_end();
       iu++)
    resolutionGraph.addStartingClause(*iu);
  for (ClauseSet::const_s_iterator is = pClauseSet->step_begin();
       is != pClauseSet->step_end();
       is++)
    resolutionGraph.addStartingClause(*is);
  for (ClauseSet::const_e_iterator ie = pClauseSet->eventuality_begin();
       ie != pClauseSet->eventuality_end();
       ie++)
    resolutionGraph.addStartingClause(*ie);

  typedef TemporalProver::Prover<ClauseSet,
    TemporalProver::LoopList<ClauseSet>,
    TemporalProver::ClassicalRule<ClauseSet> > 
    _Prover;
  _Prover prover(*pClauseSet);

  // dumping ClauseSet (if required)
  if(options.isShowInput()) {
    std::cout << "At the begining.." << std::endl;
    prover.getClauseSet().dump(std::cout);
    std::cout << std::endl;
  }
  // end dumping
        
  res = prover.search(options, resolutionGraph);

  // dumping ClauseSet (if needed)
  if(options.isShowResult()) {
    std::cout << "At the end.." << std::endl;
    prover.getClauseSet().dump(std::cout);
    std::cout << std::endl;
  }
  // end dumping

#ifdef COLLECT_STAT
  prover.outputStatistics(std::cout);
#endif // COLLECT_STAT

  if (res) {
    resolutionGraph.computeCore();
    (void) resolutionGraph.dumpStatistics(std::cout);
  }

  delete pClauseSet;

  return res;
} // MainAux::prove_core_extraction

void MainAux::dumpUnsat()
{
  std::cout << (myUnsat?"Unsatisfiable":"Satisfiable") << std::endl;
} // MainAux::dumpUnsat

MainAux::ClauseSet
MainAux::clause_handle_set_2_clause_set(UnsatCore::ClauseHandleSet const &
                                        clauseHandleSet)
{
  ClauseSet clauseSet;

  for (UnsatCore::ClauseHandleSet::iterator it = clauseHandleSet.begin();
       it != clauseHandleSet.end();
       ++it)
    {
      if (it->isIClause()) {
        clauseSet.push_back(it->getIClause());
      } else if (it->isUClause()) {
        clauseSet.push_back(it->getUClause());
      } else if (it->isSClause()) {
        clauseSet.push_back(it->getSClause());
      } else {
        Assert(it->isEClause(), "it not an EClause.");
        clauseSet.push_back(it->getEClause());
      }
    }
  return clauseSet;
} // MainAux::clause_handle_set_2_clause_set

UnsatCore::ClauseHandleSet
MainAux::clause_set_2_clause_handle_set(ClauseSet const & clauseSet,
                                        UnsatCore::Partition const & partition)
{
  UnsatCore::ClauseHandleSet clauseHandleSet;

  for (ClauseSet::const_i_iterator ii = clauseSet.initial_begin();
       ii != clauseSet.initial_end();
       ii++)
    clauseHandleSet.insert(UnsatCore::ClauseHandle(*ii, partition));
  for (ClauseSet::const_u_iterator iu = clauseSet.universal_begin();
       iu != clauseSet.universal_end();
       iu++)
    clauseHandleSet.insert(UnsatCore::ClauseHandle(*iu, partition));
  for (ClauseSet::const_s_iterator is = clauseSet.step_begin();
       is != clauseSet.step_end();
       is++)
    clauseHandleSet.insert(UnsatCore::ClauseHandle(*is, partition));
  for (ClauseSet::const_e_iterator ie = clauseSet.eventuality_begin();
       ie != clauseSet.eventuality_end();
       ie++)
    clauseHandleSet.insert(UnsatCore::ClauseHandle(*ie, partition));
  return clauseHandleSet;
} // MainAux::clause_set_2_clause_handle_set

void MainAux::enqueue_children(TreeNode * subformula,
                               bool isPositivePolarity)
{
  if (isIdentifier(subformula)) {
    Assert(subformula->childrenCount() == 0, "Unexpected number of children.");
    // nop
  } else if (isNot(subformula)) {
    Assert(subformula->childrenCount() == 1, "Unexpected number of children.");
    enqueue_children(subformula->firstChild(), !isPositivePolarity);
  } else if (isAnd(subformula)) {
    Assert(subformula->childrenCount() > 0, "Unexpected number of children.");
    if (isPositivePolarity) {
      for (TreeNodeList::const_iterator it = subformula->children().begin();
           it != subformula->children().end();
           ++it) {
        Assert(myOptions.getSharing() == SHARING_NONE ||
               myOptions.getSharing() == SHARING_SAMEPOLARITY,
               "Sharing not none or samepolarity.");
        if (myOptions.getSharing() == SHARING_NONE ||
            !has_been_enqueued(*it, isPositivePolarity)) {
          mySubformulasPQ.push(std::make_pair(*it, isPositivePolarity));
        }
      }
    } else {
      for (TreeNodeList::const_iterator it = subformula->children().begin();
           it != subformula->children().end();
           ++it) {
        enqueue_children(*it, isPositivePolarity);
      }
    }
  } else if (isOr(subformula)) {
    Assert(subformula->childrenCount() > 0, "Unexpected number of children.");
    if (isPositivePolarity) {
      for (TreeNodeList::const_iterator it = subformula->children().begin();
           it != subformula->children().end();
           ++it) {
        enqueue_children(*it, isPositivePolarity);
      }
    } else {
      for (TreeNodeList::const_iterator it = subformula->children().begin();
           it != subformula->children().end();
           ++it) {
        Assert(myOptions.getSharing() == SHARING_NONE ||
               myOptions.getSharing() == SHARING_SAMEPOLARITY,
               "Sharing not none or samepolarity.");
        if (myOptions.getSharing() == SHARING_NONE ||
            !has_been_enqueued(*it, isPositivePolarity)) {
          mySubformulasPQ.push(std::make_pair(*it, isPositivePolarity));
        }
      }
    }
  } else if (isImplication(subformula)) {
    Assert(subformula->childrenCount() == 2, "Unexpected number of children.");
    if (isPositivePolarity) {
      enqueue_children(subformula->firstChild(), !isPositivePolarity);
      enqueue_children(subformula->secondChild(), isPositivePolarity);
    } else {
      Assert(myOptions.getSharing() == SHARING_NONE ||
             myOptions.getSharing() == SHARING_SAMEPOLARITY,
             "Sharing not none or samepolarity.");
      if (myOptions.getSharing() == SHARING_NONE ||
          !has_been_enqueued(subformula->firstChild(), !isPositivePolarity)) {
        mySubformulasPQ.push(std::make_pair(subformula->firstChild(),
                                            !isPositivePolarity));
      }
      if (myOptions.getSharing() == SHARING_NONE ||
          !has_been_enqueued(subformula->secondChild(), isPositivePolarity)) {
        mySubformulasPQ.push(std::make_pair(subformula->secondChild(),
                                            isPositivePolarity));
      }
    }
  } else if (isNext(subformula) ||
             isAlways(subformula) ||
             isSometime(subformula)) {
    Assert(subformula->childrenCount() == 1, "Unexpected number of children.");
    enqueue_children(subformula->firstChild(), isPositivePolarity);
  } else if (isUntil(subformula)) {
    Assert(subformula->childrenCount() == 2, "Unexpected number of children.");
    Assert(myOptions.getSharing() == SHARING_NONE ||
           myOptions.getSharing() == SHARING_SAMEPOLARITY,
           "Sharing not none or samepolarity.");
    if (myOptions.getSharing() == SHARING_NONE ||
        !has_been_enqueued(subformula->firstChild(), isPositivePolarity)) {
      mySubformulasPQ.push(std::make_pair(subformula->firstChild(),
                                          isPositivePolarity));
    }
    enqueue_children(subformula->secondChild(), isPositivePolarity);
  } else {
    Assert(isUnless(subformula), "Unexpected subformula type.");
    Assert(subformula->childrenCount() == 2, "Unexpected number of children.");
    if (isPositivePolarity) {
      enqueue_children(subformula->firstChild(), isPositivePolarity);
      enqueue_children(subformula->secondChild(), isPositivePolarity);
    } else {
      Assert(myOptions.getSharing() == SHARING_NONE ||
             myOptions.getSharing() == SHARING_SAMEPOLARITY,
             "Sharing not none or samepolarity.");
      if (myOptions.getSharing() == SHARING_NONE ||
          !has_been_enqueued(subformula->firstChild(), isPositivePolarity)) {
        mySubformulasPQ.push(std::make_pair(subformula->firstChild(),
                                            isPositivePolarity));
      }
      if (myOptions.getSharing() == SHARING_NONE ||
          !has_been_enqueued(subformula->secondChild(), isPositivePolarity)) {
        mySubformulasPQ.push(std::make_pair(subformula->secondChild(),
                                            isPositivePolarity));
      }
    }
  }
} // MainAux::enqueue_children

// Replace all occurrences of sf with polarity sfIsPositivePolarity in
// *children* of top with an occurrence of r and build a mapping from
// occurrences of r to the replaced occurrences of sf.
//
// Note the restriction to children of top, which is justified by the
// fact that in uc extraction via deletion the top level formula need
// not be tried.
void MainAux::replace_occurrences(TreeNode * top,
                                  bool topIsPositivePolarity,
                                  TreeNode * sf,
                                  bool sfIsPositivePolarity,
                                  TreeNode * r,
                                  std::map<TreeNode *, TreeNode *> & mapORToSf)
{
  if (isIdentifier(top)) {
    Assert(top->childrenCount() == 0, "Unexpected number of children.");
    // nop
  } else if (isAnd(top) ||
             isOr(top) ||
             isNext(top) ||
             isAlways(top) ||
             isSometime(top) ||
             isUntil(top) ||
             isUnless(top)) {
    Assert(top->childrenCount() > 0, "Unexpected number of children.");
    for (TreeNodeList::const_iterator it = top->children().begin();
         it != top->children().end();
         ) {
      TreeNodeList::const_iterator itcur = it;
      ++it;
      if (topIsPositivePolarity == sfIsPositivePolarity && **itcur == *sf) {
        TreeNode * roc = r->clone();
        mapORToSf[roc] = *itcur;
        TRACE(unsatcoreModule, {
            std::cout << "MainAux: mapORToSf["
                      << roc
                      << "] = "
                      << mapORToSf[roc]
                      << std::endl;
          });
        top->replaceChild(*itcur, roc);
      } else {
        replace_occurrences(*itcur,
                            topIsPositivePolarity,
                            sf,
                            sfIsPositivePolarity,
                            r,
                            mapORToSf);
      }
    }    
  } else if (isNot(top)) {
    Assert(top->childrenCount() == 1, "Unexpected number of children.");
    for (TreeNodeList::const_iterator it = top->children().begin();
         it != top->children().end();
         ) {
      TreeNodeList::const_iterator itcur = it;
      ++it;
      if (topIsPositivePolarity != sfIsPositivePolarity && **itcur == *sf) {
        TreeNode * roc = r->clone();
        mapORToSf[roc] = *itcur;
        TRACE(unsatcoreModule, {
            std::cout << "MainAux: mapORToSf["
                      << roc
                      << "] = "
                      << mapORToSf[roc]
                      << std::endl;
          });
        top->replaceChild(*itcur, roc);
      } else {
        replace_occurrences(*itcur,
                            !topIsPositivePolarity,
                            sf,
                            sfIsPositivePolarity,
                            r,
                            mapORToSf);
      }
    }    
  } else {
    Assert(isImplication(top), "Unexpected subformula type.");
    Assert(top->childrenCount() == 2, "Unexpected number of children.");
    TreeNode * c = top->firstChild();
    if (topIsPositivePolarity != sfIsPositivePolarity && *c == *sf) {
      TreeNode * roc = r->clone();
      mapORToSf[roc] = c;
      TRACE(unsatcoreModule, {
          std::cout << "MainAux: mapORToSf["
                    << roc
                    << "] = "
                    << mapORToSf[roc]
                    << std::endl;
        });
      top->replaceChild(c, roc);
    } else {
      replace_occurrences(c,
                          !topIsPositivePolarity,
                          sf,
                          sfIsPositivePolarity,
                          r,
                          mapORToSf);
    }
    c = top->secondChild();
    if (topIsPositivePolarity == sfIsPositivePolarity && *c == *sf) {
      TreeNode * roc = r->clone();
      mapORToSf[roc] = c;
      TRACE(unsatcoreModule, {
          std::cout << "MainAux: mapORToSf["
                    << roc
                    << "] = "
                    << mapORToSf[roc]
                    << std::endl;
        });
      top->replaceChild(c, roc);
    } else {
      replace_occurrences(c,
                          topIsPositivePolarity,
                          sf,
                          sfIsPositivePolarity,
                          r,
                          mapORToSf);
    }
  }
} // MainAux::replace_occurrences

bool
MainAux::has_been_enqueued(TreeNode * f, bool isPositivePolarity)
{
  for (std::set<std::pair<TreeNode *, bool> >::iterator it =
         myWasInSubformulasPQ.begin();
       it != myWasInSubformulasPQ.end();
       ++it) {
    if ((*it).second == isPositivePolarity &&
        (*((*it).first)) == *f) {
      return true;
    }
  }
  (void) myWasInSubformulasPQ.insert(std::make_pair(f->clone(),
                                                    isPositivePolarity));
  return false;
} // MainAux::has_been_enqueued
