/************************************************************
 *    Copyright (C) 2012-2014                               *
 *    Viktor Schuppan (Viktor.Schuppan@gmx.de)              *
 *                                                          *
 *    This program is free software; you can redistribute   *
 *    it and/or modify it under the terms of the GNU        *
 *    General Public License as published by the Free       *
 *    Software Foundation, either version 3 of the License, *
 *    or (at your option) any later version.                *
 *                                                          *
 *    This program is distributed in the hope that it will  *
 *    be useful, but WITHOUT ANY WARRANTY; without even     *
 *    the implied warranty of MERCHANTABILITY or FITNESS    *
 *    FOR A PARTICULAR PURPOSE.  See the GNU General Public *
 *    License for more details.                             *
 *                                                          *
 *    You should have received a copy of the GNU General    *
 *    Public License along with this program.  If not, see  *
 *    <http://www.gnu.org/licenses/>.                       *
 ************************************************************/
#ifndef __CLAUSE_HANDLE_H__
#define __CLAUSE_HANDLE_H__

#include <functional>
#include <iostream>
#include <set>
#include <boost/any.hpp>
#include "unsat_core_private.h"

namespace UnsatCore
{
  class ClauseHandle
  {
  private:
    boost::any myClause;
    Partition myPartition;
  public:
    ClauseHandle();
    ClauseHandle(IClause const & iclause, Partition const & partition);
    ClauseHandle(UClause const & uclause, Partition const & partition);
    ClauseHandle(SClause const & sclause, Partition const & partition);
    ClauseHandle(EClause const & eclause, Partition const & partition);
    ClauseHandle(PClause const & pclause, Partition const & partition);
    bool isIClause() const;
    bool isUClause() const;
    bool isSClause() const;
    bool isEClause() const;
    bool isEmpty() const;
    IClause getIClause() const;
    UClause getUClause() const;
    SClause getSClause() const;
    EClause getEClause() const;
    Partition getPartition() const;
    PClause getPClause() const;
    bool operator<(ClauseHandle const & other) const;
    bool operator==(ClauseHandle const & other) const;
    std::ostream & dump(std::ostream & os) const;
  }; // class ClauseHandle

  typedef std::set<ClauseHandle> ClauseHandleSet;
} // namespace UnsatCore

std::ostream& operator<<(std::ostream& os,
                         const UnsatCore::ClauseHandle & clauseh);

namespace std
{
  template <>
    class hash<UnsatCore::ClauseHandle>
    {
    public :
      size_t operator()(const UnsatCore::ClauseHandle & ch) const
      {
        size_t hp = hash<UnsatCore::Partition>()(ch.getPartition());
        if (ch.isSClause()) {
          return hash<UnsatCore::SClause>()(ch.getSClause()) ^ hp;
        } else if (ch.isUClause()) {
          return hash<UnsatCore::UClause>()(ch.getUClause()) ^ hp;
        } else if (ch.isIClause()) {
          return hash<UnsatCore::IClause>()(ch.getIClause()) ^ hp;
        } else {
          Assert(ch.isEClause(), "ch not an EClause");
          return hash<TRIMs::EClause>()(ch.getEClause()) ^ hp;
        }
      }
    };
};
#endif // __CLAUSE_HANDLE_H__
