/************************************************************
 *    Copyright (C) 2014                                    *
 *    Viktor Schuppan (Viktor.Schuppan@gmx.de)              *
 *                                                          *
 *    This program is free software; you can redistribute   *
 *    it and/or modify it under the terms of the GNU        *
 *    General Public License as published by the Free       *
 *    Software Foundation, either version 3 of the License, *
 *    or (at your option) any later version.                *
 *                                                          *
 *    This program is distributed in the hope that it will  *
 *    be useful, but WITHOUT ANY WARRANTY; without even     *
 *    the implied warranty of MERCHANTABILITY or FITNESS    *
 *    FOR A PARTICULAR PURPOSE.  See the GNU General Public *
 *    License for more details.                             *
 *                                                          *
 *    You should have received a copy of the GNU General    *
 *    Public License along with this program.  If not, see  *
 *    <http://www.gnu.org/licenses/>.                       *
 ************************************************************/
#include <iostream>
#include <map>
#include <set>
#include <utility>
#include <boost/graph/adjacency_list.hpp>
#include <boost/graph/graph_traits.hpp>
#include <boost/pending/disjoint_sets.hpp>
#include <boost/tuple/tuple.hpp>
#include "getopt/options.h"
#include "misc/assert.h"
#include "misc/tracer.h"
#include "modules.h"
#include "clause_handle.h"
#include "inference_rule_edge_labeling.h"
#include "lcho_partition.h"
#include "literal_clauseh_occurrence.h"
#include "resolution_graph.h"
#include "unsat_core_private.h"
#include "lcho_partitioner.h"

namespace UnsatCore
{
  LCHOPartitioner::LCHOPartitioner(Graph & resolutionGraph,
                            std::map<Vertex, ClauseHandle> & mapVertexToClauseh,
                                   std::map<Edge, InferenceRuleEdgeLabeling> &
                                   mapEdgeToInferenceRuleEdgeLabeling,
                                   ClauseHandleSet & coreStartingClausehs,
                                MapLCHOToLCHOPartition & mapLCHOToLCHOPartition,
                                   std::map<LCHOPartition, unsigned long> &
                                   mapLCHOPartitionToIndex,
                           std::map<Partition, Literal> & mapPartitionToLiteral,
                                unsigned long & noPropositionsReachableSubgraph,
                      unsigned long & noPropositionOccurrencesReachableSubgraph,
                unsigned long & noPartitionsPerPropositionPeakReachableSubgraph,
                 unsigned long & noPartitionsPerPropositionGE2ReachableSubgraph,
                                   unsigned long & noPropositionsCore,
                                   unsigned long & noPropositionOccurrencesCore,
                             unsigned long & noPartitionsPerPropositionPeakCore,
                              unsigned long & noPartitionsPerPropositionGE2Core,
                                   unsigned long & noPropositionsCoreNonNVV,
                             unsigned long & noPropositionOccurrencesCoreNonNVV,
                       unsigned long & noPartitionsPerPropositionPeakCoreNonNVV,
                        unsigned long & noPartitionsPerPropositionGE2CoreNonNVV,
                                   TRPPPOptions const & options)
    : myResolutionGraph(resolutionGraph),
      myMapVertexToClauseh(mapVertexToClauseh),
      myMapEdgeToInferenceRuleEdgeLabeling(mapEdgeToInferenceRuleEdgeLabeling),
      myCoreStartingClausehs(coreStartingClausehs),
      myMapLCHOToLCHOPartition(mapLCHOToLCHOPartition),
      myMapLCHOPartitionToIndex(mapLCHOPartitionToIndex),
      myMapPartitionToLiteral(mapPartitionToLiteral),
      myNoPropositionsReachableSubgraph(noPropositionsReachableSubgraph),
      myNoPropositionOccurrencesReachableSubgraph(
                                     noPropositionOccurrencesReachableSubgraph),
      myNoPartitionsPerPropositionPeakReachableSubgraph(
                               noPartitionsPerPropositionPeakReachableSubgraph),
      myNoPartitionsPerPropositionGE2ReachableSubgraph(
                                noPartitionsPerPropositionGE2ReachableSubgraph),
      myNoPropositionsCore(noPropositionsCore),
      myNoPropositionOccurrencesCore(noPropositionOccurrencesCore),
      myNoPartitionsPerPropositionPeakCore(noPartitionsPerPropositionPeakCore),
      myNoPartitionsPerPropositionGE2Core(noPartitionsPerPropositionGE2Core),
      myNoPropositionsCoreNonNVV(noPropositionsCoreNonNVV),
      myNoPropositionOccurrencesCoreNonNVV(noPropositionOccurrencesCoreNonNVV),
      myNoPartitionsPerPropositionPeakCoreNonNVV(
                                      noPartitionsPerPropositionPeakCoreNonNVV),
      myNoPartitionsPerPropositionGE2CoreNonNVV(
                                       noPartitionsPerPropositionGE2CoreNonNVV),
      myOptions(options)
  { } // LCHOPartitioner::LCHOPartitioner

  void
  LCHOPartitioner::computeLCHOPartitionsSNF()
  {
    if (!myOptions.isQuiet()) {
      std::cout << "LCHOPartitioner: partitioning proposition occurrences begin"
                << std::endl;
    }

    // Prepare union find data structure
    //
    // Following
    // http://stackoverflow.com/questions/4134703/understanding-boostdisjoint-sets
    MapLCHOPartitionToSizeT rankm;
    MapLCHOPartitionToLCHOPartition parentm;
    BapmMapLCHOPartitionToSizeT rankpm(rankm);
    BapmMapLCHOPartitionToLCHOPartition parentpm(parentm);
    LCHOPartitionDisjointSets ufind(rankpm, parentpm);

    // Map each occurrence of a literal in a clause to a fresh
    // occurrence partition
    {
      VertexIterator vi, vend;
      for(boost::tie(vi,vend) = boost::vertices(myResolutionGraph);
          vi != vend;
          ++vi)
        {
          Assert(myMapVertexToClauseh.find(*vi) != myMapVertexToClauseh.end(),
                 "*vi not contained in myMapVertexToClauseh.");

          ClauseHandle const & ch = myMapVertexToClauseh[*vi];
          PClause pc;
          Clause::const_iterator it;

          // get clause body, handle eventuality literal
          if (ch.isIClause() || ch.isUClause() || ch.isSClause()) {
            pc = ch.getPClause();
          } else {
            Assert(ch.isEClause(), "it not an eventuality clause.");
            pc = ch.getEClause().getC().getInternalRep();
            // handle eventuality literal
            LiteralClausehOccurrence lcho =
              LiteralClausehOccurrence(ch);
            LCHOPartition lchop =
              LCHOPartition(ch.getEClause().getEventuality().getProposition());
            TRACE(unsatcoreModule, {
                std::cout << "LCHOPartitioner::computeLCHOPartitionsSNF: mapping lcho ";
                lcho.dump(std::cout);
                std::cout << " to partition ";
                std::cout << std::endl;
              });
            Assert(myMapLCHOToLCHOPartition.find(lcho) ==
                   myMapLCHOToLCHOPartition.end(),
                   "lcho contained in myMapLCHOToLCHOPartition.");
            myMapLCHOToLCHOPartition[lcho] = lchop;
            ufind.make_set(lchop);
            ++myNoPropositionOccurrencesReachableSubgraph;
          }

          // handle clause body
          for (it = pc->cbegin(); it != pc->cend(); ++it) {
            LiteralClausehOccurrence lcho = LiteralClausehOccurrence(ch, it);
            LCHOPartition lchop = LCHOPartition(it->getProposition());
            TRACE(unsatcoreModule, {
                std::cout << "LCHOPartitioner::computeLCHOPartitionsSNF: mapping lcho ";
                lcho.dump(std::cout);
                std::cout << " to partition ";
                lchop.dump(std::cout);
                std::cout << std::endl;
              });
            Assert(myMapLCHOToLCHOPartition.find(lcho) ==
                   myMapLCHOToLCHOPartition.end(),
                   "lcho contained in myMapLCHOToLCHOPartition.");
            myMapLCHOToLCHOPartition[lcho] = lchop;
            ufind.make_set(lchop);
            ++myNoPropositionOccurrencesReachableSubgraph;
          }
        }
    }

    // For each clause c backward reachable from the empty clause:
    // union the partitions of the lchos of literals in c with the
    // partitions of the lchos of corresponding literals in the
    // premises of c
    {
      VertexIterator vi, vend;
      for(boost::tie(vi,vend) = boost::vertices(myResolutionGraph);
          vi != vend;
          ++vi)
        {
          Assert(myMapVertexToClauseh.find(*vi) != myMapVertexToClauseh.end(),
                 "*vi not contained in myMapVertexToClauseh.");

          ClauseHandle chc = myMapVertexToClauseh[*vi];
          Partition pc = chc.getPartition();
          unsigned long noEdges = boost::out_degree(*vi, myResolutionGraph);

          // shared part for getting the edge(s)
          typedef boost::graph_traits<Graph>::out_edge_iterator oedge_iterator;
          std::pair<oedge_iterator, oedge_iterator> oot;
          oot = boost::out_edges(*vi, myResolutionGraph);

          TRACE(unsatcoreModule, {
              std::cout << "LCHOPartitioner::computeLCHOPartitionsSNF: "
                        << "unioning partitions of lchos in clauses:"
                        << std::endl;
              std::cout << "LCHOPartitioner::computeLCHOPartitionsSNF: "
                        << "conclusion: ";
              chc.dump(std::cout);
              std::cout << std::endl;
              for (std::pair<oedge_iterator, oedge_iterator> oott = oot;
                   oott.first != oott.second;
                   ++oott.first) {
                Edge et = *oott.first;
                Assert(myMapEdgeToInferenceRuleEdgeLabeling.find(et) !=
                       myMapEdgeToInferenceRuleEdgeLabeling.end(),
                   "et not contained in myMapEdgeToInferenceRuleEdgeLabeling.");
                InferenceRuleEdgeLabeling lbt =
                  myMapEdgeToInferenceRuleEdgeLabeling[et];
                Vertex vpt = boost::target(et, myResolutionGraph);
                Assert(myMapVertexToClauseh.find(vpt) !=
                       myMapVertexToClauseh.end(),
                       "vpt not contained in myMapVertexToClauseh.");
                ClauseHandle const & chpt = myMapVertexToClauseh[vpt];
                std::cout << "LCHOPartitioner::computeLCHOPartitionsSNF: "
                          << "edge: ";
                dumpInferenceRuleEdgeLabeling(std::cout, lbt);
                std::cout << std::endl;
                std::cout << "LCHOPartitioner::computeLCHOPartitionsSNF: "
                          << "premise: ";
                chpt.dump(std::cout);
                std::cout << std::endl;
              }
            });

          if (pc == mainPartition) {
            if (noEdges == 0) {
              if (myCoreStartingClausehs.find(chc) !=
                  myCoreStartingClausehs.end()) {
                // starting clause
                /* nop */
              } else {
                Assert(!myOptions.isNoRGNoEdgeAug2(),
                       "Unexpected situation in resolution graph.");
                // aug2 clause
                //
                // G(P, F(l))
                // G(not wl, X(l, wl))
                // for some premise 1 : part(l in premise 1) == part(l
                // in conclusion). Note that the partitions of
                // all premises 1's eventuality literals are unioned
                // elsewhere.
                // NOTE: edge to premise 1 not in resolution graph
                Assert(chc.isSClause(), "Not a step clause.");
                PClause pcc = chc.getSClause().getInternalRep();
                Assert(pcc->size() == 3, "Not an aug2 clause.");
                Clause::const_iterator it = find_l_aug2(pcc);
                ClauseHandle chp = find_eclauseh_l(*it);
                LiteralClausehOccurrence lchoc =
                  LiteralClausehOccurrence(chc, it);
                LiteralClausehOccurrence lchop = LiteralClausehOccurrence(chp);
                union_set_lchos(ufind, lchoc, lchop);
              }
            } else if (noEdges == 1) {
              // get the edge, label
              Edge e = *oot.first;
              Assert(myMapEdgeToInferenceRuleEdgeLabeling.find(e) !=
                     myMapEdgeToInferenceRuleEdgeLabeling.end(),
                     "e not contained in myMapEdgeToInferenceRuleEdgeLabeling.");
              InferenceRuleEdgeLabeling lb =
                myMapEdgeToInferenceRuleEdgeLabeling[e];
              Vertex vp = boost::target(e, myResolutionGraph);
              Assert(myMapVertexToClauseh.find(vp) !=
                     myMapVertexToClauseh.end(),
                     "vp not contained in myMapVertexToClauseh.");
              ClauseHandle const & chp = myMapVertexToClauseh[vp];

              if (lb == INFERENCE_RULE_EDGE_LABELING_AUG1) {
                // aug1 clause
                //
                // G(P, F(l))
                // G(P, l, wl)
                // for p in P : part(p in premise 1) == part(p in conclusion)
                // part(l in premise 1) == part(l in conclusion)
                Assert(chc.isUClause(), "Not a universal clause.");
                Assert(chp.isEClause(), "Not an eventuality clause.");
                union_set_l_n(ufind, chc, chp);
                union_set_bodies_nn(ufind, chc, chp);
              } else if (myOptions.isNoRGNoEdgeAug2() &&
                         lb == INFERENCE_RULE_EDGE_LABELING_AUG2) {
                // aug2 clause
                //
                // G(P, F(l))
                // G(not wl, X(l, wl))
                // for some premise 1 : part(l in premise 1) == part(l
                // in conclusion). Note that the partitions of
                // all premises 1's eventuality literals are unioned
                // elsewhere.
                Assert(chc.isSClause(), "Not a step clause.");
                PClause pcc = chc.getSClause().getInternalRep();
                Assert(pcc->size() == 3, "Not an aug2 clause.");
                Assert(chp.isEClause(), "Not an eventuality clause.");
                Clause::const_iterator it = find_l_aug2(pcc);
#ifdef DEBUG
                Literal l = chp.getEClause().getEventuality();
#endif
                Assert(it->getProposition() == l.getProposition() &&
                       it->getSign() == l.getSign(),
                       "l not found in aug2 clause.");
                LiteralClausehOccurrence lchoc =
                  LiteralClausehOccurrence(chc, it);
                LiteralClausehOccurrence lchop = LiteralClausehOccurrence(chp);
                union_set_lchos(ufind, lchoc, lchop);
              } else {
                Assert(!myOptions.isNoRGNoEdgeBFSLoopConclusion2E() &&
                       (lb == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION2_WEMPTYG ||
                        lb == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION2_WOEMPTYG),
                       "Unexpected situation in resolution graph.");
                // bfs-loop-conclusion2 clause
                //
                // G(P)
                // G(Q, F(l))
                // G(not wl, X(P, l))
                // for p in P : part(p in premise 1) == part(p in conclusion)
                // for some premise 2 : part(l in premise 2) == part(l
                // in conclusion). Note that the partitions of
                // all premises 2's eventuality literals are unioned
                // elsewhere.
                // NOTE: edge to premise 2 not in resolution graph
                Assert(chc.isSClause(), "Not a step clause.");
                Assert(chp.isUClause() ||
                       chp.isSClause() ||
                       (chp.isIClause() &&
                        chp.getIClause().getInternalRep()->size() == 0),
                       "Not a universal, step, or empty clause.");
                Partition p = chp.getPartition();
                Assert(myMapPartitionToLiteral.find(p) !=
                       myMapPartitionToLiteral.end(),
                       "p not contained in myMapPartitionToLiteral.");
                ClauseHandle chpe = find_eclauseh_l(myMapPartitionToLiteral[p]);
                union_set_l_x(ufind, chc, chpe);
                union_set_bodies_xn(ufind, chc, chp);
              }
            } else {
              Assert(noEdges == 2, "Unexpected number of edges.");

              // get the edges, labels
              Edge e1 = *oot.first;
              ++oot.first;
              Edge e2 = *oot.first;
              Assert(myMapEdgeToInferenceRuleEdgeLabeling.find(e1) !=
                     myMapEdgeToInferenceRuleEdgeLabeling.end(),
                   "e1 not contained in myMapEdgeToInferenceRuleEdgeLabeling.");
              Assert(myMapEdgeToInferenceRuleEdgeLabeling.find(e2) !=
                     myMapEdgeToInferenceRuleEdgeLabeling.end(),
                   "e2 not contained in myMapEdgeToInferenceRuleEdgeLabeling.");
              InferenceRuleEdgeLabeling lb1 =
                myMapEdgeToInferenceRuleEdgeLabeling[e1];
              InferenceRuleEdgeLabeling lb2 =
                myMapEdgeToInferenceRuleEdgeLabeling[e2];
              Vertex vp1 = boost::target(e1, myResolutionGraph);
              Vertex vp2 = boost::target(e2, myResolutionGraph);
              Assert(myMapVertexToClauseh.find(vp1) !=
                     myMapVertexToClauseh.end(),
                     "vp1 not contained in myMapVertexToClauseh.");
              Assert(myMapVertexToClauseh.find(vp2) !=
                     myMapVertexToClauseh.end(),
                     "vp2 not contained in myMapVertexToClauseh.");
              ClauseHandle const & chp1 = myMapVertexToClauseh[vp1];
              ClauseHandle const & chp2 = myMapVertexToClauseh[vp2];

              if (lb1 == INFERENCE_RULE_EDGE_LABELING_INIT_II &&
                  lb2 == INFERENCE_RULE_EDGE_LABELING_INIT_II) {
                // init-ii
                //
                // (P, l)
                // (not l, Q)
                // (P, Q)
                // part(l in premise 1) == part(not l in premise 2)
                // for p in P : part(p in premise 1) == part(p in conclusion)
                // for p in Q : part(p in premise 2) == part(p in conclusion)
                Assert(chc.isIClause(), "Not an init clause.");
                Assert(chp1.isIClause(), "Not an init clause.");
                Assert(chp2.isIClause(), "Not an init clause.");
                union_set_resolved_upon_lits(ufind, chp1, chp2);
                union_set_bodies_ii(ufind, chc, chp1);
                union_set_bodies_ii(ufind, chc, chp2);
              } else if ((lb1 == INFERENCE_RULE_EDGE_LABELING_INIT_INI &&
                          lb2 == INFERENCE_RULE_EDGE_LABELING_INIT_INN) ||
                         (lb1 == INFERENCE_RULE_EDGE_LABELING_INIT_INN &&
                          lb2 == INFERENCE_RULE_EDGE_LABELING_INIT_INI)) {
                // init-in clause
                //
                // (P, l)
                // G(not l, Q)
                // (P, Q)
                // part(l in premise 1) == part(not l in premise 2)
                // for p in P : part(p in premise 1) == part(p in conclusion)
                // for p in Q : part(p in premise 2) == part(p in conclusion)
                Assert(chc.isIClause() || chc.isIClause(),
                       "Not an initial clause.");
                if (lb1 == INFERENCE_RULE_EDGE_LABELING_INIT_INI) {
                  Assert(chp1.isIClause(), "Not an init clause.");
                  Assert(chp2.isUClause() || chp2.isSClause(),
                         "Not a universal or step clause.");
                } else {
                  Assert(chp1.isUClause() || chp2.isSClause(),
                         "Not a universal or step clause.");
                  Assert(chp2.isIClause(), "Not an init clause.");
                }
                union_set_resolved_upon_lits(ufind, chp1, chp2);
                if (lb1 == INFERENCE_RULE_EDGE_LABELING_INIT_INI) {
                  union_set_bodies_ii(ufind, chc, chp1);
                  union_set_bodies_in(ufind, chc, chp2);
                } else {
                  union_set_bodies_in(ufind, chc, chp1);
                  union_set_bodies_ii(ufind, chc, chp2);
                }
              } else if (lb1 == INFERENCE_RULE_EDGE_LABELING_STEP_NN &&
                         lb2 == INFERENCE_RULE_EDGE_LABELING_STEP_NN) {
                // step-nn clause
                //
                // G(P, l)
                // G(not l, Q)
                // G(P, Q)
                // part(l in premise 1) == part(not l in premise 2)
                // for p in P : part(p in premise 1) == part(p in conclusion)
                // for p in Q : part(p in premise 2) == part(p in conclusion)
                Assert(chc.isUClause() ||
                       chc.isSClause() ||
                       (chc.isIClause() &&
                        chc.getIClause().getInternalRep()->size() == 0),
                       "Not a universal, step, or empty clause.");
                Assert(chp1.isUClause() || chp1.isSClause(),
                       "Not a universal or step clause.");
                Assert(chp2.isUClause() || chp2.isSClause(),
                       "Not a universal or step clause.");
                union_set_resolved_upon_lits(ufind, chp1, chp2);
                union_set_bodies_nn(ufind, chc, chp1);
                union_set_bodies_nn(ufind, chc, chp2);
              } else if ((lb1 == INFERENCE_RULE_EDGE_LABELING_STEP_NXN &&
                          lb2 == INFERENCE_RULE_EDGE_LABELING_STEP_NXX) ||
                         (lb1 == INFERENCE_RULE_EDGE_LABELING_STEP_NXX &&
                          lb2 == INFERENCE_RULE_EDGE_LABELING_STEP_NXN)) {
                // step-nx clause
                //
                // G(P, l)
                // G(Q, X(not l, R))
                // G(Q, X(P, R))
                // part(l in premise 1) == part(not l in premise 2)
                // for p in P : part(p in premise 1) == part(p in conclusion)
                // for p in Q : part(p in premise 2) == part(p in conclusion)
                // for p in R : part(p in premise 2) == part(p in conclusion)
                Assert(chc.isUClause() ||
                       chc.isSClause() ||
                       (chc.isIClause() &&
                        chc.getIClause().getInternalRep()->size() == 0),
                       "Not a universal, step, or empty clause.");
                if (lb1 == INFERENCE_RULE_EDGE_LABELING_STEP_NXN) {
                  Assert(chp1.isUClause() || chp1.isSClause(),
                         "Not a universal or step clause.");
                  Assert(chp2.isSClause(), "Not a step clause.");
                } else {
                  Assert(chp1.isSClause(), "Not a step clause.");
                  Assert(chp2.isUClause() || chp2.isSClause(),
                         "Not a universal or step clause.");
                }
                union_set_resolved_upon_lits(ufind, chp1, chp2);
                if (lb1 == INFERENCE_RULE_EDGE_LABELING_STEP_NXN) {
                  union_set_bodies_xn(ufind, chc, chp1);
                  union_set_bodies_nn(ufind, chc, chp2);
                  union_set_bodies_xx(ufind, chc, chp2);
                } else {
                  union_set_bodies_nn(ufind, chc, chp1);
                  union_set_bodies_xx(ufind, chc, chp1);
                  union_set_bodies_xn(ufind, chc, chp2);
                }
              } else if (lb1 == INFERENCE_RULE_EDGE_LABELING_STEP_XX &&
                         lb2 == INFERENCE_RULE_EDGE_LABELING_STEP_XX) {
                // step-xx clause
                //
                // G(P, X(Q, l))
                // G(R, X(not l, S))
                // G(P, R, X(Q, S))
                // part(l in premise 1) == part(not l in premise 2)
                // for p in P : part(p in premise 1) == part(p in conclusion)
                // for p in Q : part(p in premise 1) == part(p in conclusion)
                // for p in R : part(p in premise 2) == part(p in conclusion)
                // for p in S : part(p in premise 2) == part(p in conclusion)
                Assert(chc.isUClause() ||
                       chc.isSClause() ||
                       (chc.isIClause() &&
                        chc.getIClause().getInternalRep()->size() == 0),
                       "Not a universal, step, or empty clause.");
                Assert(chp1.isSClause(), "Not a step clause.");
                Assert(chp2.isSClause(), "Not a step clause.");
                union_set_resolved_upon_lits(ufind, chp1, chp2);
                union_set_bodies_nn(ufind, chc, chp1);
                union_set_bodies_xx(ufind, chc, chp1);
                union_set_bodies_nn(ufind, chc, chp2);
                union_set_bodies_xx(ufind, chc, chp2);
              } else if (((lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION1_WEMPTYG ||
                           lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION1_WOEMPTYG) &&
                          (lb2 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION1_WEMPTYE ||
                           lb2 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION1_WOEMPTYE)) ||
                         ((lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION1_WEMPTYE ||
                           lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION1_WOEMPTYE) &&
                          (lb2 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION1_WEMPTYG ||
                           lb2 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION1_WOEMPTYG))) {
                // bfs-loop-conclusion1 clause
                //
                // G(P)
                // G(Q, F(l))
                // G(P, Q, l)
                // for p in P : part(p in premise 1) == part(p in conclusion)
                // for p in Q : part(p in premise 2) == part(p in conclusion)
                // part(l in premise 2) == part(l in conclusion)
                Assert(chc.isUClause() || chc.isSClause(),
                       "Not a universal or step clause.");
                if (lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION1_WEMPTYG ||
                    lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION1_WOEMPTYG) {
                  Assert(chp1.isUClause() ||
                         chp1.isSClause() ||
                         (chp1.isIClause() &&
                          chp1.getIClause().getInternalRep()->size() == 0),
                         "Not a universal, step, or empty clause.");
                  Assert(chp2.isEClause(), "Not an eventuality clause.");
                } else {
                  Assert(chp1.isEClause(), "Not an eventuality clause.");
                  Assert(chp2.isUClause() ||
                         chp2.isSClause() ||
                         (chp2.isIClause() &&
                          chp2.getIClause().getInternalRep()->size() == 0),
                         "Not a universal, step, or empty clause.");
                }
                if (lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION1_WEMPTYE ||
                    lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION1_WOEMPTYE) {
                  union_set_l_n(ufind, chc, chp1);
                } else {
                  union_set_l_n(ufind, chc, chp2);
                }
                union_set_bodies_nn(ufind, chc, chp1);
                union_set_bodies_nn(ufind, chc, chp2);
              } else {
                Assert((myOptions.isNoRGNoEdgeBFSLoopConclusion2E()) &&
                       (((lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION2_WEMPTYG ||
                          lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION2_WOEMPTYG) &&
                         (lb2 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION2_WEMPTYE ||
                          lb2 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION2_WOEMPTYE)) ||
                        ((lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION2_WEMPTYE ||
                          lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION2_WOEMPTYE) &&
                         (lb2 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION2_WEMPTYG ||
                          lb2 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION2_WOEMPTYG))),
                       "Unexpected situation in resolution graph.");
                // bfs-loop-conclusion2 clause
                //
                // G(P)
                // G(Q, F(l))
                // G(not wl, X(P, l))
                // for p in P : part(p in premise 1) == part(p in conclusion)
                // for some premise 2 : part(l in premise 2) == part(l
                // in conclusion). Note that the partitions of
                // all premises 2's eventuality literals are unioned
                // elsewhere.
                Assert(chc.isSClause(), "Not a step clause.");
                if (lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION2_WEMPTYG ||
                    lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION2_WOEMPTYG) {
                  Assert(chp1.isUClause() || chp1.isSClause(),
                         "Not a universal or step clause.");
                  Assert(chp2.isEClause(), "Not an eventuality clause.");
                } else {
                  Assert(chp1.isEClause(), "Not an eventuality clause.");
                  Assert(chp2.isUClause() || chp1.isSClause(),
                         "Not a universal or step clause.");
                }
                if (lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION2_WEMPTYG ||
                    lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION2_WOEMPTYG) {
                  union_set_l_x(ufind, chc, chp2);
                  union_set_bodies_xn(ufind, chc, chp1);
                } else {
                  union_set_l_x(ufind, chc, chp1);
                  union_set_bodies_xn(ufind, chc, chp2);
                }
              }
            }
          } else {
            if (noEdges == 0) {
              Assert(!myOptions.isNoRGNoEdgeBFSLoopItInitCE() &&
                     myOptions.isNoRGNoEdgeBFSLoopItInitCG(),
                     "Unexpected situation in resolution graph.");
              Assert(myOptions.isNoPruneFailedLoopSearchIts(),
                     "myNoPruneFailedLoopSearchIts is false.");
              // bfs-loop-it-init-c clause
              Assert(chc.isSClause(), "Not a step clause.");
              Assert(chc.getSClause().getInternalRep()->size() == 1,
                     "Unexpected situation in resolution graph.");
              /* nop */
            } else if (noEdges == 1) {
              // get the edge, label
              Edge e = *oot.first;
              Assert(myMapEdgeToInferenceRuleEdgeLabeling.find(e) !=
                     myMapEdgeToInferenceRuleEdgeLabeling.end(),
                    "e not contained in myMapEdgeToInferenceRuleEdgeLabeling.");
              InferenceRuleEdgeLabeling lb =
                myMapEdgeToInferenceRuleEdgeLabeling[e];
              Vertex vp = boost::target(e, myResolutionGraph);
              Assert(myMapVertexToClauseh.find(vp) !=
                     myMapVertexToClauseh.end(),
                     "vp not contained in myMapVertexToClauseh.");
              ClauseHandle const & chp = myMapVertexToClauseh[vp];

              if (lb == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_N) {
                // bfs-loop-it-init-n clause
                //
                // G(P)
                // G(X(P))
                // for p in P : part(p in premise 1) == part(p in conclusion)
                Assert(chc.isSClause(), "Not a step clause.");
                Assert(chp.isUClause() || chp.isSClause(),
                       "Not a universal or step clause.");
                union_set_bodies_xn(ufind, chc, chp);
              } else if (lb == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_X) {
                // bfs-loop-it-init-x clause
                //
                // G(P, X(Q))
                // G(P, X(Q))
                // for p in P : part(p in premise 1) == part(p in conclusion)
                // for p in Q : part(p in premise 1) == part(p in conclusion)
                Assert(chc.isSClause(), "Not a step clause.");
                Assert(chp.isSClause(), "Not a step clause.");
                union_set_bodies_nn(ufind, chc, chp);
                union_set_bodies_xx(ufind, chc, chp);
              } else if (!myOptions.isNoRGNoEdgeBFSLoopItInitCE() &&
                         lb == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_SUB) {
                // bfs-loop-it-init-c clause
                //
                // G(P)
                // G(Q, F(l))
                // G(X(P, l))
                // for some premise 2 : part(l in premise 2) == part(l
                // in conclusion). Note that the partitions of
                // all premises 2's eventuality literals are unioned
                // elsewhere.
                // NOTE: edge to premise 2 not in resolution graph
                //
                // G(P), s.t. G(P) -> G(Q)
                // G(X(Q, l)) generated by BFS-loop-it-init-c
                // for all p in P : part(p in premise 1) == part(p in conclusion)
                Assert(chc.isSClause(), "Not a step clause.");
                Assert(!myOptions.isNoRGNoEdgeBFSLoopItInitCG() ||
                       chc.getSClause().getInternalRep()->size() == 1,
                       "Unexpected situation in resolution graph.");
                Assert(chp.isUClause() ||
                       chp.isSClause() ||
                       (chp.isIClause() &&
                        chp.getIClause().getInternalRep()->size() == 0),
                       "Not a universal, step, or empty clause.");
                Partition p = chp.getPartition();
                Assert(myMapPartitionToLiteral.find(p) !=
                       myMapPartitionToLiteral.end(),
                       "p not contained in myMapPartitionToLiteral.");
                ClauseHandle chpe = find_eclauseh_l(myMapPartitionToLiteral[p]);
                union_set_l_x(ufind, chc, chpe);
                union_set_bodies_xn(ufind, chc, chp);
              } else if (!myOptions.isNoRGNoEdgeBFSLoopItInitCE() &&
                         myOptions.isNoRGNoEdgeBFSLoopItInitCG() &&
                       lb == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CG) {
                Assert(myOptions.isNoPruneFailedLoopSearchIts(),
                       "myNoPruneFailedLoopSearchIts is false.");
                // bfs-loop-it-init-c clause
                //
                // G(P)
                // G(Q, F(l))
                // G(X(P, l))
                // for p in P : part(p in premise 1) == part(p in conclusion)
                // NOTE: not necessary for correctness
                // for some premise 2 : part(l in premise 2) == part(l
                // in conclusion). Note that the partitions of
                // all premises 2's eventuality literals are unioned
                // elsewhere.
                // NOTE: edge to premise 2 not in resolution graph
                Assert(chc.isSClause(), "Not a step clause.");
                Assert(chp.isUClause() ||
                       chp.isSClause() ||
                       (chp.isIClause() &&
                        chp.getIClause().getInternalRep()->size() == 0),
                       "Not a universal, step, or empty clause.");
                Partition p = chp.getPartition();
                Assert(myMapPartitionToLiteral.find(p) !=
                       myMapPartitionToLiteral.end(),
                       "p not contained in myMapPartitionToLiteral.");
                ClauseHandle chpe = find_eclauseh_l(myMapPartitionToLiteral[p]);
                union_set_bodies_xn(ufind, chc, chp);
                union_set_l_x(ufind, chc, chpe);
              } else if (myOptions.isNoRGNoEdgeBFSLoopItInitCE() &&
                         myOptions.isNoRGNoEdgeBFSLoopItInitCG() &&
                       lb == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CE) {
                Assert(myOptions.isNoPruneFailedLoopSearchIts(),
                       "myNoPruneFailedLoopSearchIts is false.");
                // bfs-loop-it-init-c clause
                //
                // G(P)
                // G(Q, F(l))
                // G(X(P, l))
                // for some premise 2 : part(l in premise 2) == part(l
                // in conclusion). Note that the partitions of
                // all premises 2's eventuality literals are unioned
                // elsewhere.
                Assert(chc.isSClause(), "Not a step clause.");
                Assert(chp.isEClause(), "Not an eventuality clause.");
                union_set_l_x(ufind, chc, chp);
              } else {
                Assert(!myOptions.isNoPruneMainFwdReachable() &&
                       !myOptions.isNoRGNoEdgeBFSLoopItInitCE() &&
                       myOptions.isNoRGNoEdgeBFSLoopItInitCG() &&
                       lb == INFERENCE_RULE_EDGE_LABELING_STEP_XX,
                       "Unexpected situation in resolution graph.");
                Assert(myOptions.isNoPruneFailedLoopSearchIts(),
                       "myNoPruneFailedLoopSearchIts is false.");
                // step-xx clause that lost a premise X(l) during pruning
                //
                // G(X(l)) (lost)
                // G(R, X(not l, S))
                // G(R, X(S))
                // for p in R : part(p in premise 2) == part(p in conclusion)
                // for p in S : part(p in premise 2) == part(p in conclusion)
                Assert(chc.isUClause() ||
                       chc.isSClause() ||
                       (chc.isIClause() &&
                        chc.getIClause().getInternalRep()->size() == 0),
                       "Not a universal, step, or empty clause.");
                Assert(chp.isSClause(), "Not a step clause.");
                if (chc.isUClause()) {
                  Assert(chc.getUClause().getInternalRep()->size() + 1 ==
                         chp.getSClause().getInternalRep()->size(),
                         "Unexpected situation in resolution graph.");
                } else if (chc.isSClause()) {
                  Assert(chc.getSClause().getInternalRep()->size() + 1 ==
                         chp.getSClause().getInternalRep()->size(),
                         "Unexpected situation in resolution graph.");
                } else {
                  Assert(chc.getIClause().getInternalRep()->size() + 1 ==
                         chp.getSClause().getInternalRep()->size(),
                         "Unexpected situation in resolution graph.");
                }
                union_set_bodies_nn(ufind, chc, chp);
                union_set_bodies_xx(ufind, chc, chp);
              }
            } else if (noEdges == 2) {
              // get the edges, labels
              Edge e1 = *oot.first;
              ++oot.first;
              Edge e2 = *oot.first;
              Assert(myMapEdgeToInferenceRuleEdgeLabeling.find(e1) !=
                     myMapEdgeToInferenceRuleEdgeLabeling.end(),
                   "e1 not contained in myMapEdgeToInferenceRuleEdgeLabeling.");
              Assert(myMapEdgeToInferenceRuleEdgeLabeling.find(e2) !=
                     myMapEdgeToInferenceRuleEdgeLabeling.end(),
                   "e2 not contained in myMapEdgeToInferenceRuleEdgeLabeling.");
              InferenceRuleEdgeLabeling lb1 =
                myMapEdgeToInferenceRuleEdgeLabeling[e1];
              InferenceRuleEdgeLabeling lb2 =
                myMapEdgeToInferenceRuleEdgeLabeling[e2];
              Vertex vp1 = boost::target(e1, myResolutionGraph);
              Vertex vp2 = boost::target(e2, myResolutionGraph);
              Assert(myMapVertexToClauseh.find(vp1) !=
                     myMapVertexToClauseh.end(),
                     "vp1 not contained in myMapVertexToClauseh.");
              Assert(myMapVertexToClauseh.find(vp2) !=
                     myMapVertexToClauseh.end(),
                     "vp2 not contained in myMapVertexToClauseh.");
              ClauseHandle const & chp1 = myMapVertexToClauseh[vp1];
              ClauseHandle const & chp2 = myMapVertexToClauseh[vp2];

              if (lb1 == INFERENCE_RULE_EDGE_LABELING_STEP_XX &&
                  lb2 == INFERENCE_RULE_EDGE_LABELING_STEP_XX) {
                // step-xx clause
                //
                // G(P, X(Q, l))
                // G(R, X(not l, S))
                // G(P, R, X(Q, S))
                // part(l in premise 1) == part(not l in premise 2)
                // for p in P : part(p in premise 1) == part(p in conclusion)
                // for p in Q : part(p in premise 1) == part(p in conclusion)
                // for p in R : part(p in premise 2) == part(p in conclusion)
                // for p in S : part(p in premise 2) == part(p in conclusion)
                Assert(chc.isUClause() ||
                       chc.isSClause() ||
                       (chc.isIClause() &&
                        chc.getIClause().getInternalRep()->size() == 0),
                       "Not a universal, step, or empty clause.");
                Assert(chp1.isSClause(), "Not a step clause.");
                Assert(chp2.isSClause(), "Not a step clause.");
                union_set_resolved_upon_lits(ufind, chp1, chp2);
                union_set_bodies_nn(ufind, chc, chp1);
                union_set_bodies_xx(ufind, chc, chp1);
                union_set_bodies_nn(ufind, chc, chp2);
                union_set_bodies_xx(ufind, chc, chp2);
              } else if (!myOptions.isNoRGNoEdgeBFSLoopItInitCE() &&
                         myOptions.isNoRGNoEdgeBFSLoopItInitCG() &&
                         ((lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CG &&
                           lb2 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_SUB) ||
                          (lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_SUB &&
                           lb2 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CG))) {
                // bfs-loop-it-init-c clause
                //
                // G(P)
                // G(Q, F(l))
                // G(X(P, l))
                // for p in P : part(p in premise 1) == part(p in conclusion)
                // NOTE: not necessary for correctness
                // for some premise 2 : part(l in premise 2) == part(l
                // in conclusion). Note that the partitions of
                // all premises 2's eventuality literals are unioned
                // elsewhere.
                // NOTE: edge to premise 2 not in resolution graph
                //
                // G(P), s.t. G(P) -> G(Q)
                // G(X(Q, l)) generated by BFS-loop-it-init-c
                // for all p in P : part(p in premise 1) == part(p in conclusion)
                Assert(chc.isSClause(), "Not a step clause.");
                if (lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CG) {
                  Assert(chp1.isUClause() || chp1.isSClause(),
                         "Not a universal or step clause.");
                } else {
                  Assert(chp2.isUClause() || chp2.isSClause(),
                         "Not a universal or step clause.");
                }
                if (lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_SUB) {
                  Assert(chp1.isUClause() ||
                         chp1.isSClause() ||
                         (chp1.isIClause() &&
                          chp1.getIClause().getInternalRep()->size() == 0),
                         "Not a universal, step, or empty clause.");
                } else {
                  Assert(chp2.isUClause() ||
                         chp2.isSClause() ||
                         (chp2.isIClause() &&
                          chp2.getIClause().getInternalRep()->size() == 0),
                         "Not a universal, step, or empty clause.");
                }
                Partition p;
                if (lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CG) {
                  p = chp1.getPartition();
                } else {
                  p = chp2.getPartition();
                }
                Assert(myMapPartitionToLiteral.find(p) !=
                       myMapPartitionToLiteral.end(),
                       "p not contained in myMapPartitionToLiteral.");
                ClauseHandle chpe = find_eclauseh_l(myMapPartitionToLiteral[p]);
                union_set_l_x(ufind, chc, chpe);
                union_set_bodies_xn(ufind, chc, chp1);
                union_set_bodies_xn(ufind, chc, chp2);
              } else if (myOptions.isNoRGNoEdgeBFSLoopItInitCE() &&
                    ((lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CE &&
                         lb2 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_SUB) ||
                         (lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_SUB &&
                    lb2 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CE))) {
                // bfs-loop-it-init-c clause
                //
                // G(P)
                // G(Q, F(l))
                // G(X(P, l))
                // for some premise 2 : part(l in premise 2) == part(l
                // in conclusion). Note that the partitions of
                // all premises 2's eventuality literals are unioned
                // elsewhere.
                //
                // G(P), s.t. G(P) -> G(Q)
                // G(X(Q, l)) generated by BFS-loop-it-init-c
                // for all p in P : part(p in premise 1) == part(p in conclusion)
                Assert(chc.isSClause(), "Not a step clause.");
                Assert(!myOptions.isNoRGNoEdgeBFSLoopItInitCG() ||
                       chc.getSClause().getInternalRep()->size() == 1,
                       "Unexpected situation in resolution graph.");
                if (lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CE) {
                  Assert(chp1.isEClause(), "Not an eventuality clause.");
                } else {
                  Assert(chp2.isEClause(), "Not an eventuality clause.");
                }
                if (lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_SUB) {
                  Assert(chp1.isUClause() ||
                         chp1.isSClause() ||
                         (chp1.isIClause() &&
                          chp1.getIClause().getInternalRep()->size() == 0),
                         "Not a universal, step, or empty clause.");
                } else {
                  Assert(chp2.isUClause() ||
                         chp2.isSClause() ||
                         (chp2.isIClause() &&
                          chp2.getIClause().getInternalRep()->size() == 0),
                         "Not a universal, step, or empty clause.");
                }
                if (lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CE) {
                  union_set_l_x(ufind, chc, chp1);
                } else {
                  union_set_l_x(ufind, chc, chp2);
                }
                if (lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_SUB) {
                  union_set_bodies_xn(ufind, chc, chp1);
                } else {
                  union_set_bodies_xn(ufind, chc, chp2);
                }
              } else {
                Assert(myOptions.isNoRGNoEdgeBFSLoopItInitCE() &&
                       myOptions.isNoRGNoEdgeBFSLoopItInitCG() &&
                       ((lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CE &&
                         lb2 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CG) ||
                        (lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CG &&
                         lb2 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CE)),
                       "Unexpected situation in resolution graph.");
                Assert(myOptions.isNoPruneFailedLoopSearchIts(),
                       "myNoPruneFailedLoopSearchIts is false.");
                // bfs-loop-it-init-c clause
                //
                // G(P)
                // G(Q, F(l))
                // G(X(P, l))
                // for p in P : part(p in premise 1) == part(p in conclusion)
                // NOTE: not necessary for correctness
                // for some premise 2 : part(l in premise 2) == part(l
                // in conclusion). Note that the partitions of
                // all premises 2's eventuality literals are unioned
                // elsewhere.
                Assert(chc.isSClause(), "Not a step clause.");
                if (lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CE) {
                  Assert(chp1.isEClause(), "Not an eventuality clause.");
                  Assert(chp2.isUClause() ||
                         chp2.isSClause() ||
                         (chp2.isIClause() &&
                          chp2.getIClause().getInternalRep()->size() == 0),
                         "Not a universal, step, or empty clause.");
                } else {
                  Assert(chp1.isUClause() ||
                         chp1.isSClause() ||
                         (chp1.isIClause() &&
                          chp1.getIClause().getInternalRep()->size() == 0),
                         "Not a universal, step, or empty clause.");
                  Assert(chp2.isEClause(), "Not an eventuality clause.");
                }
                if (lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CE) {
                  union_set_l_x(ufind, chc, chp1);
                  union_set_bodies_xn(ufind, chc, chp2);
                } else {
                  union_set_l_x(ufind, chc, chp2);
                  union_set_bodies_xn(ufind, chc, chp1);
                }
              }
            } else {
              Assert(noEdges == 3, "Unexpected number of edges.");

              // get the edges, labels
              Edge e1 = *oot.first;
              ++oot.first;
              Edge e2 = *oot.first;
              ++oot.first;
              Edge e3 = *oot.first;
              Assert(myMapEdgeToInferenceRuleEdgeLabeling.find(e1) !=
                     myMapEdgeToInferenceRuleEdgeLabeling.end(),
                   "e1 not contained in myMapEdgeToInferenceRuleEdgeLabeling.");
              Assert(myMapEdgeToInferenceRuleEdgeLabeling.find(e2) !=
                     myMapEdgeToInferenceRuleEdgeLabeling.end(),
                   "e2 not contained in myMapEdgeToInferenceRuleEdgeLabeling.");
              Assert(myMapEdgeToInferenceRuleEdgeLabeling.find(e3) !=
                     myMapEdgeToInferenceRuleEdgeLabeling.end(),
                   "e3 not contained in myMapEdgeToInferenceRuleEdgeLabeling.");
              InferenceRuleEdgeLabeling lb1 =
                myMapEdgeToInferenceRuleEdgeLabeling[e1];
              InferenceRuleEdgeLabeling lb2 =
                myMapEdgeToInferenceRuleEdgeLabeling[e2];
#ifdef DEBUG
              InferenceRuleEdgeLabeling lb3 =
                myMapEdgeToInferenceRuleEdgeLabeling[e3];
#endif
              Vertex vp1 = boost::target(e1, myResolutionGraph);
              Vertex vp2 = boost::target(e2, myResolutionGraph);
              Vertex vp3 = boost::target(e3, myResolutionGraph);
              Assert(myMapVertexToClauseh.find(vp1) !=
                     myMapVertexToClauseh.end(),
                     "vp1 not contained in myMapVertexToClauseh.");
              Assert(myMapVertexToClauseh.find(vp2) !=
                     myMapVertexToClauseh.end(),
                     "vp2 not contained in myMapVertexToClauseh.");
              Assert(myMapVertexToClauseh.find(vp3) !=
                     myMapVertexToClauseh.end(),
                     "vp3 not contained in myMapVertexToClauseh.");
              ClauseHandle const & chp1 = myMapVertexToClauseh[vp1];
              ClauseHandle const & chp2 = myMapVertexToClauseh[vp2];
              ClauseHandle const & chp3 = myMapVertexToClauseh[vp3];

              Assert(myOptions.isNoRGNoEdgeBFSLoopItInitCE() &&
                     myOptions.isNoRGNoEdgeBFSLoopItInitCG() &&
                     ((lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CE ||
                       lb2 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CE ||
                       lb3 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CE) &&
                      (lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CG ||
                       lb2 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CG ||
                       lb3 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CG) &&
                      (lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_SUB ||
                       lb2 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_SUB ||
                       lb3 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_SUB)),
                     "Unexpected situation in resolution graph.");
              // bfs-loop-it-init-c clause
              //
              // G(P)
              // G(Q, F(l))
              // G(X(P, l))
              // for p in P : part(p in premise 1) == part(p in conclusion)
              // NOTE: not necessary for correctness
              // for some premise 2 : part(l in premise 2) == part(l
              // in conclusion). Note that the partitions of
              // all premises 2's eventuality literals are unioned
              // elsewhere.
              //
              // G(P), s.t. G(P) -> G(Q)
              // G(X(Q, l)) generated by BFS-loop-it-init-c
              // for all p in P : part(p in premise 1) == part(p in conclusion)
              Assert(chc.isSClause(), "Not a step clause.");
              if (lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CG) {
                Assert(chp1.isUClause() || chp1.isSClause(),
                       "Not a universal or step clause.");
              } else if (lb2 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CG) {
                Assert(chp2.isUClause() || chp2.isSClause(),
                       "Not a universal or step clause.");
              } else {
                Assert(chp3.isUClause() || chp3.isSClause(),
                       "Not a universal or step clause.");
              }
              if (lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CE) {
                Assert(chp1.isEClause(), "Not an eventuality clause.");
              } else if (lb2 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CE) {
                Assert(chp2.isEClause(), "Not an eventuality clause.");
              } else {
                Assert(chp3.isEClause(), "Not an eventuality clause.");
              }
              if (lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_SUB) {
                Assert(chp1.isUClause() ||
                       chp1.isSClause() ||
                       (chp1.isIClause() &&
                        chp1.getIClause().getInternalRep()->size() == 0),
                       "Not a universal, step, or empty clause.");
              } else if (lb2 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_SUB) {
                Assert(chp2.isUClause() ||
                       chp2.isSClause() ||
                       (chp2.isIClause() &&
                        chp2.getIClause().getInternalRep()->size() == 0),
                       "Not a universal, step, or empty clause.");
              } else {
                Assert(chp3.isUClause() ||
                       chp3.isSClause() ||
                       (chp3.isIClause() &&
                        chp3.getIClause().getInternalRep()->size() == 0),
                       "Not a universal, step, or empty clause.");
              }
              if (lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CE) {
                union_set_l_x(ufind, chc, chp1);
              } else if (lb2 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CE) {
                union_set_l_x(ufind, chc, chp2);
              } else {
                union_set_l_x(ufind, chc, chp3);
              }
              if (lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CG) {
                union_set_bodies_xn(ufind, chc, chp1);
              } else if (lb2 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_CG) {
                union_set_bodies_xn(ufind, chc, chp2);
              } else {
                union_set_bodies_xn(ufind, chc, chp3);
              }
              if (lb1 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_SUB) {
                union_set_bodies_xn(ufind, chc, chp1);
              } else if (lb2 == INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_SUB) {
                union_set_bodies_xn(ufind, chc, chp2);
              } else {
                union_set_bodies_xn(ufind, chc, chp3);
              }
            }
          }
        }
    }

    // Union the partitions of lchos of same eventuality literals
    {
      std::map<Literal, LCHOPartition> mapLiteralToLCHOPartition;

      ClauseHandleSet::iterator it;
      for (it = myCoreStartingClausehs.begin();
           it != myCoreStartingClausehs.end();
           ++it) {
        if (it->isEClause()) {
          Literal l = it->getEClause().getEventuality();
          LiteralClausehOccurrence lcho = LiteralClausehOccurrence(*it);
          Assert(myMapLCHOToLCHOPartition.find(lcho) !=
                 myMapLCHOToLCHOPartition.end(),
                 "lcho not contained in myMapLCHOToLCHOPartition.");
          LCHOPartition lchop = myMapLCHOToLCHOPartition[lcho];
          if (mapLiteralToLCHOPartition.find(l) ==
              mapLiteralToLCHOPartition.end()) {
            mapLiteralToLCHOPartition[l] = lchop;
          } else {
            ufind.union_set(lchop, mapLiteralToLCHOPartition[l]);
          }
        }
      }
    }

    // Replace lchops in myMapLCHOToLCHOPartition with representatives
    {
      MapLCHOToLCHOPartition::iterator it;
      for (it = myMapLCHOToLCHOPartition.begin();
           it != myMapLCHOToLCHOPartition.end();
           ++it) {
        it->second = ufind.find_set(it->second);
      }
    }

    // Map lchops to indices s.t. increase from 0 by 1 for same
    // propositions and collect statistics in reachable subgraph
    {
      std::map<Proposition, unsigned long> mapPropositionToIndex;
      MapLCHOToLCHOPartition::iterator it;
      for (it = myMapLCHOToLCHOPartition.begin();
           it != myMapLCHOToLCHOPartition.end();
           ++it) {
        LCHOPartition lchop = it->second;
        if (myMapLCHOPartitionToIndex.find(lchop) ==
            myMapLCHOPartitionToIndex.end()) {
          Proposition p = lchop.getP();
          if (mapPropositionToIndex.find(p) == mapPropositionToIndex.end()) {
            mapPropositionToIndex[p] = 0;
            ++myNoPropositionsReachableSubgraph;
          } else {
            ++mapPropositionToIndex[p];
            if (mapPropositionToIndex[p] >
                myNoPartitionsPerPropositionPeakReachableSubgraph) {
              myNoPartitionsPerPropositionPeakReachableSubgraph =
                mapPropositionToIndex[p];
            }
            if (mapPropositionToIndex[p] == 1) {
              ++myNoPartitionsPerPropositionGE2ReachableSubgraph;
            }
          }
          myMapLCHOPartitionToIndex[lchop] = mapPropositionToIndex[p];
        }
      }
      ++myNoPartitionsPerPropositionPeakReachableSubgraph;
    }

    // Collect statistics in core
    {
      std::set<Proposition> seenProposition;
      LCHOPartitionSet seenLCHOPartition;
      std::map<Proposition, unsigned long> mapPropositionToIndex;
      for (ClauseHandleSet::const_iterator it = myCoreStartingClausehs.cbegin();
           it != myCoreStartingClausehs.cend();
           ++it) {
        PClause pc = it->getPClause();
        if (it->isEClause()) {
          LiteralClausehOccurrence lcho = LiteralClausehOccurrence(*it);
          collect_statistics_in_core_lcho(lcho,
                                          seenProposition,
                                          seenLCHOPartition,
                                          mapPropositionToIndex);
        }
        for (Clause::const_iterator it2 = pc->cbegin();
             it2 != pc->cend();
             ++it2) {
          LiteralClausehOccurrence lcho = LiteralClausehOccurrence(*it, it2);
          collect_statistics_in_core_lcho(lcho,
                                          seenProposition,
                                          seenLCHOPartition,
                                          mapPropositionToIndex);
        }
      }
      ++myNoPartitionsPerPropositionPeakCore;
      ++myNoPartitionsPerPropositionPeakCoreNonNVV;
    }

    if (!myOptions.isQuiet()) {
      std::cout << "LCHOPartitioner: partitioning proposition occurrences end"
                << std::endl;
    }
  } // LCHOPartitioner::computeLCHOPartitionsSNF

  void
  LCHOPartitioner::union_set_lchos(LCHOPartitionDisjointSets & ufind,
                                   LiteralClausehOccurrence const & lcho1,
                                   LiteralClausehOccurrence const & lcho2)
  {
    TRACE(unsatcoreModule, {
        std::cout << "LCHOPartitioner::union_set_lchos: unioning partitions of ";
        lcho1.dump(std::cout);
        std::cout << " and ";
        lcho2.dump(std::cout);
        std::cout << std::endl;
      });

    Assert(myMapLCHOToLCHOPartition.find(lcho1) !=
           myMapLCHOToLCHOPartition.end(),
           "lcho1 not contained in myMapLCHOToLCHOPartition.");
    LCHOPartition lchop1 = myMapLCHOToLCHOPartition[lcho1];
    Assert(myMapLCHOToLCHOPartition.find(lcho2) !=
           myMapLCHOToLCHOPartition.end(),
           "lcho2 not contained in myMapLCHOToLCHOPartition.");
    LCHOPartition lchop2 = myMapLCHOToLCHOPartition[lcho2];
    ufind.union_set(lchop1, lchop2);
  } // LCHOPartitioner::union_set_lchos

  // Finds the l in an aug2, i.e. (G(!wl, X(l, wl))), clause
  Clause::const_iterator
  LCHOPartitioner::find_l_aug2(PClause const & pc)
  {
    Assert(pc->size() == 3, "Not an aug2 clause.");

    Clause::const_iterator it1, it2, it3, it;
    it1 = pc->cbegin();
    it2 = it1; ++it2;
    it3 = it2; ++it3;
    if (it1->getProposition() == it2->getProposition() &&
        it1->getSign() != it2->getSign()) {
      Assert(it1->getProposition() != it3->getProposition(),
             "Not an aug2 clause.");
      Assert(it1->getAttribute().getType() == Attribute::step_now ||
             it2->getAttribute().getType() == Attribute::step_now,
             "Not an aug2 clause.");
      Assert(it1->getAttribute().getType() == Attribute::step_next ||
             it2->getAttribute().getType() == Attribute::step_next,
             "Not an aug2 clause.");
      Assert(it3->getAttribute().getType() == Attribute::step_next,
             "Not an aug2 clause.");
      it = it3;
    } else if (it1->getProposition() == it3->getProposition() &&
               it1->getSign() != it3->getSign()) {
      Assert(it1->getProposition() != it2->getProposition(),
             "Not an aug2 clause.");
      Assert(it1->getAttribute().getType() == Attribute::step_now ||
             it3->getAttribute().getType() == Attribute::step_now,
             "Not an aug2 clause.");
      Assert(it1->getAttribute().getType() == Attribute::step_next ||
             it3->getAttribute().getType() == Attribute::step_next,
             "Not an aug2 clause.");
      Assert(it2->getAttribute().getType() == Attribute::step_next,
             "Not an aug2 clause.");
      it = it2;
    } else {
      Assert(it2->getProposition() == it3->getProposition() &&
             it2->getSign() != it3->getSign(),
             "Literals are not complementary.");
      Assert(it2->getProposition() != it1->getProposition(),
             "Not an aug2 clause.");
      Assert(it2->getAttribute().getType() == Attribute::step_now ||
             it3->getAttribute().getType() == Attribute::step_now,
             "Not an aug2 clause.");
      Assert(it2->getAttribute().getType() == Attribute::step_next ||
             it3->getAttribute().getType() == Attribute::step_next,
             "Not an aug2 clause.");
      Assert(it1->getAttribute().getType() == Attribute::step_next,
             "Not an aug2 clause.");
      it = it1;
    }
    return it;
  } // LCHOPartitioner::find_l_aug2

  // Performs a union_set on the lcho partitions of the first literals
  // in chp1 and chp2.
  void
  LCHOPartitioner::union_set_resolved_upon_lits(LCHOPartitionDisjointSets & ufind,
                                                ClauseHandle const & chp1,
                                                ClauseHandle const & chp2)
  {
    PClause pcp1, pcp2;
    Clause::const_iterator itp1, itp2;
    
    TRACE(unsatcoreModule, {
        std::cout << "LCHOPartitioner::union_set_resolved_upon_lits: unioning "
                  << "partitions of lchos of leading literals in" << std::endl;
        std::cout << "LCHOPartitioner::union_set_resolved_upon_lits: ";
        chp1.dump(std::cout);
        std::cout << std::endl;
        std::cout << "LCHOPartitioner::union_set_resolved_upon_lits: ";
        chp2.dump(std::cout);
        std::cout << std::endl;
      });

    Assert(chp1.isIClause() || chp1.isUClause() || chp1.isSClause(),
           "Not an init, universal, or step clause.");
    pcp1 = chp1.getPClause();
    
    Assert(chp2.isIClause() || chp2.isUClause() || chp2.isSClause(),
           "Not an init, universal, or step clause.");
    pcp2 = chp2.getPClause();
    
    itp1 = pcp1->cbegin();
    itp2 = pcp2->cbegin();

    Assert(itp1->getProposition() == itp2->getProposition() &&
           itp1->getSign() != itp2->getSign(),
           "Literals not complementary.");

    LiteralClausehOccurrence lchop1 = LiteralClausehOccurrence(chp1, itp1);
    LiteralClausehOccurrence lchop2 = LiteralClausehOccurrence(chp2, itp2);
    union_set_lchos(ufind, lchop1, lchop2);
  } // LCHOPartitioner::union_set_resolved_upon_lits

  void
  LCHOPartitioner::union_set_bodies_ii(LCHOPartitionDisjointSets & ufind,
                                       ClauseHandle const & chc,
                                       ClauseHandle const & chp)
  {
    union_set_bodies(ufind,
                     chc,
                     chp,
                     &UnsatCore::lpInit,
                     &UnsatCore::lpInit);
  } // LCHOPartitioner::union_set_bodies_ii

  void
  LCHOPartitioner::union_set_bodies_in(LCHOPartitionDisjointSets & ufind,
                                       ClauseHandle const & chc,
                                       ClauseHandle const & chp)
  {
    union_set_bodies(ufind,
                     chc,
                     chp,
                     &UnsatCore::lpInit,
                     &UnsatCore::lpNow);
  } // LCHOPartitioner::union_set_bodies_ii

  void
  LCHOPartitioner::union_set_bodies_nn(LCHOPartitionDisjointSets & ufind,
                                       ClauseHandle const & chc,
                                       ClauseHandle const & chp)
  {
    union_set_bodies(ufind,
                     chc,
                     chp,
                     &UnsatCore::lpNow,
                     &UnsatCore::lpNow);
  } // LCHOPartitioner::union_set_bodies_nn

  void
  LCHOPartitioner::union_set_bodies_xn(LCHOPartitionDisjointSets & ufind,
                                       ClauseHandle const & chc,
                                       ClauseHandle const & chp)
  {
    union_set_bodies(ufind,
                     chc,
                     chp,
                     &UnsatCore::lpNext,
                     &UnsatCore::lpNow);
  } // LCHOPartitioner::union_set_bodies_xn

  void
  LCHOPartitioner::union_set_bodies_xx(LCHOPartitionDisjointSets & ufind,
                                       ClauseHandle const & chc,
                                       ClauseHandle const & chp)
  {
    union_set_bodies(ufind,
                     chc,
                     chp,
                     &UnsatCore::lpNext,
                     &UnsatCore::lpNext);
  } // LCHOPartitioner::union_set_bodies_xx

  // For each literal in the internal rep of chp it looks for an
  // occurrence of that literal in chc and performs union_set on the
  // respective partitions. Literal attributes must return true in
  // lpc/lpp to be considered.
  void
  LCHOPartitioner::union_set_bodies(LCHOPartitionDisjointSets & ufind,
                                    ClauseHandle const & chc,
                                    ClauseHandle const & chp,
                                    LiteralPredicate const lpc,
                                    LiteralPredicate const lpp)
  {
    PClause pcc, pcp;

    TRACE(unsatcoreModule, {
        std::cout << "LCHOPartitioner::union_set_bodies: unioning partitions "
                  << "of lchos in bodies of" << std::endl;
        std::cout << "LCHOPartitioner::union_set_bodies: conclusion ";
        chc.dump(std::cout);
        std::cout << std::endl;
        std::cout << "LCHOPartitioner::union_set_bodies: premise ";
        chp.dump(std::cout);
        std::cout << std::endl;
      });

    Assert(chc.isIClause() || chc.isUClause() || chc.isSClause(),
           "Not an init, universal, or step clause.");
    pcc = chc.getPClause();

    pcp = chp.getPClause();

    // Inefficient but simple
    for (Clause::const_iterator itp = pcp->cbegin();
         itp != pcp->cend();
         ++itp) {
      if ((*lpp)(*itp)) {
        for (Clause::const_iterator itc = pcc->cbegin();
             itc != pcc->cend();
             ++itc) {
          if ((*lpc)(*itc)) {
            if (itc->getProposition() == itp->getProposition() &&
                itc->getSign() == itp->getSign()) {
              LiteralClausehOccurrence lchoc =
                LiteralClausehOccurrence(chc, itc);
              LiteralClausehOccurrence lchop =
                LiteralClausehOccurrence(chp, itp);
              union_set_lchos(ufind, lchoc, lchop);
              break;
            }
          }
        }
      }
    }
  } // LCHOPartitioner::union_set_bodies

  void LCHOPartitioner::union_set_l_n(LCHOPartitionDisjointSets & ufind,
                                      ClauseHandle const & chc,
                                      ClauseHandle const & chp)
  {
    union_set_l(ufind, chc, chp, &UnsatCore::lpNow);
  } // LCHOPartitioner::union_set_l_n

  void LCHOPartitioner::union_set_l_x(LCHOPartitionDisjointSets & ufind,
                                      ClauseHandle const & chc,
                                      ClauseHandle const & chp)
  {
    union_set_l(ufind, chc, chp, &UnsatCore::lpNext);
  } // LCHOPartitioner::union_set_l_n

  void LCHOPartitioner::union_set_l(LCHOPartitionDisjointSets & ufind,
                                    ClauseHandle const & chc,
                                    ClauseHandle const & chp,
                                    LiteralPredicate const lpc)
  {
    Assert(chc.isUClause() || chc.isSClause(),
           "Not a universal or step clause.");
    Assert(chp.isEClause(), "Not an eventuality clause.");

    TRACE(unsatcoreModule, {
        std::cout << "LCHOPartitioner::union_set_l: unioning partitions of "
                  << "lchos of l of" << std::endl;
        std::cout << "LCHOPartitioner::union_set_l: conclusion ";
        chc.dump(std::cout);
        std::cout << std::endl;
        std::cout << "LCHOPartitioner::union_set_l: premise ";
        chp.dump(std::cout);
        std::cout << std::endl;
      });

    PClause pcc;
    Literal l;
    bool foundl;

    pcc = chc.getPClause();

    l = chp.getEClause().getEventuality();
    foundl = false;
    for (Clause::const_iterator itc = pcc->cbegin();
         !foundl && itc != pcc->cend();
         ++itc) {
      if ((*lpc)(*itc)) {
        if (itc->getProposition() == l.getProposition() &&
            itc->getSign() == l.getSign()) {
          LiteralClausehOccurrence lchoc = LiteralClausehOccurrence(chc, itc);
          LiteralClausehOccurrence lchop = LiteralClausehOccurrence(chp);
          union_set_lchos(ufind, lchoc, lchop);
          foundl = true;
        }
      }
    }
    Assert(foundl, "l not found in pcc.");
  } // LCHOPartitioner::union_set_l

  ClauseHandle const & LCHOPartitioner::find_eclauseh_l(Literal const & l)
  {
    ClauseHandleSet::iterator it;

    for (it = myCoreStartingClausehs.begin();
         it != myCoreStartingClausehs.end();
         ++it) {
      if (it->isEClause()) {
        Literal le = it->getEClause().getEventuality();
        if (le.getProposition() == l.getProposition() &&
            le.getSign() == l.getSign()) {
          return *it;
        }
      }
    }
    Assert(false, "Found no eventuality clause with eventuality l.");
    return *it; // Silence compiler
  } // LCHOPartitioner::find_eclauseh_l

  void
  LCHOPartitioner::collect_statistics_in_core_lcho(
                                          LiteralClausehOccurrence const & lcho,
                                        std::set<Proposition> & seenProposition,
                                           LCHOPartitionSet & seenLCHOPartition,
                   std::map<Proposition, unsigned long> & mapPropositionToIndex)
  {
    Literal l = lcho.getLiteral();
    Proposition p = l.getProposition();
    Assert(myMapLCHOToLCHOPartition.find(lcho) !=
           myMapLCHOToLCHOPartition.end(),
           "lcho not contained in myMapLCHOToLCHOPartition.");
    LCHOPartition lchop = myMapLCHOToLCHOPartition.find(lcho)->second;
    bool isNvv = isNVVNatural(p.getName());
    ++myNoPropositionOccurrencesCore;
    if (!isNvv) {
      ++myNoPropositionOccurrencesCoreNonNVV;
    }
    if (seenProposition.find(p) == seenProposition.end()) {
      seenProposition.insert(p);
      ++myNoPropositionsCore;
      if (!isNvv) {
        ++myNoPropositionsCoreNonNVV;
      }
    }
    if (seenLCHOPartition.find(lchop) == seenLCHOPartition.end()) {
      seenLCHOPartition.insert(lchop);
      if (mapPropositionToIndex.find(p) == mapPropositionToIndex.end()) {
        mapPropositionToIndex[p] = 0;
      } else {
        ++myNoPartitionsPerPropositionGE2Core;
        if (!isNvv) {
          ++myNoPartitionsPerPropositionGE2CoreNonNVV;
        }
        ++mapPropositionToIndex[p];
        if (mapPropositionToIndex[p] >
            myNoPartitionsPerPropositionPeakCore) {
          myNoPartitionsPerPropositionPeakCore = mapPropositionToIndex[p];
        }
        if (!isNvv) {
          if (mapPropositionToIndex[p] >
              myNoPartitionsPerPropositionPeakCoreNonNVV) {
            myNoPartitionsPerPropositionPeakCoreNonNVV = mapPropositionToIndex[p];
          }
        }
      }
    }
  } // LCHOPartitioner::collect_statistics_in_core_lcho

  // returns true iff s is of the form 'NVV[0-9]+'
  bool LCHOPartitioner::isNVVNatural(std::string const & s) const
  {
    if (s.length() < 4) {
      return false;
    }
    if (s[0] != 'N' || s[1] != 'V' || s[2] != 'V') {
      return false;
    }
    for (unsigned long i = 3; i < s.length(); ++i) {
      if (s[i] < '0' || s[i] > '9') {
        return false;
      }
    }
    return true;
  } // LCHOPartitioner::isNVVNatural

  bool lpInit(Literal const & l) {
    return l.getAttribute().getType() == Attribute::initial;
  } // lpInit
  
  bool lpNow(Literal const & l) {
    return l.getAttribute().getType() == Attribute::universal ||
      l.getAttribute().getType() == Attribute::step_now;
  } // lpNow
  
  bool lpNext(Literal const & l) {
    return l.getAttribute().getType() == Attribute::step_next;
  } // lpNext
} // namespace UnsatCore
