/************************************************************
 *    Copyright (C) 2012-2014                               *
 *    Viktor Schuppan (Viktor.Schuppan@gmx.de)              *
 *                                                          *
 *    This program is free software; you can redistribute   *
 *    it and/or modify it under the terms of the GNU        *
 *    General Public License as published by the Free       *
 *    Software Foundation, either version 3 of the License, *
 *    or (at your option) any later version.                *
 *                                                          *
 *    This program is distributed in the hope that it will  *
 *    be useful, but WITHOUT ANY WARRANTY; without even     *
 *    the implied warranty of MERCHANTABILITY or FITNESS    *
 *    FOR A PARTICULAR PURPOSE.  See the GNU General Public *
 *    License for more details.                             *
 *                                                          *
 *    You should have received a copy of the GNU General    *
 *    Public License along with this program.  If not, see  *
 *    <http://www.gnu.org/licenses/>.                       *
 ************************************************************/
#include <iostream>
#include <list>
#include <map>
#include <set>
#include <string>
#include <boost/pending/disjoint_sets.hpp>
#include <boost/property_map/property_map.hpp>
#include "builder_ltl/formula.h"
#include "getopt/options.h"
#include "misc/assert.h"
#include "misc/tracer.h"
#include "modules.h"
#include "clause_handle.h"
#include "lcho_partitioner.h"
#include "literal_clauseh_occurrence.h"
#include "semi_linear_set.h"
#include "unsat_core_writer.h"
#include "unsat_core_private.h"
#include "mapper_snf_to_ltl.h"

namespace UnsatCore
{
  void
  MapperSnfToLtl::computeCoreLtl(ClauseHandleSet const & coreStartingClausehs,
                                 std::list<TreeNode*> & formulaList,
                              std::map<TreeNode*, LiteralClausehOccurrenceSet> &
                                 mapSubformulaToUseLCHOsNow,
                              std::map<TreeNode*, LiteralClausehOccurrenceSet> &
                                 mapSubformulaToUseLCHOsNext,
                              std::map<TreeNode*, LiteralClausehOccurrenceSet> &
                                 mapSubformulaToUseLCHOsSometime,
                                 LiteralClausehOccurrenceSet &
                                 setSubformulaToUseLCHOsInvalidPos,
                                MapLCHOToLCHOPartition & mapLCHOToLCHOPartition,
                                 std::map<LCHOPartition, unsigned long> &
                                 mapLCHOPartitionToIndex,
                                 std::map<ClauseHandle, SemiLinearSet> &
                                 mapClausehToSemiLinearSet,
                                 std::map<TreeNode*, SemiLinearSet> &
                                 mapSubformulaToSemiLinearSet,
                                 TRPPPOptions const & options)
  {
    MapperSnfToLtl mstl = MapperSnfToLtl(coreStartingClausehs,
                                         formulaList,
                                         mapSubformulaToUseLCHOsNow,
                                         mapSubformulaToUseLCHOsNext,
                                         mapSubformulaToUseLCHOsSometime,
                                         setSubformulaToUseLCHOsInvalidPos,
                                         mapLCHOToLCHOPartition,
                                         mapLCHOPartitionToIndex,
                                         mapClausehToSemiLinearSet,
                                         mapSubformulaToSemiLinearSet,
                                         options);
    mstl.compute_core_ltl();
  } // MapperSnfToLtl::computeCoreLtl

  void
  MapperSnfToLtl::simplifyCoreLtl(std::list<TreeNode*> & formulaList,
                                std::set<TreeNode*> & coreSimplifiedSubformulas,
                                  TRPPPOptions const & options)
  {
    Assert(options.isExtractUnsatCore(), "Unsat core extraction not enabled.");
    Assert(!options.isExtractUnsatCoreSetsOfTimePoints(),
           "Sets of time points unsat core extraction enabled.");
    Assert(!options.isExtractUnsatCorePartitionedPropositions(),
           "Partitioned propositions unsat core extraction enabled.");
    Assert(!options.isNoUnsatCoreSimplification(),
           "options.isNoUnsatCoreSimplification is true.");

    unsigned long noVerticesSyntaxTreeCoreInput = 0;
    unsigned long noVerticesSyntaxTreeCoreSimplified = 0;

    // Take implicit conjunctions into account
    noVerticesSyntaxTreeCoreInput = formulaList.size() - 1;

    for (std::list<TreeNode *>::iterator it = formulaList.begin();
         it != formulaList.end();
         )
      {
        std::list<TreeNode *>::iterator itcur = it;
        ++it;

        // Compute size before simplification
        noVerticesSyntaxTreeCoreInput += (*itcur)->size();
        
        TreeNode * tmp = *itcur;
        *itcur = simplify_core_ltl_rec(*itcur,
                                       true,
                                       coreSimplifiedSubformulas);
        if (*itcur != tmp) {
          delete(tmp);
        }

        if (!options.isNoUnsatCoreSimplification() &&
            isIdentifier(*itcur) &&
            ((Identifier *) *itcur)->isTrue() &&
            coreSimplifiedSubformulas.find(*itcur) !=
            coreSimplifiedSubformulas.end()) {
          tmp = *itcur;
          formulaList.erase(itcur);
          delete(tmp);
        } else {
          // Compute size after simplification
          noVerticesSyntaxTreeCoreSimplified += (*itcur)->size();
        }
      }

    // Take implicit conjunctions into account
    noVerticesSyntaxTreeCoreSimplified += formulaList.size() - 1;

    // Statistics
    std::cout << "MapperSnfToLtl: number of nodes in syntax tree of "
              << "input unsatisfiable core: "
              << noVerticesSyntaxTreeCoreInput
              << std::endl;
    std::cout << "MapperSnfToLtl: number of nodes in syntax tree of "
              << "simplified unsatisfiable core: "
              << noVerticesSyntaxTreeCoreSimplified
              << std::endl;
  } // MapperSnfToLtl::simplifyCoreLtl

  MapperSnfToLtl::MapperSnfToLtl(ClauseHandleSet const & coreStartingClausehs,
                                 std::list<TreeNode*> & formulaList,
                                 std::map<TreeNode*, LiteralClausehOccurrenceSet> &
                                 mapSubformulaToUseLCHOsNow,
                                 std::map<TreeNode*, LiteralClausehOccurrenceSet> &
                                 mapSubformulaToUseLCHOsNext,
                                 std::map<TreeNode*, LiteralClausehOccurrenceSet> &
                                 mapSubformulaToUseLCHOsSometime,
                                 LiteralClausehOccurrenceSet &
                                 setSubformulaToUseLCHOsInvalidPos,
                                 MapLCHOToLCHOPartition & mapLCHOToLCHOPartition,
                                 std::map<LCHOPartition, unsigned long> &
                                 mapLCHOPartitionToIndex,
                                 std::map<ClauseHandle, SemiLinearSet> &
                                 mapClausehToSemiLinearSet,
                                 std::map<TreeNode*, SemiLinearSet> &
                                 mapSubformulaToSemiLinearSet,
                                 TRPPPOptions const & options)
    : myCoreStartingClausehs(coreStartingClausehs),
      myFormulaList(formulaList),
      myMapSubformulaToUseLCHOsNow(mapSubformulaToUseLCHOsNow),
      myMapSubformulaToUseLCHOsNext(mapSubformulaToUseLCHOsNext),
      myMapSubformulaToUseLCHOsSometime(mapSubformulaToUseLCHOsSometime),
      mySetSubformulaToUseLCHOsInvalidPos(setSubformulaToUseLCHOsInvalidPos),
      myMapLCHOToLCHOPartition(mapLCHOToLCHOPartition),
      myMapLCHOPartitionToIndex(mapLCHOPartitionToIndex),
      myMapClausehToSemiLinearSet(mapClausehToSemiLinearSet),
      myMapSubformulaToSemiLinearSet(mapSubformulaToSemiLinearSet),
      myMapLCHOPartitionToIdentifier(),
      myMapPropositionToLCHOPartitionSet(),
      myOptions(options),
      myNoPropositionsLtlCore(0),
      myNoPropositionOccurrencesLtlCore(0),
      myNoPartitionsPerPropositionPeakLtlCore(0),
      myNoPartitionsPerPropositionGE2LtlCore(0)
  { } // MapperSnfToLtl::MapperSnfToLtl

  MapperSnfToLtl::~MapperSnfToLtl()
  {
    for (std::map<LCHOPartition, Identifier *>::iterator it =
           myMapLCHOPartitionToIdentifier.begin();
         it != myMapLCHOPartitionToIdentifier.end();
         ) {
      std::map<LCHOPartition, Identifier *>::iterator itcur = it;
      ++it;
      Identifier * tmp = itcur->second;
      myMapLCHOPartitionToIdentifier.erase(itcur);
      delete(tmp);
    }
  } // MapperSnfToLtl::~MapperSnfToLtl

  void
  MapperSnfToLtl::compute_core_ltl()
  {
    Assert(myOptions.isExtractUnsatCore(),
           "Unsat core extraction not enabled.");

    std::set<TreeNode*> coreSimplifiedSubformulas;
    unsigned long noVerticesSyntaxTreeInputProblem = 0;
    unsigned long noVerticesSyntaxTreeUnsatCore = 0;

    if (myOptions.isExtractUnsatCorePartitionedPropositions()) {
      compute_core_partitioned_ltl_prepare();
      TRACE(unsatcoreModule, {
          dump_maps_sets_subformula_to_use_lchos(std::cout);
          dump_myMapLCHOToLCHOPartition(std::cout);
        });
    }

    // Take implicit conjunctions into account
    noVerticesSyntaxTreeInputProblem = myFormulaList.size() - 1;

    for (std::list<TreeNode *>::iterator it = myFormulaList.begin();
         it != myFormulaList.end();
         )
      {
        std::list<TreeNode *>::iterator itcur = it;
        ++it;

        // Compute size before core computation
        noVerticesSyntaxTreeInputProblem += (*itcur)->size();
        
        if (!myOptions.isExtractUnsatCoreSetsOfTimePoints()) {
          TreeNode * tmp = *itcur;
          *itcur = compute_core_simple_ltl_rec(*itcur,
                                               true,
                                               coreSimplifiedSubformulas);
          if (tmp != *itcur) {
            delete(tmp);
          }
          if (!myOptions.isNoUnsatCoreSimplification()) {
            tmp = *itcur;
            *itcur = simplify_core_ltl_rec(*itcur,
                                           true,
                                           coreSimplifiedSubformulas);
            if (tmp != *itcur) {
              delete(tmp);
            }
          }
        } else if (myOptions.isExtractUnsatCoreSetsOfTimePoints() &&
                   !myOptions.isExtractUnsatCorePartitionedPropositions()) {
          Assert(myOptions.getSharing() == SHARING_NONE ||
                 myOptions.getSharing() == SHARING_SAMEPOLARITY,
                 "Sharing is not none or samepolarity.");
          TreeNode * tmp = *itcur;
          *itcur =
            compute_core_sets_of_time_points_ltl_rec(*itcur,
                                                     true,
                                                     coreSimplifiedSubformulas);
          if (tmp != *itcur) {
            delete(tmp);
          }
        } else {
          Assert(false, "False.");
        }

        if (!myOptions.isNoUnsatCoreSimplification() &&
            isIdentifier(*itcur) &&
            ((Identifier *) *itcur)->isTrue() &&
            coreSimplifiedSubformulas.find(*itcur) !=
            coreSimplifiedSubformulas.end()) {
          TreeNode * tmp = *itcur;
          myFormulaList.erase(itcur);
          delete(tmp);
        } else {
          // Compute size after core computation
          noVerticesSyntaxTreeUnsatCore += (*itcur)->size();
        }
      }

    // Take implicit conjunctions into account
    noVerticesSyntaxTreeUnsatCore += myFormulaList.size() - 1;

    // Statistics
    std::cout << "MapperSnfToLtl: number of nodes in syntax tree of input "
              << "problem: "
              << noVerticesSyntaxTreeInputProblem
              << std::endl;
    std::cout << "MapperSnfToLtl: number of nodes in syntax tree of "
              << "unsatisfiable core: "
              << noVerticesSyntaxTreeUnsatCore
              << std::endl;
    if (myOptions.isExtractUnsatCorePartitionedPropositions()) {
      for (std::map<Proposition, LCHOPartitionSet>::iterator it =
             myMapPropositionToLCHOPartitionSet.begin();
           it != myMapPropositionToLCHOPartitionSet.end();
           ++it) {
        if (it->second.size() > myNoPartitionsPerPropositionPeakLtlCore) {
          myNoPartitionsPerPropositionPeakLtlCore = it->second.size();
        }
        if (it->second.size() >= 2) {
          ++myNoPartitionsPerPropositionGE2LtlCore;
        }
      }
      std::cout << "MapperSnfToLtl: number of propositions in LTL core: "
                << myNoPropositionsLtlCore
                << std::endl;
      std::cout << "MapperSnfToLtl: number of proposition occurrences in LTL "
                << "core: "
                << myNoPropositionOccurrencesLtlCore
                << std::endl;
      std::cout << "MapperSnfToLtl: peak number of partitions per proposition "
                << "in LTL core: "
                << myNoPartitionsPerPropositionPeakLtlCore
                << std::endl;
      std::cout << "MapperSnfToLtl: number of propositions with 2 or more "
                << "partitions in LTL core: "
                << myNoPartitionsPerPropositionGE2LtlCore
                << std::endl;
    }
  } // MapperSnfToLtl::compute_core_ltl

  TreeNode *
  MapperSnfToLtl::compute_core_simple_ltl_rec(TreeNode * n,
                                              bool isPositivePolarity,
                                std::set<TreeNode*> & coreSimplifiedSubformulas)
  {
    TreeNode * res;

    if (!has_uses_uc(n)) {
      if (isPositivePolarity) {
        res = new Identifier(true);
      } else {
        res = new Identifier(false);
      }
      if (!myOptions.isNoUnsatCoreSimplification()) {
        (void) coreSimplifiedSubformulas.insert(res);
      }
      TRACE(unsatcoreModule, {
          std::cout << "MapperSnfToLtl::compute_core_simple_ltl_rec: replaced ";
          std::cout << n;
          std::cout << " with ";
          std::cout << res;
          std::cout << std::endl;
        });
    } else if (myOptions.isExtractUnsatCorePartitionedPropositions() &&
               isIdentifier(n) &&
               !((Identifier *) n)->isTrue() &&
               !((Identifier *) n)->isFalse()) {
      Assert(has_valid_uses_uc(n), "n has no valid uses in uc.");
      LiteralClausehOccurrence lcho = *get_valid_uses_uc(n).begin();
      Assert(myMapLCHOToLCHOPartition.find(lcho) !=
             myMapLCHOToLCHOPartition.end(),
             "lcho not contained in myMapLCHOToLCHOPartition.");
      LCHOPartition lchop = myMapLCHOToLCHOPartition[lcho];
      if (myMapLCHOPartitionToIdentifier.find(lchop) ==
          myMapLCHOPartitionToIdentifier.end()) {
        Proposition p = lchop.getP();
        Assert(myMapLCHOPartitionToIndex.find(lchop) !=
               myMapLCHOPartitionToIndex.end(),
               "lchop not contained in myMapLCHOPartitionToIndex.");
        unsigned long i = myMapLCHOPartitionToIndex[lchop];
        std::string partition_var_suffix =
          UnsatCoreWriter::getUnusedVarInfix(myCoreStartingClausehs,
                                             "HASH");
        std::string id =
          p.getName() + partition_var_suffix + std::to_string(i);
        myMapLCHOPartitionToIdentifier[lchop] =
          new Identifier(id, TreeNodeList());

        // statistics
        if (myMapPropositionToLCHOPartitionSet.find(p) ==
            myMapPropositionToLCHOPartitionSet.end()) {
          myMapPropositionToLCHOPartitionSet[p] = LCHOPartitionSet();
        }
        myMapPropositionToLCHOPartitionSet[p].insert(lchop);
        ++myNoPropositionsLtlCore;
      }
      res = myMapLCHOPartitionToIdentifier[lchop]->clone();

      // statistics
      ++myNoPropositionOccurrencesLtlCore;

      TRACE(unsatcoreModule, {
          std::cout << "MapperSnfToLtl::compute_core_simple_ltl_rec: replaced ";
          std::cout << n;
          std::cout << " with ";
          std::cout << res;
          std::cout << std::endl;
        });
    } else {
      for (TreeNodeList::const_iterator it = n->children().begin();
           it != n->children().end();
           ) {
        TreeNodeList::const_iterator itcur = it;
        ++it;
        if (isNot(n)) {
          TreeNode * tmp = *itcur;
          n->replaceChild(*itcur,
                          compute_core_simple_ltl_rec(*itcur,
                                                      !isPositivePolarity,
                                                    coreSimplifiedSubformulas));
          if (tmp != *itcur) {
            delete(tmp);
          }
        } else if (isImplication(n) && *itcur == n->firstChild()) {
          TreeNode * tmp = *itcur;
          n->replaceChild(*itcur,
                          compute_core_simple_ltl_rec(*itcur,
                                                      !isPositivePolarity,
                                                    coreSimplifiedSubformulas));
          if (tmp != *itcur) {
            delete(tmp);
          }
        } else {
          Assert(!isEquivalence(n), "Unexpected equivalence operator.");
          TreeNode * tmp = *itcur;
          n->replaceChild(*itcur,
                          compute_core_simple_ltl_rec(*itcur,
                                                      isPositivePolarity,
                                                    coreSimplifiedSubformulas));
          if (tmp != *itcur) {
            delete(tmp);
          }
        }
      }
      res = n;
    }

    return res;
  } // MapperSnfToLtl::compute_core_simple_ltl_rec

  TreeNode *
  MapperSnfToLtl::compute_core_sets_of_time_points_ltl_rec(TreeNode * n,
                                                        bool isPositivePolarity,
                                std::set<TreeNode*> & coreSimplifiedSubformulas)
  {
    TreeNode * res;

    // Form union of all semi-linear sets associated to n
    SemiLinearSet sls = SemiLinearSet();
    if (myMapSubformulaToUseLCHOsNow.find(n) !=
        myMapSubformulaToUseLCHOsNow.end()) {
      LiteralClausehOccurrenceSet lchos = myMapSubformulaToUseLCHOsNow[n];
      for (LiteralClausehOccurrenceSet::iterator it = lchos.begin();
           it != lchos.end();
           ++it)
        {
          if (myMapClausehToSemiLinearSet.find(it->getClauseh()) !=
              myMapClausehToSemiLinearSet.end()) {
            SemiLinearSet sls2 = myMapClausehToSemiLinearSet[it->getClauseh()];
            (void) sls.insert(sls2.begin(), sls2.end());
          }
        }
    }
    if (myMapSubformulaToUseLCHOsNext.find(n) !=
        myMapSubformulaToUseLCHOsNext.end()) {
      LiteralClausehOccurrenceSet lchos = myMapSubformulaToUseLCHOsNext[n];
      for (LiteralClausehOccurrenceSet::iterator it = lchos.begin();
           it != lchos.end();
           ++it)
        {
          if (myMapClausehToSemiLinearSet.find(it->getClauseh()) !=
              myMapClausehToSemiLinearSet.end()) {
            SemiLinearSet sls2 = myMapClausehToSemiLinearSet[it->getClauseh()];
            sls2.toNext();
            (void) sls.insert(sls2.begin(), sls2.end());
          }
        }
    }
    if (myMapSubformulaToUseLCHOsSometime.find(n) !=
        myMapSubformulaToUseLCHOsSometime.end()) {
      LiteralClausehOccurrenceSet lchos = myMapSubformulaToUseLCHOsSometime[n];
      for (LiteralClausehOccurrenceSet::iterator it = lchos.begin();
           it != lchos.end();
           ++it)
        {
          if (myMapClausehToSemiLinearSet.find(it->getClauseh()) !=
              myMapClausehToSemiLinearSet.end()) {
            SemiLinearSet sls2 = myMapClausehToSemiLinearSet[it->getClauseh()];
            sls2.toSometime();
            (void) sls.insert(sls2.begin(), sls2.end());
          }
        }
    }
    sls.reduce();

    if (sls.empty()) {
      if (isPositivePolarity) {
        res = new Identifier(true);
      } else {
        res = new Identifier(false);
      }
      if (!myOptions.isNoUnsatCoreSimplification()) {
        (void) coreSimplifiedSubformulas.insert(res);
      }
    } else {
      for (TreeNodeList::const_iterator it = n->children().begin();
           it != n->children().end();
           )
        {
          TreeNodeList::const_iterator itcur = it;
          ++it;
          if (isNot(n)) {
            TreeNode * tmp = *itcur;
            n->replaceChild(*itcur,
                            compute_core_sets_of_time_points_ltl_rec(*itcur,
                                                            !isPositivePolarity,
                                                    coreSimplifiedSubformulas));
            if (tmp != *itcur) {
              delete(tmp);
            }
          } else if (isImplication(n) && *itcur == n->firstChild()) {
            TreeNode * tmp = *itcur;
            n->replaceChild(*itcur,
                            compute_core_sets_of_time_points_ltl_rec(*itcur,
                                                            !isPositivePolarity,
                                                    coreSimplifiedSubformulas));
            if (tmp != *itcur) {
              delete(tmp);
            }
          } else {
            Assert(!isEquivalence(n), "Unexpected equivalence operator.");
            TreeNode * tmp = *itcur;
            n->replaceChild(*itcur,
                            compute_core_sets_of_time_points_ltl_rec(*itcur,
                                                             isPositivePolarity,
                                                    coreSimplifiedSubformulas));
            if (tmp != *itcur) {
              delete(tmp);
            }
          }
        }
      res = n;
    }

    myMapSubformulaToSemiLinearSet[res] = sls;

    return res;
  } // MapperSnfToLtl::compute_core_sets_of_time_points_ltl_rec

  void MapperSnfToLtl::compute_core_partitioned_ltl_prepare()
  {
    // Prepare union find data structure
    //
    // Copied from lcho_partitioner.cc
    LCHOPartitioner::MapLCHOPartitionToSizeT rankm;
    LCHOPartitioner::MapLCHOPartitionToLCHOPartition parentm;
    LCHOPartitioner::BapmMapLCHOPartitionToSizeT rankpm(rankm);
    LCHOPartitioner::BapmMapLCHOPartitionToLCHOPartition parentpm(parentm);
    LCHOPartitioner::LCHOPartitionDisjointSets ufind(rankpm, parentpm);

    // Put each relevant partition in a singleton set
    for (std::list<TreeNode *>::iterator it = myFormulaList.begin();
         it != myFormulaList.end();
         ++it){
      compute_core_partitioned_ltl_prepare_uf_make_set(ufind, *it);
    }

    // Union sets for partitions of uses
    for (std::list<TreeNode *>::iterator it = myFormulaList.begin();
         it != myFormulaList.end();
         ++it){
      compute_core_partitioned_ltl_prepare_uf_union_set(ufind, *it);
    }

    // Replace partitions of uses with representatives
    for (std::list<TreeNode *>::iterator it = myFormulaList.begin();
         it != myFormulaList.end();
         ++it){
      compute_core_partitioned_ltl_prepare_uf_find_set(ufind, *it);
    }
  } // MapperSnfToLtl::compute_core_partitioned_ltl_prepare

  void MapperSnfToLtl::compute_core_partitioned_ltl_prepare_uf_make_set(
                             LCHOPartitioner::LCHOPartitionDisjointSets & ufind,
                                                                   TreeNode * n)
  {
    if (has_uses_uc(n)) {
      if (isIdentifier(n) &&
          !((Identifier *) n)->isTrue() &&
          !((Identifier *) n)->isFalse()) {
        LiteralClausehOccurrenceSet lchos = get_valid_uses_uc(n);
        for (LiteralClausehOccurrenceSet::iterator it = lchos.begin();
             it != lchos.end();
             ++it) {
          Assert(myMapLCHOToLCHOPartition.find(*it) !=
                 myMapLCHOToLCHOPartition.end(),
                 "*it not contained in myMapLCHOToLCHOPartition.");
          LCHOPartition lchop = myMapLCHOToLCHOPartition[*it];
          ufind.make_set(lchop);
        }
      } else {
        for (TreeNodeList::const_iterator it = n->children().begin();
             it != n->children().end();
             ++it) {
          compute_core_partitioned_ltl_prepare_uf_make_set(ufind, *it);
        }
      }
    }
  } // MapperSnfToLtl::compute_core_partitioned_ltl_prepare_uf_make_set

  void MapperSnfToLtl::compute_core_partitioned_ltl_prepare_uf_union_set(
                             LCHOPartitioner::LCHOPartitionDisjointSets & ufind,
                                                                   TreeNode * n)
  {
    if (has_uses_uc(n)) {
      if (isIdentifier(n) &&
          !((Identifier *) n)->isTrue() &&
          !((Identifier *) n)->isFalse()) {
        LiteralClausehOccurrenceSet lchos = get_valid_uses_uc(n);
        if (!lchos.empty()) {
          LiteralClausehOccurrenceSet::iterator it = lchos.begin();
          Assert(myMapLCHOToLCHOPartition.find(*it) !=
                 myMapLCHOToLCHOPartition.end(),
                 "*it not contained in myMapLCHOToLCHOPartition.");
          LCHOPartition lchop = myMapLCHOToLCHOPartition[*it];
          ++it;
          for (;
               it != lchos.end();
               ++it) {
            Assert(myMapLCHOToLCHOPartition.find(*it) !=
                   myMapLCHOToLCHOPartition.end(),
                   "*it not contained in myMapLCHOToLCHOPartition.");
            LCHOPartition lchoq = myMapLCHOToLCHOPartition[*it];
            ufind.union_set(lchop, lchoq);
          }
        }
      } else {
        for (TreeNodeList::const_iterator it = n->children().begin();
             it != n->children().end();
             ++it) {
          compute_core_partitioned_ltl_prepare_uf_union_set(ufind, *it);
        }
      }
    }
  } // MapperSnfToLtl::compute_core_partitioned_ltl_prepare_uf_union_set

  void MapperSnfToLtl::compute_core_partitioned_ltl_prepare_uf_find_set(
                             LCHOPartitioner::LCHOPartitionDisjointSets & ufind,
                                                                   TreeNode * n)
  {
    if (has_uses_uc(n)) {
      if (isIdentifier(n) &&
          !((Identifier *) n)->isTrue() &&
          !((Identifier *) n)->isFalse()) {
        LiteralClausehOccurrenceSet lchos = get_valid_uses_uc(n);
        for (LiteralClausehOccurrenceSet::iterator it = lchos.begin();
             it != lchos.end();
             ++it) {
          Assert(myMapLCHOToLCHOPartition.find(*it) !=
                 myMapLCHOToLCHOPartition.end(),
                 "*it not contained in myMapLCHOToLCHOPartition.");
          myMapLCHOToLCHOPartition[*it] =
            ufind.find_set(myMapLCHOToLCHOPartition[*it]);
        }
      } else {
        for (TreeNodeList::const_iterator it = n->children().begin();
             it != n->children().end();
             ++it) {
          compute_core_partitioned_ltl_prepare_uf_find_set(ufind, *it);
        }
      }
    }
  } // MapperSnfToLtl::compute_core_partitioned_ltl_prepare_uf_find_set

  bool
  MapperSnfToLtl::has_uses_uc(TreeNode * n)
  {
    if (myMapSubformulaToUseLCHOsNow.find(n) !=
        myMapSubformulaToUseLCHOsNow.end()) {
      LiteralClausehOccurrenceSet lchos = myMapSubformulaToUseLCHOsNow[n];
      for (LiteralClausehOccurrenceSet::iterator it = lchos.begin();
           it != lchos.end();
           ++it)
        {
          if (myCoreStartingClausehs.find(it->getClauseh()) !=
              myCoreStartingClausehs.end()) {
            return true;
          }
        }
    }
    if (myMapSubformulaToUseLCHOsNext.find(n) !=
        myMapSubformulaToUseLCHOsNext.end()) {
      LiteralClausehOccurrenceSet lchos = myMapSubformulaToUseLCHOsNext[n];
      for (LiteralClausehOccurrenceSet::iterator it = lchos.begin();
           it != lchos.end();
           ++it)
        {
          if (myCoreStartingClausehs.find(it->getClauseh()) !=
              myCoreStartingClausehs.end()) {
            return true;
          }
        }
    }
    if (myMapSubformulaToUseLCHOsSometime.find(n) !=
        myMapSubformulaToUseLCHOsSometime.end()) {
      LiteralClausehOccurrenceSet lchos = myMapSubformulaToUseLCHOsSometime[n];
      for (LiteralClausehOccurrenceSet::iterator it = lchos.begin();
           it != lchos.end();
           ++it)
        {
          if (myCoreStartingClausehs.find(it->getClauseh()) !=
              myCoreStartingClausehs.end()) {
            return true;
          }
        }
    }
    return false;
  } // MapperSnfToLtl::has_uses_uc

  bool
  MapperSnfToLtl::has_valid_uses_uc(TreeNode * n)
  {
    if (myMapSubformulaToUseLCHOsNow.find(n) !=
        myMapSubformulaToUseLCHOsNow.end()) {
      for (LiteralClausehOccurrenceSet::iterator it =
             myMapSubformulaToUseLCHOsNow[n].begin();
           it != myMapSubformulaToUseLCHOsNow[n].end();
           ++it) {
        if (myCoreStartingClausehs.find(it->getClauseh()) !=
            myCoreStartingClausehs.end() &&
            mySetSubformulaToUseLCHOsInvalidPos.find(*it) ==
            mySetSubformulaToUseLCHOsInvalidPos.end()) {
          return true;
        }
      }
    }
    if (myMapSubformulaToUseLCHOsNext.find(n) !=
        myMapSubformulaToUseLCHOsNext.end()) {
      for (LiteralClausehOccurrenceSet::iterator it =
             myMapSubformulaToUseLCHOsNext[n].begin();
           it != myMapSubformulaToUseLCHOsNext[n].end();
           ++it) {
        if (myCoreStartingClausehs.find(it->getClauseh()) !=
            myCoreStartingClausehs.end() &&
            mySetSubformulaToUseLCHOsInvalidPos.find(*it) ==
            mySetSubformulaToUseLCHOsInvalidPos.end()) {
          return true;
        }
      }
    }
    if (myMapSubformulaToUseLCHOsSometime.find(n) !=
        myMapSubformulaToUseLCHOsSometime.end()) {
      for (LiteralClausehOccurrenceSet::iterator it =
             myMapSubformulaToUseLCHOsSometime[n].begin();
           it != myMapSubformulaToUseLCHOsSometime[n].end();
           ++it) {
        if (myCoreStartingClausehs.find(it->getClauseh()) !=
            myCoreStartingClausehs.end() &&
            mySetSubformulaToUseLCHOsInvalidPos.find(*it) ==
            mySetSubformulaToUseLCHOsInvalidPos.end()) {
          return true;
        }
      }
    }
    return false;
  } // MapperSnfToLtl::has_valid_uses_uc

  LiteralClausehOccurrenceSet
  MapperSnfToLtl::get_valid_uses_uc(TreeNode * n)
  {
    LiteralClausehOccurrenceSet lchos;
    if (myMapSubformulaToUseLCHOsNow.find(n) !=
        myMapSubformulaToUseLCHOsNow.end()) {
      for (LiteralClausehOccurrenceSet::iterator it =
             myMapSubformulaToUseLCHOsNow[n].begin();
           it != myMapSubformulaToUseLCHOsNow[n].end();
           ++it) {
        if (myCoreStartingClausehs.find(it->getClauseh()) !=
            myCoreStartingClausehs.end() &&
            mySetSubformulaToUseLCHOsInvalidPos.find(*it) ==
            mySetSubformulaToUseLCHOsInvalidPos.end()) {
          (void) lchos.insert(*it);
        }
      }
    }
    if (myMapSubformulaToUseLCHOsNext.find(n) !=
        myMapSubformulaToUseLCHOsNext.end()) {
      for (LiteralClausehOccurrenceSet::iterator it =
             myMapSubformulaToUseLCHOsNext[n].begin();
           it != myMapSubformulaToUseLCHOsNext[n].end();
           ++it) {
        if (myCoreStartingClausehs.find(it->getClauseh()) !=
            myCoreStartingClausehs.end() &&
            mySetSubformulaToUseLCHOsInvalidPos.find(*it) ==
            mySetSubformulaToUseLCHOsInvalidPos.end()) {
          (void) lchos.insert(*it);
        }
      }
    }
    if (myMapSubformulaToUseLCHOsSometime.find(n) !=
        myMapSubformulaToUseLCHOsSometime.end()) {
      for (LiteralClausehOccurrenceSet::iterator it =
             myMapSubformulaToUseLCHOsSometime[n].begin();
           it != myMapSubformulaToUseLCHOsSometime[n].end();
           ++it) {
        if (myCoreStartingClausehs.find(it->getClauseh()) !=
            myCoreStartingClausehs.end() &&
            mySetSubformulaToUseLCHOsInvalidPos.find(*it) ==
            mySetSubformulaToUseLCHOsInvalidPos.end()) {
          (void) lchos.insert(*it);
        }
      }
    }
    return lchos;
  } // MapperSnfToLtl::get_valid_uses_uc

  TreeNode * MapperSnfToLtl::simplify_core_ltl_rec(TreeNode * n,
                                                   bool isPositivePolarity,
                                std::set<TreeNode*> & coreSimplifiedSubformulas)
  {
    TreeNode * res = NULL;
    
    // simplify children
    {
      for (TreeNodeList::const_iterator it = n->children().begin();
           it != n->children().end();
           )
        {
          TreeNodeList::const_iterator itcur = it;
          ++it;
          TreeNode * childSimplified;
          if (isNot(n)) {
            childSimplified = simplify_core_ltl_rec(*itcur,
                                                    !isPositivePolarity,
                                                    coreSimplifiedSubformulas);
            if (*itcur != childSimplified) {
              TreeNode * tmp = *itcur;
              n->replaceChild(*itcur, childSimplified);
              erase_tn_from_set_rec(tmp, coreSimplifiedSubformulas);
              delete(tmp);
            }
          } else if (isImplication(n) && *itcur == n->firstChild()) {
            childSimplified = simplify_core_ltl_rec(*itcur,
                                                    !isPositivePolarity,
                                                    coreSimplifiedSubformulas);
            if (*itcur != childSimplified) {
              TreeNode * tmp = *itcur;
              n->replaceChild(*itcur, childSimplified);
              erase_tn_from_set_rec(tmp, coreSimplifiedSubformulas);
              delete(tmp);
            }
          } else {
            Assert(!isEquivalence(n), "Unexpected equivalence operator.");
            childSimplified = simplify_core_ltl_rec(*itcur,
                                                    isPositivePolarity,
                                                    coreSimplifiedSubformulas);
            if (*itcur != childSimplified) {
              TreeNode * tmp = *itcur;
              n->replaceChild(*itcur, childSimplified);
              erase_tn_from_set_rec(tmp, coreSimplifiedSubformulas);
              delete(tmp);
            }
          }
        }
    }

    // simplify self
    if (isIdentifier(n)) {
      Assert(n->childrenCount() == 0, "Unexpected number of children.");
      res = n;
    } else if (isNot(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      res = n;
    } else if (isAnd(n)) {
      Assert(n->childrenCount() > 0, "Unexpected number of children.");

      // remove True
      {
        for (TreeNodeList::const_iterator it = n->children().begin();
             it != n->children().end();
             )
          {
            TreeNodeList::const_iterator itcur = it;
            ++it;
            if (isIdentifier(*itcur) &&
                ((Identifier *) *itcur)->isTrue() &&
                coreSimplifiedSubformulas.find(*itcur) !=
                coreSimplifiedSubformulas.end()) {
              TreeNode * tmp = *itcur;
              n->removeChild(*itcur);
              delete(tmp);
            }
          }
        Assert(n->childrenCount() > 0, "Unexpected number of children.");
      }

      // single child
      if (n->childrenCount() == 1) {
        res = n->firstChild()->clone();
        TreeNode * tmp = n->firstChild();
        n->removeChild(n->firstChild());
        delete(tmp);
        return res;
      }
      
      res = n;
    } else if (isOr(n)) {
      Assert(n->childrenCount() > 0, "Unexpected number of children.");

      // remove False
      {
        for (TreeNodeList::const_iterator it = n->children().begin();
             it != n->children().end();
             )
          {
            TreeNodeList::const_iterator itcur = it;
            ++it;
            if (isIdentifier(*itcur) &&
                ((Identifier *) *itcur)->isFalse() &&
                coreSimplifiedSubformulas.find(*itcur) !=
                coreSimplifiedSubformulas.end()) {
              TreeNode * tmp = *itcur;
              n->removeChild(*itcur);
              delete(tmp);
            }
          }
        Assert(n->childrenCount() > 0, "Unexpected number of children.");
      }

      // single child
      if (n->childrenCount() == 1) {
        res = n->firstChild()->clone();
        TreeNode * tmp = n->firstChild();
        n->removeChild(n->firstChild());
        delete(tmp);
        return res;
      }
      
      res = n;
    } else if (isImplication(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      TreeNode * fst = n->firstChild();
      TreeNode * snd = n->secondChild();
      if (isIdentifier(fst) &&
          ((Identifier *) fst)->isTrue() &&
          coreSimplifiedSubformulas.find(fst) !=
          coreSimplifiedSubformulas.end()) {
        res = snd->clone();
        n->removeChild(snd);
        delete(snd);
        return res;
      } else if (isIdentifier(snd) &&
                 ((Identifier *) snd)->isFalse() &&
                 coreSimplifiedSubformulas.find(snd) !=
                 coreSimplifiedSubformulas.end()) {
        res = Not(fst->clone());
        n->removeChild(fst);
        delete(fst);
        return res;
      }
      res = n;
    } else if (isNext(n) || isAlways(n) || isSometime(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      res = n;
    } else if (isUntil(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      TreeNode * fst = n->firstChild();
      TreeNode * snd = n->secondChild();
      if (isIdentifier(fst) &&
          ((Identifier *) fst)->isTrue() &&
          coreSimplifiedSubformulas.find(fst) !=
          coreSimplifiedSubformulas.end()) {
        ((Operator *) n)->setOperator(SOMETIME);
        n->removeChild(fst);
        delete(fst);
        res = n;
        return res;
      } else if (isIdentifier(fst) &&
                 ((Identifier *) fst)->isFalse() &&
                 coreSimplifiedSubformulas.find(fst) !=
                 coreSimplifiedSubformulas.end()) {
        res = snd->clone();
        n->removeChild(snd);
        delete(snd);
        return res;
      }
      res = n;
    } else if (isUnless(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      TreeNode * fst = n->firstChild();
      TreeNode * snd = n->secondChild();
      if (isIdentifier(snd) &&
          ((Identifier *) snd)->isFalse() &&
          coreSimplifiedSubformulas.find(snd) !=
          coreSimplifiedSubformulas.end()) {
        ((Operator *) n)->setOperator(ALWAYS);
        n->removeChild(snd);
        delete(snd);
        res = n;
        return res;
      } else if (isIdentifier(fst) &&
                 ((Identifier *) fst)->isFalse() &&
                 coreSimplifiedSubformulas.find(fst) !=
                 coreSimplifiedSubformulas.end()) {
        res = snd->clone();
        n->removeChild(snd);
        delete(snd);
        return res;
      }
      res = n;
    } else {
      Assert(false, "n has unexpected type.");
    }

    return res;
  } // MapperSnfToLtl::simplify_core_ltl_rec

  void
  MapperSnfToLtl::erase_tn_from_set_rec(TreeNode * n,
                                        std::set<TreeNode*> & s)
  {
    for (TreeNodeList::const_iterator it = n->children().begin();
         it != n->children().end();
         ++it) {
      erase_tn_from_set_rec(*it, s);
    }
    if (s.find(n) != s.end()) {
      (void) s.erase(n);
    }
  } // MapperSnfToLtl::erase_tn_from_set_rec

  std::ostream &
  MapperSnfToLtl::dump_maps_sets_subformula_to_use_lchos(std::ostream & os)
    const
  {
    std::map<TreeNode*, LiteralClausehOccurrenceSet>::iterator mapit;
    LiteralClausehOccurrenceSet::iterator setit;

    // myMapSubformulaToUseLCHOsNow
    os << "MapperSnfToLtl: dumping myMapSubformulaToUseLCHOsNow begin"
       << std::endl;
    for (mapit = myMapSubformulaToUseLCHOsNow.begin();
         mapit != myMapSubformulaToUseLCHOsNow.end();
         ++mapit) {
      os << "MapperSnfToLtl: subformula "
         << mapit->first
         << " is mapped to { ";
      for (LiteralClausehOccurrenceSet::iterator it = mapit->second.begin();
           it != mapit->second.end();
           ) {
        it->dump(os);
        ++it;
        if (it != mapit->second.end()) {
          os << ", ";
        }
      }
      os << " }"
         << std::endl;
    }
    os << "MapperSnfToLtl: dumping myMapSubformulaToUseLCHOsNow end"
       << std::endl;

    // myMapSubformulaToUseLCHOsNext
    os << "MapperSnfToLtl: dumping myMapSubformulaToUseLCHOsNext begin"
       << std::endl;
    for (mapit = myMapSubformulaToUseLCHOsNext.begin();
         mapit != myMapSubformulaToUseLCHOsNext.end();
         ++mapit) {
      os << "MapperSnfToLtl: subformula "
         << mapit->first
         << " is mapped to { ";
      for (LiteralClausehOccurrenceSet::iterator it = mapit->second.begin();
           it != mapit->second.end();
           ) {
        it->dump(os);
        ++it;
        if (it != mapit->second.end()) {
          os << ", ";
        }
      }
      os << " }"
         << std::endl;
    }
    os << "MapperSnfToLtl: dumping myMapSubformulaToUseLCHOsNext end"
       << std::endl;

    // myMapSubformulaToUseLCHOsSometime
    os << "MapperSnfToLtl: dumping myMapSubformulaToUseLCHOsSometime begin"
       << std::endl;
    for (mapit = myMapSubformulaToUseLCHOsSometime.begin();
         mapit != myMapSubformulaToUseLCHOsSometime.end();
         ++mapit) {
      os << "MapperSnfToLtl: subformula "
         << mapit->first
         << " is mapped to { ";
      for (LiteralClausehOccurrenceSet::iterator it = mapit->second.begin();
           it != mapit->second.end();
           ) {
        it->dump(os);
        ++it;
        if (it != mapit->second.end()) {
          os << ", ";
        }
      }
      os << " }"
         << std::endl;
    }
    os << "MapperSnfToLtl: dumping myMapSubformulaToUseLCHOsSometime end"
       << std::endl;

    // mySetSubformulaToUseLCHOsInvalidPos
    os << "MapperSnfToLtl: dumping mySetSubformulaToUseLCHOsInvalidPos begin"
       << std::endl;
    os << "MapperSnfToLtl: { ";
    for (setit = mySetSubformulaToUseLCHOsInvalidPos.begin();
         setit != mySetSubformulaToUseLCHOsInvalidPos.end();
         ) {
      setit->dump(os);
      ++setit;
      if (setit != mySetSubformulaToUseLCHOsInvalidPos.end()) {
        os << ", ";
      }
    }
    os << " }"
       << std::endl;
    os << "MapperSnfToLtl: dumping mySetSubformulaToUseLCHOsInvalidPos end"
       << std::endl;

    return os;
  }  // MapperSnfToLtl::dump_maps_sets_subformula_to_use_lchos

  std::ostream &
  MapperSnfToLtl::dump_myMapLCHOToLCHOPartition(std::ostream & os) const
  {
    os << "MapperSnfToLtl: dumping myMapLCHOToLCHOPartition begin"
       << std::endl;
    for (MapLCHOToLCHOPartition::iterator it = myMapLCHOToLCHOPartition.begin();
         it != myMapLCHOToLCHOPartition.end();
         ++it) {
      if (myCoreStartingClausehs.find(it->first.getClauseh()) !=
          myCoreStartingClausehs.end()) {
        os << "MapperSnfToLtl: ";
        it->first.dump(os);
        os << " is mapped to ";
        it->second.dump(os);
        os << std::endl;
      }
    }
    os << "MapperSnfToLtl: dumping myMapLCHOToLCHOPartition end"
       << std::endl;
    return os;
  } // MapperSnfToLtl::dump_myMapLCHOToLCHOPartition
} // namespace UnsatCore
