/************************************************************
 *    Copyright (C) 2012                                    *
 *    Viktor Schuppan (Viktor.Schuppan@gmx.de)              *
 *                                                          *
 *    This program is free software; you can redistribute   *
 *    it and/or modify it under the terms of the GNU        *
 *    General Public License as published by the Free       *
 *    Software Foundation, either version 3 of the License, *
 *    or (at your option) any later version.                *
 *                                                          *
 *    This program is distributed in the hope that it will  *
 *    be useful, but WITHOUT ANY WARRANTY; without even     *
 *    the implied warranty of MERCHANTABILITY or FITNESS    *
 *    FOR A PARTICULAR PURPOSE.  See the GNU General Public *
 *    License for more details.                             *
 *                                                          *
 *    You should have received a copy of the GNU General    *
 *    Public License along with this program.  If not, see  *
 *    <http://www.gnu.org/licenses/>.                       *
 ************************************************************/
#include <iostream>
#include <map>
#include <set>
#include <stdlib.h>
#include "adaptor/iclause.h"
#include "adaptor/sclause.h"
#include "adaptor/uclause.h"
#include "misc/assert.h"
#include "misc/tracer.h"
#include "modules.h"
#include "stl_tim/attribute.h"
#include "stl_tim/literal.h"
#include "stl_tim/pclause.h"
#include "stl_tim/proposition.h"
#include "trim/eclause.h"
#include "unsat_core/clause_handle.h"
#include "formula.h"
#include "simplifier.h"

namespace TrpLtlParser
{
  void
  replaceEquivalencesFormulasLtl(TreeNodeList & formulaList)
  {
    for (TreeNodeList::const_iterator it = formulaList.begin();
         it != formulaList.end();
         ++it)
      {
        replaceEquivalencesFormulaLtl(*it);
      }
  } // replaceEquivalencesFormulasLtl

  void
  replaceEquivalencesFormulaLtl(TreeNode * n)
  {
    replace_equivalences_formula_ltl_rec(n);
  } // replaceEquivalencesFormulaLtl

  void
  nnfFormulasLtl(TreeNodeList & formulaList)
  {
    for (TreeNodeList::iterator it = formulaList.begin();
         it != formulaList.end();
         )
      {
        TreeNodeList::iterator itcur = it;
        ++it;
        *itcur = nnfFormulaLtl(*itcur);
      }
  } // nnfFormulasLtl

  TreeNode *
  nnfFormulaLtl(TreeNode * n)
  {
    return nnf_formula_ltl_rec(n, true);
  } // nnfFormulaLtl

  void
  simplifyFormulasLtl(TreeNodeList & formulaList, TRPPPOptions const & options)
  {
    for (TreeNodeList::iterator it = formulaList.begin();
         it != formulaList.end();
         )
      {
        TreeNodeList::iterator itcur = it;
        ++it;
        *itcur = simplifyFormulaLtl(*itcur, options);
      }
  } // simplifyFormulasLtl

  TreeNode *
  simplifyFormulaLtl(TreeNode * n, TRPPPOptions const & options)
  {
    TreeNode * res = n;
    bool changed = true;

    while (changed) {
      TRACE(builderltlModule, {
          std::cout << "TrpLtlParser: LTL formula before one round of "
                       "simplification:"
                    << std::endl;
          std::cout << res << std::endl;
      });

      res = simplify_formula_ltl_rec(res, true, true, options, changed);

      TRACE(builderltlModule, {
          std::cout << "TrpLtlParser: LTL formula after one round of "
                       "simplification:"
                    << std::endl;
          std::cout << res << std::endl;
      });

    }

    return res;
  } // simplifyFormulaLtl

  void
  simplifyClausehsPureLiteral(UnsatCore::ClauseHandleSet & clausehs)
  {
    // the set of propositions
    std::set<PropositionalProver::Proposition> propositions;

    // maps from positive/negative occurrences to the respective clausehs
    MapPToChs mInitPosToChs;
    MapPToChs mInitNegToChs;
    MapPToChs mUnivNowEvPosToChs;
    MapPToChs mUnivNowEvNegToChs;
    MapPToChs mNextPosToChs;
    MapPToChs mNextNegToChs;

    // fill set of propositions and maps
    for (UnsatCore::ClauseHandleSet::iterator it = clausehs.begin();
         it != clausehs.end();
         ++it)
      {
        if (it->isIClause()) {
          Adaptors::IClause ic = it->getIClause();
          for (Adaptors::IClause::const_iterator it2 = ic.begin();
               it2 != ic.end();
               ++it2)
            {
              Assert(it2->getAttribute() == PropositionalProver::initial_attr,
                     "it2 is not an initial literal.");
              simplify_clausehs_pure_literal_fill(*it2,
                                                  *it,
                                                  propositions,
                                                  mInitPosToChs,
                                                  mInitNegToChs);
            }
        } else if (it->isUClause()) {
          Adaptors::UClause uc = it->getUClause();
          for (Adaptors::UClause::const_iterator it2 = uc.begin();
               it2 != uc.end();
               ++it2)
            {
              Assert(it2->getAttribute() == PropositionalProver::universal_attr,
                     "it2 is not a universal literal.");
              simplify_clausehs_pure_literal_fill(*it2,
                                                  *it,
                                                  propositions,
                                                  mUnivNowEvPosToChs,
                                                  mUnivNowEvNegToChs);
            }
        } else if (it->isSClause()) {
          Adaptors::SClause sc = it->getSClause();
          PropositionalProver::PClause pc = sc.getInternalRep();
          for (PropositionalProver::Clause::iterator it2 = pc->begin();
               it2 != pc->end();
               ++it2)
            {
              Assert(it2->getAttribute() ==
                     PropositionalProver::step_now_attr ||
                     it2->getAttribute() ==
                     PropositionalProver::step_next_attr,
                     "it2 is not a step_now or step_next literal.");
              if (it2->getAttribute() == PropositionalProver::step_now_attr) {
                simplify_clausehs_pure_literal_fill(*it2,
                                                    *it,
                                                    propositions,
                                                    mUnivNowEvPosToChs,
                                                    mUnivNowEvNegToChs);
              } else {
                simplify_clausehs_pure_literal_fill(*it2,
                                                    *it,
                                                    propositions,
                                                    mNextPosToChs,
                                                    mNextNegToChs);
              }
            }
        } else {
          Assert(it->isEClause(), "*it is not an EClause.");
          TRIMs::EClause ec = it->getEClause();
          for (TRIMs::EClause::const_present_iterator it2 = ec.present_begin();
               it2 != ec.present_end();
               ++it2)
            {
              Assert(it2->getAttribute() ==
                     PropositionalProver::universal_attr ||
                     it2->getAttribute() ==
                     PropositionalProver::step_now_attr,
                     "it2 is not a universal or step_now literal.");
              simplify_clausehs_pure_literal_fill(*it2,
                                                  *it,
                                                  propositions,
                                                  mUnivNowEvPosToChs,
                                                  mUnivNowEvNegToChs);
            }
          PropositionalProver::Literal l = ec.getEventuality();
          simplify_clausehs_pure_literal_fill(l,
                                              *it,
                                              propositions,
                                              mUnivNowEvPosToChs,
                                              mUnivNowEvNegToChs);
        }
      }
    
    // find and remove clauses with pure literals
    {
      bool changed = true;
      while (changed) {
        changed = false;
        for (std::set<PropositionalProver::Proposition>::iterator it =
               propositions.begin();
             it != propositions.end();
             )
          {
            std::set<PropositionalProver::Proposition>::iterator itcur = it;
            ++it;
            PropositionalProver::Proposition p = *itcur;

            // initial_attr cannot resolve with step_next_attr. Hence,
            // check initial literals only against initial, universal,
            // step_now, eventuality literals. Similarly, check
            // step_next literals only against universal, step_now,
            // step_next, eventuality literals.
            if (mInitPosToChs.find(p) != mInitPosToChs.end() &&
                mInitNegToChs.find(p) == mInitNegToChs.end() &&
                mUnivNowEvNegToChs.find(p) == mUnivNowEvNegToChs.end()) {
              changed = true;
              simplify_clausehs_pure_literal_remove_clausehs(mInitPosToChs[p],
                                                             clausehs,
                                                             mInitPosToChs,
                                                             mInitNegToChs,
                                                             mUnivNowEvPosToChs,
                                                             mUnivNowEvNegToChs,
                                                             mNextPosToChs,
                                                             mNextNegToChs);
            }
            if (mInitNegToChs.find(p) != mInitNegToChs.end() &&
                mInitPosToChs.find(p) == mInitPosToChs.end() &&
                mUnivNowEvPosToChs.find(p) == mUnivNowEvPosToChs.end()) {
              changed = true;
              simplify_clausehs_pure_literal_remove_clausehs(mInitNegToChs[p],
                                                             clausehs,
                                                             mInitPosToChs,
                                                             mInitNegToChs,
                                                             mUnivNowEvPosToChs,
                                                             mUnivNowEvNegToChs,
                                                             mNextPosToChs,
                                                             mNextNegToChs);
            }
            if (mUnivNowEvPosToChs.find(p) != mUnivNowEvPosToChs.end() &&
                mInitNegToChs.find(p) == mInitNegToChs.end() &&
                mUnivNowEvNegToChs.find(p) == mUnivNowEvNegToChs.end() &&
                mNextNegToChs.find(p) == mNextNegToChs.end()) {
              changed = true;
              simplify_clausehs_pure_literal_remove_clausehs(mUnivNowEvPosToChs[p],
                                                             clausehs,
                                                             mInitPosToChs,
                                                             mInitNegToChs,
                                                             mUnivNowEvPosToChs,
                                                             mUnivNowEvNegToChs,
                                                             mNextPosToChs,
                                                             mNextNegToChs);
            }
            if (mUnivNowEvNegToChs.find(p) != mUnivNowEvNegToChs.end() &&
                mInitPosToChs.find(p) == mInitPosToChs.end() &&
                mUnivNowEvPosToChs.find(p) == mUnivNowEvPosToChs.end() &&
                mNextPosToChs.find(p) == mNextPosToChs.end()) {
              changed = true;
              simplify_clausehs_pure_literal_remove_clausehs(mUnivNowEvNegToChs[p],
                                                             clausehs,
                                                             mInitPosToChs,
                                                             mInitNegToChs,
                                                             mUnivNowEvPosToChs,
                                                             mUnivNowEvNegToChs,
                                                             mNextPosToChs,
                                                             mNextNegToChs);
            }
            if (mNextPosToChs.find(p) != mNextPosToChs.end() &&
                mUnivNowEvNegToChs.find(p) == mUnivNowEvNegToChs.end() &&
                mNextNegToChs.find(p) == mNextNegToChs.end()) {
              changed = true;
              simplify_clausehs_pure_literal_remove_clausehs(mNextPosToChs[p],
                                                             clausehs,
                                                             mInitPosToChs,
                                                             mInitNegToChs,
                                                             mUnivNowEvPosToChs,
                                                             mUnivNowEvNegToChs,
                                                             mNextPosToChs,
                                                             mNextNegToChs);
            }
            if (mNextNegToChs.find(p) != mNextNegToChs.end() &&
                mUnivNowEvPosToChs.find(p) == mUnivNowEvPosToChs.end() &&
                mNextPosToChs.find(p) == mNextPosToChs.end()) {
              changed = true;
              simplify_clausehs_pure_literal_remove_clausehs(mNextNegToChs[p],
                                                             clausehs,
                                                             mInitPosToChs,
                                                             mInitNegToChs,
                                                             mUnivNowEvPosToChs,
                                                             mUnivNowEvNegToChs,
                                                             mNextPosToChs,
                                                             mNextNegToChs);
            }

            // if there are no more clauses containing p, then remove
            // p from propositions
            if (mInitPosToChs.find(p) == mInitPosToChs.end() &&
                mInitNegToChs.find(p) == mInitNegToChs.end() &&
                mUnivNowEvPosToChs.find(p) == mUnivNowEvPosToChs.end() &&
                mUnivNowEvNegToChs.find(p) == mUnivNowEvNegToChs.end() &&
                mNextPosToChs.find(p) == mNextPosToChs.end() &&
                mNextNegToChs.find(p) == mNextNegToChs.end()) {
              (void) propositions.erase(itcur);
            }
          }
      }
    }
  } // simplifyClausehsPureLiteral

  /*
    True
            nothing
    False
            nothing
    p
            nothing
    !
            !True           False
            !False          True
            !!f             f
    &
            f & (g & h)     f & g & h
            f & g           g & f when f < g
            f & True        f
            f & f           f
            f & !f          False
            f & False       False
            single child    child
            no child        True
#if 0
            (X f) & (X g)   X (f & g)
            (G f) & (G g)   G (f & g)
#endif
    |
            f | (g | h)     f | g | h
            f | g           g | f when f < g
            f | (g -> h)    f | !g | h
            f | False       f
            p | p           p
            p | !p          True
            f | True        True
            single child    child
            no child        True
#if 0
            (X f) | (X g)   X (f | g)
            (F f) | (F g)   F (f | g)
#endif
    ->
            f -> True       True
            False -> f      True
            True -> f       f
            f -> False      !f
            f -> f          True
            (X f) -> (X g)  X (f -> g)
            f -> !f         !f
            !f -> f         f
            f -> (g | h)    !f | g | h
            f -> (g & h)    (f -> g) & (f -> h) when isPositivePolarity
            (f | g) -> h    (f -> h) & (g -> h) when isPositivePolarity
            (f & g) -> h    !f | !g | h
            f -> (g -> h)   !f | !g | h
    X
            X True          True    
            X False         False
    G
            G True          True    
            G False         False
            G G f           G f
            G (f W g)       G (f | g)
            G F G f         F G f
            G (f & g)       (G f) & (G g) when isPositivePolarity and isInitial
    F
            F True          True    
            F False         False
            F F f           F f
            F (f U g)       F g
            F G F f         G F f
    U
            f U True        True
            f U False       False
            True U f        F f
            False U f       f
            f U f           f
            f U F g         F g
            (X f) U (X g)   X (f U g)
            f U !f          F !f
            !f U f          F f
    W
            f W True        True
            f W False       G f
            True W f        True
            False W f       f
            f W f           f
            (X f) W (X g)   X (f W g)
            (G f) W g       (G f) | g
            f W !f          True
            !f W f          True
  */
  TreeNode *
  simplify_formula_ltl_rec(TreeNode * n,
                           bool isPositivePolarity,
                           bool isInitial,
                           TRPPPOptions const & options,
                           bool & changed)
  {
    TreeNode * res = NULL;

    changed = false;

    // simplify children
    {
      bool isChildInitial = isInitial;
      if (isOperator(n) &&
          isTemporalOperator(((Operator *) n)->getOperator())) {
        isChildInitial = false;
      }
      
      for (TreeNodeList::const_iterator it = n->children().begin();
           it != n->children().end();
           )
        {
          TreeNodeList::const_iterator itcur = it;
          ++it;
          bool changedChild = false;
          TreeNode * childSimplified;
          if (isNot(n)) {
            childSimplified = simplify_formula_ltl_rec(*itcur,
                                                       !isPositivePolarity,
                                                       isChildInitial,
                                                       options,
                                                       changedChild);
            if (*itcur != childSimplified) {
              TreeNode * tmp = *itcur;
              n->replaceChild(*itcur, childSimplified);
              delete(tmp);
            }
          } else if (isImplication(n) && *itcur == n->firstChild()) {
            childSimplified = simplify_formula_ltl_rec(*itcur,
                                                       !isPositivePolarity,
                                                       isChildInitial,
                                                       options,
                                                       changedChild);
            if (*itcur != childSimplified) {
              TreeNode * tmp = *itcur;
              n->replaceChild(*itcur, childSimplified);
              delete(tmp);
            }
          } else {
            Assert(!isEquivalence(n), "Unexpected equivalence operator.");
            childSimplified = simplify_formula_ltl_rec(*itcur,
                                                       isPositivePolarity,
                                                       isChildInitial,
                                                       options,
                                                       changedChild);
            if (*itcur != childSimplified) {
              TreeNode * tmp = *itcur;
              n->replaceChild(*itcur, childSimplified);
              delete(tmp);
            }
          }
          changed |= changedChild;
        }
    }

    // simplify self
    if (isIdentifier(n)) {
      Assert(n->childrenCount() == 0, "Unexpected number of children.");
      res = n;
    } else if (isNot(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      TreeNode * child = n->firstChild();
      if (isIdentifier(child) && ((Identifier *) child)->isTrue()) {
        changed = true;
        res = new Identifier(false);
      } else if (isIdentifier(child) && ((Identifier *) child)->isFalse()) {
        changed = true;
        res = new Identifier(true);
      } else if (isNot(child)) {
        changed = true;
        res = child->firstChild()->clone();
        child->removeChild(child->firstChild());
      } else {
        res = n;
      }
    } else if (isAnd(n)) {
      Assert(n->childrenCount() > 0, "Unexpected number of children.");

      // lift children & up
      for (TreeNodeList::const_iterator it = n->children().begin();
           it != n->children().end();
           )
        {
          TreeNodeList::const_iterator itcur = it;
          ++it;
          if (isAnd(*itcur)) {
            changed = true;
            for (TreeNodeList::const_iterator it2 = (*itcur)->children().begin();
                 it2 != (*itcur)->children().end();
                 )
              {
                TreeNodeList::const_iterator it2cur = it2;
                ++it2;
                n->addChild((*it2cur)->clone());
                (*itcur)->removeChild(*it2cur);
              }
            n->removeChild(*itcur);
          }
        }

      // sort children
      if (!options.isNoSorting()) {
        TreeNodeList tnl = TreeNodeList();

        while (n->childrenCount() > 0) {
          TreeNodeList::const_iterator smallest = n->children().begin();
          for (TreeNodeList::const_iterator it = n->children().begin();
               it != n->children().end();
               ++it)
            {
              if (*(*it) < *(*smallest)) {
                changed = true;
                smallest = it;
              }            
            }
          tnl.push_back((*smallest)->clone());
          n->removeChild(*smallest);
        }

        n->setChildren(tnl);
      }

      // remove True
      {
        for (TreeNodeList::const_iterator it = n->children().begin();
             it != n->children().end();
             )
          {
            TreeNodeList::const_iterator itcur = it;
            ++it;
            if (isIdentifier(*itcur) && ((Identifier *) *itcur)->isTrue()) {
              changed = true;
              n->removeChild(*itcur);
            }
          }
        if (n->childrenCount() == 0) {
          res = new Identifier(true);
          return res;
        }
      }

      // remove duplicates
      {
        std::set<TreeNode *> toRemove = std::set<TreeNode *>();
        for (TreeNodeList::const_iterator it = n->children().begin();
             it != n->children().end();
             )
          {
            TreeNodeList::const_iterator itcur = it;
            ++it;
            for (TreeNodeList::const_iterator it2 = it;
                 it2 != n->children().end();
                 )
              {
                TreeNodeList::const_iterator it2cur = it2;
                ++it2;
                if ((*(*itcur)) == (*(*it2cur))) {
                  changed = true;
                  (void) toRemove.insert(*it2cur);
                }
              }
          }
        for (std::set<TreeNode *>::iterator it = toRemove.begin();
             it != toRemove.end();
             ++it)
          {
            n->removeChild(*it);
          }
      }

      // replace f & !f with False
      {
        std::set<TreeNode *> toRemove = std::set<TreeNode *>();
        bool found = false;
        for (TreeNodeList::const_iterator it = n->children().begin();
             !found && it != n->children().end();
             )
          {
            TreeNodeList::const_iterator itcur = it;
            ++it;
            for (TreeNodeList::const_iterator it2 = it;
                 !found && it2 != n->children().end();
                 )
              {
                TreeNodeList::const_iterator it2cur = it2;
                ++it2;
                TreeNode * n1a = Not((*itcur)->clone());
                TreeNode * n2a = Not((*it2cur)->clone());
                TreeNode * n1b = nnf_formula_ltl_rec((*itcur)->clone(), !isPositivePolarity);
                TreeNode * n2b = nnf_formula_ltl_rec((*it2cur)->clone(), !isPositivePolarity);
                if ((*(*itcur)) == (*n2a) ||
                    (*n1a) == (*(*it2cur)) ||
                    (*(*itcur)) == (*n2b) ||
                    (*n1b) == (*(*it2cur))) {
                  delete(n1a);
                  delete(n2a);
                  delete(n1b);
                  delete(n2b);

                  changed = true;
                  found = true;
                  (void) toRemove.insert(*it2cur);
                  TreeNode * tmp = *itcur;
                  n->replaceChild(*itcur, new Identifier(false));
                  delete(tmp);
                } else {
                  delete(n1a);
                  delete(n2a);
                  delete(n1b);
                  delete(n2b);
                }
              }
          }
        for (std::set<TreeNode *>::iterator it = toRemove.begin();
             it != toRemove.end();
             ++it)
          {
            n->removeChild(*it);
          }
      }

      // False dominates
      for (TreeNodeList::const_iterator it = n->children().begin();
           it != n->children().end();
           )
        {
          TreeNodeList::const_iterator itcur = it;
          ++it;
          if (isIdentifier(*itcur) && ((Identifier *) *itcur)->isFalse()) {
            changed = true;
            res = new Identifier(false);
            return res;
          }
        }

      // single child
      if (n->childrenCount() == 1) {
        changed = true;
        res = n->firstChild()->clone();
        n->removeChild(n->firstChild());
        return res;
      }

      // no child
      if (n->childrenCount() == 0) {
        changed = true;
        res = new Identifier(true);
        return res;
      }

      // all children next
#if 0
      {
        bool allChildrenNext = true;
        for (TreeNodeList::const_iterator it = n->children().begin();
             allChildrenNext && it != n->children().end();
             ++it)
          {
            if (!isNext(*it)) {
              allChildrenNext = false;
            }
          }

        if (allChildrenNext) {
          TreeNodeList tnl = TreeNodeList();
          for (TreeNodeList::const_iterator it = n->children().begin();
               it != n->children().end();
               ++it)
            {
              tnl.push_back((*it)->firstChild()->clone());
            }
          res = Next(And(tnl));
          return res;
        }
      }

      // all children always
      {
        bool allChildrenAlways = true;
        for (TreeNodeList::const_iterator it = n->children().begin();
             allChildrenAlways && it != n->children().end();
             ++it)
          {
            if (!isAlways(*it)) {
              allChildrenAlways = false;
            }
          }

        if (allChildrenAlways) {
          TreeNodeList tnl = TreeNodeList();
          for (TreeNodeList::const_iterator it = n->children().begin();
               it != n->children().end();
               ++it)
            {
              tnl.push_back((*it)->firstChild()->clone());
            }
          res = Always(And(tnl));
          return res;
        }
      }
#endif

      res = n;
    } else if (isOr(n)) {
      Assert(n->childrenCount() > 0, "Unexpected number of children.");

      // lift children |, -> up
      for (TreeNodeList::const_iterator it = n->children().begin();
           it != n->children().end();
           )
        {
          TreeNodeList::const_iterator itcur = it;
          ++it;
          if (isOr(*itcur)) {
            changed = true;
            for (TreeNodeList::const_iterator it2 = (*itcur)->children().begin();
                 it2 != (*itcur)->children().end();
                 )
              {
                TreeNodeList::const_iterator it2cur = it2;
                ++it2;
                n->addChild((*it2cur)->clone());
                (*itcur)->removeChild(*it2cur);
              }
            n->removeChild(*itcur);
          } else if (isImplication(*itcur)) {
            changed = true;
            TreeNode * fst, * snd;
            fst = (*itcur)->firstChild();
            snd = (*itcur)->secondChild();
            n->addChild(Not(fst->clone()));
            n->addChild(snd->clone());
            (*itcur)->removeChild(fst);
            (*itcur)->removeChild(snd);
            n->removeChild(*itcur);
          }
        }

      // sort children
      if (!options.isNoSorting()) {
        TreeNodeList tnl = TreeNodeList();

        while (n->childrenCount() > 0) {
          TreeNodeList::const_iterator smallest = n->children().begin();
          for (TreeNodeList::const_iterator it = n->children().begin();
               it != n->children().end();
               ++it)
            {
              if (*(*it) < *(*smallest)) {
                changed = true;
                smallest = it;
              }
            }
          tnl.push_back((*smallest)->clone());
          n->removeChild(*smallest);
        }

        n->setChildren(tnl);
      }

      // remove False
      {
        for (TreeNodeList::const_iterator it = n->children().begin();
             it != n->children().end();
             )
          {
            TreeNodeList::const_iterator itcur = it;
            ++it;
            if (isIdentifier(*itcur) && ((Identifier *) *itcur)->isFalse()) {
              changed = true;
              n->removeChild(*itcur);
            }
          }
        if (n->childrenCount() == 0) {
          res = new Identifier(false);
          return res;
        }
      }

      // remove duplicates
      {
        std::set<TreeNode *> toRemove = std::set<TreeNode *>();
        for (TreeNodeList::const_iterator it = n->children().begin();
             it != n->children().end();
             )
          {
            TreeNodeList::const_iterator itcur = it;
            ++it;
            for (TreeNodeList::const_iterator it2 = it;
                 it2 != n->children().end();
                 )
              {
                TreeNodeList::const_iterator it2cur = it2;
                ++it2;
                if ((*(*itcur)) == (*(*it2cur))) {
                  changed = true;
                  (void) toRemove.insert(*it2cur);
                }
              }
          }
        for (std::set<TreeNode *>::iterator it = toRemove.begin();
             it != toRemove.end();
             ++it)
          {
            n->removeChild(*it);
          }
      }

      // replace f | !f with True
      {
        std::set<TreeNode *> toRemove = std::set<TreeNode *>();
        bool found = false;
        for (TreeNodeList::const_iterator it = n->children().begin();
             !found && it != n->children().end();
             )
          {
            TreeNodeList::const_iterator itcur = it;
            ++it;
            for (TreeNodeList::const_iterator it2 = it;
                 !found && it2 != n->children().end();
                 )
              {
                TreeNodeList::const_iterator it2cur = it2;
                ++it2;
                TreeNode * n1a = Not((*itcur)->clone());
                TreeNode * n2a = Not((*it2cur)->clone());
                TreeNode * n1b = nnf_formula_ltl_rec((*itcur)->clone(), !isPositivePolarity);
                TreeNode * n2b = nnf_formula_ltl_rec((*it2cur)->clone(), !isPositivePolarity);
                if ((*(*itcur)) == (*n2a) ||
                    (*n1a) == (*(*it2cur)) ||
                    (*(*itcur)) == (*n2b) ||
                    (*n1b) == (*(*it2cur))) {
                  delete(n1a);
                  delete(n2a);
                  delete(n1b);
                  delete(n2b);

                  changed = true;
                  found = true;
                  (void) toRemove.insert(*it2cur);
                  TreeNode * tmp = *itcur;
                  n->replaceChild(*itcur, new Identifier(true));
                  delete(tmp);
                } else {
                  delete(n1a);
                  delete(n2a);
                  delete(n1b);
                  delete(n2b);
                }
              }
          }
        for (std::set<TreeNode *>::iterator it = toRemove.begin();
             it != toRemove.end();
             ++it)
          {
            n->removeChild(*it);
          }
      }

      // True dominates
      for (TreeNodeList::const_iterator it = n->children().begin();
           it != n->children().end();
           )
        {
          TreeNodeList::const_iterator itcur = it;
          ++it;
          if (isIdentifier(*itcur) && ((Identifier *) *itcur)->isTrue()) {
            changed = true;
            res = new Identifier(true);
            return res;
          }
        }

      // single child
      if (n->childrenCount() == 1) {
        changed = true;
        res = n->firstChild()->clone();
        n->removeChild(n->firstChild());
        return res;
      }

      // no child
      if (n->childrenCount() == 0) {
        changed = true;
        res = new Identifier(true);
        return res;
      }

      // all children next
#if 0
      {
        bool allChildrenNext = true;
        for (TreeNodeList::const_iterator it = n->children().begin();
             allChildrenNext && it != n->children().end();
             ++it)
          {
            if (!isNext(*it)) {
              allChildrenNext = false;
            }
          }

        if (allChildrenNext) {
          TreeNodeList tnl = TreeNodeList();
          for (TreeNodeList::const_iterator it = n->children().begin();
               it != n->children().end();
               ++it)
            {
              tnl.push_back((*it)->firstChild()->clone());
            }
          res = Next(Or(tnl));
          return res;
        }
      }

      // all children sometime
      {
        bool allChildrenSometime = true;
        for (TreeNodeList::const_iterator it = n->children().begin();
             allChildrenSometime && it != n->children().end();
             ++it)
          {
            if (!isSometime(*it)) {
              allChildrenSometime = false;
            }
          }

        if (allChildrenSometime) {
          TreeNodeList tnl = TreeNodeList();
          for (TreeNodeList::const_iterator it = n->children().begin();
               it != n->children().end();
               ++it)
            {
              tnl.push_back((*it)->firstChild()->clone());
            }
          res = Sometime(Or(tnl));
          return res;
        }
      }
#endif

      res = n;
    } else if (isImplication(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      TreeNode * fst = n->firstChild();
      TreeNode * snd = n->secondChild();
      if (isIdentifier(snd) && ((Identifier *) snd)->isTrue()) {
        changed = true;
        res = new Identifier(true);
        return res;
      } else if (isIdentifier(fst) && ((Identifier *) fst)->isFalse()) {
        changed = true;
        res = new Identifier(true);
        return res;
      } else if (isIdentifier(fst) && ((Identifier *) fst)->isTrue()) {
        changed = true;
        res = snd->clone();
        n->removeChild(snd);
        return res;
      } else if (isIdentifier(snd) && ((Identifier *) snd)->isFalse()) {
        changed = true;
        res = Not(fst->clone());
        n->removeChild(fst);
        return res;
      } else if ((*fst) == (*snd)) {
        changed = true;
        res = new Identifier(true);
        return res;
      } else if (isNext(fst) && isNext(snd)) {
        changed = true;
        res = Next(Implication(fst->clone(), snd->clone()));
        n->removeChild(fst);
        n->removeChild(snd);
        return res;
      }

      {
        TreeNode * n1a = Not(fst->clone());
        TreeNode * n2a = Not(snd->clone());
        TreeNode * n1b = nnf_formula_ltl_rec(fst->clone(), !isPositivePolarity);
        TreeNode * n2b = nnf_formula_ltl_rec(snd->clone(), !isPositivePolarity);
        if ((*fst) == (*n2a) ||
            (*n1a) == (*snd) ||
            (*fst) == (*n2b) ||
            (*n1b) == (*snd)) {
          delete(n1a);
          delete(n2a);
          delete(n1b);
          delete(n2b);

          changed = true;
          res = snd->clone();
          n->removeChild(snd);
          return res;
        } else {
          delete(n1a);
          delete(n2a);
          delete(n1b);
          delete(n2b);
        }
      }

      if (isOr(snd)) {
        changed = true;
        ((Operator *) n)->setOperator(OR);
        TreeNode * tmp = fst;
        n->replaceChild(fst, Not(fst->clone()));
        delete(tmp);
        for (TreeNodeList::const_iterator it = snd->children().begin();
             it != snd->children().end();
             )
          {
            TreeNodeList::const_iterator itcur = it;
            ++it;
            n->addChild((*itcur)->clone());
            snd->removeChild(*itcur);
          }
        n->removeChild(snd);
        res = n;
        return res;
      }

      // Split f -> (g & h) into (f -> g) & (f -> h) to facilitate
      // direct encoding into SNF clauses
      if (isAnd(snd) && isPositivePolarity) {
        changed = true;
        TreeNodeList children = TreeNodeList();
        for (TreeNodeList::const_iterator it = snd->children().begin();
             it != snd->children().end();
             )
          {
            TreeNodeList::const_iterator itcur = it;
            ++it;
            children.push_back(Implication(fst->clone(), (*itcur)->clone()));
            snd->removeChild(*itcur);
          }
        n->removeChild(fst);
        n->removeChild(snd);
        res = And(children);
        return res;
      }

      // Split (f | g) -> h into (f -> h) & (g -> h) to facilitate
      // direct encoding into SNF clauses
      if (isOr(fst) && isPositivePolarity) {
        changed = true;
        TreeNodeList children = TreeNodeList();
        for (TreeNodeList::const_iterator it = fst->children().begin();
             it != fst->children().end();
             )
          {
            TreeNodeList::const_iterator itcur = it;
            ++it;
            children.push_back(Implication((*itcur)->clone(), snd->clone()));
            fst->removeChild(*itcur);
          }
        n->removeChild(fst);
        n->removeChild(snd);
        res = And(children);
        return res;
      }

      if (isAnd(fst)) {
        changed = true;
        ((Operator *) n)->setOperator(OR);
        for (TreeNodeList::const_iterator it = fst->children().begin();
             it != fst->children().end();
             )
          {
            TreeNodeList::const_iterator itcur = it;
            ++it;
            n->addChild(Not((*itcur)->clone()));
            fst->removeChild(*itcur);
          }
        n->removeChild(fst);
        res = n;
        return res;
      }

      if (isImplication(snd)) {
        changed = true;
        ((Operator *) n)->setOperator(OR);
        TreeNode * tmp = fst;
        n->replaceChild(fst, Not(fst->clone()));
        delete(tmp);
        TreeNode * sndfst = snd->firstChild();
        TreeNode * sndsnd = snd->secondChild();
        n->addChild(Not(sndfst->clone()));
        n->addChild(sndsnd->clone());
        snd->removeChild(sndfst);
        snd->removeChild(sndsnd);
        n->removeChild(snd);
        res = n;
        return res;
      }

      res = n;
    } else if (isNext(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      TreeNode * child = n->firstChild();
      if (isIdentifier(child) && ((Identifier *) child)->isTrue()) {
        changed = true;
        res = new Identifier(true);
      } else if (isIdentifier(child) && ((Identifier *) child)->isFalse()) {
        changed = true;
        res = new Identifier(false);
      } else {
        res = n;
      }
    } else if (isAlways(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      TreeNode * child = n->firstChild();
      if (isIdentifier(child) && ((Identifier *) child)->isTrue()) {
        changed = true;
        res = new Identifier(true);
        return res;
      } else if (isIdentifier(child) && ((Identifier *) child)->isFalse()) {
        changed = true;
        res = new Identifier(false);
        return res;
      } else if (isAlways(child)) {
        changed = true;
        res = child->clone();
        n->removeChild(child);
        return res;
      } else if (isSometime(child)) {
        Assert(child->childrenCount() == 1, "Unexpected number of children.");
        TreeNode * grandChild = child->firstChild();
        if (isAlways(grandChild)) {
          changed = true;
          res = child->clone();
          n->removeChild(child);
        } else {
          res = n;
        }
        return res;
      } else if (isUnless(child)) {
        changed = true;
        ((Operator *) child)->setOperator(OR);
        res = n;
        return res;
      }

      // Split G (f & g) into (G f) & (G g) to facilitate direct
      // encoding into SNF clauses
      if (isAnd(child) && isPositivePolarity && isInitial) {
        changed = true;
        TreeNodeList children = TreeNodeList();
        for (TreeNodeList::const_iterator it = child->children().begin();
             it != child->children().end();
             )
          {
            TreeNodeList::const_iterator itcur = it;
            ++it;
            children.push_back(Always((*itcur)->clone()));
            child->removeChild(*itcur);
          }
        n->removeChild(child);
        res = And(children);
        return res;
      }

      res = n;
    } else if (isSometime(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      TreeNode * child = n->firstChild();
      if (isIdentifier(child) && ((Identifier *) child)->isTrue()) {
        changed = true;
        res = new Identifier(true);
      } else if (isIdentifier(child) && ((Identifier *) child)->isFalse()) {
        changed = true;
        res = new Identifier(false);
      } else if (isSometime(child)) {
        changed = true;
        res = child->clone();
        n->removeChild(child);
      } else if (isAlways(child)) {
        Assert(child->childrenCount() == 1, "Unexpected number of children.");
        TreeNode * grandChild = child->firstChild();
        if (isSometime(grandChild)) {
          changed = true;
          res = child->clone();
          n->removeChild(child);
        } else {
          res = n;
        }
      } else if (isUntil(child)) {
        changed = true;
        TreeNode * tmp = child->firstChild();
        child->removeChild(tmp);
        ((Operator *) child)->setOperator(SOMETIME);
        res = n;
      } else {
        res = n;
      }
    } else if (isUntil(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      TreeNode * fst = n->firstChild();
      TreeNode * snd = n->secondChild();
      if (isIdentifier(snd) && ((Identifier *) snd)->isTrue()) {
        changed = true;
        res = new Identifier(true);
        return res;
      } else if (isIdentifier(snd) && ((Identifier *) snd)->isFalse()) {
        changed = true;
        res = new Identifier(false);
        return res;
      } else if (isIdentifier(fst) && ((Identifier *) fst)->isTrue()) {
        changed = true;
        ((Operator *) n)->setOperator(SOMETIME);
        n->removeChild(fst);
        res = n;
        return res;
      } else if (isIdentifier(fst) && ((Identifier *) fst)->isFalse()) {
        changed = true;
        res = snd->clone();
        n->removeChild(snd);
        return res;
      } else if ((*fst) == (*snd)) {
        changed = true;
        res = fst->clone();
        n->removeChild(fst);
        return res;
      } else if (isSometime(snd)) {
        changed = true;
        res = snd->clone();
        n->removeChild(snd);
        return res;
      } else if (isNext(fst) && isNext(snd)) {
        changed = true;
        res = Next(Until(fst->firstChild()->clone(),
                         snd->firstChild()->clone()));
        fst->removeChild(fst->firstChild());
        snd->removeChild(snd->firstChild());
        return res;
      }

      {
        TreeNode * n1a = Not(fst->clone());
        TreeNode * n2a = Not(snd->clone());
        TreeNode * n1b = nnf_formula_ltl_rec(fst->clone(), !isPositivePolarity);
        TreeNode * n2b = nnf_formula_ltl_rec(snd->clone(), !isPositivePolarity);
        if ((*fst) == (*n2a) ||
            (*n1a) == (*snd) ||
            (*fst) == (*n2b) ||
            (*n1b) == (*snd)) {
          delete(n1a);
          delete(n2a);
          delete(n1b);
          delete(n2b);

          changed = true;
          ((Operator *) n)->setOperator(SOMETIME);
          n->removeChild(fst);
          res = n;
          return res;
        } else {
          delete(n1a);
          delete(n2a);
          delete(n1b);
          delete(n2b);
        }
      }

      res = n;
    } else if (isUnless(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      TreeNode * fst = n->firstChild();
      TreeNode * snd = n->secondChild();
      if (isIdentifier(snd) && ((Identifier *) snd)->isTrue()) {
        changed = true;
        res = new Identifier(true);
        return res;
      } else if (isIdentifier(snd) && ((Identifier *) snd)->isFalse()) {
        changed = true;
        ((Operator *) n)->setOperator(ALWAYS);
        n->removeChild(snd);
        res = n;
        return res;
      } else if (isIdentifier(fst) && ((Identifier *) fst)->isTrue()) {
        changed = true;
        res = new Identifier(true);
        return res;
      } else if (isIdentifier(fst) && ((Identifier *) fst)->isFalse()) {
        changed = true;
        res = snd->clone();
        n->removeChild(snd);
        return res;
      } else if ((*fst) == (*snd)) {
        changed = true;
        res = fst->clone();
        n->removeChild(fst);
        return res;
      } else if (isNext(fst) && isNext(snd)) {
        changed = true;
        res = Next(Unless(fst->firstChild()->clone(),
                          snd->firstChild()->clone()));
        fst->removeChild(fst->firstChild());
        snd->removeChild(snd->firstChild());
        return res;
      } else if (isAlways(fst)) {
        changed = true;
        ((Operator *) n)->setOperator(OR);
        res = n;
        return res;
      }

      {
        TreeNode * n1a = Not(fst->clone());
        TreeNode * n2a = Not(snd->clone());
        TreeNode * n1b = nnf_formula_ltl_rec(fst->clone(), !isPositivePolarity);
        TreeNode * n2b = nnf_formula_ltl_rec(snd->clone(), !isPositivePolarity);
        if ((*fst) == (*n2a) ||
            (*n1a) == (*snd) ||
            (*fst) == (*n2b) ||
            (*n1b) == (*snd)) {
          delete(n1a);
          delete(n2a);
          delete(n1b);
          delete(n2b);

          changed = true;
          res = new Identifier(true);
          return res;
        } else {
          delete(n1a);
          delete(n2a);
          delete(n1b);
          delete(n2b);
        }
      }

      res = n;
    } else {
      Assert(false, "n has unexpected type.");
    }

    return res;
  } // simplify_formula_ltl_rec

  void
  replace_equivalences_formula_ltl_rec(TreeNode * n)
  {
    if (!isEquivalence(n)) {
      for (TreeNodeList::const_iterator it = n->children().begin();
           it != n->children().end();
           ++it)
        {
          replace_equivalences_formula_ltl_rec(*it);
        }
    } else {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");

      TreeNode * fst = n->firstChild();
      TreeNode * snd = n->secondChild();

      replace_equivalences_formula_ltl_rec(fst);
      replace_equivalences_formula_ltl_rec(snd);

      TreeNode * fstfst = fst->clone();
      TreeNode * fstsnd = snd->clone();
      TreeNode * sndfst = snd->clone();
      TreeNode * sndsnd = fst->clone();
      fst = Implication(fstfst, fstsnd);
      snd = Implication(sndfst, sndsnd);

      ((Operator *) n)->setOperator(AND);

      TreeNode * tmp = n->firstChild();
      n->replaceChild(n->firstChild(), fst);
      delete(tmp);

      tmp = n->secondChild();
      n->replaceChild(n->secondChild(), snd);
      delete(tmp);
    }
  } // replace_equivalences_formula_ltl_rec

  TreeNode * nnf_formula_ltl_rec(TreeNode * n, bool isPositivePolarity)
  {
    TreeNode * res = NULL;

    if (isPositivePolarity) {
      if (isIdentifier(n)) {
        Assert(n->childrenCount() == 0, "Unexpected number of children.");
        res = n->clone();
      } else if (isNot(n)) {
        Assert(n->childrenCount() == 1, "Unexpected number of children.");
        res = nnf_formula_ltl_rec(n->firstChild(), false);
      } else if (isAnd(n) || isOr(n)) {
        Assert(n->childrenCount() > 0, "Unexpected number of children.");
        TreeNodeList tnl = TreeNodeList();
        for (TreeNodeList::const_iterator it = n->children().begin();
             it != n->children().end();
             ++it)
          {
            tnl.push_back(nnf_formula_ltl_rec(*it, true));
          }
        if (isAnd(n)) {
          res = And(tnl);
        } else {
          Assert(isOr(n), "n is not Or.");
          res = Or(tnl);
        }
      } else if (isImplication(n)) {
        Assert(n->childrenCount() == 2, "Unexpected number of children.");
        TreeNode * fstNnfed = nnf_formula_ltl_rec(n->firstChild(), false);
        TreeNode * sndNnfed = nnf_formula_ltl_rec(n->secondChild(), true);
        res = Or(fstNnfed, sndNnfed);
      } else if (isNext(n)) {
        Assert(n->childrenCount() == 1, "Unexpected number of children.");
        res = Next(nnf_formula_ltl_rec(n->firstChild(), true));
      } else if (isAlways(n)) {
        Assert(n->childrenCount() == 1, "Unexpected number of children.");
        res = Always(nnf_formula_ltl_rec(n->firstChild(), true));
      } else if (isSometime(n)) {
        Assert(n->childrenCount() == 1, "Unexpected number of children.");
        res = Sometime(nnf_formula_ltl_rec(n->firstChild(), true));
      } else if (isUntil(n)) {
        Assert(n->childrenCount() == 2, "Unexpected number of children.");
        res = Until(nnf_formula_ltl_rec(n->firstChild(), true),
                    nnf_formula_ltl_rec(n->secondChild(), true));
      } else if (isUnless(n)) {
        Assert(n->childrenCount() == 2, "Unexpected number of children.");
        res = Unless(nnf_formula_ltl_rec(n->firstChild(), true),
                     nnf_formula_ltl_rec(n->secondChild(), true));
      } else {
        Assert(false, "n has unexpected type.");
      }
    } else {
      Assert(!isPositivePolarity, "isPositivePolarity.");
      if (isIdentifier(n)) {
        Assert(n->childrenCount() == 0, "Unexpected number of children.");
        if (((Identifier *) n)->isTrue()) {
          res = new Identifier(false);
        } else if (((Identifier *) n)->isFalse()) {
          res = new Identifier(true);
        } else {
          res = Not(n->clone());
        }
      } else if (isNot(n)) {
        Assert(n->childrenCount() == 1, "Unexpected number of children.");
        res = nnf_formula_ltl_rec(n->firstChild(), true);
      } else if (isAnd(n) || isOr(n)) {
        Assert(n->childrenCount() > 0, "Unexpected number of children.");
        TreeNodeList tnl = TreeNodeList();
        for (TreeNodeList::const_iterator it = n->children().begin();
             it != n->children().end();
             ++it)
          {
            tnl.push_back(nnf_formula_ltl_rec(*it, false));
          }
        if (isAnd(n)) {
          res = Or(tnl);
        } else {
          Assert(isOr(n), "n is not Or.");
          res = And(tnl);
        }
      } else if (isImplication(n)) {
        Assert(n->childrenCount() == 2, "Unexpected number of children.");
        TreeNode * fstNnfed = nnf_formula_ltl_rec(n->firstChild(), true);
        TreeNode * sndNnfed = nnf_formula_ltl_rec(n->secondChild(), false);
        res = And(fstNnfed, sndNnfed);
      } else if (isNext(n)) {
        Assert(n->childrenCount() == 1, "Unexpected number of children.");
        res = Next(nnf_formula_ltl_rec(n->firstChild(), false));
      } else if (isAlways(n)) {
        Assert(n->childrenCount() == 1, "Unexpected number of children.");
        res = Sometime(nnf_formula_ltl_rec(n->firstChild(), false));
      } else if (isSometime(n)) {
        Assert(n->childrenCount() == 1, "Unexpected number of children.");
        res = Always(nnf_formula_ltl_rec(n->firstChild(), false));
      } else if (isUntil(n)) {
        Assert(n->childrenCount() == 2, "Unexpected number of children.");
        TreeNode * tmp = nnf_formula_ltl_rec(n->secondChild(), false);
        res = Unless(tmp,
                     And(nnf_formula_ltl_rec(n->firstChild(), false),
                         tmp->clone()));
      } else if (isUnless(n)) {
        Assert(n->childrenCount() == 2, "Unexpected number of children.");
        TreeNode * tmp = nnf_formula_ltl_rec(n->secondChild(), false);
        res = Until(tmp,
                    And(nnf_formula_ltl_rec(n->firstChild(), false),
                        tmp->clone()));
      } else {
        Assert(false, "n has unexpected type.");
      }
    }

    return res;
  } // nnf_formula_ltl_rec

  void
  simplify_clausehs_pure_literal_fill
  (PropositionalProver::Literal const & l,
   UnsatCore::ClauseHandle const & ch,
   std::set<PropositionalProver::Proposition> & propositions,
   MapPToChs & mPosToChs,
   MapPToChs & mNegToChs)
  {
    PropositionalProver::Proposition p = l.getProposition();
    (void) propositions.insert(p);
    
    if (l.isPositive()) {
      if (mPosToChs.find(p) == mPosToChs.end()) {
        mPosToChs[p] = UnsatCore::ClauseHandleSet();
      }
      (void) mPosToChs[p].insert(ch);
    } else {
      if (mNegToChs.find(p) == mNegToChs.end()) {
        mNegToChs[p] = UnsatCore::ClauseHandleSet();
      }
      (void) mNegToChs[p].insert(ch);
    }
  } // simplify_clausehs_pure_literal_fill

  void simplify_clausehs_pure_literal_remove_clausehs
  (UnsatCore::ClauseHandleSet const chs,
   UnsatCore::ClauseHandleSet & clausehs,
   MapPToChs & mInitPosToChs,
   MapPToChs & mInitNegToChs,
   MapPToChs & mUnivNowEvPosToChs,
   MapPToChs & mUnivNowEvNegToChs,
   MapPToChs & mNextPosToChs,
   MapPToChs & mNextNegToChs)
  {
    for (UnsatCore::ClauseHandleSet::iterator it2 = chs.begin();
         it2 != chs.end();
         )
      {
        UnsatCore::ClauseHandle ch = *it2;
        ++it2;

        // remove ch from clausehs
        UnsatCore::ClauseHandleSet::iterator tmp = clausehs.find(ch);
        Assert(tmp != clausehs.end(), "tmp not present in clausehs.");
        (void) clausehs.erase(tmp);
    
        // remove ch from maps
        if (ch.isIClause()) {
          Adaptors::IClause ic = ch.getIClause();
          for (Adaptors::IClause::const_iterator it3 = ic.begin();
               it3 != ic.end();
               ++it3)
            {
              PropositionalProver::Literal l = *it3;
              Assert(l.getAttribute() == PropositionalProver::initial_attr,
                     "l is not an initial literal.");
              simplify_clausehs_pure_literal_remove_clauseh_from_maps(ch,
                                                                      l,
                                                                  mInitPosToChs,
                                                                 mInitNegToChs);
            }
        } else if (ch.isUClause()) {
          Adaptors::UClause uc = ch.getUClause();
          for (Adaptors::UClause::const_iterator it3 = uc.begin();
               it3 != uc.end();
               ++it3)
            {
              PropositionalProver::Literal l = *it3;
              Assert(l.getAttribute() == PropositionalProver::universal_attr,
                     "l is not a universal literal.");
              simplify_clausehs_pure_literal_remove_clauseh_from_maps(ch,
                                                                      l,
                                                             mUnivNowEvPosToChs,
                                                            mUnivNowEvNegToChs);
            }
        } else if (ch.isSClause()) {
          Adaptors::SClause sc = ch.getSClause();
          PropositionalProver::PClause pc = sc.getInternalRep();
          for (PropositionalProver::Clause::iterator it3 = pc->begin();
               it3 != pc->end();
               ++it3)
            {
              PropositionalProver::Literal l = *it3;
              Assert(l.getAttribute() ==
                     PropositionalProver::step_now_attr ||
                     l.getAttribute() ==
                     PropositionalProver::step_next_attr,
                     "l is not a step_now or step_next literal.");
              if (l.getAttribute() == PropositionalProver::step_now_attr) {
                simplify_clausehs_pure_literal_remove_clauseh_from_maps(ch,
                                                                        l,
                                                             mUnivNowEvPosToChs,
                                                            mUnivNowEvNegToChs);
              } else {
                simplify_clausehs_pure_literal_remove_clauseh_from_maps(ch,
                                                                        l,
                                                                  mNextPosToChs,
                                                                 mNextNegToChs);
              }
            }
        } else {
          Assert(ch.isEClause(), "ch is not an EClause.");
          TRIMs::EClause ec = ch.getEClause();
          for (TRIMs::EClause::const_present_iterator it3 = ec.present_begin();
               it3 != ec.present_end();
               ++it3)
            {
              PropositionalProver::Literal l = *it3;
              Assert(l.getAttribute() == PropositionalProver::universal_attr ||
                     l.getAttribute() == PropositionalProver::step_now_attr,
                     "l is not a universal or step_now literal.");
              simplify_clausehs_pure_literal_remove_clauseh_from_maps(ch,
                                                                      l,
                                                             mUnivNowEvPosToChs,
                                                            mUnivNowEvNegToChs);
            }
          PropositionalProver::Literal l = ec.getEventuality();
          simplify_clausehs_pure_literal_remove_clauseh_from_maps(ch,
                                                                  l,
                                                             mUnivNowEvPosToChs,
                                                            mUnivNowEvNegToChs);
        }
      }
  } // simplify_clausehs_pure_literal_remove_clausehs

  void simplify_clausehs_pure_literal_remove_clauseh_from_maps
  (UnsatCore::ClauseHandle const & ch,
   PropositionalProver::Literal const & l,
   MapPToChs & mPosToChs,
   MapPToChs & mNegToChs)
  {
    MapPToChs * m;
    PropositionalProver::Proposition p = l.getProposition();
    if (l.isPositive()) {
      m = &mPosToChs;
    } else {
      m = &mNegToChs;
    }
    MapPToChs::iterator tmp = m->find(p);
    Assert(tmp != m->end(), "p not present in *m.");
    UnsatCore::ClauseHandleSet::iterator tmp2 = tmp->second.find(ch);
    Assert(tmp2 != tmp->second.end(), "ch not present in *m[p].");
    (void) tmp->second.erase(tmp2);
    if (tmp->second.empty()) {
      m->erase(tmp);
    }
  } // simplify_clausehs_pure_literal_remove_clauseh_from_maps
} // namespace TrpLtlParser
