/************************************************************
 *    Copyright (C) 2012                                    *
 *    Viktor Schuppan (Viktor.Schuppan@gmx.de)              *
 *                                                          *
 *    This program is free software; you can redistribute   *
 *    it and/or modify it under the terms of the GNU        *
 *    General Public License as published by the Free       *
 *    Software Foundation, either version 3 of the License, *
 *    or (at your option) any later version.                *
 *                                                          *
 *    This program is distributed in the hope that it will  *
 *    be useful, but WITHOUT ANY WARRANTY; without even     *
 *    the implied warranty of MERCHANTABILITY or FITNESS    *
 *    FOR A PARTICULAR PURPOSE.  See the GNU General Public *
 *    License for more details.                             *
 *                                                          *
 *    You should have received a copy of the GNU General    *
 *    Public License along with this program.  If not, see  *
 *    <http://www.gnu.org/licenses/>.                       *
 ************************************************************/
#include <iostream>
#include <string>
#include <stdio.h>
#include <stdlib.h>
#include "adaptor/stl_tim_adaptor.h"
#include "builder/builder.h"
#include "getopt/options.h"
#include "misc/assert.h"
#include "misc/tracer.h"
#include "modules.h"
#include "stl_tim/pclause.h"
#include "trim/trim.h"
#include "unsat_core/clause_handle.h"
#include "formula.h"
#include "helpers.h"
#include "snf_translator.h"
#include "simplifier.h"
#include "trp_ltl_parser.h"

extern "C"
{
  extern FILE *trpltl_yyin;
}
extern void trpltl_yyerror(const char *error);
extern int trpltl_yyparse(void);
extern struct list *formula_list;

namespace TrpLtlParser
{
  TrpLtlParser::TrpLtlParser(TRPPPOptions const & options)
    : myFormulaList(),
      myClausehSet(),
      myClauseSet(),
      myMapSubformulaToClausehsNow(),
      myMapSubformulaToClausehsNext(),
      myMapSubformulaToClausehsSometime(),
      myOptions(options)
  { } // TrpLtlParser::TrpLtlParser

  void
  TrpLtlParser::parseLtl()
  {
    std::cout << "TrpLtlParser: Parsing input LTL formula begin."
              << std::endl;

    trpltl_yyin = fopen(myOptions.getFileName().c_str(), "r");
    if(!trpltl_yyin) {
      perror("Input file");
      exit(1);
    }

    trpltl_yyparse();

    fclose(trpltl_yyin);

    for(struct list *it = formula_list; !list_IsEmpty(it); it = list_Tail(it)) {
      tree* t = (tree*)list_Element(it);
      if(t) {
        myFormulaList.push_back(transformToTree((tree*)list_Element(it)));
      }
      tree_Delete(t);
    }
    list_Delete(formula_list);

    std::cout << "TrpLtlParser: Parsing input LTL formula end."
              << std::endl;
  } // TrpLtlParser::parseLtl

  void
  TrpLtlParser::replaceEquivalencesLtl()
  {
    std::cout << "TrpLtlParser: Replacing equivalences in input LTL formula begin."
              << std::endl;

    TRACE(builderltlModule, {
        std::cout << "TrpLtlParser: LTL formula before replacing equivalences:"
                  << std::endl;
        for (TreeNodeList::const_iterator it = myFormulaList.begin();
             it != myFormulaList.end();
             ++it)
          {
            std::cout << *it << std::endl;
          }
      });

    replaceEquivalencesFormulasLtl(myFormulaList);

    TRACE(builderltlModule, {
        std::cout << "TrpLtlParser: LTL formula after replacing equivalences:"
                  << std::endl;
        for (TreeNodeList::const_iterator it = myFormulaList.begin();
             it != myFormulaList.end();
             ++it)
          {
            std::cout << *it << std::endl;
          }
      });

    std::cout << "TrpLtlParser: Replacing equivalences in input LTL formula end."
              << std::endl;
  } // TrpLtlParser::reqplaceEquivalencesLtl

  void
  TrpLtlParser::nnfLtl()
  {
    std::cout << "TrpLtlParser: Transforming input LTL formula to NNF begin."
              << std::endl;

    TRACE(builderltlModule, {
        std::cout << "TrpLtlParser: LTL formula before transforming to NNF:"
                  << std::endl;
        for (TreeNodeList::const_iterator it = myFormulaList.begin();
             it != myFormulaList.end();
             ++it)
          {
            std::cout << *it << std::endl;
          }
      });

    nnfFormulasLtl(myFormulaList);

    TRACE(builderltlModule, {
        std::cout << "TrpLtlParser: LTL formula after transforming to NNF:"
                  << std::endl;
        for (TreeNodeList::const_iterator it = myFormulaList.begin();
             it != myFormulaList.end();
             ++it)
          {
            std::cout << *it << std::endl;
          }
      });

    std::cout << "TrpLtlParser: Transforming input LTL formula to NNF end."
              << std::endl;
  } // TrpLtlParser::nnfLtl

  void
  TrpLtlParser::simplifyLtl()
  {
    std::cout << "TrpLtlParser: Simplifying input LTL formula begin."
              << std::endl;

    TRACE(builderltlModule, {
        std::cout << "TrpLtlParser: LTL formula before simplification:"
                  << std::endl;
        for (TreeNodeList::const_iterator it = myFormulaList.begin();
             it != myFormulaList.end();
             ++it)
          {
            std::cout << *it << std::endl;
          }
      });

    unsigned long myNoVerticesSyntaxTreeBeforeLtlSimplification =
      sizeTreeNodeList(myFormulaList) + myFormulaList.size() - 1;
    simplifyFormulasLtl(myFormulaList, myOptions);
    unsigned long myNoVerticesSyntaxTreeAfterLtlSimplification =
      sizeTreeNodeList(myFormulaList) + myFormulaList.size() - 1;

    TRACE(builderltlModule, {
        std::cout << "TrpLtlParser: LTL formula after simplification:"
                  << std::endl;
        for (TreeNodeList::const_iterator it = myFormulaList.begin();
             it != myFormulaList.end();
             ++it)
          {
            std::cout << *it << std::endl;
          }
      });

    std::cout << "TrpLtlParser: number of nodes in syntax tree before LTL "
                 "simplification: "
              << myNoVerticesSyntaxTreeBeforeLtlSimplification
              << std::endl;
    std::cout << "TrpLtlParser: number of nodes in syntax tree after LTL "
                 "simplification: "
              << myNoVerticesSyntaxTreeAfterLtlSimplification
              << std::endl;

    std::cout << "TrpLtlParser: Simplifying input LTL formula end."
              << std::endl;
  } // TrpLtlParser::simplifyLtl

  void
  TrpLtlParser::translateLtlToSnf()
  {
    std::cout << "TrpLtlParser: Translating input LTL formula to SNF begin."
              << std::endl;

    SnfTranslator snfTranslator = SnfTranslator(myFormulaList,
                                                myClausehSet,
                                                myMapSubformulaToClausehsNow,
                                                myMapSubformulaToClausehsNext,
                                              myMapSubformulaToClausehsSometime,
                                                myOptions);
    snfTranslator.translate();

    std::cout << "TrpLtlParser: number of top level LTL subformulas directly "
                 "turned into SNF clauses: "
              << snfTranslator.myNoTopLevelSubformulaClauses
              << std::endl;
    std::cout << "TrpLtlParser: number of formula to SNF cache hits: "
              << snfTranslator.myNoFormulaToSnfCacheHits
              << std::endl;
    std::cout << "TrpLtlParser: number of formula to SNF cache misses: "
              << snfTranslator.myNoFormulaToSnfCacheMisses
              << std::endl;

    std::cout << "TrpLtlParser: Translating input LTL formula to SNF end."
              << std::endl;
  } // TrpLtlParser::translateLtlToSnf

  void
  TrpLtlParser::simplifySnf()
  {
    std::cout << "TrpLtlParser: Simplifying SNF begin."
              << std::endl;

    unsigned long myNoClausehsBeforePureLiteral = myClausehSet.size();
    simplifyClausehsPureLiteral(myClausehSet);
    unsigned long myNoClausehsAfterPureLiteral = myClausehSet.size();

    std::cout << "TrpLtlParser: number of clauses before pure literal simplification: "
              << myNoClausehsBeforePureLiteral
              << std::endl;
    std::cout << "TrpLtlParser: number of clauses after pure literal simplification: "
              << myNoClausehsAfterPureLiteral
              << std::endl;

    std::cout << "TrpLtlParser: Simplifying SNF end."
              << std::endl;
  } // TrpLtlParser::simplifySnf

  void
  TrpLtlParser::checkSnfForEmptyClause()
  {
    for (UnsatCore::ClauseHandleSet::iterator it = myClausehSet.begin();
         it != myClausehSet.end();
         ++it)
      {
        PropositionalProver::PClause pc;
        if (it->isIClause()) {
          pc = it->getIClause().getInternalRep();
        } else if (it->isUClause()) {
          pc = it->getUClause().getInternalRep();
        } else if (it->isSClause()) {
          pc = it->getSClause().getInternalRep();
        } else {
          Assert(it->isEClause(), "*it is not an EClause.");
          // EClause can't be empty
          continue;
        }
        if (pc->size() == 0) {
          std::cout << "Input formula is or has been simplified to False."
                    << std::endl;
          throw Exceptions::EmptyClauseInInput();
        }
      }
  } // TrpLtlParser::checkSnfForEmptyClause

  TreeNodeList
  TrpLtlParser::getLtlFormulaList()
  {
    return myFormulaList;
  } // TrpLtlParser::getLtlFormulaList

  TrpLtlParser::ClauseSet
  TrpLtlParser::getSnfClauseSet()
  {
    if (!myClausehSet.empty() && myClauseSet.empty()) {
      for (UnsatCore::ClauseHandleSet::iterator it = myClausehSet.begin();
           it != myClausehSet.end();
           ++it)
        {
          if (it->isIClause()) {
            myClauseSet.push_back(it->getIClause());
          } else if (it->isUClause()) {
            myClauseSet.push_back(it->getUClause());
          } else if (it->isSClause()) {
            myClauseSet.push_back(it->getSClause());
          } else {
            Assert(it->isEClause(), "it not an EClause.");
            myClauseSet.push_back(it->getEClause());
          }
        }
    }
    return myClauseSet;
  } // TrpLtlParser::getSnfClauseSet

  std::map<TreeNode*, UnsatCore::ClauseHandleSet>
  TrpLtlParser::getMapSubformulaToClausehsNow()
  {
    return myMapSubformulaToClausehsNow;
  } // TrpLtlParser::getMapSubformulaToClausehsNow

  std::map<TreeNode*, UnsatCore::ClauseHandleSet>
  TrpLtlParser::getMapSubformulaToClausehsNext()
  {
    return myMapSubformulaToClausehsNext;
  } // TrpLtlParser::getMapSubformulaToClausehsNext

  std::map<TreeNode*, UnsatCore::ClauseHandleSet>
  TrpLtlParser::getMapSubformulaToClausehsSometime()
  {
    return myMapSubformulaToClausehsSometime;
  } // TrpLtlParser::getMapSubformulaToClausehsSometime
} // namespace TrpLtlParser
