/**************************************************************/
/* ********************************************************** */
/* *                                                        * */
/* *  Copyright (C) 2001-2011                               * */
/* *  Boris Konev                                           * */
/* *  The University of Liverpool                           * */
/* *                                                        * */
/* *  Copyright (C) 2012                                    * */
/* *  Viktor Schuppan                                       * */
/* *                                                        * */
/* *  This program is free software; you can redistribute   * */
/* *  it and/or modify it under the terms of the GNU        * */
/* *  General Public License as published by the Free       * */
/* *  Software Foundation; either version 2 of the License, * */
/* *  or (at your option) any later version.                * */
/* *                                                        * */
/* *  This program is distributed in the hope that it will  * */
/* *  be useful, but WITHOUT ANY WARRANTY; without even     * */
/* *  the implied warranty of MERCHANTABILITY or FITNESS    * */
/* *  FOR A PARTICULAR PURPOSE.  See the GNU General Public * */
/* *  License for more details.                             * */
/* *                                                        * */
/* *  You should have received a copy of the GNU General    * */
/* *  Public License along with this program; if not, write * */
/* *  to the Free Software Foundation, Inc., 59 Temple      * */
/* *  Place, Suite 330, Boston, MA  02111-1307  USA         * */
/* *                                                        * */
/* *                                                        * */
/* ********************************************************** */
/**************************************************************/
#include "options.h"
#include <boost/tokenizer.hpp>
#include "config.h"
#include "modules.h"
#include "misc/tracer.h"

TRPPPOptions::TRPPPOptions(int argc, char** argv)
{
    gengetopt_args_info args_info; 

    if(cmdline_parser(argc, argv, &args_info) != 0)
    {
        reportFailure("Failed to parse the command line\n");
    }

    if(args_info.inputs_num != 1)
    {
        reportFailure("");
    }

    myFileName = args_info.inputs[0];

    if(args_info.format_input_given)
    {
        if(std::string("snf") == args_info.format_input_arg)
        {
            myInputFormat = INPUT_FORMAT_SNF;
        }
        else
        {
            if(std::string("ltl") == args_info.format_input_arg)
            {
                myInputFormat = INPUT_FORMAT_LTL;
            }
            else
            {
                reportFailure(std::string("Illegal value for the \"format-input\" option: ")+ args_info.format_input_arg);
            }
        }
    }
    else
    {
        myInputFormat = INPUT_FORMAT_SNF; // default value
    }

    myShowInput  = args_info.show_input_flag;
    myShowResult = args_info.show_result_flag;
    myQuiet      = args_info.quiet_flag;

    if(myQuiet)
    {
        if (myShowInput || myShowResult)
        {
            reportFailure("Incompatible options\n");
        }
        else
        {
            myShowInput = false;
            myShowResult = false;
        }
    }

    if(args_info.select_by_given)
    {
        if(std::string("DFS") == args_info.select_by_arg)
        {
            myDFS = true;
        }
        else 
            if(std::string("BFS") == args_info.select_by_arg)
            {
                myDFS = false;
            }
            else
            {
                reportFailure(std::string("Illegal value for the \"select-by\" option: ")+ args_info.select_by_arg);
            }
    }
    else
    {
        myDFS = true; // default value
    }

    myUsePretest = args_info.pretest_flag;

    myUseFSR = args_info.FSR_flag;

    if(args_info.order_given)
    {
        myOrderFile = args_info.order_arg;
    }
    else 
    {
        myOrderFile = "";
    }

    if(args_info.extract_uc_given)
    {
        if(std::string("none") == args_info.extract_uc_arg)
        {
            myExtractUnsatCoreMode = EXTRACT_UNSAT_CORE_MODE_NONE;
        }
        else
        {
            if(myQuiet || myUseFSR)
            {
                reportFailure("Incompatible options\n");
            }
            if(std::string("simple") == args_info.extract_uc_arg)
            {
                myExtractUnsatCoreMode = EXTRACT_UNSAT_CORE_MODE_SIMPLE;
            }
            else
            {
                if(std::string("positionsets") == args_info.extract_uc_arg)
                {
                    myExtractUnsatCoreMode = EXTRACT_UNSAT_CORE_MODE_POSITIONSETS;
                }
                else
                {
                    reportFailure(std::string("Illegal value for the \"extract-uc\" option: ")+ args_info.extract_uc_arg);
                }
            }
        }
    }
    else
    {
        myExtractUnsatCoreMode = EXTRACT_UNSAT_CORE_MODE_NONE; // default value
    }

    if(args_info.write_uc_given)
    {
        if (myExtractUnsatCoreMode == EXTRACT_UNSAT_CORE_MODE_NONE)
        {
            reportFailure(std::string("The \"write-uc\" option requires the \"extract-uc\" option."));
        }
        else
        {
            myExtractUnsatCoreFileName = args_info.write_uc_arg;
        }
    }
    else 
    {
        myExtractUnsatCoreFileName = ""; // default value
    }

    if(args_info.format_uc_given)
    {
        if (myExtractUnsatCoreMode == EXTRACT_UNSAT_CORE_MODE_NONE)
        {
            reportFailure(std::string("The \"format-uc\" option requires the \"extract-uc\" option."));
        } 
        else
        {
            if(std::string("internal") == args_info.format_uc_arg)
            {
                myExtractUnsatCoreFormat = EXTRACT_UNSAT_CORE_FORMAT_INTERNAL;
            }
            else
            {
                if(std::string("snf") == args_info.format_uc_arg)
                {
                    myExtractUnsatCoreFormat = EXTRACT_UNSAT_CORE_FORMAT_SNF;
                }
                else
                {
                    if(std::string("ltlc") == args_info.format_uc_arg)
                    {
                        myExtractUnsatCoreFormat = EXTRACT_UNSAT_CORE_FORMAT_LTLC;
                    }
                    else
                    {
                        if(std::string("ltl") == args_info.format_uc_arg)
                        {
                            if (myInputFormat == INPUT_FORMAT_SNF)
                            {
                                reportFailure(std::string("The \"format-uc ltl\" option requires the \"format-input ltl\" option."));
                            }
                            else
                            {
                                myExtractUnsatCoreFormat = EXTRACT_UNSAT_CORE_FORMAT_LTL;
                            }
                        }
                        else
                        {
                            reportFailure(std::string("Illegal value for the \"format-uc\" option: ")+ args_info.extract_uc_arg);
                        }
                    }
                }
            }
        }
    }
    else
    {
        myExtractUnsatCoreFormat = EXTRACT_UNSAT_CORE_FORMAT_INTERNAL; // default value
    }

    if(args_info.Parikh_algorithm_given)
    {
        if (myExtractUnsatCoreMode != EXTRACT_UNSAT_CORE_MODE_POSITIONSETS)
        {
            reportFailure(std::string("The \"Parikh-algorithm\" option requires the \"extract-uc positionsets\" option."));
        }
        else
        {
            if(std::string("Gawrychowski") == args_info.Parikh_algorithm_arg)
            {
                myExtractUnsatCoreParikhAlgorithm = EXTRACT_UNSAT_CORE_PARIKH_ALGORITHM_GAWRYCHOWSKI;
            }
            else
            {
                if(std::string("Sawa") == args_info.Parikh_algorithm_arg)
                {
                    myExtractUnsatCoreParikhAlgorithm = EXTRACT_UNSAT_CORE_PARIKH_ALGORITHM_SAWA;
                }
                else
                {
                    reportFailure(std::string("Illegal value for the \"Parikh-algorithm\" option: ")+ args_info.Parikh_algorithm_arg);
                }
            }
        }
    }
    else
    {
      myExtractUnsatCoreParikhAlgorithm = EXTRACT_UNSAT_CORE_PARIKH_ALGORITHM_GAWRYCHOWSKI; // default value
    }

    myNoNnf = args_info.no_nnf_flag;
    if (myNoNnf && myInputFormat != INPUT_FORMAT_LTL) {
      reportFailure(std::string("The \"no-nnf\" option requires the \"format-input ltl\" option."));
    }

    myNoLTLSimplification = args_info.no_ltl_simplification_flag;
    if (myNoLTLSimplification && myInputFormat != INPUT_FORMAT_LTL) {
      reportFailure(std::string("The \"no-ltl-simplification\" option requires the \"format-input ltl\" option."));
    }

    myNoDirectClauses = args_info.no_direct_clauses_flag;
    if (myNoDirectClauses && myInputFormat != INPUT_FORMAT_LTL) {
      reportFailure(std::string("The \"no-direct-clauses\" option requires the \"format-input ltl\" option."));
    }

    myNoEclauseReduction = args_info.no_eclause_reduction_flag;
    if (myNoEclauseReduction && myInputFormat != INPUT_FORMAT_LTL) {
      reportFailure(std::string("The \"no-eclause-reduction\" option requires the \"format-input ltl\" option."));
    }

    myNoSharing = args_info.no_sharing_flag;
    if (myNoSharing && myInputFormat != INPUT_FORMAT_LTL) {
      reportFailure(std::string("The \"no-sharing\" option requires the \"format-input ltl\" option."));
    }

    myNoSorting = args_info.no_sorting_flag;
    if (myNoSorting && myInputFormat != INPUT_FORMAT_LTL) {
      reportFailure(std::string("The \"no-sorting\" option requires the \"format-input ltl\" option."));
    }

    // Might want to make this available also for reading SNF
    myNoSNFSimplification = args_info.no_snf_simplification_flag;
    if (myNoSNFSimplification && myInputFormat != INPUT_FORMAT_LTL) {
      reportFailure(std::string("The \"no-snf-simplification\" option requires the \"format-input ltl\" option."));
    }

    myNoUnsatCoreSimplification = args_info.no_uc_simplification_flag;
    if (myNoUnsatCoreSimplification && myExtractUnsatCoreMode == EXTRACT_UNSAT_CORE_MODE_NONE) {
      reportFailure(std::string("The \"no-uc-simplification\" option requires the \"extract-uc\" option."));
    } 

#ifdef USE_TRACING
    if(args_info.trace_given)
    {
        std::string argString(args_info.trace_arg);
        boost::tokenizer<> tok(argString);

        for(boost::tokenizer<>::iterator beg=tok.begin(); beg!=tok.end();++beg)
        {
            if(*beg == "index")
            {
                setTrace(indexModule);
            }
            else
                if(*beg == "tempres")
                {
                    setTrace(tempresModule);
                }
                else
                    if(*beg == "loopsearch")
                    {
                        setTrace(loopsearchModule);
                    }
                    else
                        if(*beg == "resolution")
                        {
                            setTrace(resolutionModule);
                        }
                        else
                            if(*beg == "subsumption")
                            {
                                setTrace(subsumptionModule);
                            }
                            else
                                if(*beg == "forward")
                                {
                                    setTrace(forwardsubsumptionModule);
                                }
                                else
                                    if(*beg == "backward")
                                    {
                                        setTrace(backwardsubsumptionModule);
                                    }
                                    else 
                                        if(*beg == "pretest")
                                        {
                                            setTrace(pretestModule);
                                        }
                                        else
                                            if(*beg == "FSR")
                                            {
                                                setTrace(FSRModule);
                                            } 
                                            else 
                                                if(*beg == "unsatcore")
                                                {
                                                    setTrace(unsatcoreModule);
                                                } 
                                                else 
                                                    if(*beg == "builderltl")
                                                    {
                                                        setTrace(builderltlModule);
                                                    } 
                                                    else
                                                    {
                                                        reportFailure(std::string("Tracing of ") + *beg + " unsupported");
                                                    }
        }
    }
#endif
}

void
TRPPPOptions::reportFailure(std::string s )
{
    cmdline_parser_print_help();
    throw Exceptions::WrongOption(std::string("\n") + s);
}
