/**************************************************************/
/* ********************************************************** */
/* *                                                        * */
/* *  Copyright (C) 2001-2011                               * */
/* *  Boris Konev                                           * */
/* *  The University of Liverpool                           * */
/* *                                                        * */
/* *  Copyright (C) 2012                                    * */
/* *  Viktor Schuppan                                       * */
/* *                                                        * */
/* *  This program is free software; you can redistribute   * */
/* *  it and/or modify it under the terms of the GNU        * */
/* *  General Public License as published by the Free       * */
/* *  Software Foundation; either version 2 of the License, * */
/* *  or (at your option) any later version.                * */
/* *                                                        * */
/* *  This program is distributed in the hope that it will  * */
/* *  be useful, but WITHOUT ANY WARRANTY; without even     * */
/* *  the implied warranty of MERCHANTABILITY or FITNESS    * */
/* *  FOR A PARTICULAR PURPOSE.  See the GNU General Public * */
/* *  License for more details.                             * */
/* *                                                        * */
/* *  You should have received a copy of the GNU General    * */
/* *  Public License along with this program; if not, write * */
/* *  to the Free Software Foundation, Inc., 59 Temple      * */
/* *  Place, Suite 330, Boston, MA  02111-1307  USA         * */
/* *                                                        * */
/* *                                                        * */
/* ********************************************************** */
/**************************************************************/
#include <iostream>
#include <iomanip>
#include "adaptor/stl_tim_adaptor.h"
#include "trim/trim.h"
#include "misc/timer.h"  
#include "builder/parse.h"
#include "builder/generic_builder.h"
#include "builder/builder.h"
#include "loop_search/loopsearch.h"
#include "prover/search.h"
#include "loop_search/loopsearch.h"
#include "temporal_resolution/classicalrule.h"
//#include "temporal_resolution/DFKrule.h"
#include "prover/search.h"
#include "prover/temporalstat.h"
#include "interrupthandler.h"
#include "getopt/options.h"
#include "modules.h"
#include "misc/tracer.h"
#include "misc/exception.h"
#include "builder_ltl/trp_ltl_parser.h"
#include "unsat_core/unsat_core.h"

/*
#ifdef DEBUG
#define BOOST_INCLUDE_MAIN 
#include <boost/test/test_tools.hpp>
#endif // DEBUG
*/
#define BENCHMARKING

using namespace TRIMs;
using namespace Parsers;

typedef TRIM<Adaptors::Adaptor> ClauseSet;
typedef ClauseSet::Proposition Proposition;
typedef ClauseSet::Literal Literal;
typedef ClauseSet::LiteralList LiteralList;
typedef ClauseSet::IClause IClause;
typedef ClauseSet::UClause UClause;
typedef ClauseSet::SClause SClause;
typedef ClauseSet::EClause EClause;

typedef Builders::Builder<ClauseSet> Builder;

using std::cout;

// I use this function instead of main since I emply boost's mechanism
// to catch "uncought" exceptions. If something bad happens, and I do
// not catch this, hopefully, boost will do. This usually gives more
// information than just "segmentation fault" of "abort"

//#ifdef DEBUG
//int test_main(int argc, char **argv)
//#else
int main(int argc, char **argv)
//#endif
{
    Timer timer;

    extern int trp_yydebug;
    trp_yydebug = 0;

    // Initiating data structures
    // (The reason for this operator is that TRIM<> provides 
    //  with some static functions that might be used befoure
    //  any clause set is constructed (e.g. to construct such a set),
    //  and these function relay on some (static) data structures)
    //
    ClauseSet::InitiateStatic();

    try 
    {
        // parse command line options
        TRPPPOptions cmdOptions(argc, argv);

        // the set of clauses
        ClauseSet * pclauseSet;

        // the resolution graph
        UnsatCore::ResolutionGraph resolutionGraph(cmdOptions);

        // parse the file
        if (cmdOptions.getInputFormat() == INPUT_FORMAT_SNF) {
          TRPParser<Builder> parser(cmdOptions.getFileName(),
                                    cmdOptions.getOrderFileName());
          if (parser.parseOk()) {
            //get the set of clauses
            pclauseSet = new ClauseSet(parser.getClauseSet());
          } else {
            return 1;
          }
        } else {
          Assert(cmdOptions.getInputFormat() == INPUT_FORMAT_LTL,
                 "Unexpected input format.");
          TrpLtlParser::TrpLtlParser trpLtlParser(cmdOptions);
          trpLtlParser.parseLtl();
          trpLtlParser.replaceEquivalencesLtl();
          if (!cmdOptions.isNoNnf()) {
            trpLtlParser.nnfLtl();
          }
          if (!cmdOptions.isNoLTLSimplification()) {
            trpLtlParser.simplifyLtl();
          }
          trpLtlParser.translateLtlToSnf();
          if (!cmdOptions.isNoSNFSimplification()) {
            trpLtlParser.simplifySnf();
          }
          trpLtlParser.checkSnfForEmptyClause();
          pclauseSet = new ClauseSet(trpLtlParser.getSnfClauseSet());
          if (cmdOptions.isExtractUnsatCore()) {
            std::list<TreeNode *> tmp = trpLtlParser.getLtlFormulaList();
            resolutionGraph.setFormulaList(tmp);
            resolutionGraph.setMapSubformulaToClausehsNow(
                                  trpLtlParser.getMapSubformulaToClausehsNow());
            resolutionGraph.setMapSubformulaToClausehsNext(
                                 trpLtlParser.getMapSubformulaToClausehsNext());
            resolutionGraph.setMapSubformulaToClausehsSometime(
                             trpLtlParser.getMapSubformulaToClausehsSometime());
          }
        }

        // construct the prover object (for convinience, first, I 
        // give a name to the type
        typedef TemporalProver::Prover<ClauseSet,
          TemporalProver::LoopList<ClauseSet>,
          TemporalProver::ClassicalRule<ClauseSet> > 
            _Prover;
    
        // then, make an instance)
        _Prover prover(*pclauseSet);
        // dumping ClauseSet (if required)
        if(cmdOptions.isShowInput())
        {
            std::cout << "At the begining.." << std::endl;
            prover.getClauseSet().dump(std::cout);
            std::cout << std::endl;
        }
        // end dumping
        
        // this captures ^C if a user presses it. 
        // allows to see prover statistics on the fly
        HandleSIGINT<Timer, _Prover> handleSIGINT(&timer, &prover);
        
        if (cmdOptions.isExtractUnsatCore()) {
          // Add initial clauses to resolution graph
          for (ClauseSet::const_i_iterator ii = pclauseSet->initial_begin();
               ii != pclauseSet->initial_end();
               ii++)
            resolutionGraph.addStartingClause(*ii);

          // Add universal clauses to resolution graph
          for (ClauseSet::const_u_iterator iu = pclauseSet->universal_begin();
               iu != pclauseSet->universal_end();
               iu++)
            resolutionGraph.addStartingClause(*iu);

          // Add step clauses to resolution graph
          for (ClauseSet::const_s_iterator is = pclauseSet->step_begin();
               is != pclauseSet->step_end();
               is++)
            resolutionGraph.addStartingClause(*is);

          // Add eventuality clauses to resolution graph
          for (ClauseSet::const_e_iterator ie = pclauseSet->eventuality_begin();
               ie != pclauseSet->eventuality_end();
               ie++)
            resolutionGraph.addStartingClause(*ie);
        }

        // call the prover
        timer.start();
        bool result = prover.search(cmdOptions, resolutionGraph);
        timer.stop();
    
        // compute unsat core
        if (cmdOptions.isExtractUnsatCore() && result) {
          resolutionGraph.computeCore();
        }

        std::cout << (result?"Unsatisfiable":"Satisfiable") << std::endl;     
        if(!cmdOptions.isQuiet())
        {
            std::cout << "Elapsed time " << std::fixed << std::setprecision(3) << timer.elapsedSeconds() << "s" << std::endl;
            // statistics:
#ifdef COLLECT_STAT
            prover.outputStatistics(std::cout);
#endif // COLLECT_STAT
        }
        // dump unsat core (if needed)
        if (cmdOptions.isExtractUnsatCore() && result) {
#ifdef BENCHMARKING
          resolutionGraph.dumpCoreBenchmarking();
#else
          resolutionGraph.dumpCore();
#endif
        }
        // dumping ClauseSet (if needed)
        if(cmdOptions.isShowResult())
        {
            std::cout << "At the end.." << std::endl;
            prover.getClauseSet().dump(std::cout);
            std::cout << std::endl;
        }

        delete pclauseSet;

        // end dumping
        return 0;
    }
    catch(const char * ex)
    {
        std::cerr << ex << std::endl;
        exit(1);
    }
    catch (const std::string& s) 
    {
        std::cerr << s << std::endl;
        exit(1);
    }
    catch (Exceptions::EmptyClauseInInput &e)
    {
        std::cout << ("Unsatisfiable") << std::endl;     
        std::cout << ("Empty clause in the input") << std::endl;     
    }
    catch (Exceptions::Exception &e)
    {
        std::cout << e.module() << ": " << e.what() << '\n';
    }
    // basically, we never reach this point, 
    // put here in order to switch off warnings
    return 0;
}
