/************************************************************
 *    Copyright (C) 2012                                    *
 *    Viktor Schuppan (Viktor.Schuppan@gmx.de)              *
 *                                                          *
 *    This program is free software; you can redistribute   *
 *    it and/or modify it under the terms of the GNU        *
 *    General Public License as published by the Free       *
 *    Software Foundation, either version 3 of the License, *
 *    or (at your option) any later version.                *
 *                                                          *
 *    This program is distributed in the hope that it will  *
 *    be useful, but WITHOUT ANY WARRANTY; without even     *
 *    the implied warranty of MERCHANTABILITY or FITNESS    *
 *    FOR A PARTICULAR PURPOSE.  See the GNU General Public *
 *    License for more details.                             *
 *                                                          *
 *    You should have received a copy of the GNU General    *
 *    Public License along with this program.  If not, see  *
 *    <http://www.gnu.org/licenses/>.                       *
 ************************************************************/
#include <fstream>
#include <iostream>
#include <limits>
#include <map>
#include <set>
#include <string>
#include <utility>
#include <vector>
#include <boost/graph/adjacency_list.hpp>
#include <boost/graph/breadth_first_search.hpp>
#include <boost/graph/filtered_graph.hpp>
#include <boost/graph/graph_traits.hpp>
#include <boost/graph/graphviz.hpp>
#include <boost/graph/properties.hpp>
#include <boost/graph/reverse_graph.hpp>
#include <boost/graph/strong_components.hpp>
#include <boost/property_map/property_map.hpp>
#include <boost/tuple/tuple.hpp>
#include "builder_ltl/formula.h"
#include "getopt/options.h"
#include "misc/assert.h"
#include "misc/tracer.h"
#include "modules.h"
#include "bfs_reachables_recorder.h"
#include "clause_handle.h"
#include "misc.h"
#include "pclause_triple_store.h"
#include "resolution_graph.h"
#include "unsat_core_private.h"
#include "unsat_core_writer.h"

#define BENCHMARKING

namespace UnsatCore
{
  ResolutionGraph::ResolutionGraph(TRPPPOptions const & options)
    : myResolutionGraph(),
      myMapVertexToClauseh(),
      myMapClausehToVertex(),
      myMapEdgeToInferenceRuleEdgeLabeling(),
      myStartingClausehs(),
      myCoreStartingClausehs(),
      myMapClausehToSemiLinearSet(),
      myLastEmptyClause(),
      myLastEmptyClausePartition(invalidPartition),
      myPClauseTripleStore(),
      myCurrentPartition(mainPartition),
      myLoopInitializationSUClauseStore(),
      myLoopInitializationSClauseStore(),
      myLoopInitializationCClauseMap(),
      myMapVertexTo0StarReachableVertices(),
      myLoopItVertexStore(),
      myLoopItVertexStoreActive(false),
      myMainPartition(),
      myFormulaList(),
      myMapSubformulaToClausehsNow(),
      myMapSubformulaToClausehsNext(),
      myMapSubformulaToClausehsSometime(),
      myMapSubformulaToSemiLinearSet(),
      myCoreSimplifiedSubformulas(),
      myNoVerticesBeforeReachable(0),
      myNoVerticesAfterReachable(0),
      myNoEdgesBeforeReachable(0),
      myNoEdgesAfterReachable(0),
      myNoVerticesSyntaxTreeInputProblem(0),
      myNoVerticesSyntaxTreeUnsatCore(0),
      myNoEdgesAfterUnaryNFA(0),
      myNoSccs(0),
      myNoFwdImgAccelerated(0),
      myNoFwdImgNonAccelerated(0),
      myNoBwdImgAccelerated(0),
      myNoBwdImgNonAccelerated(0),
      myNoFwdImgCacheHits(0),
      myNoFwdImgCacheMisses(0),
      myNoBwdImgCacheHits(0),
      myNoBwdImgCacheMisses(0),
      myNoFwdImgRestrictedCacheHits(0),
      myNoFwdImgRestrictedCacheMisses(0),
      myOptions(options)
  { }

  void
  ResolutionGraph::addStartingClause(IClause const & startingClause)
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");
    Assert(myCurrentPartition == mainPartition,
           "Partition is not mainPartition");

    // Check for empty clause
    if (isPClauseEmpty(startingClause.getInternalRep())) {
      setLastEmptyClause(startingClause.getInternalRep(), false);
    }

    // Obtain clause handle
    ClauseHandle clauseh(startingClause, mainPartition);

    // Now that we've got handle, do work generically
    add_starting_clauseh(clauseh);
  } // ResolutionGraph::addStartingClause

  void
  ResolutionGraph::addStartingClause(UClause const & startingClause)
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");
    Assert(myCurrentPartition == mainPartition,
           "Partition is not mainPartition");

    // Check for empty clause
    if (isPClauseEmpty(startingClause.getInternalRep())) {
      setLastEmptyClause(startingClause.getInternalRep(), false);
    }

    // Obtain clause handle
    ClauseHandle clauseh(startingClause, mainPartition);

    // Now that we've got handle, do work generically
    add_starting_clauseh(clauseh);
  } // ResolutionGraph::addStartingClause

  void
  ResolutionGraph::addStartingClause(SClause const & startingClause)
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");
    Assert(myCurrentPartition == mainPartition,
           "Partition is not mainPartition");
    Assert(isPClauseStep(startingClause.getInternalRep()),
           "Not a step clause.");
    Assert(!isPClauseEmpty(startingClause.getInternalRep()),
           "SClause is the empty clause.");

    // Obtain clause handle
    ClauseHandle clauseh(startingClause, mainPartition);

    // Now that we've got handle, do work generically
    add_starting_clauseh(clauseh);
  } // ResolutionGraph::addStartingClause

  void
  ResolutionGraph::addStartingClause(EClause const & startingClause)
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");
    Assert(myCurrentPartition == mainPartition,
           "Partition is not mainPartition");

    // Obtain clause handle
    ClauseHandle clauseh(startingClause, mainPartition);

    // Now that we've got handle, do work generically
    add_starting_clauseh(clauseh);
  } // ResolutionGraph::addStartingClause

  void
  ResolutionGraph::addSaturationClause(PClause const & conclusion,
                                       PClause const & premise1,
                                       PClause const & premise2,
                                       bool const loopSearch)
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");

    // Obtain clause handles
    Partition partition = loopSearch ? myCurrentPartition : mainPartition;
    ClauseHandle conclusionh(conclusion, partition);
    ClauseHandle premise1h(premise1, partition);
    ClauseHandle premise2h(premise2, partition);

    // Determine edge labelings.
    InferenceRuleEdgeLabeling p1label, p2label;
    if (isPClauseInitial(conclusion)) {
      if (isPClauseInitial(premise1) && isPClauseInitial(premise2)) {
        p1label = INFERENCE_RULE_EDGE_LABELING_INIT_II;
        p2label = INFERENCE_RULE_EDGE_LABELING_INIT_II;
      } else if (isPClauseInitial(premise1) && isPClauseUniversal(premise2)) {
        p1label = INFERENCE_RULE_EDGE_LABELING_INIT_INI;
        p2label = INFERENCE_RULE_EDGE_LABELING_INIT_INN;
      } else if (isPClauseUniversal(premise1) && isPClauseInitial(premise2)) {
        p1label = INFERENCE_RULE_EDGE_LABELING_INIT_INN;
        p2label = INFERENCE_RULE_EDGE_LABELING_INIT_INI;
      } else {
        Assert(isPClauseEmpty(conclusion), "Conclusion not the empty clause.");
        if (isPClauseUniversal(premise1) && isPClauseUniversal(premise2)) {
          p1label = INFERENCE_RULE_EDGE_LABELING_STEP_NN;
          p2label = INFERENCE_RULE_EDGE_LABELING_STEP_NN;
        } else if (isPClauseUniversal(premise1) && isPClauseStep(premise2)) {
          p1label = INFERENCE_RULE_EDGE_LABELING_STEP_NXN;
          p2label = INFERENCE_RULE_EDGE_LABELING_STEP_NXX;
        } else if (isPClauseStep(premise1) && isPClauseUniversal(premise2)) {
          p1label = INFERENCE_RULE_EDGE_LABELING_STEP_NXX;
          p2label = INFERENCE_RULE_EDGE_LABELING_STEP_NXN;
        } else {
          Assert(isPClauseStep(premise1) && isPClauseStep(premise2),
                 "Premise1 or premsie2 not a step clause.");
          p1label = INFERENCE_RULE_EDGE_LABELING_STEP_XX;
          p2label = INFERENCE_RULE_EDGE_LABELING_STEP_XX;
        }
      }
    } else if (isPClauseUniversal(conclusion)) {
      if (isPClauseUniversal(premise1) && isPClauseUniversal(premise2)) {
        p1label = INFERENCE_RULE_EDGE_LABELING_STEP_NN;
        p2label = INFERENCE_RULE_EDGE_LABELING_STEP_NN;
      } else if (isPClauseUniversal(premise1) && isPClauseStep(premise2)) {
        p1label = INFERENCE_RULE_EDGE_LABELING_STEP_NXN;
        p2label = INFERENCE_RULE_EDGE_LABELING_STEP_NXX;
      } else if (isPClauseStep(premise1) && isPClauseUniversal(premise2)) {
        p1label = INFERENCE_RULE_EDGE_LABELING_STEP_NXX;
        p2label = INFERENCE_RULE_EDGE_LABELING_STEP_NXN;
      } else {
        Assert(isPClauseStep(premise1) && isPClauseStep(premise2),
               "Premise1 or premsie2 not a step clause.");
        p1label = INFERENCE_RULE_EDGE_LABELING_STEP_XX;
        p2label = INFERENCE_RULE_EDGE_LABELING_STEP_XX;
        }
    } else {
      Assert(isPClauseStep(conclusion), "Conclusion not a step clause.");
      if (isPClauseUniversal(premise1) && isPClauseStep(premise2)) {
        p1label = INFERENCE_RULE_EDGE_LABELING_STEP_NXN;
        p2label = INFERENCE_RULE_EDGE_LABELING_STEP_NXX;
      } else if (isPClauseStep(premise1) && isPClauseUniversal(premise2)) {
        p1label = INFERENCE_RULE_EDGE_LABELING_STEP_NXX;
        p2label = INFERENCE_RULE_EDGE_LABELING_STEP_NXN;
      } else {
        Assert(isPClauseStep(premise1) && isPClauseStep(premise2),
               "Premise1 or premsie2 not a step clause.");
        p1label = INFERENCE_RULE_EDGE_LABELING_STEP_XX;
        p2label = INFERENCE_RULE_EDGE_LABELING_STEP_XX;
      }
    }

    // Create vertex and edges, insert into graph, and update vertex
    // <-> clauseh and edge -> labeling maps
    add_clauseh_two_premises(conclusionh,
                             premise1h,
                             p1label,
                             premise2h,
                             p2label);
  } // ResolutionGraph::addSaturationClause

  void
  ResolutionGraph::addAugmentation1Clause(UClause const & conclusion,
                                          EClause const & premise)
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");
    Assert(myCurrentPartition == mainPartition,
           "Partition is not mainPartition");

    // Obtain clause handles
    ClauseHandle conclusionh(conclusion, mainPartition);
    ClauseHandle premiseh(premise, mainPartition);

    // Create vertex and edges, insert into graph, and update vertex
    // <-> clauseh and edge -> labeling maps
    add_clauseh_one_premise(conclusionh,
                            premiseh,
                            INFERENCE_RULE_EDGE_LABELING_AUG1);
  } // ResolutionGraph::addAugmentation1Clause

  void
  ResolutionGraph::addAugmentation2Clause(SClause const & conclusion)
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");
    Assert(myCurrentPartition == mainPartition,
           "Partition is not mainPartition");

    // Obtain clause handle
    ClauseHandle conclusionh(conclusion, mainPartition);

    // Create vertex, insert into graph, and update vertex <-> clauseh maps
    add_clauseh_no_premise(conclusionh);
  } // ResolutionGraph::addAugmentation1Clause

  void
  ResolutionGraph::clearLoopInitializationCClauseMap()
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");
    Assert(myCurrentPartition != mainPartition, "Partition is mainPartition");

    myLoopInitializationCClauseMap.clear();
  } // ResolutionGraph::clearLoopInitializationCClauseMap

  void
  ResolutionGraph::addLoopItInitializationCClause(SClause const & conclusion,
                                                  UClause const & premise)
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");
    Assert(myCurrentPartition != mainPartition, "Partition is mainPartition");

    // Add mapping from premise to conclusion to myLoopInitializationCClauseMap
    TRACE(unsatcoreModule, {
        std::cout << "ResolutionGraph: Adding mapping from premise ";
        premise.getInternalRep()->dump(std::cout);
        std::cout << " to conclusion ";
        conclusion.getInternalRep()->dump(std::cout);
        std::cout << " to myLoopInitializationClauseMap" << std::endl;
      });
    Assert(myLoopInitializationCClauseMap.find(premise) ==
           myLoopInitializationCClauseMap.end(),
           "premise already contained in myLoopInitializationCClauseMap.");
    myLoopInitializationCClauseMap[premise] = conclusion;

    // Obtain clause handle
    ClauseHandle conclusionh(conclusion, myCurrentPartition);

    // Create vertex, insert into graph, and update vertex <-> clauseh maps
    add_clauseh_no_premise(conclusionh);
  } // ResolutionGraph::addLoopItInitializationCClause

  void
  ResolutionGraph::addLoopItSubsumption(UClause const & conclusion,
                                        UClause const & premise)
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");
    Assert(myCurrentPartition != mainPartition, "Partition is mainPartition");

    TRACE(unsatcoreModule, {
        std::cout << "ResolutionGraph: Adding subsumption between premise ";
        premise.getInternalRep()->dump(std::cout);
        std::cout << " and conclusion ";
        conclusion.getInternalRep()->dump(std::cout);
        std::cout << std::endl;
      });

    // Obtain real conclusion
    Assert(myLoopInitializationCClauseMap.find(conclusion) !=
           myLoopInitializationCClauseMap.end(),
           "conclusion not contained in myLoopInitializationCClauseMap.");
    SClause real_conclusion = myLoopInitializationCClauseMap[conclusion];

    // Obtain clause handles
    ClauseHandle conclusionh(real_conclusion, myCurrentPartition);
    ClauseHandle premiseh(premise, myCurrentPartition);

    // Create edge, insert into graph, and update edge -> labeling map
    (void) add_edge(conclusionh,
                    premiseh,
                    INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_SUB);
  } // ResolutionGraph::addLoopItSubsumption


  void
  ResolutionGraph::addLoopItSubsumptionByEmptyClause(UClause const & conclusion)
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");
    Assert(myCurrentPartition != mainPartition, "Partition is mainPartition");

    TRACE(unsatcoreModule, {
        std::cout << "ResolutionGraph: Adding subsumption between the empty clause ";
        myLastEmptyClause->dump(std::cout);
        std::cout << " and conclusion ";
        conclusion.getInternalRep()->dump(std::cout);
        std::cout << std::endl;
      });

    // Obtain real conclusion
    Assert(myLoopInitializationCClauseMap.find(conclusion) !=
           myLoopInitializationCClauseMap.end(),
           "conclusion not contained in myLoopInitializationCClauseMap.");
    SClause real_conclusion = myLoopInitializationCClauseMap[conclusion];

    // Obtain clause handles
    ClauseHandle conclusionh(real_conclusion, myCurrentPartition);
    ClauseHandle premiseh(myLastEmptyClause, myCurrentPartition);

    // Create edge, insert into graph, and update edge -> labeling map
    (void) add_edge(conclusionh,
                    premiseh,
                    INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_SUB);
  } // ResolutionGraph::addLoopItSubsumptionByEmptyClause

  PClauseTripleStore &
  ResolutionGraph::getPClauseTripleStore()
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");

    return myPClauseTripleStore;
  } // ResolutionGraph::getPClauseTripleStore

  void
  ResolutionGraph::newPartition()
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");

    myCurrentPartition++;
  } // ResolutionGraph::newPartition

  void
  ResolutionGraph::clearLoopInitializationStores()
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");

    myLoopInitializationSUClauseStore.clear();
    myLoopInitializationSClauseStore.clear();
  } // ResolutionGraph::clearLoopInitializationStores

  void
  ResolutionGraph::addLoopInitializationSUClauseStore(SClause const & conclusion,
                                                      UClause const & premise)
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");

    myLoopInitializationSUClauseStore.push_back(std::make_pair(conclusion,
                                                               premise));
  } // ResolutionGraph::addLoopInitializationSUClauseStore

  void
  ResolutionGraph::addLoopInitializationSClauseStore(SClause const & conclusion)
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");

    myLoopInitializationSClauseStore.push_back(conclusion);
  } // ResolutionGraph::addLoopInitializationSClauseStore

  void
  ResolutionGraph::addLoopItInitializationNXClauses()
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");
    Assert(myCurrentPartition != mainPartition, "Partition is mainPartition");

    {
      std::vector<SClause>::iterator it;
      for (it = myLoopInitializationSClauseStore.begin();
           it != myLoopInitializationSClauseStore.end();
           ++it)
        {
          // Obtain clause handles
          ClauseHandle conclusionh(*it, myCurrentPartition);
          ClauseHandle premiseh(*it, mainPartition);
          
          // Create vertex and edges, insert into graph, and update vertex
          // <-> clauseh and edge -> labeling maps
          add_clauseh_one_premise(conclusionh,
                                  premiseh,
                               INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_X);
        }
    }

    {
      std::vector<std::pair<SClause, UClause> >::iterator it;
      for (it = myLoopInitializationSUClauseStore.begin();
           it != myLoopInitializationSUClauseStore.end();
           ++it)
        {
          // Obtain clause handles
          ClauseHandle conclusionh(it->first, myCurrentPartition);
          ClauseHandle premiseh(it->second, mainPartition);
          
          // Create vertex and edges, insert into graph, and update vertex
          // <-> clauseh and edge -> labeling maps
          add_clauseh_one_premise(conclusionh,
                                  premiseh,
                               INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_N);
        }
    }
  } // ResolutionGraph::addLoopItInitializationNXClauses

  void
  ResolutionGraph::addLoopConclusion1WEmptyClause(UClause const & conclusion,
                                                  EClause const & premisee)
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");
    Assert(myCurrentPartition != mainPartition, "Partition is mainPartition");
    Assert(myLastEmptyClause.get() != 0, "myLastEmptyClause is 0.");
    Assert(myLastEmptyClausePartition != invalidPartition,
           "myLastEmptyClausePartition is invalidPartition.");
    Assert(myLastEmptyClausePartition != mainPartition,
           "myLastEmptyClausePartition is mainPartition.");

    // Obtain clause handles
    ClauseHandle conclusionh(conclusion, mainPartition);
    ClauseHandle premisegh(myLastEmptyClause, myCurrentPartition);
    ClauseHandle premiseeh(premisee, mainPartition);

    // Create vertex and edges, insert into graph, and update vertex
    // <-> clauseh and edge -> labeling maps
    add_clauseh_two_premises(conclusionh,
                             premisegh,
                      INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION1_WEMPTYG,
                             premiseeh,
                     INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION1_WEMPTYE);
  } // ResolutionGraph::addLoopConclusion1WEmptyClause

  void
  ResolutionGraph::addLoopConclusion1WOEmptyClause(UClause const & conclusion,
                                                   UClause const & premiseg,
                                                   EClause const & premisee)
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");
    Assert(myCurrentPartition != mainPartition, "Partition is mainPartition");
    Assert(myLastEmptyClause.get() == 0, "myLastEmptyClause is not 0.");
    Assert(myLastEmptyClausePartition == invalidPartition,
           "myLastEmptyClausePartition is not invalidPartition.");

    // Obtain clause handles
    ClauseHandle conclusionh(conclusion, mainPartition);
    ClauseHandle premisegh(premiseg, myCurrentPartition);
    ClauseHandle premiseeh(premisee, mainPartition);

    // Create vertex and edges, insert into graph, and update vertex
    // <-> clauseh and edge -> labeling maps
    add_clauseh_two_premises(conclusionh,
                             premisegh,
                     INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION1_WOEMPTYG,
                             premiseeh,
                    INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION1_WOEMPTYE);
  } // ResolutionGraph::addLoopConclusion1WOEmptyClause

  void
  ResolutionGraph::addLoopConclusion2WEmptyClause(SClause const & conclusion)
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");
    Assert(myCurrentPartition != mainPartition, "Partition is mainPartition");
    Assert(myLastEmptyClause.get() != 0, "myLastEmptyClause is 0.");
    Assert(myLastEmptyClausePartition != invalidPartition,
           "myLastEmptyClausePartition is invalidPartition.");
    Assert(myLastEmptyClausePartition != mainPartition,
           "myLastEmptyClausePartition is mainPartition.");

    // Obtain clause handles
    ClauseHandle conclusionh(conclusion, mainPartition);
    ClauseHandle premiseh(myLastEmptyClause, myCurrentPartition);

    // Create vertex and edges, insert into graph, and update vertex
    // <-> clauseh and edge -> labeling maps
    add_clauseh_one_premise(conclusionh,
                            premiseh,
                     INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION2_WEMPTYG);
  } // ResolutionGraph::addLoopConclusion2WEmptyClause

  void
  ResolutionGraph::addLoopConclusion2WOEmptyClause(SClause const & conclusion,
                                                   UClause const & premise)
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");
    Assert(myCurrentPartition != mainPartition, "Partition is mainPartition");
    Assert(myLastEmptyClause.get() == 0, "myLastEmptyClause is not 0.");
    Assert(myLastEmptyClausePartition == invalidPartition,
           "myLastEmptyClausePartition is not invalidPartition.");

    // Obtain clause handles
    ClauseHandle conclusionh(conclusion, mainPartition);
    ClauseHandle premiseh(premise, myCurrentPartition);

    // Create vertex and edges, insert into graph, and update vertex
    // <-> clauseh and edge -> labeling maps
    add_clauseh_one_premise(conclusionh,
                            premiseh,
                    INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION2_WOEMPTYG);
  } // ResolutionGraph::addLoopConclusion2WOEmptyClause

  void
  ResolutionGraph::setLastEmptyClause(PClause const & clause,
                                      bool const loopSearch)
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");

    myLastEmptyClause = clause;
    if (clause.get() == 0) {
      myLastEmptyClausePartition = invalidPartition;
    } else if (loopSearch) {
      myLastEmptyClausePartition = myCurrentPartition;
    } else {
      myLastEmptyClausePartition = mainPartition;
    }
  } // ResolutionGraph::setLastEmptyClause

  void
  ResolutionGraph::activateLoopItVertexStore()
  {
    Assert(!myLoopItVertexStoreActive, "myLoopItVertexStoreActive is true.");
    Assert(myLoopItVertexStore.size() == 0,
           "myLoopItVertexStore is not empty.");

    myLoopItVertexStoreActive = true;
  } // ResolutionGraph::activateLoopItVertexStore

  void
  ResolutionGraph::removeFromLoopItVertexStore()
  {
    Assert(myLoopItVertexStoreActive, "myLoopItVertexStoreActive is false.");

    VertexSet::iterator it, itnext;
    it = myLoopItVertexStore.begin();
    for (itnext = it;
         it != myLoopItVertexStore.end();
         it = itnext)
      {
        ++itnext;
        remove_vertex(*it);
      }
  } // ResolutionGraph::removeFromLoopItVertexStore

  void
  ResolutionGraph::deactivateLoopItVertexStore()
  {
    Assert(myLoopItVertexStoreActive, "myLoopItVertexStoreActive is false.");

    myLoopItVertexStoreActive = false;
    myLoopItVertexStore.clear();
  } // ResolutionGraph::deactivateLoopItVertexStore

  void
  ResolutionGraph::computeReachableSubgraphFromEmptyClause()
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");
    Assert(myLastEmptyClause.get() != 0, "myLastEmptyClause is 0.");
    Assert(myLastEmptyClausePartition != invalidPartition,
           "myLastEmptyClausePartition is invalidPartition");
    Assert(myLastEmptyClausePartition == mainPartition,
           "myLastEmptyClausePartition is not mainPartition");
    Assert(myCoreStartingClausehs.empty(),
           "myCoreStartingClausehs is not empty.");

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing reachable subgraph from empty "
                   "clause begin"
                << std::endl;
    }

    // Get statistics before pruning
    myNoVerticesBeforeReachable = boost::num_vertices(myResolutionGraph);
    myNoEdgesBeforeReachable = boost::num_edges(myResolutionGraph);

    // Obtain start vertex
    ClauseHandle startClauseh(myLastEmptyClause, myLastEmptyClausePartition);
    Assert(myMapClausehToVertex.find(startClauseh) !=
           myMapClausehToVertex.end(),
           "myMapClauseHandleToVertex does not contain startClauseh.");
    VertexSet start = VertexSet();
    (void) start.insert(myMapClausehToVertex[startClauseh]);

    compute_reachable_subgraph(start);

    // Get statistics after pruning
    myNoVerticesAfterReachable = boost::num_vertices(myResolutionGraph);
    myNoEdgesAfterReachable = boost::num_edges(myResolutionGraph);

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing reachable subgraph from empty "
                   "clause end"
                << std::endl;
    }
  } // ResolutionGraph::computeReachableSubgraphFromEmptyClause

  void
  ResolutionGraph::computeReachableSubgraphFromMainPartition()
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");
    Assert(myCoreStartingClausehs.empty(),
           "myCoreStartingClausehs is not empty.");

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing reachable subgraph from main "
                   "partition begin"
                << std::endl;
    }

    // Get statistics before pruning
    unsigned long noVerticesBeforeReachable;
    noVerticesBeforeReachable = boost::num_vertices(myResolutionGraph);
    unsigned long noEdgesBeforeReachable;
    noEdgesBeforeReachable = boost::num_edges(myResolutionGraph);

    compute_reachable_subgraph(myMainPartition);

    // Get statistics after pruning
    unsigned long noVerticesAfterReachable;
    noVerticesAfterReachable = boost::num_vertices(myResolutionGraph);
    unsigned long noEdgesAfterReachable;
    noEdgesAfterReachable = boost::num_edges(myResolutionGraph);

    TRACE(unsatcoreModule, {
        std::cout << "ResolutionGraph: number of vertices before pruning to "
                     "main partition reachable vertices: "
                  << noVerticesBeforeReachable
                  << std::endl;
        std::cout << "ResolutionGraph: number of edges before pruning to "
                     "main partition reachable vertices: "
                  << noEdgesBeforeReachable
                  << std::endl;
        std::cout << "ResolutionGraph: number of vertices after pruning to "
                     "main partition reachable vertices: "
                  << noVerticesAfterReachable
                  << std::endl;
        std::cout << "ResolutionGraph: number of edges after pruning to "
                     "main partition reachable vertices: "
                  << noEdgesAfterReachable
                  << std::endl;
      });

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing reachable subgraph from main "
                   "partition end"
                << std::endl;
    }
  } // ResolutionGraph::computeReachableSubgraphFromMainPartition

  void
  ResolutionGraph::computeCoreStartingClauses()
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");
    Assert(myLastEmptyClause.get() != 0, "myLastEmptyClause is 0.");
    Assert(myLastEmptyClausePartition != invalidPartition,
           "myLastEmptyClausePartition is invalidPartition");
    Assert(myLastEmptyClausePartition == mainPartition,
           "myLastEmptyClausePartition is not mainPartition");
    Assert(myCoreStartingClausehs.empty(),
           "myCoreStartingClausehs is not empty.");

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing core starting clauses begin"
                << std::endl;
    }

    std::set<ClauseHandle>::iterator it;
    for (it = myStartingClausehs.begin();
         it != myStartingClausehs.end();
         ++it)
      {
        std::map<ClauseHandle, Vertex>::iterator it2 =
          myMapClausehToVertex.find(*it);
        if (it2 != myMapClausehToVertex.end()) {
          bool inserted;
          std::set<ClauseHandle>::iterator dummy;
          boost::tie(dummy, inserted) = myCoreStartingClausehs.insert(*it);
          Assert(inserted, "*it already present in myCoreStartingClauses.");
        }
      }

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing core starting clauses end"
                << std::endl;
    }
  } // ResolutionGraph::computeCoreStartingClauses

  // Construct filtered graph with only 0 edges
  template <typename EdgeWeightMap> struct ZeroEdgeWeight {
    ZeroEdgeWeight() { }
    ZeroEdgeWeight(EdgeWeightMap weight) : m_weight(weight) { }
    template <typename Edge>
    bool operator()(const Edge& e) const {
      return 0 == get(m_weight, e);
    }
    EdgeWeightMap m_weight;
  };

  void
  ResolutionGraph::transformToUnaryNFA()
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");
    Assert(myLastEmptyClause.get() != 0, "myLastEmptyClause is 0.");
    Assert(myLastEmptyClausePartition != invalidPartition,
           "myLastEmptyClausePartition is invalidPartition");
    Assert(myLastEmptyClausePartition == mainPartition,
           "myLastEmptyClausePartition is not mainPartition");
    Assert(!myCoreStartingClausehs.empty(), "myCoreStartingClausehs is empty.");
    Assert(boost::num_vertices(myResolutionGraph) <
           std::numeric_limits<unsigned long>::max(),
           "Number of vertices in graph is larger than maximum value for"
           "unsigned long.");

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: transforming to unary NFA begin"
                << std::endl;
    }

    TRACE(unsatcoreModule, {
        std::ofstream myUnaryNFAFile;
        std::string const fileName = "/tmp/rg.dot";
        myUnaryNFAFile.open(fileName.data());
        if (!myUnaryNFAFile.is_open()) {
          std::cout << "Could not open file for writing unsatisfiable core "
                    << fileName
                    << ". Writing to stdout."
                    << std::endl;
        }
        std::ostream & os =
          myUnaryNFAFile.is_open() ? myUnaryNFAFile : std::cout;
        write_graphviz(os, myResolutionGraph);
        myUnaryNFAFile.close();
      });

    // The vertex index
    myVertexIndex = boost::get(boost::vertex_index, myResolutionGraph);
    {
      VertexIterator vi, vend;
      unsigned long cnt = 0;
      for(boost::tie(vi,vend) = boost::vertices(myResolutionGraph);
          vi != vend;
          ++vi)
        {
          boost::put(myVertexIndex, *vi, cnt++);
        }
    }

    // The edge weights map
    typedef boost::property_map<Graph, boost::edge_weight_t>::type EdgeWeightMap;
    EdgeWeightMap edgeWeight;
    edgeWeight = boost::get(boost::edge_weight, myResolutionGraph);
    {
      boost::graph_traits<Graph>::edge_iterator ei, eend;
      for(boost::tie(ei,eend) = boost::edges(myResolutionGraph);
          ei != eend;
          ++ei)
        {
          Assert(myMapEdgeToInferenceRuleEdgeLabeling.find(*ei) !=
                 myMapEdgeToInferenceRuleEdgeLabeling.end(),
                 "*ei is not present in myMapEdgeToInferenceRuleEdgeLabeling");
          InferenceRuleEdgeLabeling l;
          long w = std::numeric_limits<long>::min();
          l = myMapEdgeToInferenceRuleEdgeLabeling[*ei];
          switch (l) {
          case INFERENCE_RULE_EDGE_LABELING_INIT_II:
          case INFERENCE_RULE_EDGE_LABELING_INIT_INI:
          case INFERENCE_RULE_EDGE_LABELING_INIT_INN:
          case INFERENCE_RULE_EDGE_LABELING_STEP_NN:
          case INFERENCE_RULE_EDGE_LABELING_STEP_NXX:
          case INFERENCE_RULE_EDGE_LABELING_STEP_XX:
          case INFERENCE_RULE_EDGE_LABELING_AUG1:
          case INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_X:
          case INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION1_WEMPTYG:
          case INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION1_WEMPTYE:
          case INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION1_WOEMPTYG:
          case INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION1_WOEMPTYE:
            w = 0;
            break;
          case INFERENCE_RULE_EDGE_LABELING_STEP_NXN:
          case INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_N:
          case INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_SUB:
          case INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION2_WEMPTYG:
          case INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION2_WOEMPTYG:
            w = 1;
            break;
          default:
            Assert(false, "Unexpected label.");
          }
          boost::put(edgeWeight, *ei, w);
        }
    }

    // Construct filtered graph with only 0 edges
    ZeroEdgeWeight<EdgeWeightMap>
      zeroEdgeWeightFilter(boost::get(boost::edge_weight, myResolutionGraph));
    boost::filtered_graph<Graph, ZeroEdgeWeight<EdgeWeightMap> >
      zeroEdgeWeightGraph(myResolutionGraph, zeroEdgeWeightFilter);

    // Determine the sets of vertices reachable via 0* edges
    {
      VertexIterator vi, vend;
      for(boost::tie(vi,vend) = boost::vertices(myResolutionGraph);
          vi != vend;
          ++vi)
        {
          Assert(myMapVertexTo0StarReachableVertices.find(*vi) ==
                 myMapVertexTo0StarReachableVertices.end(),
                 "*vi present in myMapVertexTo0StarReachableVertices.");
          myMapVertexTo0StarReachableVertices[*vi] = VertexSet();

          BfsReachablesRecorder<std::set<Vertex> >
            vis(&(myMapVertexTo0StarReachableVertices[*vi]));
          boost::breadth_first_search(zeroEdgeWeightGraph,
                                      *vi,
                                      boost::visitor(vis));

          TRACE(unsatcoreModule, {
              std::cout << "ResolutionGraph: myMapVertexTo0StarReachableVertices of vertex "
                        << *vi
                        << " has size "
                        << myMapVertexTo0StarReachableVertices[*vi].size()
                        << std::endl;
            });
        }
    }

    // Determine the sets of vertices reachable via 10* edges
    std::map<Vertex, VertexSet> mapVertexTo10StarReachableVertices;
    {
      VertexIterator vi, vend;
      for(boost::tie(vi,vend) = boost::vertices(myResolutionGraph);
          vi != vend;
          ++vi)
        {
          Assert(mapVertexTo10StarReachableVertices.find(*vi) ==
                 mapVertexTo10StarReachableVertices.end(),
                 "*vi present in mapVertexTo10StarReachableVertices.");
          mapVertexTo10StarReachableVertices[*vi] = VertexSet();

          boost::graph_traits<Graph>::out_edge_iterator ei, eend;
          for (boost::tie(ei, eend) = boost::out_edges(*vi, myResolutionGraph);
               ei != eend;
               ++ei)
            {
              if (boost::get(edgeWeight, *ei) == 1) {
                Vertex dest = boost::target(*ei, myResolutionGraph);
                Assert(myMapVertexTo0StarReachableVertices.find(dest) !=
                       myMapVertexTo0StarReachableVertices.end(),
                       "dest not present in myMapVertexTo0StarReachableVertices.");
                mapVertexTo10StarReachableVertices[*vi].
                  insert(myMapVertexTo0StarReachableVertices[dest].begin(),
                         myMapVertexTo0StarReachableVertices[dest].end());
              }
            }

          TRACE(unsatcoreModule, {
              std::cout << "ResolutionGraph: mapVertexTo10StarReachableVertices of vertex "
                        << *vi
                        << " has size "
                        << mapVertexTo10StarReachableVertices[*vi].size()
                        << std::endl;
            });
        }
    }

    // Determine the sets of vertices reachable via 0*10* edges
    std::map<Vertex, VertexSet> mapVertexTo0Star10StarReachableVertices;
    {
      VertexIterator vi, vend;
      for(boost::tie(vi,vend) = boost::vertices(myResolutionGraph);
          vi != vend;
          ++vi)
        {
          Assert(mapVertexTo0Star10StarReachableVertices.find(*vi) ==
                 mapVertexTo0Star10StarReachableVertices.end(),
                 "*vi present in mapVertexTo0Star10StarReachableVertices.");
          mapVertexTo0Star10StarReachableVertices[*vi] = VertexSet();

          Assert(myMapVertexTo0StarReachableVertices.find(*vi) !=
                 myMapVertexTo0StarReachableVertices.end(),
                 "*vi not present in myMapVertexTo0StarReachableVertices.");
          for (VertexSet::iterator it =
                 myMapVertexTo0StarReachableVertices[*vi].begin();
               it != myMapVertexTo0StarReachableVertices[*vi].end();
               ++it)
            {
              Assert(mapVertexTo10StarReachableVertices.find(*it) !=
                     mapVertexTo10StarReachableVertices.end(),
                     "*it not present in mapVertexTo10StarReachableVertices.");
              mapVertexTo0Star10StarReachableVertices[*vi].
                insert(mapVertexTo10StarReachableVertices[*it].begin(),
                       mapVertexTo10StarReachableVertices[*it].end());
            }

          TRACE(unsatcoreModule, {
              std::cout << "ResolutionGraph: mapVertexTo0Star10StarReachableVertices of vertex "
                        << *vi
                        << " has size "
                        << mapVertexTo0Star10StarReachableVertices[*vi].size()
                        << std::endl;
            });
        }
    }

    // Remove all existing edges
    {
      myMapEdgeToInferenceRuleEdgeLabeling.clear();

      VertexIterator vi, vend;
      for(boost::tie(vi,vend) = boost::vertices(myResolutionGraph);
          vi != vend;
          ++vi)
        {
          boost::clear_vertex(*vi, myResolutionGraph);
        }
    }

    // Add new edges between vertices according to
    // mapVertexTo0Star10StarReachableVertices
    {
      VertexIterator vi, vend;
      for(boost::tie(vi,vend) = boost::vertices(myResolutionGraph);
          vi != vend;
          ++vi)
        {
          Assert(mapVertexTo0Star10StarReachableVertices.find(*vi) !=
                 mapVertexTo0Star10StarReachableVertices.end(),
                 "*vi not present in mapVertexTo0Star10StarReachableVertices.");
          for (VertexSet::iterator it =
                 mapVertexTo0Star10StarReachableVertices[*vi].begin();
               it != mapVertexTo0Star10StarReachableVertices[*vi].end();
               ++it)
            {
              Edge edge;
              bool inserted;
              boost::tie(edge, inserted) = boost::add_edge(*vi,
                                                           *it,
                                                           myResolutionGraph);
              Assert(inserted, "edge already present in myResolutionGraph.");
              TRACE(unsatcoreModule, {
                  std::cout << "ResolutionGraph: Adding unary NFA edge between "
                            << *vi
                            << " and "
                            << *it
                            << std::endl;
                });
            }
        }
    }

    TRACE(unsatcoreModule, {
        std::ofstream myUnaryNFAFile;
        std::string const fileName = "/tmp/nfa.dot";
        myUnaryNFAFile.open(fileName.data());
        if (!myUnaryNFAFile.is_open()) {
          std::cout << "Could not open file for writing unsatisfiable core "
                    << fileName
                    << ". Writing to stdout."
                    << std::endl;
        }
        std::ostream & os =
          myUnaryNFAFile.is_open() ? myUnaryNFAFile : std::cout;
        write_graphviz(os, myResolutionGraph);
        myUnaryNFAFile.close();
      });

    // Set myNoEdgesAfterUnaryNFA
    myNoEdgesAfterUnaryNFA = num_edges(myResolutionGraph);

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: transforming to unary NFA end"
                << std::endl;
    }
  } // ResolutionGraph::transformToUnaryNFA

  /*
    An optimized version of the following algorithm

    Z. Sawa: "Efficient Construction of Semilinear Representations of
    Languages Accepted by Unary Nondeterministic Finite
    Automata". Accepted for publication in Fundamenta Informaticae.
   */
  void
  ResolutionGraph::computeParikhImages_Sawa()
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");
    Assert(myLastEmptyClause.get() != 0, "myLastEmptyClause is 0.");
    Assert(myLastEmptyClausePartition != invalidPartition,
           "myLastEmptyClausePartition is invalidPartition");
    Assert(myLastEmptyClausePartition == mainPartition,
           "myLastEmptyClausePartition is not mainPartition");
    Assert(!myCoreStartingClausehs.empty(), "myCoreStartingClausehs is empty.");
    Assert(boost::num_vertices(myResolutionGraph) *
           boost::num_vertices(myResolutionGraph) <
           std::numeric_limits<unsigned long>::max(),
           "Number of vertices in graph squared is larger than maximum value for"
           "unsigned long.");

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing Parikh images Sawa begin"
                << std::endl;
    }

    /* Preparation */

    // The clauses with more than 1 backward reachable vertex.
    //
    // Note that Sawa's algorithm treats automata of size 1 as a
    // special case. O.t.o.h., note that, upon reaching this point,
    // all starting clausehs will be non-empty. That implies that if a
    // starting clauseh has only 1 backward reachable vertex, then
    // this starting clauseh is not reachable from the final empty
    // clause in the unary NFA. Hence, such clauseh will be assigned
    // {(0 N, 0)} as its only sls.
    std::set<ClauseHandle> coreStartingClausehsBrVGt1;

    // Needed a couple of times
    unsigned long numVertices = boost::num_vertices(myResolutionGraph);
    unsigned long infty = std::numeric_limits<unsigned long>::max();

    // Find vertex in main partition labeled with the empty clause
    Assert(myMapClausehToVertex.find(ClauseHandle(myLastEmptyClause,
                                                  mainPartition)) !=
           myMapClausehToVertex.end(),
           "myLastEmptyClause not present in myMapClausehToVertex.");
    Vertex emptyClauseVertex =
      myMapClausehToVertex[ClauseHandle(myLastEmptyClause, mainPartition)];

    // Compute numbers of states backward reachable from final vertices
    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing numbers of backward reachable states begin"
                << std::endl;
    }
    std::map<Vertex, unsigned long> numBackwardReachableVertices;
    unsigned long maxNumBackwardReachableVertices =
      std::numeric_limits<unsigned long>::min();
    unsigned long minNumBackwardReachableVertices =
      std::numeric_limits<unsigned long>::max();
    {
      std::set<ClauseHandle>::iterator it;
      for (it = myCoreStartingClausehs.begin();
           it != myCoreStartingClausehs.end();
           ++it)
        {
          Assert(myMapClausehToVertex.find(*it) != myMapClausehToVertex.end(),
                 "*it is not present in myMapClausehToVertex");

          Vertex finalVertex = myMapClausehToVertex[*it];

          // This set will contain the reachable vertices
          std::set<Vertex> reachable;

          BfsReachablesRecorder<std::set<Vertex> > vis(&(reachable));
          boost::breadth_first_search(boost::make_reverse_graph(myResolutionGraph),
                                      finalVertex,
                                      boost::visitor(vis));

          unsigned long numReachable = reachable.size();

          numBackwardReachableVertices[finalVertex] = numReachable;
          if (numReachable > 1) {
            (void) coreStartingClausehsBrVGt1.insert(*it);
          }
          if (numReachable > maxNumBackwardReachableVertices) {
            maxNumBackwardReachableVertices = numReachable;
          }
          if (numReachable < minNumBackwardReachableVertices) {
            minNumBackwardReachableVertices = numReachable;
          }

          if (!myOptions.isQuiet()) {
            std::cout << "ResolutionGraph: from clauseh ";
            it->dump(std::cout);
            std::cout << " there are "
                      << numReachable
                      << " out of "
                      << numVertices
                      << " vertices backward reachable."
                      << std::endl;
          }
        }
    }
    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: maximal number of backward reachable vertices: "
                << maxNumBackwardReachableVertices
                << std::endl;
      std::cout << "ResolutionGraph: minimal number of backward reachable vertices: "
                << minNumBackwardReachableVertices
                << std::endl;
      std::cout << "ResolutionGraph: computing numbers of backward reachable states end"
                << std::endl;
    }

    // Prepare semi-linear sets
    {
      LinearSet ls00 = LinearSet(0, 0);

      std::set<ClauseHandle>::iterator it;
      for (it = myCoreStartingClausehs.begin();
           it != myCoreStartingClausehs.end();
           ++it)
        {
          Assert(myMapClausehToVertex.find(*it) != myMapClausehToVertex.end(),
                 "*it is not present in myMapClausehToVertex");

          Vertex finalVertex = myMapClausehToVertex[*it];

          // Avoid repeated calls to myMapClausehToSemiLinearSet[*it]
          SemiLinearSet & slsOfFinalVertex = myMapClausehToSemiLinearSet[*it];

          // Initialize semi-linear set
          slsOfFinalVertex = SemiLinearSet();

          // If finalVertex is reachable only via 0-edges, then need
          // to add (0 N, 0)
          Assert(myMapVertexTo0StarReachableVertices.find(emptyClauseVertex) !=
                 myMapVertexTo0StarReachableVertices.end(),
                 "emptyClauseVertex not present in "
                 "myMapVertexTo0StarReachableVertices.");
          if (myMapVertexTo0StarReachableVertices[emptyClauseVertex].
              find(finalVertex) !=
              myMapVertexTo0StarReachableVertices[emptyClauseVertex].end()) {
            slsOfFinalVertex.addLinearSet(ls00);
            TRACE(unsatcoreModule, {
                std::cout << "ResolutionGraph: Added linear set";
                ls00.dump(std::cout);
                std::cout << " to semilinear set of clauseh ";
                it->dump(std::cout);
                std::cout << std::endl;
              });
          }
        }
    }

    if (coreStartingClausehsBrVGt1.size() == 0) {
      // Clear cache
      myFwdImgCache.clear();

      if (!myOptions.isQuiet()) {
        std::cout << "ResolutionGraph: no vertex with more than 1 backward "
                     "reachable vertices. Returning."
                  << std::endl;
        std::cout << "ResolutionGraph: computing Parikh images Sawa end"
                  << std::endl;
      }
    }

    /**************************************************************************/
    /* Final vertex independent part                                          */
    /**************************************************************************/

    /* Strongly connected components */

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing Parikh images Sawa - sccs begin"
                << std::endl;
    }

    // The component map that will map vertices to their sccs and its reverse
    std::map<Vertex, unsigned long> componentMap;
    std::map<unsigned long, VertexSet> mapSccNoToVertices;

    // Execute scc algorithm
    myNoSccs = boost::strong_components(myResolutionGraph,
                                  boost::make_assoc_property_map(componentMap));

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing Parikh images Sawa - sccs end"
                << std::endl;
    }

    // Compute the reverse mapping
    {
      for (unsigned long i = 0; i < myNoSccs; ++i)
        {
          mapSccNoToVertices[i] = VertexSet();
        }
      for (std::map<Vertex, unsigned long>::iterator it = componentMap.begin();
           it != componentMap.end();
           ++it)
        {
          (void) mapSccNoToVertices[it->second].insert(it->first);
        }
    }

    /* sawaSl */

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing Parikh images Sawa - sawaSl begin"
                << std::endl;
    }

    // sl in Sawa's algorithm
    std::map<Vertex, unsigned long> sawaSl;
    {
      VertexIterator vi, vend;
      for(boost::tie(vi,vend) = boost::vertices(myResolutionGraph);
          vi != vend;
          ++vi)
        {
          Assert(sawaSl.find(*vi) == sawaSl.end(), "*vi present in sawaSl.");
          sawaSl[*vi] = infty;

          // Forward bfs to determine sawaSl[*vi]
          unsigned long distance = 0;
          {
            bool found = false;
            VertexSet reached = VertexSet();
            VertexSet frontier = VertexSet();
            VertexSet image;
            VertexSet scc;

            {
              unsigned long sccNo;
              Assert(componentMap.find(*vi) != componentMap.end(),
                     "*vi not present in componentMap");
              sccNo = componentMap[*vi];
              Assert(mapSccNoToVertices.find(sccNo) != mapSccNoToVertices.end(),
                     "sccNo not present in mapSccNoToVertices.");
              scc = mapSccNoToVertices[sccNo];
            }
            (void) frontier.insert(*vi);
            while (!found && !frontier.empty()) {
              image = VertexSet();
              forward_image(frontier, image);
              ++myNoFwdImgNonAccelerated;
              for (VertexSet::iterator it = image.begin(); // restrict to scc
                   it != image.end();
                   )
                {
                  VertexSet::iterator itcur = it;
                  ++it;
                  if (scc.find(*itcur) == scc.end()) {
                    image.erase(*itcur);
                  }
                }
              ++distance;
              found = (image.find(*vi) != image.end());
              if (!found) {
                frontier = image;
                for (VertexSet::iterator it = reached.begin();
                     it != reached.end();
                     ++it)
                  {
                    frontier.erase(*it);
                  }
                reached.insert(frontier.begin(), frontier.end());
              }
            }
            if (found) {
              sawaSl[*vi] = distance;
            }
          }

          TRACE(unsatcoreModule, {
              std::cout << "ResolutionGraph: The shortest loop from vertex "
                        << *vi
                        << " has length "
                        << sawaSl[*vi]
                        << std::endl;
            });
        }
    }

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing Parikh images Sawa - sawaSl end"
                << std::endl;
    }

    /* Map from scc ids to lengths of shortest loops in scc */
    std::map<unsigned long, unsigned long> mapSccIdToShortestLoopLength;

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing Parikh images Sawa - map sccids -> sl begin"
                << std::endl;
    }

    // Initialize with infty
    for (unsigned long i = 0; i < myNoSccs; ++i)
      {
        mapSccIdToShortestLoopLength[i] = infty;
      }
    
    // Compute values
    {
      VertexIterator vi, vend;
      for(boost::tie(vi,vend) = boost::vertices(myResolutionGraph);
          vi != vend;
          ++vi)
        {
          Assert(sawaSl.find(*vi) != sawaSl.end(),
                 "*vi not present in sawaSl.");
          Assert(componentMap.find(*vi) != componentMap.end(),
                 "*vi not present in componentMap.");

          unsigned long sl = sawaSl[*vi];
          unsigned long cid = componentMap[*vi];
          if (sl < mapSccIdToShortestLoopLength[cid]) {
            mapSccIdToShortestLoopLength[cid] = sl;
          }
        }
    }
    TRACE(unsatcoreModule, {
        for (unsigned long i = 0; i < myNoSccs; ++i)
          {
            std::cout << "ResolutionGraph: The shortest loop from scc "
                      << i
                      << " has length "
                      << mapSccIdToShortestLoopLength[i]
                      << std::endl;
          }
      });
    
    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing Parikh images Sawa - map sccids -> sl end"
                << std::endl;
    }

    /* Compute set of important vertices sawaImp */
    VertexSet sawaImp = VertexSet();
    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing Parikh images Sawa - sawaImp begin"
                << std::endl;
    }

    {
      VertexIterator vi, vend;
      for(boost::tie(vi,vend) = boost::vertices(myResolutionGraph);
          vi != vend;
          ++vi)
        {
          Assert(sawaSl.find(*vi) != sawaSl.end(),
                 "*vi not present in sawaSl.");
          Assert(componentMap.find(*vi) != componentMap.end(),
                 "*vi not present in componentMap.");

          unsigned long sl = sawaSl[*vi];

          if (sl < infty) {
            unsigned long cid = componentMap[*vi];

            Assert(mapSccIdToShortestLoopLength.find(cid) !=
                   mapSccIdToShortestLoopLength.end(),
                   "cid not present in mapSccIdToShortestLoopLength.");

            unsigned long cl = mapSccIdToShortestLoopLength[cid];

            if (sl == cl) {
              bool inserted;
              VertexSet::iterator dummy;
              boost::tie(dummy, inserted) = sawaImp.insert(*vi);
              Assert(inserted, "*vi already present in sawaImp.");
              TRACE(unsatcoreModule, {
                  std::cout << "ResolutionGraph: Added vertex "
                            << *vi
                            << " to the set of important vertices."
                            << std::endl;
                });
            }
          }
        }
    }

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing Parikh images Sawa - sawaImp end"
                << std::endl;
    }

    /* sawaSi */

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing Parikh images Sawa - sawaSi "
                   "part 1 begin"
                << std::endl;
    }

    // S_i in Sawa's algorithm
    std::vector<VertexSet> sawaSi(numVertices * numVertices);

    // sawaSi[0] is the singleton { emptyClauseVertex }
    sawaSi[0] = VertexSet();
    (void) sawaSi[0].insert(emptyClauseVertex);

    // Compute sawaSi for 0 < i < maxNumBwdReachableVertices
    {
      // To enable loop acceleration for sawaSi below we need to
      // detect loops, i.e., start with a fresh cache
      myFwdImgCache.clear();

      for (unsigned long i = 1;
           i < maxNumBackwardReachableVertices;
           ++i)
        {
          Assert(i < sawaSi.size(), "i out of range for sawaSi.");

          sawaSi[i] = VertexSet();
          forward_image(sawaSi[i - 1], sawaSi[i]);

          ++myNoFwdImgNonAccelerated;
        }
    }
    
    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing Parikh images Sawa - sawaSi "
                   "part 1 end"
                << std::endl;
    }

    /**************************************************************************/
    /* Final vertex dependent part                                            */
    /**************************************************************************/
    {
      std::set<ClauseHandle>::iterator it;
      unsigned long itnum;
      for (it = coreStartingClausehsBrVGt1.begin(), itnum = 1;
           it != coreStartingClausehsBrVGt1.end();
           ++it, ++itnum)
        {
          Assert(myMapClausehToVertex.find(*it) != myMapClausehToVertex.end(),
                 "*it is not present in myMapClausehToVertex");

          if (!myOptions.isQuiet()) {
            std::cout << "ResolutionGraph: computing Parikh images Sawa - vertex "
                         "dependent part begin, final vertex "
                      << itnum
                      << " out of "
                      << coreStartingClausehsBrVGt1.size()
                      << std::endl;
          }

          // Avoid repeated calls to myMapClausehToSemiLinearSet[*it]
          SemiLinearSet & slsOfFinalVertex = myMapClausehToSemiLinearSet[*it];

          Vertex finalVertex = myMapClausehToVertex[*it];

          Assert(numBackwardReachableVertices.find(finalVertex) !=
                 numBackwardReachableVertices.end(),
                 "finalVertex not present in numBackwardReachableVertices.");
          unsigned long numBRVertices =
            numBackwardReachableVertices[finalVertex];
          Assert(numBRVertices > 1, "numBRVertices <= 1.");

          /* compute Q_imp */

          // Q_imp in Sawa's algorithm
          VertexSet sawaQimp = VertexSet();
          
          if (!myOptions.isQuiet()) {
            std::cout << "ResolutionGraph: computing Parikh images Sawa - Q_imp "
                         "begin"
                      << std::endl;
          }

          {
            Assert(numBRVertices - 1 < sawaSi.size(),
                   "numBRVertices - 1 out of range for sawaSi.");
            for (VertexSet::iterator vi = sawaImp.begin();
                 vi != sawaImp.end();
                 ++vi)
              {
                if (sawaSi[numBRVertices - 1].find(*vi) != 
                    sawaSi[numBRVertices - 1].end()) {
                  bool inserted;
                  VertexSet::iterator dummy;
                  boost::tie(dummy, inserted) = sawaQimp.insert(*vi);
                  Assert(inserted, "*vi already present in sawaQimp.");
                  TRACE(unsatcoreModule, {
                      std::cout << "ResolutionGraph: Added vertex "
                                << *vi
                                << " to the set Q_imp."
                                << std::endl;
                    });
                }
              }
          }

          if (!myOptions.isQuiet()) {
            std::cout << "ResolutionGraph: computing Parikh images Sawa - Q_imp "
                         "end"
                      << std::endl;
          }

          /* Compute mapping from shortest scc loop lengths to
             subsets of Q_imp */
          std::map<unsigned long, VertexSet>
            mapSccLoopLengthsToQimpVertices;

          if (!myOptions.isQuiet()) {
            std::cout << "ResolutionGraph: computing Parikh images Sawa - map scc "
                         "loop lengths to Q_imp begin"
                      << std::endl;
          }

          {
            VertexSet::iterator it2;
            for(it2 = sawaQimp.begin(); it2 != sawaQimp.end(); ++it2)
              {
                Assert(sawaSl.find(*it2) != sawaSl.end(),
                       "*it2 not present in sawaSl.");
                Assert(sawaSl[*it2] > 0 && sawaSl[*it2] < infty,
                       "Unexpected value of sawaSl[*it2].");
                  
                unsigned long l = sawaSl[*it2];
                if (mapSccLoopLengthsToQimpVertices.find(l) ==
                    mapSccLoopLengthsToQimpVertices.end()) {
                  mapSccLoopLengthsToQimpVertices[l] = VertexSet();
                }
                {
                  bool inserted;
                  VertexSet::iterator dummy;
                  boost::tie(dummy, inserted) =
                    mapSccLoopLengthsToQimpVertices[l].insert(*it2);
                  Assert(inserted,
                  "*it2 already present in mapSccLoopLengthsToQimpVertices[l].");
                }
              }
          }

          if (!myOptions.isQuiet()) {
            std::cout << "ResolutionGraph: computing Parikh images Sawa - map scc "
                         "loop lengths to Q_imp end"
                      << std::endl;
          }

          /* Compute T_i */

          // T_i in Sawa's algorithm
          std::vector<VertexSet> sawaTi(numBRVertices * numBRVertices);

          if (!myOptions.isQuiet()) {
            std::cout << "ResolutionGraph: computing Parikh images Sawa - T_i begin"
                      << std::endl;
          }

          // sawaTi[0] is the singleton { finalVertex }
          sawaTi[0] = VertexSet();
          (void) sawaTi[0].insert(finalVertex);

          // Compute sawaTi for i > 0
          if (numBRVertices > 1) {
            // To enable loop acceleration we need to detect loops,
            // i.e., start with a fresh cache
            myBwdImgCache.clear();

            // Accelerate only once
            bool accelerated = false;

            // k is the first index where we really need sawaTi
            unsigned long k = numBRVertices * numBRVertices - 2 * numBRVertices;

            for (unsigned long i = 1;
                 i <= numBRVertices * numBRVertices - numBRVertices - 1;
                 ++i)
              {
                // Detect loop by observing cache hit
                unsigned long old = myNoBwdImgCacheHits;

                sawaTi[i] = VertexSet();
                backward_image(sawaTi[i - 1], sawaTi[i]);

                ++myNoBwdImgNonAccelerated;

                if (myNoBwdImgCacheHits != old && !accelerated && i < k) {
                  // Cache hit: now search loop and accelerate
                  accelerated = true;

                  // j will be the length of the loop
                  unsigned long j = 1;

                  while (j <= i && sawaTi[i - j] != sawaTi[i])
                    {
                      ++j;
                    }
                  Assert(j <= i, "Expected j <= i.");

                  // Actually accelerate part 1
                  unsigned long jump = ((k - i) / j) * j;
                  sawaTi[i + jump] = sawaTi[i];
                  myNoBwdImgAccelerated += jump;

                  // Clean sawaTi not needed any more
                  for (unsigned long l = 0; l <= i; ++l) {
                    sawaTi[l].clear();
                  }
                    
                  // Actually accelerate part 2
                  i += jump;
                } else if (accelerated && i - 1 < k) {
                  sawaTi[i - 1].clear();
                }
              }
          }

          if (!myOptions.isQuiet()) {
            std::cout << "ResolutionGraph: computing Parikh images Sawa - T_i end"
                      << std::endl;
          }

          /* Compute R_2 */

          if (!myOptions.isQuiet()) {
            std::cout << "ResolutionGraph: computing Parikh images Sawa - R_2 begin"
                      << std::endl;
          }

          {
            // Iterate over D
            for (unsigned long d = 1; d <= numBRVertices; ++d)
              {
                if (mapSccLoopLengthsToQimpVertices.find(d) !=
                    mapSccLoopLengthsToQimpVertices.end()) {
                  VertexSet vs = mapSccLoopLengthsToQimpVertices[d];
                  for (unsigned long cprime =
                         numBRVertices * numBRVertices - numBRVertices - d;
                       cprime <=
                         numBRVertices * numBRVertices - numBRVertices - 1;
                       ++cprime)
                    {
                      Assert(cprime < sawaTi.size(),
                             "cprime out of range for sawaTi.");
                      for (VertexSet::iterator vsit = vs.begin();
                           vsit != vs.end();
                           ++vsit)
                        {
                          if (sawaTi[cprime].find(*vsit) !=
                              sawaTi[cprime].end()) {
                            LinearSet ls =
                              LinearSet(d, numBRVertices - 1 + cprime);
                            slsOfFinalVertex.addLinearSet(ls);
                            TRACE(unsatcoreModule, {
                                std::cout <<
                                  "ResolutionGraph: Added linear set";
                                ls.dump(std::cout);
                                std::cout << " to semilinear set of clauseh ";
                                it->dump(std::cout);
                                std::cout << std::endl;
                              });
                          }
                        }
                    }
                }
              } // Iterate over D
          }

          if (!myOptions.isQuiet()) {
            std::cout << "ResolutionGraph: computing Parikh images Sawa - R_2 end"
                      << std::endl;
          }

          if (!myOptions.isQuiet()) {
            std::cout << "ResolutionGraph: computing Parikh images Sawa - vertex "
                         "dependent part end, final vertex "
                      << itnum
                      << " out of "
                      << coreStartingClausehsBrVGt1.size()
                      << std::endl;
          }
        }
    }

    // Below, in order to accelerate, we need to work on reduced
    // semi-linear sets
    reduceParikhImages();

    // Clear cache
    myBwdImgCache.clear();

    /**************************************************************************/
    /* Final vertex independent part                                          */
    /**************************************************************************/

    /* R_1 from 0 to maxNumBwdReachableVertices */

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing Parikh images Sawa - R_1 "
                   "part 1 begin"
                << std::endl;
    }

    // Compute R_1 for 0 < i < maxNumBwdReachableVertices

    // For speed up precompute vectors of those rather than
    // accessing naively in inner loop
    std::vector<Vertex> finalVertices;
    std::vector<SemiLinearSet*> slsOfFinalVertices;
    std::vector<unsigned long> numBwdReachablesSq;
    unsigned long numFinalVertices;
    {
      std::set<ClauseHandle>::iterator it;
      for (it = coreStartingClausehsBrVGt1.begin();
           it != coreStartingClausehsBrVGt1.end();
           ++it)
        {
          Assert(myMapClausehToVertex.find(*it) != myMapClausehToVertex.end(),
                 "*it is not present in myMapClausehToVertex.");
          Assert(myMapClausehToSemiLinearSet.find(*it) !=
                 myMapClausehToSemiLinearSet.end(),
                 "*it is not present in myMapClausehToSemiLinearSet.");
          Assert(numBackwardReachableVertices.find(myMapClausehToVertex[*it]) !=
                 numBackwardReachableVertices.end(),
                 "*it is not present in numBackwardReachableVertices.");
            
          Vertex finalVertex = myMapClausehToVertex[*it];
          SemiLinearSet * slsOfFinalVertex = &(myMapClausehToSemiLinearSet[*it]);
          unsigned long numBRVertices =
            numBackwardReachableVertices[finalVertex];
          Assert(numBRVertices > 1, "numBRVertices <= 1.");

          finalVertices.push_back(finalVertex);
          slsOfFinalVertices.push_back(slsOfFinalVertex);
          numBwdReachablesSq.push_back(numBRVertices * numBRVertices);
        }
      numFinalVertices = finalVertices.size();

      for (unsigned long i = 1;
           i < maxNumBackwardReachableVertices;
           ++i)
        {
          Assert(i < sawaSi.size(), "i out of range for sawaSi.");

          LinearSet ls0i = LinearSet(0, i);

          for (unsigned long j = 0; j < numFinalVertices; ++j)
            {
              if (i < numBwdReachablesSq[j] &&
                  sawaSi[i].find(finalVertices[j]) != sawaSi[i].end()) {
                slsOfFinalVertices[j]->addLinearSet(ls0i);
                TRACE(unsatcoreModule, {
                    std::cout << "ResolutionGraph: Added linear set";
                    ls0i.dump(std::cout);
                    std::cout << " to semilinear set of final vertex no "
                              << j
                              << std::endl;
                  });
              }
            }
        }
    }
    
    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing Parikh images Sawa - R_1 "
                   "part 1 end"
                << std::endl;
    }

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing Parikh images Sawa - sawaSi and R_1 "
                   "part 2 begin"
                << std::endl;
    }

    // Compute sawaSi and R_1 for maxNumBwdReachableVertices <= i <
    // maxNumBwdReachableVertices^2
    {
      // Accelerate only once
      bool accelerated = false;

      // True if it is known that all final vertices can be accelerated
      bool accelerateAll = false;
            
      // True if it is known that that final vertex can be accelerated
      std::vector<bool> accelerateFinalVertices =
        std::vector<bool>(numFinalVertices, false);

      for (unsigned long i = maxNumBackwardReachableVertices;
           !accelerateAll &&
             i < maxNumBackwardReachableVertices *
                 maxNumBackwardReachableVertices;
           ++i)
        {
          Assert(i < sawaSi.size(), "i out of range for sawaSi.");

          // Detect loop by observing cache hit
          unsigned long old = myNoFwdImgCacheHits;

          sawaSi[i] = VertexSet();
          forward_image(sawaSi[i - 1], sawaSi[i]);

          ++myNoFwdImgNonAccelerated;

          if (!accelerated && myNoFwdImgCacheHits != old) {
            // Cache hit: now search loop and accelerate
            accelerated = true;

            // after the while loopLength will be the length of the loop
            unsigned long loopLength = 1;

            while (loopLength <= i - 1 &&
                   sawaSi[i - 1 - loopLength] != sawaSi[i - 1])
              {
                ++loopLength;
              }
            Assert(loopLength <= i - 1, "Expected loopLength <= i - 1.");

            if (!myOptions.isQuiet()) {
              std::cout << "ResolutionGraph: found loop at "
                        << i
                        << " of length "
                        << loopLength
                        << std::endl;
            }

            unsigned long loopStart = i - loopLength; // inclusive
            unsigned long loopEnd = i - 1; // inclusive

            accelerateAll = true;

            for (unsigned long j = 0; j < numFinalVertices; ++j)
              {
                if (i < numBwdReachablesSq[j]) {
                  SemiLinearSet toDelete = SemiLinearSet();
                  SemiLinearSet toAdd = SemiLinearSet();

                  for (SemiLinearSet::iterator it = slsOfFinalVertices[j]->begin();
                       it != slsOfFinalVertices[j]->end();
                       ++it)
                    {
                      unsigned long p = it->getPeriod();
                      if (p != 0 && loopLength % p == 0) {
                        unsigned long o = it->getOffset();
                        unsigned long checkStart;
                        unsigned long checkEnd;
                        if ((loopStart % p) <= (o % p)) {
                          checkStart = loopStart + ((o % p) - (loopStart % p));
                        } else {
                          checkStart = loopStart + p - ((loopStart % p) - (o % p));
                        }
                        checkEnd = checkStart + (((loopLength / p) - 1) * p);

                        bool foundAll = true;
                        for (unsigned long k = checkStart;
                             foundAll && k <= checkEnd;
                             k += p)
                          {
                            LinearSet ls = LinearSet(0, k);
                            if (slsOfFinalVertices[j]->find(ls) ==
                                slsOfFinalVertices[j]->end()) {
                              foundAll = false;
                            }
                          }

                        if (foundAll) {
                          for (unsigned long k = checkStart;
                               k <= checkEnd;
                               k += p)
                            {
                              LinearSet ls = LinearSet(0, k);
                              (void) toDelete.insert(ls);
                            }
                          (void) toDelete.insert(*it);
                          (void) toAdd.insert(LinearSet(p, checkStart));
                        }
                      }
                    }
                  {
                    bool foundAll = true;
                    for (unsigned long k = loopStart;
                         foundAll && k <= loopEnd;
                         ++k)
                      {
                        LinearSet ls = LinearSet(0, k);
                        if ((slsOfFinalVertices[j]->find(ls) !=
                             slsOfFinalVertices[j]->end()) &&
                            (toDelete.find(ls) == toDelete.end())) {
                          foundAll = false;
                        }
                      }
                    if (foundAll) {
                      accelerateFinalVertices[j] = true;
                      for (SemiLinearSet::iterator it2 = toDelete.begin();
                           it2 != toDelete.end();
                           ++it2)
                        {
                          Assert(slsOfFinalVertices[j]->find(*it2) !=
                                 slsOfFinalVertices[j]->end(),
                                 "*it2 not present in slsOfFinalVertices[j].");
                          slsOfFinalVertices[j]->erase(*it2);
                        }
                      for (SemiLinearSet::iterator it2 = toAdd.begin();
                           it2 != toAdd.end();
                           ++it2)
                        {
                          (void) slsOfFinalVertices[j]->insert(*it2);
                        }

                      if (!myOptions.isQuiet()) {
                        std::cout << "ResolutionGraph: accelerating final vertex "
                                  << j
                                  << std::endl;
                      }

                    } else {
                      accelerateAll = false;

                      if (!myOptions.isQuiet()) {
                        std::cout << "ResolutionGraph: not accelerating final vertex "
                                  << j
                                  << std::endl;
                      }

                      slsOfFinalVertices[j]->dump(std::cout);
                      std::cout << std::endl;
                    }
                  }
                }
              }
            if (accelerateAll) {
              myNoFwdImgAccelerated +=
                maxNumBackwardReachableVertices *
                maxNumBackwardReachableVertices - i - 1;
              if (!myOptions.isQuiet()) {
                std::cout << "ResolutionGraph: accelerating all final vertices"
                          << std::endl;
              }
            }
          }

          if (!accelerateAll) {
            LinearSet ls0i = LinearSet(0, i);

            for (unsigned long j = 0; j < numFinalVertices; ++j)
              {
                if (!accelerateFinalVertices[j] &&
                    i < numBwdReachablesSq[j] &&
                    sawaSi[i].find(finalVertices[j]) != sawaSi[i].end()) {
                  slsOfFinalVertices[j]->addLinearSet(ls0i);
                  TRACE(unsatcoreModule, {
                      std::cout << "ResolutionGraph: Added linear set";
                      ls0i.dump(std::cout);
                      std::cout << " to semilinear set of final vertex no "
                                << j
                                << std::endl;
                    });
                }
              }
          }
        }
    }
    
    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing Parikh images Sawa - sawaSi and R_1 "
                   "part 2 end"
                << std::endl;
    }

    // Clear cache
    myFwdImgCache.clear();

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing Parikh images Sawa end"
                << std::endl;
    }
  } // ResolutionGraph::computeParikhImages_Sawa

  /*
    An optimized version of the following algorithm

    P. Gawrychowski: "Chrobak Normal Form Revisited, with
    Applications". CIAA'11.

    Remark: In the setting of this program
    - the initial state is a trivial scc,
    - all final states are trivial sccs, and
    - the initial state and the final states are distinct.
   */
  void
  ResolutionGraph::computeParikhImages_Gawrychowski()
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");
    Assert(myLastEmptyClause.get() != 0, "myLastEmptyClause is 0.");
    Assert(myLastEmptyClausePartition != invalidPartition,
           "myLastEmptyClausePartition is invalidPartition");
    Assert(myLastEmptyClausePartition == mainPartition,
           "myLastEmptyClausePartition is not mainPartition");
    Assert(!myCoreStartingClausehs.empty(), "myCoreStartingClausehs is empty.");
    Assert(boost::num_vertices(myResolutionGraph) *
           boost::num_vertices(myResolutionGraph) <
           std::numeric_limits<unsigned long>::max(),
           "Number of vertices in graph squared is larger than maximum value for"
           "unsigned long.");

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing Parikh images Gawrychowski begin"
                << std::endl;
    }

    /***************************************************************************/
    /* Preparation                                                             */
    /***************************************************************************/

    // The clauses with more than 1 backward reachable vertex.
    //
    // Note that, upon reaching this point, all starting clausehs will
    // be non-empty. That implies that if a starting clauseh has only
    // 1 backward reachable vertex, then this starting clauseh is not
    // reachable from the final empty clause in the unary NFA. Hence,
    // such clauseh will be assigned {(0 N, 0)} as its only sls.
    std::set<ClauseHandle> coreStartingClausehsBrVGt1;
    std::set<Vertex> finalVerticesBrVGt1;

    // Needed a couple of times
    unsigned long numVertices = boost::num_vertices(myResolutionGraph);
    unsigned long infty = std::numeric_limits<unsigned long>::max();
    std::vector<bool> forbidden(numVertices, false);

    // Find vertex in main partition labeled with the empty clause
    Assert(myMapClausehToVertex.find(ClauseHandle(myLastEmptyClause,
                                                  mainPartition)) !=
           myMapClausehToVertex.end(),
           "myLastEmptyClause not present in myMapClausehToVertex.");
    Vertex emptyClauseVertex =
      myMapClausehToVertex[ClauseHandle(myLastEmptyClause, mainPartition)];
    {
      // Assert that emptyClauseVertex is different from each finalVertex
      std::set<ClauseHandle>::iterator it;
      for (it = myCoreStartingClausehs.begin();
           it != myCoreStartingClausehs.end();
           ++it)
        {
          Assert(myMapClausehToVertex.find(*it) != myMapClausehToVertex.end(),
                 "*it is not present in myMapClausehToVertex");

          Vertex finalVertex;
          finalVertex = myMapClausehToVertex[*it];
          Assert(finalVertex != emptyClauseVertex,
                 "finalVertex == emptyClauseVertex.");
        }
    }

    // Compute which final vertices are reachable from the initial vertex
    //
    // Note that due to how the unary NFA is constructed it is
    // sufficient to check whether or not a vertex has incoming edges.
    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing Parikh images Gawrychowski - "
                   "computing final vertices reachable from initial vertex "
                   "begin"
                << std::endl;
    }
    {
      std::set<ClauseHandle>::iterator it;
      for (it = myCoreStartingClausehs.begin();
           it != myCoreStartingClausehs.end();
           ++it)
        {
          Assert(myMapClausehToVertex.find(*it) != myMapClausehToVertex.end(),
                 "*it is not present in myMapClausehToVertex");

          Vertex finalVertex = myMapClausehToVertex[*it];

          if (boost::in_degree(finalVertex, myResolutionGraph) > 0) {
            (void) coreStartingClausehsBrVGt1.insert(*it);
            (void) finalVerticesBrVGt1.insert(finalVertex);
              if (!myOptions.isQuiet()) {
              std::cout << "ResolutionGraph: clauseh ";
              it->dump(std::cout);
              std::cout << " is reachable from the empty clause."
                        << std::endl;
            }
          }
        }
    }
    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing Parikh images Gawrychowski - "
                   "computing final vertices reachable from initial vertex end"
                << std::endl;
    }

    // Prepare semi-linear sets
    {
      LinearSet ls00 = LinearSet(0, 0);

      std::set<ClauseHandle>::iterator it;
      for (it = myCoreStartingClausehs.begin();
           it != myCoreStartingClausehs.end();
           ++it)
        {
          Assert(myMapClausehToVertex.find(*it) != myMapClausehToVertex.end(),
                 "*it is not present in myMapClausehToVertex");

          Vertex finalVertex = myMapClausehToVertex[*it];

          // Avoid repeated calls to myMapClausehToSemiLinearSet[*it]
          SemiLinearSet & slsOfFinalVertex = myMapClausehToSemiLinearSet[*it];

          // Initialize semi-linear set
          slsOfFinalVertex = SemiLinearSet();

          // If finalVertex is reachable only via 0-edges, then need
          // to add (0 N, 0)
          Assert(myMapVertexTo0StarReachableVertices.find(emptyClauseVertex) !=
                 myMapVertexTo0StarReachableVertices.end(),
                 "emptyClauseVertex not present in "
                 "myMapVertexTo0StarReachableVertices.");
          if (myMapVertexTo0StarReachableVertices[emptyClauseVertex].
              find(finalVertex) !=
              myMapVertexTo0StarReachableVertices[emptyClauseVertex].end()) {
            slsOfFinalVertex.addLinearSet(ls00);
            TRACE(unsatcoreModule, {
                std::cout << "ResolutionGraph: computing Parikh images "
                             "Gawrychowski - added linear set";
                ls00.dump(std::cout);
                std::cout << " to semilinear set of clauseh ";
                it->dump(std::cout);
                std::cout << std::endl;
              });
          }
        }
    }

    if (coreStartingClausehsBrVGt1.size() == 0) {
      // Clear cache
      myFwdImgCache.clear();

      if (!myOptions.isQuiet()) {
        std::cout << "ResolutionGraph: computing Parikh images Gawrychowski - "
                     "no final vertex reachable from initial state. Returning."
                  << std::endl;
        std::cout << "ResolutionGraph: computing Parikh images Gawrychowski end"
                  << std::endl;
      }
    }

    /* Strongly connected components */

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing Parikh images Gawrychowski - sccs begin"
                << std::endl;
    }

    // The component map that will map vertices to their sccs and its reverse
    std::map<Vertex, unsigned long> componentMap;
    std::map<unsigned long, VertexSet> mapSccNoToVertices;

    // Execute scc algorithm
    myNoSccs = boost::strong_components(myResolutionGraph,
                                  boost::make_assoc_property_map(componentMap));

    // Compute the reverse mapping
    {
      for (unsigned long i = 0; i < myNoSccs; ++i)
        {
          mapSccNoToVertices[i] = VertexSet();
        }
      for (std::map<Vertex, unsigned long>::iterator it = componentMap.begin();
           it != componentMap.end();
           ++it)
        {
          (void) mapSccNoToVertices[it->second].insert(it->first);
        }
    }

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing Parikh images Gawrychowski - sccs end"
                << std::endl;
    }

    /* Compute the shortest cycle in each scc */

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing Parikh images Gawrychowski - "
                   "shortest cycles per scc begin"
                << std::endl;
    }

    // Compute the shortest cycle for each state
    std::map<Vertex, unsigned long> shortestCycleLengths;
    {
      VertexIterator vi, vend;
      for(boost::tie(vi,vend) = boost::vertices(myResolutionGraph);
          vi != vend;
          ++vi)
        {
          Assert(shortestCycleLengths.find(*vi) == shortestCycleLengths.end(),
                 "*vi present in shortestCycleLengths.");
          shortestCycleLengths[*vi] = infty;

          // Forward bfs to determine shortestCycleLengths[*vi]
          unsigned long distance = 0;
          {
            bool found = false;
            VertexSet reached = VertexSet();
            VertexSet frontier = VertexSet();
            VertexSet image;
            VertexSet scc;

            {
              unsigned long sccNo;
              Assert(componentMap.find(*vi) != componentMap.end(),
                     "*vi not present in componentMap");
              sccNo = componentMap[*vi];
              Assert(mapSccNoToVertices.find(sccNo) != mapSccNoToVertices.end(),
                     "sccNo not present in mapSccNoToVertices.");
              scc = mapSccNoToVertices[sccNo];
            }
            (void) frontier.insert(*vi);
            while (!found && !frontier.empty()) {
              image = VertexSet();
              forward_image(frontier, image);
              ++myNoFwdImgNonAccelerated;
              for (VertexSet::iterator it = image.begin(); // restrict to scc
                   it != image.end();
                   )
                {
                  VertexSet::iterator itcur = it;
                  ++it;
                  if (scc.find(*itcur) == scc.end()) {
                    image.erase(*itcur);
                  }
                }
              ++distance;
              found = (image.find(*vi) != image.end());
              if (!found) {
                frontier = image;
                for (VertexSet::iterator it = reached.begin();
                     it != reached.end();
                     ++it)
                  {
                    frontier.erase(*it);
                  }
                reached.insert(frontier.begin(), frontier.end());
              }
            }
            if (found) {
              shortestCycleLengths[*vi] = distance;
            }
          }

          if (*vi == emptyClauseVertex) {
            Assert(shortestCycleLengths[*vi] == infty,
                   "shortestCycleLength of emptyClauseVertex not infty.");
          }
          if (finalVerticesBrVGt1.find(*vi) != finalVerticesBrVGt1.end()) {
            Assert(shortestCycleLengths[*vi] == infty,
                   "shortestCycleLength of a final vertex not infty.");
          }

          TRACE(unsatcoreModule, {
              std::cout << "ResolutionGraph: computing Parikh images "
                           "Gawrychowski - the shortest loop from vertex "
                        << *vi
                        << " has length "
                        << shortestCycleLengths[*vi]
                        << std::endl;
            });
        }
    }

    // Map from scc ids to lengths of shortest loops in scc
    std::map<unsigned long, unsigned long> mapSccIdToShortestLoopLength;

    // Initialize with infty
    for (unsigned long i = 0; i < myNoSccs; ++i)
      {
        mapSccIdToShortestLoopLength[i] = infty;
      }
    
    // Compute values
    {
      VertexIterator vi, vend;
      for(boost::tie(vi,vend) = boost::vertices(myResolutionGraph);
          vi != vend;
          ++vi)
        {
          Assert(shortestCycleLengths.find(*vi) != shortestCycleLengths.end(),
                 "*vi not present in shortestCycleLengths.");
          Assert(componentMap.find(*vi) != componentMap.end(),
                 "*vi not present in componentMap.");

          unsigned long sl = shortestCycleLengths[*vi];
          unsigned long cid = componentMap[*vi];
          if (sl < mapSccIdToShortestLoopLength[cid]) {
            mapSccIdToShortestLoopLength[cid] = sl;
          }
        }
    }
    TRACE(unsatcoreModule, {
        for (unsigned long i = 0; i < myNoSccs; ++i)
          {
            std::cout << "ResolutionGraph: computing Parikh images "
                         "Gawrychowski - the shortest loop from scc "
                      << i
                      << " has length "
                      << mapSccIdToShortestLoopLength[i]
                      << std::endl;
          }
      });
    
    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing Parikh images Gawrychowski - "
                   "shortest cycles per scc end"
                << std::endl;
    }

    /***************************************************************************/
    /* Non-trivial sccs                                                        */
    /***************************************************************************/

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing Parikh images Gawrychowski - "
                   "non-trivial sccs begin"
                << std::endl;
    }

    {
      myFwdImgRestrictedCache.clear();
      for (unsigned long sccNo = 0; sccNo < myNoSccs; ++sccNo) {
        // Get length of shortest cycle
        Assert(mapSccIdToShortestLoopLength.find(sccNo) !=
               mapSccIdToShortestLoopLength.end(),
               "sccNo not present in mapSccIdToShortestLoopLength.");
        unsigned long d = mapSccIdToShortestLoopLength[sccNo];

        // Skip trivial sccs
        if (d == infty) {
          continue;
        }

        // Get scc
        Assert(mapSccNoToVertices.find(sccNo) !=
               mapSccNoToVertices.end(),
               "sccNo not present in mapSccNoToVertices.");
        VertexSet scc = mapSccNoToVertices[sccNo];

        // The sets of states already reached
        // reached1 is w/o having seen an scc state
        // reached2 is w/ having seen an scc state
        std::vector<VertexSet> reached1(d, VertexSet());
        std::vector<VertexSet> reached2(d, VertexSet());

        // The current distance from the initial vertex
        unsigned long l = 0;

        // Initialize reached1
        (void) reached1[0].insert(emptyClauseVertex);

        // One frontier each for w/o and w/ having seen an scc state
        VertexSet frontier1 = reached1[0];
        VertexSet frontier2 = reached2[0];

        // Forward state space exploration
        while (true) {
          ++l;
          Assert(l <= 2 * d * numVertices, "l unexpectedly large.");
          TRACE(unsatcoreModule, {
              std::cout << "ResolutionGraph: computing Parikh images "
                           "Gawrychowski - forward interation "
                        << l
                        << " begin"
                        << std::endl;
            });

          // Forward image
          VertexSet fwdImage1;
          VertexSet fwdImage2;
          forward_image_restricted(frontier1, fwdImage1, forbidden);
          ++myNoFwdImgNonAccelerated;
          forward_image_restricted(frontier2, fwdImage2, forbidden);
          ++myNoFwdImgNonAccelerated;
          // Move states in scc from fwdImage1 to fwdImage2
          for (VertexSet::iterator it = scc.begin();
               it != scc.end();
               ++it)
            {
              if (fwdImage1.find(*it) != fwdImage1.end()) {
                Vertex v = *it;
                (void) fwdImage1.erase(v);
                (void) fwdImage2.insert(v);
              }
            }

          // Frontiers
          frontier1 = fwdImage1;
          for (VertexSet::iterator it = reached1[l % d].begin();
               it != reached1[l % d].end();
               ++it)
            {
              (void) frontier1.erase(*it);
            }
          frontier2 = fwdImage2;
          for (VertexSet::iterator it = reached2[l % d].begin();
               it != reached2[l % d].end();
               ++it)
            {
              (void) frontier2.erase(*it);
            }

          // Terminate?
          if (frontier1.empty() && frontier2.empty()) {
            break;
          }

          // Add frontiers to reached
          reached1[l % d].insert(frontier1.begin(), frontier1.end());
          reached2[l % d].insert(frontier2.begin(), frontier2.end());

          // Add new linear set (d N, l) where appropriate
          for (std::set<ClauseHandle>::iterator it =
                 coreStartingClausehsBrVGt1.begin();
               it != coreStartingClausehsBrVGt1.end();
               ++it)
            {
              Assert(myMapClausehToVertex.find(*it) !=
                     myMapClausehToVertex.end(),
                     "*it is not present in myMapClausehToVertex");

              Vertex finalVertex = myMapClausehToVertex[*it];

              if (frontier2.find(finalVertex) != frontier2.end()) {
                Assert(myMapClausehToSemiLinearSet.find(*it) !=
                       myMapClausehToSemiLinearSet.end(),
                       "*it not present in myMapClausehToSemiLinearSet.");
                myMapClausehToSemiLinearSet[*it].addLinearSet(LinearSet(d, l));
                TRACE(unsatcoreModule, {
                    std::cout << "ResolutionGraph: Added linear set";
                    LinearSet(d, l).dump(std::cout);
                    std::cout << " to semilinear set of clauseh ";
                    it->dump(std::cout);
                    std::cout << std::endl;
                  });
              }
            }

          TRACE(unsatcoreModule, {
              std::cout << "ResolutionGraph: computing Parikh images "
                           "Gawrychowski - forward interation "
                        << l
                        << " end"
                        << std::endl;
            });
        }

        // Rm scc
        for (VertexSet::iterator it = scc.begin();
             it != scc.end();
             ++it)
          {
            forbidden[boost::get(myVertexIndex, *it)] = true;
            --numVertices;
            myFwdImgRestrictedCache.clear();
          }
      }
    }

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing Parikh images Gawrychowski - "
                   "non-trivial sccs end"
                << std::endl;
    }

    /***************************************************************************/
    /* Trivial sccs                                                            */
    /***************************************************************************/

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing Parikh images Gawrychowski - "
                   "trivial sccs begin"
                << std::endl;
    }

    {
      VertexSet fwdImage = VertexSet();
      (void) fwdImage.insert(emptyClauseVertex);

      // Has changed, need to recompute
      numVertices = boost::num_vertices(myResolutionGraph);

      // Forward state space exploration
      for (unsigned long l = 0; l <= numVertices; ++l)
        {
          // Add new linear set (0 N, l) where appropriate
          for (std::set<ClauseHandle>::iterator it =
                 coreStartingClausehsBrVGt1.begin();
               it != coreStartingClausehsBrVGt1.end();
               ++it)
            {
              Assert(myMapClausehToVertex.find(*it) !=
                     myMapClausehToVertex.end(),
                     "*it is not present in myMapClausehToVertex");

              Vertex finalVertex = myMapClausehToVertex[*it];

              if (fwdImage.find(finalVertex) != fwdImage.end()) {
                Assert(myMapClausehToSemiLinearSet.find(*it) !=
                       myMapClausehToSemiLinearSet.end(),
                       "*it not present in myMapClausehToSemiLinearSet.");
                myMapClausehToSemiLinearSet[*it].addLinearSet(LinearSet(0, l));
                TRACE(unsatcoreModule, {
                    std::cout << "ResolutionGraph: Added linear set";
                    LinearSet(0, l).dump(std::cout);
                    std::cout << " to semilinear set of clauseh ";
                    it->dump(std::cout);
                    std::cout << std::endl;
                  });
              }
            }

          // Forward image
          {
            VertexSet tmp;
            forward_image_restricted(fwdImage, tmp, forbidden);
            ++myNoFwdImgNonAccelerated;
            fwdImage = tmp;
          }
        }
    }

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing Parikh images Gawrychowski - "
                   "trivial sccs end"
                << std::endl;
    }

    // Clear cache
    myFwdImgCache.clear();

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: computing Parikh images Gawrychowski end"
                << std::endl;
    }
  } // ResolutionGraph::computeParikhImages_Gawrychowski

  // Only do this for clauses in the unsat core
  void
  ResolutionGraph::reduceParikhImages()
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");
    Assert(myLastEmptyClause.get() != 0, "myLastEmptyClause is 0.");
    Assert(myLastEmptyClausePartition != invalidPartition,
           "myLastEmptyClausePartition is invalidPartition");
    Assert(myLastEmptyClausePartition == mainPartition,
           "myLastEmptyClausePartition is not mainPartition");
    Assert(!myCoreStartingClausehs.empty(), "myCoreStartingClausehs is empty.");
    Assert(!myMapClausehToSemiLinearSet.empty() ||
           myCoreStartingClausehs.empty(),
     "myMapClausehToSemiLinearSet is empty but myCoreStartingClausehs is not.");

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: reducing Parikh images begin"
                << std::endl;
    }

    std::set<ClauseHandle>::iterator it;
    for (it = myCoreStartingClausehs.begin();
         it != myCoreStartingClausehs.end();
         ++it)
      {
        Assert(myMapClausehToSemiLinearSet.find(*it) !=
               myMapClausehToSemiLinearSet.end(),
               "*it not present in myMapClausehToSemiLinearSet.");
        SemiLinearSet sls = myMapClausehToSemiLinearSet[*it];
        sls.reduce();
        myMapClausehToSemiLinearSet[*it] = sls;
      }

    if (!myOptions.isQuiet()) {
      std::cout << "ResolutionGraph: reducing Parikh images end"
                << std::endl;
    }
  } // ResolutionGraph::reduceParikhImages

  void
  ResolutionGraph::computeCoreLtl()
  {
    // Take implicit conjunctions into account
    myNoVerticesSyntaxTreeInputProblem += myFormulaList.size() - 1;

    for (std::list<TreeNode *>::iterator it = myFormulaList.begin();
         it != myFormulaList.end();
         )
      {
        std::list<TreeNode *>::iterator itcur = it;
        ++it;

        // Compute size before core computation
        myNoVerticesSyntaxTreeInputProblem += (*itcur)->size();
        
        if (myOptions.getExtractUnsatCoreMode() ==
            EXTRACT_UNSAT_CORE_MODE_SIMPLE) {
          *itcur = compute_core_simple_ltl_rec(*itcur, true);
          if (!myOptions.isNoUnsatCoreSimplification()) {
            *itcur = simplify_core_ltl_rec(*itcur, true);
          }
        } else {
          Assert(myOptions.getExtractUnsatCoreMode() ==
                 EXTRACT_UNSAT_CORE_MODE_POSITIONSETS,
                 "Unexpected core extraction mode.");
          *itcur = compute_core_position_sets_ltl_rec(*itcur, true);
        }

        if (!myOptions.isNoUnsatCoreSimplification() &&
            isIdentifier(*itcur) &&
            ((Identifier *) *itcur)->isTrue() &&
            myCoreSimplifiedSubformulas.find(*itcur) !=
            myCoreSimplifiedSubformulas.end()) {
          myFormulaList.erase(itcur);
        } else {
          // Compute size after core computation
          myNoVerticesSyntaxTreeUnsatCore += (*itcur)->size();
        }
      }

    // Take implicit conjunctions into account
    myNoVerticesSyntaxTreeUnsatCore += myFormulaList.size() - 1;
  } // ResolutionGraph::computeCoreLtl

  /* Convenience method to avoid doing too many details in main.cc */
  void
  ResolutionGraph::computeCore()
  {
    computeReachableSubgraphFromEmptyClause();
    computeCoreStartingClauses();
    if (myOptions.getExtractUnsatCoreMode() ==
        EXTRACT_UNSAT_CORE_MODE_POSITIONSETS) {
      transformToUnaryNFA();
      if (myOptions.getExtractUnsatCoreParikhAlgorithm() ==
          EXTRACT_UNSAT_CORE_PARIKH_ALGORITHM_GAWRYCHOWSKI) {
        computeParikhImages_Gawrychowski();
      } else if (myOptions.getExtractUnsatCoreParikhAlgorithm() ==
                 EXTRACT_UNSAT_CORE_PARIKH_ALGORITHM_SAWA) {
        computeParikhImages_Sawa();
      } else {
        Assert(false,
               "Unexpected value for myExtractUnsatCoreParikhAlgorithm.");
      }
      reduceParikhImages();
    }
#ifndef BENCHMARKING
    if (myOptions.getExtractUnsatCoreFormat() ==
        EXTRACT_UNSAT_CORE_FORMAT_LTL) {
#endif
      computeCoreLtl();
#ifndef BENCHMARKING
    }
#endif
  } // ResolutionGraph::computeCore

  /* Convenience method to avoid doing too many details in main.cc */
  void
  ResolutionGraph::dumpCore()
  {
    if (myOptions.getExtractUnsatCoreFileName() == std::string("")) {
      std::cout << "The following is an unsatisfiable core:"
                << std::endl;
    } else {
      std::cout << "An unsatisfiable core is written to "
                << myOptions.getExtractUnsatCoreFileName()
                << "."
                << std::endl;
    }

    {
      std::ofstream myUnsatCoreFile;
      if (myOptions.getExtractUnsatCoreFileName() != std::string("")) {
        myUnsatCoreFile.open(myOptions.getExtractUnsatCoreFileName().c_str());
        if (!myUnsatCoreFile.is_open()) {
          std::cout << "Could not open file for writing unsatisfiable core "
                    << myOptions.getExtractUnsatCoreFileName()
                    << ". Writing to stdout."
                    << std::endl;
        }
      }
      std::ostream & os =
        (myOptions.getExtractUnsatCoreFileName() != std::string("") &&
         myUnsatCoreFile.is_open()) ?
        myUnsatCoreFile :
        std::cout;

      {
        UnsatCoreWriter ucw;
        ucw.dumpClausehs(os,
                         myCoreStartingClausehs,
                         myMapClausehToSemiLinearSet,
                         myFormulaList,
                         myMapSubformulaToSemiLinearSet,
                         myOptions);
      }

      if (myOptions.getExtractUnsatCoreFileName() != std::string("")) {
        myUnsatCoreFile.close();
      }
    }

    if (myOptions.getExtractUnsatCoreFileName() == std::string("")) {
      std::cout << "End of unsatisfiable core." << std::endl;
    }            

    (void) dumpStatistics(std::cout);
  } // ResolutionGraph::dumpCore

  void
  ResolutionGraph::dumpCoreBenchmarking()
  {
    std::ofstream myUnsatCoreFile;
    if (myOptions.getExtractUnsatCoreMode() ==
        EXTRACT_UNSAT_CORE_MODE_SIMPLE) {
      // Internal
      {
        std::string fn = myOptions.getFileName();
        fn += ".simple.internal";
        myUnsatCoreFile.open(fn.c_str());
        if (!myUnsatCoreFile.is_open()) {
          std::cout << "Could not open file for writing unsatisfiable core "
                    << fn
                    << ". Writing to stdout."
                    << std::endl;
        }
        std::ostream & os =
          myUnsatCoreFile.is_open() ? myUnsatCoreFile : std::cout;
        UnsatCoreWriter ucw;
        ucw.dumpClausehsSimpleInternal(os,
                                       myCoreStartingClausehs);
        myUnsatCoreFile.close();
      }
      // SNF
      {
        std::string fn = myOptions.getFileName();
        fn += ".simple.snf";
        myUnsatCoreFile.open(fn.c_str());
        if (!myUnsatCoreFile.is_open()) {
          std::cout << "Could not open file for writing unsatisfiable core "
                    << fn
                    << ". Writing to stdout."
                    << std::endl;
        }
        std::ostream & os =
          myUnsatCoreFile.is_open() ? myUnsatCoreFile : std::cout;
        UnsatCoreWriter ucw;
        ucw.dumpClausehsSimpleSNF(os,
                                  myCoreStartingClausehs);
        myUnsatCoreFile.close();
      }
      // LTLC
      {
        std::string fn = myOptions.getFileName();
        fn += ".simple.ltlc";
        myUnsatCoreFile.open(fn.c_str());
        if (!myUnsatCoreFile.is_open()) {
          std::cout << "Could not open file for writing unsatisfiable core "
                    << fn
                    << ". Writing to stdout."
                    << std::endl;
        }
        std::ostream & os =
          myUnsatCoreFile.is_open() ? myUnsatCoreFile : std::cout;
        UnsatCoreWriter ucw;
        ucw.dumpClausehsSimpleLTLC(os,
                                   myCoreStartingClausehs);
        myUnsatCoreFile.close();
      }
      // LTL
      {
        std::string fn = myOptions.getFileName();
        fn += ".simple.ltl";
        myUnsatCoreFile.open(fn.c_str());
        if (!myUnsatCoreFile.is_open()) {
          std::cout << "Could not open file for writing unsatisfiable core "
                    << fn
                    << ". Writing to stdout."
                    << std::endl;
        }
        std::ostream & os =
          myUnsatCoreFile.is_open() ? myUnsatCoreFile : std::cout;
        UnsatCoreWriter ucw;
        ucw.dumpClausehsSimpleLTL(os,
                                  myFormulaList);
        myUnsatCoreFile.close();
      }
    } else {
      // Internal
      {
        std::string fn = myOptions.getFileName();
        fn += ".positionsets.internal";
        myUnsatCoreFile.open(fn.c_str());
        if (!myUnsatCoreFile.is_open()) {
          std::cout << "Could not open file for writing unsatisfiable core "
                    << fn
                    << ". Writing to stdout."
                    << std::endl;
        }
        std::ostream & os =
          myUnsatCoreFile.is_open() ? myUnsatCoreFile : std::cout;
        UnsatCoreWriter ucw;
        ucw.dumpClausehsPositionSetsInternal(os,
                                             myCoreStartingClausehs,
                                             myMapClausehToSemiLinearSet);
        myUnsatCoreFile.close();
      }
      // SNF
      {
        std::string fn = myOptions.getFileName();
        fn += ".positionsets.snf";
        myUnsatCoreFile.open(fn.c_str());
        if (!myUnsatCoreFile.is_open()) {
          std::cout << "Could not open file for writing unsatisfiable core "
                    << fn
                    << ". Writing to stdout."
                    << std::endl;
        }
        std::ostream & os =
          myUnsatCoreFile.is_open() ? myUnsatCoreFile : std::cout;
        UnsatCoreWriter ucw;
        ucw.dumpClausehsPositionSetsSNF(os,
                                        myCoreStartingClausehs,
                                        myMapClausehToSemiLinearSet);
        myUnsatCoreFile.close();
      }
      // LTLC
      {
        std::string fn = myOptions.getFileName();
        fn += ".positionsets.ltlc";
        myUnsatCoreFile.open(fn.c_str());
        if (!myUnsatCoreFile.is_open()) {
          std::cout << "Could not open file for writing unsatisfiable core "
                    << fn
                    << ". Writing to stdout."
                    << std::endl;
        }
        std::ostream & os =
          myUnsatCoreFile.is_open() ? myUnsatCoreFile : std::cout;
        UnsatCoreWriter ucw;
        ucw.dumpClausehsPositionSetsLTLC(os,
                                         myCoreStartingClausehs,
                                         myMapClausehToSemiLinearSet);
        myUnsatCoreFile.close();
      }
      // LTL
      {
        std::string fn = myOptions.getFileName();
        fn += ".positionsets.ltl";
        myUnsatCoreFile.open(fn.c_str());
        if (!myUnsatCoreFile.is_open()) {
          std::cout << "Could not open file for writing unsatisfiable core "
                    << fn
                    << ". Writing to stdout."
                    << std::endl;
        }
        std::ostream & os =
          myUnsatCoreFile.is_open() ? myUnsatCoreFile : std::cout;
        UnsatCoreWriter ucw;
        ucw.dumpClausehsPositionSetsLTL(os,
                                        myCoreStartingClausehs,
                                        myMapClausehToSemiLinearSet,
                                        myFormulaList,
                                        myMapSubformulaToSemiLinearSet);
        myUnsatCoreFile.close();
      }
    }
    (void) dumpStatistics(std::cout);
  } // ResolutionGraph::dumpCoreBenchmarking

  std::ostream &
  ResolutionGraph::dumpStatistics(std::ostream & os)
  {
    os << "ResolutionGraph: number of vertices before pruning to reachable vertices: "
       << myNoVerticesBeforeReachable
       << std::endl;
    os << "ResolutionGraph: number of edges before pruning to reachable vertices: "
       << myNoEdgesBeforeReachable
       << std::endl;
    os << "ResolutionGraph: number of vertices after pruning to reachable vertices: "
       << myNoVerticesAfterReachable
       << std::endl;
    os << "ResolutionGraph: number of edges after pruning to reachable vertices: "
       << myNoEdgesAfterReachable
       << std::endl;
    os << "ResolutionGraph: number of clauses in input problem: "
       << myStartingClausehs.size()
       << std::endl;
    os << "ResolutionGraph: number of clauses in unsatisfiable core: "
       << myCoreStartingClausehs.size()
       << std::endl;
#ifndef BENCHMARKING
    if (myOptions.getExtractUnsatCoreFormat() ==
        EXTRACT_UNSAT_CORE_FORMAT_LTL) {
#endif
      os << "ResolutionGraph: number of nodes in syntax tree of input problem: "
         << myNoVerticesSyntaxTreeInputProblem
         << std::endl;
      os << "ResolutionGraph: number of nodes in syntax tree of unsatisfiable core: "
         << myNoVerticesSyntaxTreeUnsatCore
         << std::endl;
#ifndef BENCHMARKING
    }
#endif
    if (myOptions.getExtractUnsatCoreMode() ==
        EXTRACT_UNSAT_CORE_MODE_POSITIONSETS) {
      os << "ResolutionGraph: number of edges after transforming to unary NFA: "
         << myNoEdgesAfterUnaryNFA
         << std::endl;
      os << "ResolutionGraph: number of strongly connected components in unary NFA: "
         << myNoSccs
         << std::endl;
      if (myOptions.getExtractUnsatCoreParikhAlgorithm() ==
          EXTRACT_UNSAT_CORE_PARIKH_ALGORITHM_SAWA) {
        os << "ResolutionGraph: number of accelerated forward image computations: "
           << myNoFwdImgAccelerated
           << std::endl;
        os << "ResolutionGraph: number of non-accelerated forward image computations: "
           << myNoFwdImgNonAccelerated
           << std::endl;
      } else {
        os << "ResolutionGraph: number of forward image computations: "
           << myNoFwdImgNonAccelerated
           << std::endl;
      }
      if (myOptions.getExtractUnsatCoreParikhAlgorithm() ==
          EXTRACT_UNSAT_CORE_PARIKH_ALGORITHM_SAWA) {
        os << "ResolutionGraph: number of accelerated backward image computations: "
           << myNoBwdImgAccelerated
           << std::endl;
        os << "ResolutionGraph: number of non-accelerated backward image computations: "
           << myNoBwdImgNonAccelerated
           << std::endl;
      } else {
        os << "ResolutionGraph: number of backward image computations: "
           << myNoBwdImgNonAccelerated
           << std::endl;
      }
      os << "ResolutionGraph: number of forward image cache hits: "
         << myNoFwdImgCacheHits
         << std::endl;
      os << "ResolutionGraph: number of forward image cache misses: "
         << myNoFwdImgCacheMisses
         << std::endl;
      os << "ResolutionGraph: number of backward image cache hits: "
         << myNoBwdImgCacheHits
         << std::endl;
      os << "ResolutionGraph: number of backward image cache misses: "
         << myNoBwdImgCacheMisses
         << std::endl;
      os << "ResolutionGraph: number of forward image restricted cache hits: "
         << myNoFwdImgRestrictedCacheHits
         << std::endl;
      os << "ResolutionGraph: number of forward image restricted cache misses: "
         << myNoFwdImgRestrictedCacheMisses
         << std::endl;
    }
    return os;
  } // ResolutionGraph::dumpStatistics

  void
  ResolutionGraph::setFormulaList(std::list<TreeNode*> & formulaList)
  {
    myFormulaList = formulaList;
  } // ResolutionGraph::setFormulaList

  void
  ResolutionGraph::setMapSubformulaToClausehsNow(
        std::map<TreeNode*, ClauseHandleSet> const & mapSubformulaToClausehsNow)
  {
    myMapSubformulaToClausehsNow = mapSubformulaToClausehsNow;
  } // ResolutionGraph::setMapSubformulaToClausehsNow

  void
  ResolutionGraph::setMapSubformulaToClausehsNext(
        std::map<TreeNode*, ClauseHandleSet> const & mapSubformulaToClausehsNext)
  {
    myMapSubformulaToClausehsNext = mapSubformulaToClausehsNext;
  } // ResolutionGraph::setMapSubformulaToClausehsNext

  void
  ResolutionGraph::setMapSubformulaToClausehsSometime(
   std::map<TreeNode*, ClauseHandleSet> const & mapSubformulaToClausehsSometime)
  {
    myMapSubformulaToClausehsSometime = mapSubformulaToClausehsSometime;
  } // ResolutionGraph::setMapSubformulaToClausehsSometime

  void
  ResolutionGraph::add_starting_clauseh(ClauseHandle const & clauseh)
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");
    Assert(myCurrentPartition == mainPartition,
           "Partition is not mainPartition");
    Assert(!clauseh.isEmpty(), "clauseh is empty.");

    // Insert clauseh into set of starting clausehs
    bool inserted;
    std::set<ClauseHandle>::iterator dummy;
    boost::tie(dummy, inserted) = myStartingClausehs.insert(clauseh);
    Assert(inserted, "Starting clauseh already present in myStartingClausehs.");

    // Create vertex, insert into graph, and update vertex <-> clauseh maps
    add_clauseh_no_premise(clauseh);
  } // ResolutionGraph::add_starting_clauseh

  void
  ResolutionGraph::add_clauseh_no_premise(ClauseHandle const & clauseh)
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");
    Assert(!clauseh.isEmpty(), "clauseh is empty.");

    // Create vertex, insert into graph, and update vertex <-> clauseh maps
    (void) add_vertex(clauseh);
  } // ResolutionGraph::add_clauseh_no_premise

  void
  ResolutionGraph::add_clauseh_one_premise(ClauseHandle const & conclusionh,
                                           ClauseHandle const & premiseh,
                                           InferenceRuleEdgeLabeling const plabel)
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");
    Assert(!conclusionh.isEmpty(), "conclusionh is empty.");
    Assert(!premiseh.isEmpty(), "premiseh is empty.");

    // Create vertex, insert into graph, and update vertex <-> clauseh maps
    (void) add_vertex(conclusionh);

    // Create edge, insert into graph, and update edge -> labeling map
    (void) add_edge(conclusionh, premiseh, plabel);
  } // ResolutionGraph::add_clauseh_two_premises

  void
  ResolutionGraph::add_clauseh_two_premises(ClauseHandle const & conclusionh,
                                            ClauseHandle const & premise1h,
                                            InferenceRuleEdgeLabeling const p1label,
                                            ClauseHandle const & premise2h,
                                            InferenceRuleEdgeLabeling const p2label)
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");
    Assert(!conclusionh.isEmpty(), "conclusionh is empty.");
    Assert(!premise1h.isEmpty(), "premise1h is empty.");
    Assert(!premise2h.isEmpty(), "premise2h is empty.");

    // Create vertex, insert into graph, and update vertex <-> clauseh maps
    (void) add_vertex(conclusionh);

    // Create edges, insert into graph, and update edge -> labeling map
    (void) add_edge(conclusionh, premise1h, p1label);
    (void) add_edge(conclusionh, premise2h, p2label);
  } // ResolutionGraph::add_clauseh_two_premises

  ResolutionGraph::Vertex
  ResolutionGraph::add_vertex(ClauseHandle const & clauseh)
  {
    TRACE(unsatcoreModule, {
        std::cout << "ResolutionGraph: Adding vertex for clause ";
        clauseh.dump(std::cout);
        std::cout << std::endl;
      });

    Assert(clauseh.getPartition() != invalidPartition,
           "clauseh has invalid partition.");
    Assert(!clauseh.isEmpty(), "clauseh is empty.");

    // Create vertex and insert into graph
    Vertex vertex = boost::add_vertex(myResolutionGraph);

    // Update vertex <-> clauseh maps
    Assert(myMapVertexToClauseh.find(vertex) == myMapVertexToClauseh.end(),
           "Vertex already present in myMapVertexToClauseh.");
    myMapVertexToClauseh[vertex] = clauseh;
    Assert(myMapClausehToVertex.find(clauseh) == myMapClausehToVertex.end(),
           "Clauseh already present in myMapClausehToVertex.");
    myMapClausehToVertex[clauseh] = vertex;

    // Update myMainPartition
    if (clauseh.getPartition() == mainPartition) {
      bool inserted;
      VertexSet::iterator dummy;
      boost::tie(dummy, inserted) = myMainPartition.insert(vertex);
      Assert(inserted, "vertex pesent in myMainPartition.");
    }

    // Update myLoopItVertexStore
    if (myLoopItVertexStoreActive) {
      bool inserted;
      VertexSet::iterator dummy;
      boost::tie(dummy, inserted) = myLoopItVertexStore.insert(vertex);
      Assert(inserted, "vertex pesent in myLoopItVertexStore.");
    }

    return vertex;
  } // ResolutionGraph::add_vertex

  void
  ResolutionGraph::remove_vertex(Vertex v)
  {
    ClauseHandle ch = myMapVertexToClauseh[v];
    Assert(myMapClausehToVertex.find(ch) !=
           myMapClausehToVertex.end(),
           "ch not present in myMapClausehToVertex.");
    Assert(myMapClausehToVertex[ch] == v,
           "ch is not mapped to v in myMapClausehToVertex.");
          
    // Remove v->ch from myMapVertexToClauseh
    std::map<Vertex, ClauseHandle>::iterator itvc;
    itvc = myMapVertexToClauseh.find(v);
    myMapVertexToClauseh.erase(itvc);

    // Remove ch->v from myMapClausehToVertex
    std::map<ClauseHandle, Vertex>::iterator itcv;
    itcv = myMapClausehToVertex.find(ch);
    myMapClausehToVertex.erase(itcv);

    // Remove outgoing edges of v from
    // myMapEdgeToInferenceRuleEdgeLabeling
    typedef boost::graph_traits<Graph>::out_edge_iterator oedge_iterator;
    std::pair<oedge_iterator, oedge_iterator> oit;
    for (oit = boost::out_edges(v, myResolutionGraph);
         oit.first != oit.second;
         ++oit.first)
      {
        Assert(myMapEdgeToInferenceRuleEdgeLabeling.find(*oit.first) !=
               myMapEdgeToInferenceRuleEdgeLabeling.end(),
             "*oit.first not present in myMapEdgeToInferenceRuleEdgeLabeling.");

        std::map<Edge, InferenceRuleEdgeLabeling>::iterator it;
        it = myMapEdgeToInferenceRuleEdgeLabeling.find(*oit.first);
        myMapEdgeToInferenceRuleEdgeLabeling.erase(it);
      }

    // Remove incoming edges of v from
    // myMapEdgeToInferenceRuleEdgeLabeling
    typedef boost::graph_traits<Graph>::in_edge_iterator iedge_iterator;
    std::pair<iedge_iterator, iedge_iterator> iit;
    for (iit = boost::in_edges(v, myResolutionGraph);
         iit.first != iit.second;
         ++iit.first)
      {
        Assert(myMapEdgeToInferenceRuleEdgeLabeling.find(*iit.first) !=
               myMapEdgeToInferenceRuleEdgeLabeling.end(),
             "*iit.first not present in myMapEdgeToInferenceRuleEdgeLabeling.");

        std::map<Edge, InferenceRuleEdgeLabeling>::iterator it;
        it = myMapEdgeToInferenceRuleEdgeLabeling.find(*iit.first);
        myMapEdgeToInferenceRuleEdgeLabeling.erase(it);
      }

    // Remove v from myResolutionGraph
    boost::clear_vertex(v, myResolutionGraph);
    boost::remove_vertex(v, myResolutionGraph);
  } // ResolutionGraph::remove_vertex

  ResolutionGraph::Edge
  ResolutionGraph::add_edge(ClauseHandle const & sourceh,
                            ClauseHandle const & desth,
                            InferenceRuleEdgeLabeling const label)
  {
    TRACE(unsatcoreModule, {
        std::cout << "ResolutionGraph: Adding edge between ";
        sourceh.dump(std::cout);
        std::cout << " and ";
        desth.dump(std::cout);
        std::cout << " labeled ";
        dumpInferenceRuleEdgeLabeling(std::cout, label);        
        std::cout << std::endl;
      });

    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");
    Assert(!sourceh.isEmpty(), "sourceh is empty.");
    Assert(!desth.isEmpty(), "desth is empty.");
    Assert(sourceh.getPartition() != invalidPartition,
           "sourceh has invalid partition.");
    Assert(desth.getPartition() != invalidPartition,
           "desth has invalid partition.");
    Assert(myMapClausehToVertex.find(sourceh) != myMapClausehToVertex.end(),
           "sourceh not present in myMapClausehToVertex.");
    Assert(myMapClausehToVertex.find(desth) != myMapClausehToVertex.end(),
           "desth not present in myMapClausehToVertex.");

    // Get vertices for clausehs
    Vertex sourcev = (myMapClausehToVertex.find(sourceh))->second;
    Vertex destv = (myMapClausehToVertex.find(desth))->second;

    // Create edge and insert into graph
    Edge edge;
    bool inserted;
    boost::tie(edge, inserted) = boost::add_edge(sourcev, destv, myResolutionGraph);
    Assert(inserted, "edge already present in myResolutionGraph.");

    // Update edge -> labeling map
    Assert(myMapEdgeToInferenceRuleEdgeLabeling.find(edge) ==
           myMapEdgeToInferenceRuleEdgeLabeling.end(),
           "edge already present in myMapEdgeToInferenceRuleEdgeLabeling");
    myMapEdgeToInferenceRuleEdgeLabeling[edge] = label;

    return edge;
  } // ResolutionGraph::add_edge

  void
  ResolutionGraph::compute_reachable_subgraph(VertexSet const & vs)
  {
    // This set will contain the reachable vertices
    std::set<Vertex> reachable;

    // The vertex index required for boost::breadth_first_visit
    //
    // myVertexIndex isreserved for the point when the resolution
    // graph is pruned to the reachable subgraph.
    VertexIndex vertexIndex = boost::get(boost::vertex_index,
                                         myResolutionGraph);
    {
      VertexIterator vi, vend;
      unsigned long cnt = 0;
      for(boost::tie(vi,vend) = boost::vertices(myResolutionGraph);
          vi != vend;
          ++vi)
        {
          boost::put(vertexIndex, *vi, cnt++);
        }
    }

    // The color map required for boost::breadth_first_visit
    typedef boost::color_traits < boost::default_color_type > Color;
    typedef boost::property_map<Graph, boost::vertex_color_t>::type ColorMap;
    ColorMap colorMap = boost::get(boost::vertex_color, myResolutionGraph);
    {
      VertexIterator vi, vend;
      for(boost::tie(vi,vend) = boost::vertices(myResolutionGraph);
          vi != vend;
          ++vi)
        {
          boost::put(colorMap, *vi, Color::white());
        }
    }

    // Execute bfs
    BfsReachablesRecorder<std::set<Vertex> > vis(&reachable);
    {
      for (VertexSet::iterator it = vs.begin(); it != vs.end(); ++it)
        {
          if (boost::get(colorMap, *it) == Color::white()) {
            boost::breadth_first_visit(myResolutionGraph,
                                       *it,
                                       boost::visitor(vis));
          }
        }
    }

    // Delete unreachable vertices, associated edges, and map entries
    VertexIterator vi, vend, vnext;
    boost::tie(vi, vend) = boost::vertices(myResolutionGraph);
    for (vnext = vi; // See Boost libs/graph/doc/adjacency_list.html
         vi != vend;
         vi = vnext)
      {
        vnext++;
        Assert(myMapVertexToClauseh.find(*vi) !=
               myMapVertexToClauseh.end(),
               "*vi not present in myMapVertexToClauseh.");

        if (reachable.find(*vi) == reachable.end()) {
          // *vi is unreachable
          remove_vertex(*vi);
        }
      }
  } // ResolutionGraph::compute_reachable_subgraph

  void
  ResolutionGraph::forward_image(VertexSet const & vs, VertexSet & result)
  {
    std::map<VertexSet, VertexSet>::iterator it = myFwdImgCache.find(vs);
    if (it != myFwdImgCache.end()) {
      result = it->second;
      ++myNoFwdImgCacheHits;
    } else {
      for (VertexSet::iterator it2 = vs.begin(); it2 != vs.end(); ++it2)
        {
          boost::graph_traits<Graph>::adjacency_iterator ab, ae;
          boost::tie(ab, ae) = boost::adjacent_vertices(*it2, myResolutionGraph);
          result.insert(ab, ae);
        }

      myFwdImgCache[vs] = result;
      ++myNoFwdImgCacheMisses;
    }
  } // ResolutionGraph::forward_image

  void
  ResolutionGraph::backward_image(VertexSet const & vs, VertexSet & result)
  {
    std::map<VertexSet, VertexSet>::iterator it = myBwdImgCache.find(vs);
    if (it != myBwdImgCache.end()) {
      result = it->second;
      ++myNoBwdImgCacheHits;
    } else {
      for (VertexSet::iterator it2 = vs.begin(); it2 != vs.end(); ++it2)
        {
          Graph::inv_adjacency_iterator ab, ae;
          boost::tie(ab, ae) = boost::inv_adjacent_vertices(*it2, myResolutionGraph);
          result.insert(ab, ae);
        }

      myBwdImgCache[vs] = result;
      ++myNoBwdImgCacheMisses;
    }
  } // ResolutionGraph::backward_image

  /* Note: caching does not take forbidden into account. Hence, cache
     must be cleared before this method is called with a new forbidden
     set. */
  void
  ResolutionGraph::forward_image_restricted(VertexSet const & vs,
                                            VertexSet & result,
                                            std::vector<bool> & forbidden)
  {
    std::map<VertexSet, VertexSet>::iterator it = myFwdImgRestrictedCache.find(vs);
    if (it != myFwdImgRestrictedCache.end()) {
      result = it->second;
      ++myNoFwdImgRestrictedCacheHits;
    } else {
      for (VertexSet::iterator it2 = vs.begin(); it2 != vs.end(); ++it2)
        {
          boost::graph_traits<Graph>::adjacency_iterator ab, ae;
          boost::tie(ab, ae) = boost::adjacent_vertices(*it2, myResolutionGraph);
          for (boost::graph_traits<Graph>::adjacency_iterator it3 = ab;
               it3 != ae;
               ++it3)
            {
              if (!forbidden[boost::get(myVertexIndex, *it3)]) {
                result.insert(*it3);
              }
            }
        }

      myFwdImgRestrictedCache[vs] = result;
      ++myNoFwdImgRestrictedCacheMisses;
    }
  } // ResolutionGraph::forward_image_restricted

  TreeNode *
  ResolutionGraph::compute_core_simple_ltl_rec(TreeNode * n, bool isPositivePolarity)
  {
    TreeNode * res;

    // Determine whether n is part of the unsat core
    bool found = false;
    if (myMapSubformulaToClausehsNow.find(n) !=
        myMapSubformulaToClausehsNow.end()) {
      ClauseHandleSet chs = myMapSubformulaToClausehsNow[n];
      for (ClauseHandleSet::iterator it = chs.begin();
           !found && it != chs.end();
           ++it)
        {
          if (myCoreStartingClausehs.find(*it) != myCoreStartingClausehs.end()) {
            found = true;
          }
        }
    }
    if (myMapSubformulaToClausehsNext.find(n) !=
        myMapSubformulaToClausehsNext.end()) {
      ClauseHandleSet chs = myMapSubformulaToClausehsNext[n];
      for (ClauseHandleSet::iterator it = chs.begin();
           !found && it != chs.end();
           ++it)
        {
          if (myCoreStartingClausehs.find(*it) != myCoreStartingClausehs.end()) {
            found = true;
          }
        }
    }
    if (myMapSubformulaToClausehsSometime.find(n) !=
        myMapSubformulaToClausehsSometime.end()) {
      ClauseHandleSet chs = myMapSubformulaToClausehsSometime[n];
      for (ClauseHandleSet::iterator it = chs.begin();
           !found && it != chs.end();
           ++it)
        {
          if (myCoreStartingClausehs.find(*it) != myCoreStartingClausehs.end()) {
            found = true;
          }
        }
    }

    if (!found) {
      if (isPositivePolarity) {
        res = new Identifier(true);
      } else {
        res = new Identifier(false);
      }
      if (!myOptions.isNoUnsatCoreSimplification()) {
        (void) myCoreSimplifiedSubformulas.insert(res);
      }
    } else {
      for (TreeNodeList::const_iterator it = n->children().begin();
           it != n->children().end();
           )
        {
          TreeNodeList::const_iterator itcur = it;
          ++it;
          if (isNot(n)) {
            n->replaceChild(*itcur, compute_core_simple_ltl_rec(*itcur, false));
          } else if (isImplication(n) && *itcur == n->firstChild()) {
            n->replaceChild(*itcur, compute_core_simple_ltl_rec(*itcur, false));
          } else {
            Assert(!isEquivalence(n), "Unexpected equivalence operator.");
            n->replaceChild(*itcur, compute_core_simple_ltl_rec(*itcur, true));
          }
        }
      res = n;
    }

    return res;
  } // ResolutionGraph::compute_core_simple_ltl_rec

  TreeNode *
  ResolutionGraph::compute_core_position_sets_ltl_rec(TreeNode * n,
                                                      bool isPositivePolarity)
  {
    TreeNode * res;

    // Form union of all semi-linear sets associated to n
    SemiLinearSet sls = SemiLinearSet();
    if (myMapSubformulaToClausehsNow.find(n) !=
        myMapSubformulaToClausehsNow.end()) {
      ClauseHandleSet chs = myMapSubformulaToClausehsNow[n];
      for (ClauseHandleSet::iterator it = chs.begin();
           it != chs.end();
           ++it)
        {
          if (myMapClausehToSemiLinearSet.find(*it) !=
              myMapClausehToSemiLinearSet.end()) {
            SemiLinearSet sls2 = myMapClausehToSemiLinearSet[*it];
            (void) sls.insert(sls2.begin(), sls2.end());
          }
        }
    }
    if (myMapSubformulaToClausehsNext.find(n) !=
        myMapSubformulaToClausehsNext.end()) {
      ClauseHandleSet chs = myMapSubformulaToClausehsNext[n];
      for (ClauseHandleSet::iterator it = chs.begin();
           it != chs.end();
           ++it)
        {
          if (myMapClausehToSemiLinearSet.find(*it) !=
              myMapClausehToSemiLinearSet.end()) {
            SemiLinearSet sls2 = myMapClausehToSemiLinearSet[*it];
            sls2.toNext();
            (void) sls.insert(sls2.begin(), sls2.end());
          }
        }
    }
    if (myMapSubformulaToClausehsSometime.find(n) !=
        myMapSubformulaToClausehsSometime.end()) {
      ClauseHandleSet chs = myMapSubformulaToClausehsSometime[n];
      for (ClauseHandleSet::iterator it = chs.begin();
           it != chs.end();
           ++it)
        {
          if (myMapClausehToSemiLinearSet.find(*it) !=
              myMapClausehToSemiLinearSet.end()) {
            SemiLinearSet sls2 = myMapClausehToSemiLinearSet[*it];
            sls2.toSometime();
            (void) sls.insert(sls2.begin(), sls2.end());
          }
        }
    }
    sls.reduce();

    if (sls.empty()) {
      if (isPositivePolarity) {
        res = new Identifier(true);
      } else {
        res = new Identifier(false);
      }
      if (!myOptions.isNoUnsatCoreSimplification()) {
        (void) myCoreSimplifiedSubformulas.insert(res);
      }
    } else {
      for (TreeNodeList::const_iterator it = n->children().begin();
           it != n->children().end();
           )
        {
          TreeNodeList::const_iterator itcur = it;
          ++it;
          if (isNot(n)) {
            n->replaceChild(*itcur,
                            compute_core_position_sets_ltl_rec(*itcur, false));
          } else if (isImplication(n) && *itcur == n->firstChild()) {
            n->replaceChild(*itcur,
                            compute_core_position_sets_ltl_rec(*itcur, false));
          } else {
            Assert(!isEquivalence(n), "Unexpected equivalence operator.");
            n->replaceChild(*itcur,
                            compute_core_position_sets_ltl_rec(*itcur, true));
          }
        }
      res = n;
    }

    myMapSubformulaToSemiLinearSet[res] = sls;

    return res;
  } // ResolutionGraph::compute_core_position_sets_ltl_rec

  TreeNode * ResolutionGraph::simplify_core_ltl_rec(TreeNode * n,
                                                    bool isPositivePolarity)
  {
    TreeNode * res = NULL;
    
    // simplify children
    {
      for (TreeNodeList::const_iterator it = n->children().begin();
           it != n->children().end();
           )
        {
          TreeNodeList::const_iterator itcur = it;
          ++it;
          TreeNode * childSimplified;
          if (isNot(n)) {
            childSimplified = simplify_core_ltl_rec(*itcur, !isPositivePolarity);
            if (*itcur != childSimplified) {
              TreeNode * tmp = *itcur;
              n->replaceChild(*itcur, childSimplified);
              delete(tmp);
            }
          } else if (isImplication(n) && *itcur == n->firstChild()) {
            childSimplified = simplify_core_ltl_rec(*itcur, !isPositivePolarity);
            if (*itcur != childSimplified) {
              TreeNode * tmp = *itcur;
              n->replaceChild(*itcur, childSimplified);
              delete(tmp);
            }
          } else {
            Assert(!isEquivalence(n), "Unexpected equivalence operator.");
            childSimplified = simplify_core_ltl_rec(*itcur, isPositivePolarity);
            if (*itcur != childSimplified) {
              TreeNode * tmp = *itcur;
              n->replaceChild(*itcur, childSimplified);
              delete(tmp);
            }
          }
        }
    }

    // simplify self
    if (isIdentifier(n)) {
      Assert(n->childrenCount() == 0, "Unexpected number of children.");
      res = n;
    } else if (isNot(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      TreeNode * child; child = n->firstChild(); // silence compiler
      Assert(myCoreSimplifiedSubformulas.find(child) ==
             myCoreSimplifiedSubformulas.end(),
             "child present in myCoreSimplifiedSubformulas.");
      res = n;
    } else if (isAnd(n)) {
      Assert(n->childrenCount() > 0, "Unexpected number of children.");

      // remove True
      {
        for (TreeNodeList::const_iterator it = n->children().begin();
             it != n->children().end();
             )
          {
            TreeNodeList::const_iterator itcur = it;
            ++it;
            if (isIdentifier(*itcur) &&
                ((Identifier *) *itcur)->isTrue() &&
                myCoreSimplifiedSubformulas.find(*itcur) !=
                myCoreSimplifiedSubformulas.end()) {
              n->removeChild(*itcur);
            }
          }
        Assert(n->childrenCount() > 0, "Unexpected number of children.");
      }

      // single child
      if (n->childrenCount() == 1) {
        Assert(myCoreSimplifiedSubformulas.find(n->firstChild()) ==
               myCoreSimplifiedSubformulas.end(),
               "n->firstChild() present in myCoreSimplifiedSubformulas.");
        res = n->firstChild()->clone();
        n->removeChild(n->firstChild());
        return res;
      }
      
      res = n;
    } else if (isOr(n)) {
      Assert(n->childrenCount() > 0, "Unexpected number of children.");

      // remove False
      {
        for (TreeNodeList::const_iterator it = n->children().begin();
             it != n->children().end();
             )
          {
            TreeNodeList::const_iterator itcur = it;
            ++it;
            if (isIdentifier(*itcur) &&
                ((Identifier *) *itcur)->isFalse() &&
                myCoreSimplifiedSubformulas.find(*itcur) !=
                myCoreSimplifiedSubformulas.end()) {
              n->removeChild(*itcur);
            }
          }
        Assert(n->childrenCount() > 0, "Unexpected number of children.");
      }

      // single child
      if (n->childrenCount() == 1) {
        Assert(myCoreSimplifiedSubformulas.find(n->firstChild()) ==
               myCoreSimplifiedSubformulas.end(),
               "n->firstChild() present in myCoreSimplifiedSubformulas.");
        res = n->firstChild()->clone();
        n->removeChild(n->firstChild());
        return res;
      }
      
      res = n;
    } else if (isImplication(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      TreeNode * fst = n->firstChild();
      TreeNode * snd = n->secondChild();
      if (isIdentifier(fst) &&
          ((Identifier *) fst)->isTrue() &&
          myCoreSimplifiedSubformulas.find(fst) !=
          myCoreSimplifiedSubformulas.end()) {
        Assert(myCoreSimplifiedSubformulas.find(snd) ==
               myCoreSimplifiedSubformulas.end(),
               "snd present in myCoreSimplifiedSubformulas.");
        res = snd->clone();
        n->removeChild(snd);
        return res;
      } else if (isIdentifier(snd) &&
                 ((Identifier *) snd)->isFalse() &&
                 myCoreSimplifiedSubformulas.find(snd) !=
                 myCoreSimplifiedSubformulas.end()) {
        Assert(myCoreSimplifiedSubformulas.find(fst) ==
               myCoreSimplifiedSubformulas.end(),
               "fst present in myCoreSimplifiedSubformulas.");
        res = Not(fst->clone());
        n->removeChild(fst);
        return res;
      }
      res = n;
    } else if (isNext(n) || isAlways(n) || isSometime(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      TreeNode * child; child = n->firstChild(); // silence compiler
      Assert(myCoreSimplifiedSubformulas.find(child) ==
             myCoreSimplifiedSubformulas.end(),
             "child present in myCoreSimplifiedSubformulas.");
      res = n;
    } else if (isUntil(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      TreeNode * fst = n->firstChild();
      TreeNode * snd = n->secondChild();
      if (isIdentifier(fst) &&
          ((Identifier *) fst)->isTrue() &&
          myCoreSimplifiedSubformulas.find(fst) !=
          myCoreSimplifiedSubformulas.end()) {
        Assert(myCoreSimplifiedSubformulas.find(snd) ==
               myCoreSimplifiedSubformulas.end(),
               "snd present in myCoreSimplifiedSubformulas.");
        ((Operator *) n)->setOperator(SOMETIME);
        n->removeChild(fst);
        res = n;
        return res;
      } else if (isIdentifier(fst) &&
                 ((Identifier *) fst)->isFalse() &&
                 myCoreSimplifiedSubformulas.find(fst) !=
                 myCoreSimplifiedSubformulas.end()) {
        Assert(myCoreSimplifiedSubformulas.find(snd) ==
               myCoreSimplifiedSubformulas.end(),
               "snd present in myCoreSimplifiedSubformulas.");
        res = snd->clone();
        n->removeChild(snd);
        return res;
      }
      res = n;
    } else if (isUnless(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      TreeNode * fst = n->firstChild();
      TreeNode * snd = n->secondChild();
      if (isIdentifier(snd) &&
          ((Identifier *) snd)->isFalse() &&
          myCoreSimplifiedSubformulas.find(snd) !=
          myCoreSimplifiedSubformulas.end()) {
        Assert(myCoreSimplifiedSubformulas.find(fst) ==
               myCoreSimplifiedSubformulas.end(),
               "fst present in myCoreSimplifiedSubformulas.");
        ((Operator *) n)->setOperator(ALWAYS);
        n->removeChild(snd);
        res = n;
        return res;
      } else if (isIdentifier(fst) &&
                 ((Identifier *) fst)->isFalse() &&
                 myCoreSimplifiedSubformulas.find(fst) !=
                 myCoreSimplifiedSubformulas.end()) {
        Assert(myCoreSimplifiedSubformulas.find(snd) ==
               myCoreSimplifiedSubformulas.end(),
               "snd present in myCoreSimplifiedSubformulas.");
        res = snd->clone();
        n->removeChild(snd);
        return res;
      }
      res = n;
    } else {
      Assert(false, "n has unexpected type.");
    }

    return res;
  } // ResolutionGraph::simplify_core_ltl_rec

  std::ostream &
  ResolutionGraph::dumpInferenceRuleEdgeLabeling(std::ostream & os,
                                    InferenceRuleEdgeLabeling const label) const
  {
    Assert(myCurrentPartition != invalidPartition,
           "Partition is invalidPartition");

    switch (label) {
    case INFERENCE_RULE_EDGE_LABELING_INIT_II:
      os << "init-ii";
      break;
    case INFERENCE_RULE_EDGE_LABELING_INIT_INI:
      os << "init-ini";
      break;
    case INFERENCE_RULE_EDGE_LABELING_INIT_INN:
      os << "init-inn";
      break;
    case INFERENCE_RULE_EDGE_LABELING_STEP_NN:
      os << "step-nn";
      break;
    case INFERENCE_RULE_EDGE_LABELING_STEP_NXN:
      os << "step-nxn";
      break;
    case INFERENCE_RULE_EDGE_LABELING_STEP_NXX:
      os << "step-nxx";
      break;
    case INFERENCE_RULE_EDGE_LABELING_STEP_XX:
      os << "step-xx";
      break;
    case INFERENCE_RULE_EDGE_LABELING_AUG1:
      os << "aug1";
      break;
    case INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_X:
      os << "BFS-loop-it-init-x";
      break;
    case INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_INIT_N:
      os << "BFS-loop-it-init-n";
      break;
    case INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_IT_SUB:
      os << "BFS-loop-it-sub";
      break;
    case INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION1_WEMPTYG:
      os << "BFS-loop-conclusion1-wemptyg";
      break;
    case INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION1_WEMPTYE:
      os << "BFS-loop-conclusion1-wemptye";
      break;
    case INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION2_WEMPTYG:
      os << "BFS-loop-conclusion2-wemptyg";
      break;
    case INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION1_WOEMPTYG:
      os << "BFS-loop-conclusion1-woemptyg";
      break;
    case INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION1_WOEMPTYE:
      os << "BFS-loop-conclusion1-woemptye";
      break;
    case INFERENCE_RULE_EDGE_LABELING_BFS_LOOP_CONCLUSION2_WOEMPTYG :
      os << "BFS-loop-conclusion2-woemptyg";
      break;
    default:
      Assert(false, "Unexpected label.");
    }
    return os;
  } // ResolutionGraph::dumpInferenceRuleEdgeLabeling
} // namespace UnsatCore
