/************************************************************
 *    Copyright (C) 2012                                    *
 *    Viktor Schuppan (Viktor.Schuppan@gmx.de)              *
 *                                                          *
 *    This program is free software; you can redistribute   *
 *    it and/or modify it under the terms of the GNU        *
 *    General Public License as published by the Free       *
 *    Software Foundation, either version 3 of the License, *
 *    or (at your option) any later version.                *
 *                                                          *
 *    This program is distributed in the hope that it will  *
 *    be useful, but WITHOUT ANY WARRANTY; without even     *
 *    the implied warranty of MERCHANTABILITY or FITNESS    *
 *    FOR A PARTICULAR PURPOSE.  See the GNU General Public *
 *    License for more details.                             *
 *                                                          *
 *    You should have received a copy of the GNU General    *
 *    Public License along with this program.  If not, see  *
 *    <http://www.gnu.org/licenses/>.                       *
 ************************************************************/
#include <algorithm>
#include <limits>
#include <list>
#include <set>
#include <boost/tuple/tuple.hpp>
#include "misc/assert.h"
#include "misc/tracer.h"
#include "modules.h"
#include "unsat_core_private.h"
#include "linear_set.h"
#include "semi_linear_set.h"

namespace UnsatCore
{
  void
  SemiLinearSet::addLinearSet(LinearSet const & ls)
  {
    bool inserted;
    SemiLinearSet::iterator dummy;
    boost::tie(dummy, inserted) = insert(ls);
  } // SemiLinearSet::addLinearSet

  // make more generic and mathematical by having general add or shift
  // method for linear and semilinear sets; might want to rename
  // addLinearSet to avoid confusion
  //
  // shift each linear set by +1
  void
  SemiLinearSet::toNext()
  {
    SemiLinearSet sls = SemiLinearSet();

    for (iterator it = begin(); it != end(); ++it)
      {
        unsigned long p = it->getPeriod();
        unsigned long o = it->getOffset();
        Assert(o < std::numeric_limits<unsigned long>::max() - 1,
               "o >= std::numeric_limits<unsigned long>::max().");
        (void) sls.insert(LinearSet(p, o + 1));
      }

    this->swap(sls);
  } // SemiLinearSet::toNext

  // turn semilinear set sls into (1 N, min(sls))
  void
  SemiLinearSet::toSometime()
  {
    if (!empty()) {
      // find minimum
      unsigned long m =
        std::numeric_limits<unsigned long>::max();
      for (iterator it = begin(); it != end(); ++it) {
        if (it->getOffset() < m) {
          m = it->getOffset();
        }
      }

      clear();
      Assert(m < std::numeric_limits<unsigned long>::max(),
             "m >= std::numeric_limits<unsigned long>::max().");
      (void) insert(LinearSet(1, m));
    }
  } // SemiLinearSet::toSometime

  std::ostream &
  SemiLinearSet::dump(std::ostream & os) const
  {
    os << "{";
    for (iterator it = begin(); it != end();)
      {
        (void) it->dump(os);
        ++it;
        if (it != end()) {
          os << ", ";
        }
      }
    os << "}";
    return os;
  } // SemiLinearSet::dump

  void
  SemiLinearSet::reduce()
  {
    bool changed1 = true;
    std::list<LinearSet> wlist = std::list<LinearSet>(begin(), end());;
    std::list<LinearSet>::iterator it1cur, it1next, it2cur, it2next;

    wlist.reverse();
    while (changed1) {
      changed1 = false;
      for (it1next = it1cur = wlist.begin();
           it1cur != wlist.end();
           it1cur = it1next)
        {
          ++it1next;
          for (it2next = it2cur = it1next;
               it2cur != wlist.end();
               it2cur = it2next)
            {
              ++it2next;
              bool changed2;
              LinearSet ls1 = *it1cur;
              LinearSet ls2 = *it2cur;
              changed2 = LinearSet::reduce(*it1cur, *it2cur);
              TRACE(unsatcoreModule, {
                  if (changed2) {
                    std::cout << "ResolutionGraph: Reducing linear sets ";
                    ls1.dump(std::cout);
                    std::cout << ", ";
                    ls2.dump(std::cout);
                    std::cout << " to ";
                    it1cur->dump(std::cout);
                    std::cout << ", ";
                    it2cur->dump(std::cout);
                    std::cout << "." << std::endl;
                  }
                });
              if (it2cur->isInvalid()) {
                if (it2cur == it1next) {
                  ++it1next;
                }
                wlist.erase(it2cur);
              }
              if (it1cur->isInvalid()) {
                wlist.erase(it1cur);
                it2next = wlist.end();
              }
              changed1 = changed1 || changed2;
            }
        }
      /* Simplify sequences (c N, d), (c N, d + 1), ...,
         (c N, d + c - 1) to (1 N, d) */
      {
        SemiLinearSet sls = SemiLinearSet();
        sls.insert(wlist.begin(), wlist.end());

        for (std::list<LinearSet>::iterator it = wlist.begin();
             it != wlist.end();
             ++it)
          {
            if (it->getPeriod() > 1) {
              unsigned long p = it->getPeriod();
              unsigned long o = it->getOffset();
              bool failed = false;
              for (unsigned long i = 1; !failed && i < p; ++i)
                {
                  Assert(o + i < std::numeric_limits<unsigned long>::max(),
                         "o + i >= std::numeric_limits<unsigned long>::max()");
                  failed = (sls.find(LinearSet(p, o + i)) == sls.end());
                }
              if (!failed) {
                changed1 = true;
                for (unsigned long i = 0; i < p; ++i)
                  {
                    (void) sls.erase(LinearSet(p, o + i));
                  }
                sls.insert(LinearSet(1, o));
              }
            }
          }
        wlist.assign(sls.begin(), sls.end());
      }
      wlist.sort();
      wlist.reverse();
    }

    TRACE(unsatcoreModule, {
        std::cout << "ResolutionGraph: Reducing semi-linear set" << std::endl;
        std::cout << "ResolutionGraph: ";
        dump(std::cout);
        std::cout << std::endl << "ResolutionGraph: to " << std::endl;
      });
    clear();
    insert(wlist.begin(), wlist.end());
    TRACE(unsatcoreModule, {
        std::cout << "ResolutionGraph: ";
        dump(std::cout);
        std::cout << std::endl;
      });
  } // SemiLinearSet::reduce
} // namespace UnsatCore



