/************************************************************
 *    Copyright (C) 2012-2013                               *
 *    Viktor Schuppan (Viktor.Schuppan@gmx.de)              *
 *                                                          *
 *    This program is free software; you can redistribute   *
 *    it and/or modify it under the terms of the GNU        *
 *    General Public License as published by the Free       *
 *    Software Foundation, either version 3 of the License, *
 *    or (at your option) any later version.                *
 *                                                          *
 *    This program is distributed in the hope that it will  *
 *    be useful, but WITHOUT ANY WARRANTY; without even     *
 *    the implied warranty of MERCHANTABILITY or FITNESS    *
 *    FOR A PARTICULAR PURPOSE.  See the GNU General Public *
 *    License for more details.                             *
 *                                                          *
 *    You should have received a copy of the GNU General    *
 *    Public License along with this program.  If not, see  *
 *    <http://www.gnu.org/licenses/>.                       *
 ************************************************************/
#include <iostream>
#include <map>
#include <string>
#include <utility>
#include <stdlib.h>
#include <boost/lexical_cast.hpp>
#include "adaptor/iclause.h"
#include "adaptor/sclause.h"
#include "adaptor/stl_tim_adaptor.h"
#include "adaptor/uclause.h"
#include "getopt/options.h"
#include "misc/assert.h"
#include "misc/tracer.h"
#include "modules.h"
#include "stl_tim/attribute.h"
#include "stl_tim/literal.h"
#include "stl_tim/literallist.h"
#include "stl_tim/proposition.h"
#include "trim/eclause.h"
#include "trim/trim.h"
#include "unsat_core/clause_handle.h"
#include "formula.h"
#include "snf_translator.h"

namespace BuilderLtl
{
  SnfTranslator::SnfTranslator(TreeNodeList & formulaList,
                               UnsatCore::ClauseHandleSet & clausehSet,
                               std::map<TreeNode*, UnsatCore::ClauseHandleSet> &
                                 mapSubformulaToClausehsNow,
                               std::map<TreeNode*, UnsatCore::ClauseHandleSet> &
                                 mapSubformulaToClausehsNext,
                               std::map<TreeNode*, UnsatCore::ClauseHandleSet> &
                                 mapSubformulaToClausehsSometime,
                               TRPPPOptions const & options)
    : myFormulaList(formulaList),
      myClausehSet(clausehSet),
      myMapSubformulaToClausehsNow(mapSubformulaToClausehsNow),
      myMapSubformulaToClausehsNext(mapSubformulaToClausehsNext),
      myMapSubformulaToClausehsSometime(mapSubformulaToClausehsSometime),
      myOptions(options),
      myMapEventualityLiteralToLiteral(),
      myFormulaToSnfCachePosInit(),
      myFormulaToSnfCacheNegInit(),
      myFormulaToSnfCachePosNoInit(),
      myFormulaToSnfCacheNegNoInit(),
      myNvvCounter(0),
      myNoTopLevelSubformulaClauses(0),
      myNoFormulaToSnfCacheHits(0),
      myNoFormulaToSnfCacheMisses(0)
  { } // SnfTranslator::SnfTranslator

  void
  SnfTranslator::translate()
  {
    // turn top level conjunction into list of conjuncts
    for (TreeNodeList::iterator it = myFormulaList.begin();
         it != myFormulaList.end();
         )
      {
        TRACE(builderltlModule, {
            std::cout << "BuilderLtl: checking whether the following top "
                         "level conjunct can be lifted: "
                      << *it
                      << std::endl;
          });
        if (isAnd(*it)) {
          for (TreeNodeList::const_iterator it2 = (*it)->children().begin();
               it2 != (*it)->children().end();
               )
            {
              TreeNodeList::const_iterator it2cur = it2;
              ++it2;
              TreeNode * tmp = *it2cur;
              (*it)->removeChild(*it2cur);
              myFormulaList.push_back(tmp);
            }
          TreeNodeList::iterator itcur = it;
          ++it;
          TreeNode * tmp = *itcur;
          myFormulaList.erase(itcur);
          delete(tmp);
        } else {
          ++it;
        }
      }
    for (TreeNodeList::const_iterator it = myFormulaList.begin();
         it != myFormulaList.end();
         ++it)
      {
        if (!myOptions.isNoDirectClauses() &&
            is_initial_clause(*it)) {
          // treat initial clauses specially
          TRACE(builderltlModule, {
              std::cout << "BuilderLtl: Treating "
                        << *it
                        << " as an initial clause."
                        << std::endl;
            });

          ++myNoTopLevelSubformulaClauses;
          make_and_add_iclause(*it);
        } else if (!myOptions.isNoDirectClauses() &&
                   is_universal_clause(*it)) {
          // treat universal clauses specially
          TRACE(builderltlModule, {
              std::cout << "BuilderLtl: Treating "
                        << *it
                        << " as a universal clause."
                        << std::endl;
            });

          ++myNoTopLevelSubformulaClauses;
          make_and_add_uclause(*it);
        } else if (!myOptions.isNoDirectClauses() &&
                   is_step_clause(*it)) {
          // treat step clauses specially
          TRACE(builderltlModule, {
              std::cout << "BuilderLtl: Treating "
                        << *it
                        << " as a step clause."
                        << std::endl;
            });

          ++myNoTopLevelSubformulaClauses;
          make_and_add_sclause(*it);
        } else if (!myOptions.isNoDirectClauses() &&
                   is_eventuality_clause(*it)) {
          // treat eventuality clauses specially
          TRACE(builderltlModule, {
              std::cout << "BuilderLtl: Treating "
                        << *it
                        << " as an eventuality clause."
                        << std::endl;
            });

          ++myNoTopLevelSubformulaClauses;
          make_and_add_eclause(*it);
        } else {
          // the general case
          Identifier id = Identifier("dummy");

          id = translate_to_snf_rec(*it, true, true);

          if (id.isTrue()) {
            // no clause
          } else if (id.isFalse()) {
            // empty clause
            Adaptors::IClause ic =
            Adaptors::Adaptor::make_iclause(PropositionalProver::LiteralList());
            (void) myClausehSet.insert(UnsatCore::ClauseHandle(ic,
                                                     UnsatCore::mainPartition));
            if (myOptions.isExtractUnsatCore()) {
              translate_to_snf_add_to_msftoch_now(*it,
                         UnsatCore::ClauseHandle(ic, UnsatCore::mainPartition));
            }
          } else {
            PropositionalProver::Proposition p =
              PropositionalProver::Proposition(id.getIdentifier());
            PropositionalProver::Literal l =
              PropositionalProver::Literal(p,
                                           true,
                                           PropositionalProver::initial_attr);
            PropositionalProver::LiteralList ll =
              PropositionalProver::LiteralList();
            ll.push_back(l);

            Adaptors::IClause ic = Adaptors::Adaptor::make_iclause(ll);
            
            (void) myClausehSet.insert(UnsatCore::ClauseHandle(ic,
                                                     UnsatCore::mainPartition));
            if (myOptions.isExtractUnsatCore()) {
              translate_to_snf_add_to_msftoch_now(*it,
                         UnsatCore::ClauseHandle(ic, UnsatCore::mainPartition));
            }
          }
        }
      }
  } // SnfTranslator::translate

  bool
  SnfTranslator::is_initial_clause(TreeNode * n)
  {
    return is_propositional_next_disjunction(n, false);
  } // SnfTranslator::is_initial_clause

  bool
  SnfTranslator::is_universal_clause(TreeNode * n)
  {
    bool res = false;

    if (isAlways(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      res = is_propositional_next_disjunction(n->firstChild(), false);
    }

    return res;
  } // SnfTranslator::is_universal_clause

  bool
  SnfTranslator::is_step_clause(TreeNode * n)
  {
    bool res = false;

    if (isAlways(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      res = is_propositional_next_disjunction(n->firstChild(), true);
    }

    return res;
  } // SnfTranslator::is_step_clause

  bool
  SnfTranslator::is_eventuality_clause(TreeNode * n)
  {
    bool res = false;

    if (isAlways(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      TreeNode * nc = n->firstChild();
      if (isOr(nc)) {
        Assert(nc->childrenCount() > 0, "Unexpected number of children.");
        int no_literal_eventualities = 0;
        res = true;
        for (TreeNodeList::const_iterator it = n->children().begin();
             res &&
               no_literal_eventualities < 2 &&
               it != n->children().end();
             ++it)
          {
            if (is_literal_eventuality(*it)) {
              ++no_literal_eventualities;
            } else {
              res = res && is_propositional_next_disjunction(*it, false);
            }
          }
        res = res && (no_literal_eventualities == 1);
      } else if (isImplication(nc)) {
        Assert(nc->childrenCount() == 2, "Unexpected number of children.");
        res = is_literal_eventuality(nc->secondChild()) &&
          is_propositional_next_conjunction(nc->firstChild(), false);
      } else if (isSometime(nc)) {
        Assert(nc->childrenCount() == 1, "Unexpected number of children.");
        res = is_literal_eventuality(nc);
      }
    }

    return res;
  } // SnfTranslator::is_eventuality_clause

  bool
  SnfTranslator::is_propositional_next_conjunction(TreeNode * n,
                                                   bool allowNext)
  {
    bool res = false;

    if (isIdentifier(n)) {
      Assert(n->childrenCount() == 0, "Unexpected number of children.");
      res = true;
    } else if (isNot(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      res = is_propositional_next_disjunction(n->firstChild(), allowNext);
    } else if (isAnd(n)) {
      Assert(n->childrenCount() > 0, "Unexpected number of children.");
      res = true;
      for (TreeNodeList::const_iterator it = n->children().begin();
           res && it != n->children().end();
           ++it)
        {
          res = res && is_propositional_next_conjunction(*it, allowNext);
        }
    } else if (isOr(n)) {
      Assert(n->childrenCount() > 0, "Unexpected number of children.");
      if (n->childrenCount() == 1) {
        res = is_propositional_next_conjunction(n->firstChild(), allowNext);
      } else {
        res = false;
      }
    } else if (isImplication(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      res = false;
    } else if (isNext(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      if (allowNext) {
        res = is_propositional_next_conjunction(n->firstChild(), false);
      } else {
        res = false;
      }
    } else {
      res = false;
    }

    return res;
  } // SnfTranslator::is_propositional_next_conjunction

  bool
  SnfTranslator::is_propositional_next_disjunction(TreeNode * n,
                                                   bool allowNext)
  {
    bool res = false;

    if (isIdentifier(n)) {
      Assert(n->childrenCount() == 0, "Unexpected number of children.");
      res = true;
    } else if (isNot(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      res = is_propositional_next_conjunction(n->firstChild(), allowNext);
    } else if (isAnd(n)) {
      Assert(n->childrenCount() > 0, "Unexpected number of children.");
      if (n->childrenCount() == 1) {
        res = is_propositional_next_disjunction(n->firstChild(), allowNext);
      } else {
        res = false;
      }
    } else if (isOr(n)) {
      Assert(n->childrenCount() > 0, "Unexpected number of children.");
      res = true;
      for (TreeNodeList::const_iterator it = n->children().begin();
           res && it != n->children().end();
           ++it)
        {
          res = res && is_propositional_next_disjunction(*it, allowNext);
        }
    } else if (isImplication(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      res = is_propositional_next_conjunction(n->firstChild(), allowNext);
      res = res && is_propositional_next_disjunction(n->secondChild(), allowNext);
    } else if (isNext(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      if (allowNext) {
        res = is_propositional_next_disjunction(n->firstChild(), false);
      } else {
        res = false;
      }
    } else {
      res = false;
    }

    return res;
  } // SnfTranslator::is_propositional_next_disjunction

  bool
  SnfTranslator::is_literal_eventuality(TreeNode * n)
  {
    bool res = false;

    if (isSometime(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      if (isIdentifier(n->firstChild())) {
        res = true;
      } else if (isNot(n->firstChild())) {
        Assert(n->childrenCount() == 1, "Unexpected number of children.");
        if (isIdentifier(n->firstChild()->firstChild())) {
          res = true;
        }
      }
    }

    return res;
  } // SnfTranslator::is_literal_eventuality

  void
  SnfTranslator::make_and_add_iclause(TreeNode * n)
  {
    Assert(is_initial_clause(n), "n is not an initial clause.");

    bool is_true = false;

    PropositionalProver::LiteralList ll = PropositionalProver::LiteralList();
    PropositionalProver::LiteralList ll_next = PropositionalProver::LiteralList();
    
    collect_literals_clause(n,
                            true,
                            false,
                            PropositionalProver::initial_attr,
                            ll,
                            ll_next,
                            is_true);

    Assert(ll_next.empty(), "ll_next not empty for initial clause.");
    if (!is_true) {
      if (ll.empty()) {
        // add empty clause
        Adaptors::IClause ic =
          Adaptors::Adaptor::make_iclause(PropositionalProver::LiteralList());
        (void) myClausehSet.insert(UnsatCore::ClauseHandle(ic,
                                                     UnsatCore::mainPartition));
        translate_to_snf_msftoch_clause(n,
                          UnsatCore::ClauseHandle(ic, UnsatCore::mainPartition),
                                        false,
                                        false);
        TRACE(builderltlModule, {
            std::cout << "BuilderLtl: Initial clause is empty"
                      << std::endl;
          });
      } else {
        Adaptors::IClause ic = Adaptors::Adaptor::make_iclause(ll);
        (void) myClausehSet.insert(UnsatCore::ClauseHandle(ic,
                                                     UnsatCore::mainPartition));
        translate_to_snf_msftoch_clause(n,
                          UnsatCore::ClauseHandle(ic, UnsatCore::mainPartition),
                                        false,
                                        false);
        TRACE(builderltlModule, {
            std::cout << "BuilderLtl: Initial clause is ";
            ic.getInternalRep()->dump(std::cout);
            std::cout << std::endl;
          });
      }
    } else {
      TRACE(builderltlModule, {
          std::cout << "BuilderLtl: Initial clause is true"
                    << std::endl;
        });
    }
  } // SnfTranslator::make_and_add_iclause

  void
  SnfTranslator::make_and_add_uclause(TreeNode * n)
  {
    Assert(is_universal_clause(n), "n is not an universal clause.");

    Adaptors::UClause uc =
      Adaptors::Adaptor::make_uclause(PropositionalProver::LiteralList());
    bool is_true = false;

    PropositionalProver::LiteralList ll = PropositionalProver::LiteralList();
    PropositionalProver::LiteralList ll_next = PropositionalProver::LiteralList();

    collect_literals_clause(n,
                            true,
                            false,
                            PropositionalProver::universal_attr,
                            ll,
                            ll_next,
                            is_true);

    Assert(ll_next.empty(), "ll_next not empty for universal clause.");
    if (!is_true) {
      if (ll.empty()) {
        // add empty clause
        Adaptors::IClause ic =
          Adaptors::Adaptor::make_iclause(PropositionalProver::LiteralList());
        (void) myClausehSet.insert(UnsatCore::ClauseHandle(ic,
                                                     UnsatCore::mainPartition));
        translate_to_snf_msftoch_clause(n,
                          UnsatCore::ClauseHandle(ic, UnsatCore::mainPartition),
                                        false,
                                        false);
        TRACE(builderltlModule, {
            std::cout << "BuilderLtl: Universal clause is empty"
                      << std::endl;
          });
      } else {
        uc = Adaptors::Adaptor::make_uclause(ll);
        (void) myClausehSet.insert(UnsatCore::ClauseHandle(uc,
                                                     UnsatCore::mainPartition));
        translate_to_snf_msftoch_clause(n,
                          UnsatCore::ClauseHandle(uc, UnsatCore::mainPartition),
                                        false,
                                        false);
        TRACE(builderltlModule, {
            std::cout << "BuilderLtl: Universal clause is ";
            uc.getInternalRep()->dump(std::cout);
            std::cout << std::endl;
          });
      }
    } else {
      TRACE(builderltlModule, {
          std::cout << "BuilderLtl: Universal clause is true"
                    << std::endl;
        });
    }
  } // SnfTranslator::make_and_add_uclause

  void
  SnfTranslator::make_and_add_sclause(TreeNode * n)
  {
    Assert(is_step_clause(n), "n is not a step clause.");

    Adaptors::SClause sc =
      Adaptors::Adaptor::make_sclause(PropositionalProver::LiteralList(),
                                      PropositionalProver::LiteralList());
    bool is_true = false;

    PropositionalProver::LiteralList ll = PropositionalProver::LiteralList();
    PropositionalProver::LiteralList ll_next = PropositionalProver::LiteralList();

    collect_literals_clause(n,
                            true,
                            false,
                            PropositionalProver::step_now_attr,
                            ll,
                            ll_next,
                            is_true);

    if (!is_true) {
      if (ll.empty() && ll_next.empty()) {
        // add empty clause
        Adaptors::IClause ic =
          Adaptors::Adaptor::make_iclause(PropositionalProver::LiteralList());
        (void) myClausehSet.insert(UnsatCore::ClauseHandle(ic,
                                                     UnsatCore::mainPartition));
        translate_to_snf_msftoch_clause(n,
                          UnsatCore::ClauseHandle(ic, UnsatCore::mainPartition),
                                        false,
                                        false);
        TRACE(builderltlModule, {
            std::cout << "BuilderLtl: Step clause is empty"
                      << std::endl;
          });
      } else {
        sc = Adaptors::Adaptor::make_sclause(ll, ll_next);
        (void) myClausehSet.insert(UnsatCore::ClauseHandle(sc,
                                                     UnsatCore::mainPartition));
        translate_to_snf_msftoch_clause(n,
                          UnsatCore::ClauseHandle(sc, UnsatCore::mainPartition),
                                        false,
                                        false);
        TRACE(builderltlModule, {
            std::cout << "BuilderLtl: Step clause is ";
            sc.getInternalRep()->dump(std::cout);
            std::cout << std::endl;
          });
      }
    } else {
      TRACE(builderltlModule, {
          std::cout << "BuilderLtl: Step clause is true"
                    << std::endl;
        });
    }
  } // SnfTranslator::make_and_add_sclause

  void
  SnfTranslator::make_and_add_eclause(TreeNode * n)
  {
    Assert(is_eventuality_clause(n), "n is not an eventuality clause.");

    PropositionalProver::LiteralList ll =
      PropositionalProver::LiteralList();
    PropositionalProver::LiteralList ll_next =
      PropositionalProver::LiteralList();
    bool is_true = false;

    if (isAlways(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      TreeNode * nc = n->firstChild();
      if (isOr(nc)) {
        Assert(nc->childrenCount() > 0, "Unexpected number of children.");
        TreeNode * litev = NULL;
        for (TreeNodeList::const_iterator it = nc->children().begin();
             it != nc->children().end();
             ++it)
          {
            if (is_literal_eventuality(*it)) {
              litev = *it;
            } else {
              collect_literals_clause(*it,
                                      true,
                                      false,
                                      PropositionalProver::universal_attr,
                                      ll,
                                      ll_next,
                                      is_true);
            }
          }
        Assert(ll_next.empty(), "ll_next is not empty.");
        Assert(litev, "litev is NULL.");
        make_and_add_eclause_aux(n, ll, litev, is_true);
      } else if (isImplication(nc)) {
        Assert(nc->childrenCount() == 2, "Unexpected number of children.");
        collect_literals_clause(nc->firstChild(),
                                false,
                                false,
                                PropositionalProver::universal_attr,
                                ll,
                                ll_next,
                                is_true);
        Assert(ll_next.empty(), "ll_next is not empty.");
        make_and_add_eclause_aux(n, ll, nc->secondChild(), is_true);
      } else if (isSometime(nc)) {
        Assert(nc->childrenCount() == 1, "Unexpected number of children.");
        make_and_add_eclause_aux(n, ll, nc, false);
      }
    }
  } // SnfTranslator::make_and_add_eclause

  void
  SnfTranslator::make_and_add_eclause_aux(TreeNode * n_clause,
                                          PropositionalProver::LiteralList & ll,
                                          TreeNode * n_evlit,
                                          bool isTrue)
  {
    Assert(is_literal_eventuality(n_evlit), "n_evlit is not a literal eventuality.");

    TreeNode * nc = n_evlit->firstChild();
    if (isTrue ||
        (isIdentifier(nc) && ((Identifier *) nc)->isTrue()) ||
        (isNot(nc) &&
         isIdentifier(nc->firstChild()) &&
         ((Identifier *) nc->firstChild())->isFalse())) {
      // add no clause
      TRACE(builderltlModule, {
          std::cout << "BuilderLtl: Eventuality clause is true"
                    << std::endl;
        });
    } else if ((isIdentifier(nc) && ((Identifier *) nc)->isFalse()) ||
               (isNot(nc) &&
                isIdentifier(nc->firstChild()) &&
                ((Identifier *) nc->firstChild())->isTrue())) {
      if (ll.empty()) {
        // add empty clause
        Adaptors::IClause ic =
          Adaptors::Adaptor::make_iclause(PropositionalProver::LiteralList());
        (void) myClausehSet.insert(UnsatCore::ClauseHandle(ic,
                                                     UnsatCore::mainPartition));
        translate_to_snf_msftoch_clause(n_clause,
                          UnsatCore::ClauseHandle(ic, UnsatCore::mainPartition),
                                        false,
                                        false);
        TRACE(builderltlModule, {
            std::cout << "BuilderLtl: Eventuality clause is empty"
                      << std::endl;
          });
      } else {
        // add "degenerated" universal clause
        Adaptors::UClause uc = Adaptors::Adaptor::make_uclause(ll);
        (void) myClausehSet.insert(UnsatCore::ClauseHandle(uc,
                                                     UnsatCore::mainPartition));
        translate_to_snf_msftoch_clause(n_clause,
                          UnsatCore::ClauseHandle(uc, UnsatCore::mainPartition),
                                        false,
                                        false);
        TRACE(builderltlModule, {
            std::cout << "BuilderLtl: Eventuality clause degenerates to "
                         "universal clause "
                      << std::endl;
            uc.getInternalRep()->dump(std::cout);
            std::cout << std::endl;
          });
      }
    } else {
      Assert(isIdentifier(nc) || (isNot(nc) && isIdentifier(nc->firstChild())),
             "Not an identifier or a negated identifier.");
      if (isIdentifier(nc)) {
        PropositionalProver::Proposition p =
          PropositionalProver::Proposition(((Identifier *) nc)->getIdentifier());
        PropositionalProver::Literal l = PropositionalProver::Literal(p, true);
        TRIMs::EClause ec = TRIMs::TRIM<Adaptors::Adaptor>::make_eclause(ll, l);
        (void) myClausehSet.insert(UnsatCore::ClauseHandle(ec,
                                                     UnsatCore::mainPartition));
        translate_to_snf_msftoch_clause(n_clause,
                          UnsatCore::ClauseHandle(ec, UnsatCore::mainPartition),
                                        false,
                                        false);
        TRACE(builderltlModule, {
            std::cout << "BuilderLtl: Eventuality clause is ";
            ec.dump(std::cout);
          });
      } else {
        PropositionalProver::Proposition p = PropositionalProver::Proposition
          (((Identifier *) nc->firstChild())->getIdentifier());
        PropositionalProver::Literal l = PropositionalProver::Literal(p, false);
        TRIMs::EClause ec = TRIMs::TRIM<Adaptors::Adaptor>::make_eclause(ll, l);
        (void) myClausehSet.insert(UnsatCore::ClauseHandle(ec,
                                                     UnsatCore::mainPartition));
        translate_to_snf_msftoch_clause(n_clause,
                          UnsatCore::ClauseHandle(ec, UnsatCore::mainPartition),
                                        false,
                                        false);
        TRACE(builderltlModule, {
            std::cout << "BuilderLtl: Eventuality clause is ";
            ec.dump(std::cout);
            std::cout << std::endl;
          });
      }
    }
  } // SnfTranslator::make_and_add_eclause_aux

  void
  SnfTranslator::collect_literals_clause(TreeNode * n,
                                         bool isPositivePolarity,
                                         bool isUnderNext,
                                         PropositionalProver::Attribute a,
                                         PropositionalProver::LiteralList & ll,
                                     PropositionalProver::LiteralList & ll_next,
                                         bool & isTrue)
  {
    Assert(!isUnderNext || a == PropositionalProver::step_now_attr,
           "isUnderNext is true while a is not step_now_attr.");

    if (isTrue) {
      return;
    }

    if (isIdentifier(n)) {
      if ((((Identifier *) n)->isTrue() && isPositivePolarity) ||
          (((Identifier *) n)->isFalse() && !isPositivePolarity)) {
        isTrue = true;
        return;
      } else if ((((Identifier *) n)->isFalse() && isPositivePolarity) ||
                 (((Identifier *) n)->isTrue() && !isPositivePolarity)) {
        return;
      } else {
        PropositionalProver::Proposition p =
          PropositionalProver::Proposition(((Identifier *) n)->getIdentifier());
        
        PropositionalProver::Attribute real_a = a;
        if (a == PropositionalProver::step_now_attr && isUnderNext) {
          real_a = PropositionalProver::step_next_attr;
        }
          
        PropositionalProver::Literal l =
          PropositionalProver::Literal(p, isPositivePolarity, real_a);

        if (isUnderNext) {
          ll_next.push_back(l);
        } else {
          ll.push_back(l);
        }
      }
    } else if (isNot(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      collect_literals_clause(n->firstChild(),
                              !isPositivePolarity,
                              isUnderNext,
                              a,
                              ll,
                              ll_next,
                              isTrue);
    } else if (isAnd(n) || isOr(n)) {
      Assert(n->childrenCount() > 0, "Unexpected number of children.");
      for (TreeNodeList::const_iterator it = n->children().begin();
           it != n->children().end();
           ++it)
        {
          collect_literals_clause(*it,
                                  isPositivePolarity,
                                  isUnderNext,
                                  a,
                                  ll,
                                  ll_next,
                                  isTrue);
        }
    } else if (isImplication(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      collect_literals_clause(n->firstChild(),
                              !isPositivePolarity,
                              isUnderNext,
                              a,
                              ll,
                              ll_next,
                              isTrue);
      collect_literals_clause(n->secondChild(),
                              isPositivePolarity,
                              isUnderNext,
                              a,
                              ll,
                              ll_next,
                              isTrue);
    } else if (isNext(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      Assert(!isUnderNext, "Nested next.");
      collect_literals_clause(n->firstChild(),
                              isPositivePolarity,
                              true,
                              a,
                              ll,
                              ll_next,
                              isTrue);
    } else if (isAlways(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      collect_literals_clause(n->firstChild(),
                              isPositivePolarity,
                              isUnderNext,
                              a,
                              ll,
                              ll_next,
                              isTrue);
    } else {
      Assert(false, "Unexpected operator.");
    }
  } // SnfTranslator::collect_literals_clause

  void
  SnfTranslator::translate_to_snf_msftoch_clause(TreeNode * n,
                                                 UnsatCore::ClauseHandle ch,
                                                 bool isUnderNext,
                                                 bool isUnderSometime)
  {
    Assert(!isUnderNext || !isUnderSometime,
           "isUnderNext and isUnderSometime are true.");

    if (isUnderNext) {
      translate_to_snf_add_to_msftoch_next(n, ch);
    } else if (isUnderSometime) {
      translate_to_snf_add_to_msftoch_sometime(n, ch);
    } else {
      translate_to_snf_add_to_msftoch_now(n, ch);
    }
      
    if (isIdentifier(n)) {
      Assert(n->childrenCount() == 0, "Unexpected number of children.");
      // nop
    } else if (isNot(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      translate_to_snf_msftoch_clause(n->firstChild(),
                                      ch,
                                      isUnderNext,
                                      isUnderSometime);
    } else if (isAnd(n) || isOr(n)) {
      Assert(n->childrenCount() > 0, "Unexpected number of children.");
      for (TreeNodeList::const_iterator it = n->children().begin();
           it != n->children().end();
           ++it)
        {
          translate_to_snf_msftoch_clause(*it,
                                          ch,
                                          isUnderNext,
                                          isUnderSometime);
        }
    } else if (isImplication(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      translate_to_snf_msftoch_clause(n->firstChild(),
                                      ch,
                                      isUnderNext,
                                      isUnderSometime);
      translate_to_snf_msftoch_clause(n->secondChild(),
                                      ch,
                                      isUnderNext,
                                      isUnderSometime);
    } else if (isNext(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      translate_to_snf_msftoch_clause(n->firstChild(),
                                      ch,
                                      true,
                                      isUnderSometime);
    } else if (isAlways(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      Assert(!isUnderNext && !isUnderSometime,
             "Unexpected always inside next or sometime.");
      translate_to_snf_msftoch_clause(n->firstChild(),
                                      ch,
                                      isUnderNext,
                                      isUnderSometime);
    } else if (isSometime(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      translate_to_snf_msftoch_clause(n->firstChild(),
                                      ch,
                                      isUnderNext,
                                      true);
    } else {
      Assert(false, "Unexpected operator.");
    }
  } // SnfTranslator::translate_to_snf_msftoch_clause

   /*
    For-    Pol-    Change   SNF
    mula    arity   Pol.

    True    +       -        id = true
    True    -       -        id = true
    False   +       -        id = false
    False   -       -        id = false
    a       +       -        id = a
    a       -       -        id = a
    ! f     +       y        (NVV -> ! NVVf)
    ! f     -       y        (! NVV -> NVVf)
    f & g   +       nn       (NVV -> NVVf), (NVV -> NVVg)
    f & g   -       nn       (! NVV -> ! NVVf | ! NVVg)
    f | g   +       nn       (NVV -> NVVf | NVVg)
    f | g   -       nn       (! NVV -> ! NVVf), (! NVV -> ! NVVg)
    f -> g  +       yn       (NVV -> ! NVVf | NVVg)
    f -> g  -       yn       (! NVV -> NVVf), (! NVV -> ! NVVg)
    X f     +       n        (NVV -> X NVVf)
    X f     -       n        (! NVV -> X ! NVVf)
    G f     +       n        (NVV -> X NVV), (NVV -> NVVf)
    G f     -       n        (! NVV -> F ! NVVf)
    F f     +       n        (NVV -> F NVVf)
    F f     -       n        (! NVV -> X ! NVV), (! NVV -> ! NVVf)
    f U g   +       nn       (NVV -> NVVg | NVVf), (NVV -> NVVg | X NVV), (NVV -> F NVVg)
    f U g   -       nn       (! NVV -> ! NVVg), (! NVV -> ! NVVf | X ! NVV)
    f W g   +       nn       (NVV -> NVVg | NVVf), (NVV -> NVVg | X NVV)
    f W g   -       nn       (! NVV -> ! NVVg), (!NVV -> ! NVVf | X ! NVV), (! NVV -> F ! NVVf)
  */
  Identifier
  SnfTranslator::translate_to_snf_rec(TreeNode * n,
                                      bool isPositivePolarity,
                                      bool isInitial)
  {
    Identifier result = get_fresh_identifier();
    UnsatCore::ClauseHandle ch;

    std::map<TreeNode *,
      std::pair<Identifier,
      TreeNode *>, less_tree_node> * formulaToSnfCache;
    if (isPositivePolarity && isInitial) {
      formulaToSnfCache = &myFormulaToSnfCachePosInit;
    } else if (!isPositivePolarity && isInitial) {
      formulaToSnfCache = &myFormulaToSnfCacheNegInit;
    } else if (isPositivePolarity && !isInitial) {
      formulaToSnfCache = &myFormulaToSnfCachePosNoInit;
    } else {
      formulaToSnfCache = &myFormulaToSnfCacheNegNoInit;
    }

    // Check cache
    TreeNode * k = n;
    if (!myOptions.isNoSharing() &&
        formulaToSnfCache->find(k) != formulaToSnfCache->end()) {
      myNoFormulaToSnfCacheHits++;
      result = (*formulaToSnfCache)[k].first;
      translate_to_snf_copy_msftoch(n, (*formulaToSnfCache)[k].second);
      return result;
    }
    myNoFormulaToSnfCacheMisses++;

    PropositionalProver::LiteralList ll_res, ll_res_next;

    PropositionalProver::Proposition p_res =
      PropositionalProver::Proposition(result.getIdentifier());
    Identifier id_aux = get_fresh_identifier();

    bool skip = false;
    bool empty = false;

    if (isIdentifier(n) && ((Identifier *) n)->isTrue()) {
      Assert(n->childrenCount() == 0, "Unexpected number of children.");
      result = Identifier(true);
    } else if (isIdentifier(n) && ((Identifier *) n)->isFalse()) {
      Assert(n->childrenCount() == 0, "Unexpected number of children.");
      result = Identifier(false);
    } else if (isIdentifier(n)) {
      Assert(n->childrenCount() == 0, "Unexpected number of children.");
      result = Identifier(((Identifier *) n)->getIdentifier(), TreeNodeList());
    } else if (isNot(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      if (isPositivePolarity) {
        ll_res = PropositionalProver::LiteralList();
        skip = false;
        translate_to_snf_rec_pbl_prop(p_res,
                                      false,
                                      PropositionalProver::universal_attr,
                                      ll_res);
        Identifier id_1 = translate_to_snf_rec(n->firstChild(),
                                               !isPositivePolarity,
                                               isInitial);
        translate_to_snf_rec_pbl_id(id_1,
                                    false,
                                    PropositionalProver::universal_attr,
                                    ll_res,
                                    skip);
        if (!skip) {
          ch = translate_to_snf_rec_make_uclause(ll_res,
                                                 myClausehSet,
                                                 isInitial);
          if (myOptions.isExtractUnsatCore()) {
            translate_to_snf_add_to_msftoch_now(n->firstChild(), ch);
          }
        }
      } else {
        ll_res = PropositionalProver::LiteralList();
        skip = false;
        translate_to_snf_rec_pbl_prop(p_res,
                                      true,
                                      PropositionalProver::universal_attr,
                                      ll_res);
        Identifier id_1 = translate_to_snf_rec(n->firstChild(),
                                               !isPositivePolarity,
                                               isInitial);
        translate_to_snf_rec_pbl_id(id_1,
                                    true,
                                    PropositionalProver::universal_attr,
                                    ll_res,
                                    skip);
        if (!skip) {
          ch = translate_to_snf_rec_make_uclause(ll_res,
                                                 myClausehSet,
                                                 isInitial);
          if (myOptions.isExtractUnsatCore()) {
            translate_to_snf_add_to_msftoch_now(n->firstChild(), ch);
          }
        }
      }
    } else if (isAnd(n)) {
      Assert(n->childrenCount() > 0, "Unexpected number of children.");
      if (isPositivePolarity) {
        PropositionalProver::Literal l_res =
          PropositionalProver::Literal(p_res,
                                       false,
                                       PropositionalProver::universal_attr);

        for (TreeNodeList::const_iterator it = n->children().begin();
             it != n->children().end();
             ++it)
        {
          ll_res = PropositionalProver::LiteralList();
          skip = false;
          ll_res.push_back(l_res);

          Identifier id_1 = translate_to_snf_rec(*it,
                                                 isPositivePolarity,
                                                 isInitial);
          translate_to_snf_rec_pbl_id(id_1,
                                      true,
                                      PropositionalProver::universal_attr,
                                      ll_res,
                                      skip);
          if (!skip) {
            ch = translate_to_snf_rec_make_uclause(ll_res,
                                                   myClausehSet,
                                                   isInitial);
            if (myOptions.isExtractUnsatCore()) {
              translate_to_snf_add_to_msftoch_now(*it, ch);
            }
          }
        }
      } else {
        ll_res = PropositionalProver::LiteralList();
        skip = false;
        translate_to_snf_rec_pbl_prop(p_res,
                                      true,
                                      PropositionalProver::universal_attr,
                                      ll_res);

        for (TreeNodeList::const_iterator it = n->children().begin();
             it != n->children().end();
             ++it)
        {
          Identifier id_1 = translate_to_snf_rec(*it,
                                                 isPositivePolarity,
                                                 isInitial);
          translate_to_snf_rec_pbl_id(id_1,
                                      false,
                                      PropositionalProver::universal_attr,
                                      ll_res,
                                      skip);
        }

        if (!skip) {
          ch = translate_to_snf_rec_make_uclause(ll_res,
                                                 myClausehSet,
                                                 isInitial);
          if (myOptions.isExtractUnsatCore()) {
            for (TreeNodeList::const_iterator it = n->children().begin();
                 it != n->children().end();
                 ++it)
              {
                translate_to_snf_add_to_msftoch_now(*it, ch);
              }
          }
        }
      }
    } else if (isOr(n)) {
      Assert(n->childrenCount() > 0, "Unexpected number of children.");
      if (isPositivePolarity) {
        ll_res = PropositionalProver::LiteralList();
        skip = false;
        translate_to_snf_rec_pbl_prop(p_res,
                                      false,
                                      PropositionalProver::universal_attr,
                                      ll_res);

        for (TreeNodeList::const_iterator it = n->children().begin();
             it != n->children().end();
             ++it)
        {
          Identifier id_1 = translate_to_snf_rec(*it,
                                                 isPositivePolarity,
                                                 isInitial);
          translate_to_snf_rec_pbl_id(id_1,
                                      true,
                                      PropositionalProver::universal_attr,
                                      ll_res,
                                      skip);
        }

        if (!skip) {
          ch = translate_to_snf_rec_make_uclause(ll_res,
                                                 myClausehSet,
                                                 isInitial);
          if (myOptions.isExtractUnsatCore()) {
            for (TreeNodeList::const_iterator it = n->children().begin();
                 it != n->children().end();
                 ++it)
              {
                translate_to_snf_add_to_msftoch_now(*it, ch);
              }
          }
        }
      } else {
        PropositionalProver::Literal l_res =
          PropositionalProver::Literal(p_res,
                                       true,
                                       PropositionalProver::universal_attr);

        for (TreeNodeList::const_iterator it = n->children().begin();
             it != n->children().end();
             ++it)
        {
          ll_res = PropositionalProver::LiteralList();
          skip = false;
          ll_res.push_back(l_res);

          Identifier id_1 = translate_to_snf_rec(*it,
                                                 isPositivePolarity,
                                                 isInitial);
          translate_to_snf_rec_pbl_id(id_1,
                                      false,
                                      PropositionalProver::universal_attr,
                                      ll_res,
                                      skip);
          if (!skip) {
            ch = translate_to_snf_rec_make_uclause(ll_res,
                                                   myClausehSet,
                                                   isInitial);
            if (myOptions.isExtractUnsatCore()) {
              translate_to_snf_add_to_msftoch_now(*it, ch);
            }
          }
        }
      }
    } else if (isImplication(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      if (isPositivePolarity) {
        ll_res = PropositionalProver::LiteralList();
        skip = false;
        translate_to_snf_rec_pbl_prop(p_res,
                                      false,
                                      PropositionalProver::universal_attr,
                                      ll_res);
        Identifier id_1 = translate_to_snf_rec(n->firstChild(),
                                               !isPositivePolarity,
                                               isInitial);
        translate_to_snf_rec_pbl_id(id_1,
                                    false,
                                    PropositionalProver::universal_attr,
                                    ll_res,
                                    skip);
        Identifier id_2 = translate_to_snf_rec(n->secondChild(),
                                               isPositivePolarity,
                                               isInitial);
        translate_to_snf_rec_pbl_id(id_2,
                                    true,
                                    PropositionalProver::universal_attr,
                                    ll_res,
                                    skip);
        if (!skip) {
          ch = translate_to_snf_rec_make_uclause(ll_res,
                                                 myClausehSet,
                                                 isInitial);
          if (myOptions.isExtractUnsatCore()) {
            translate_to_snf_add_to_msftoch_now(n->firstChild(), ch);
            translate_to_snf_add_to_msftoch_now(n->secondChild(), ch);
          }
        }
      } else {
        ll_res = PropositionalProver::LiteralList();
        skip = false;
        translate_to_snf_rec_pbl_prop(p_res,
                                      true,
                                      PropositionalProver::universal_attr,
                                      ll_res);
        Identifier id_1 = translate_to_snf_rec(n->firstChild(),
                                               !isPositivePolarity,
                                               isInitial);
        translate_to_snf_rec_pbl_id(id_1,
                                    true,
                                    PropositionalProver::universal_attr,
                                    ll_res,
                                    skip);
        if (!skip) {
          ch = translate_to_snf_rec_make_uclause(ll_res,
                                                 myClausehSet,
                                                 isInitial);
          if (myOptions.isExtractUnsatCore()) {
            translate_to_snf_add_to_msftoch_now(n->firstChild(), ch);
          }
        }

        ll_res.clear();
        skip = false;
        translate_to_snf_rec_pbl_prop(p_res,
                                      true,
                                      PropositionalProver::universal_attr,
                                      ll_res);
        Identifier id_2 = translate_to_snf_rec(n->secondChild(),
                                               isPositivePolarity,
                                               isInitial);
        translate_to_snf_rec_pbl_id(id_2,
                                    false,
                                    PropositionalProver::universal_attr,
                                    ll_res,
                                    skip);
        if (!skip) {
          ch = translate_to_snf_rec_make_uclause(ll_res,
                                                 myClausehSet,
                                                 isInitial);
          if (myOptions.isExtractUnsatCore()) {
            translate_to_snf_add_to_msftoch_now(n->secondChild(), ch);
          }
        }
      }
    } else if (isNext(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      if (isPositivePolarity) {
        ll_res = PropositionalProver::LiteralList();
        ll_res_next = PropositionalProver::LiteralList();
        skip = false;
        translate_to_snf_rec_pbl_prop(p_res,
                                      false,
                                      PropositionalProver::step_now_attr,
                                      ll_res);
        Identifier id_1 = translate_to_snf_rec(n->firstChild(),
                                               isPositivePolarity,
                                               false);
        translate_to_snf_rec_pbl_id(id_1,
                                    true,
                                    PropositionalProver::step_next_attr,
                                    ll_res_next,
                                    skip);
        if (!skip) {
          ch = translate_to_snf_rec_make_sclause(ll_res,
                                                 ll_res_next,
                                                 myClausehSet);
          if (myOptions.isExtractUnsatCore()) {
            translate_to_snf_add_to_msftoch_next(n->firstChild(), ch);
          }
        }
      } else {
        ll_res = PropositionalProver::LiteralList();
        ll_res_next = PropositionalProver::LiteralList();
        skip = false;
        translate_to_snf_rec_pbl_prop(p_res,
                                      true,
                                      PropositionalProver::step_now_attr,
                                      ll_res);
        Identifier id_1 = translate_to_snf_rec(n->firstChild(),
                                               isPositivePolarity,
                                               false);
        translate_to_snf_rec_pbl_id(id_1,
                                    false,
                                    PropositionalProver::step_next_attr,
                                    ll_res_next,
                                    skip);
        if (!skip) {
          ch = translate_to_snf_rec_make_sclause(ll_res,
                                                 ll_res_next,
                                                 myClausehSet);
          if (myOptions.isExtractUnsatCore()) {
            translate_to_snf_add_to_msftoch_next(n->firstChild(), ch);
          }
        }
      }
    } else if (isAlways(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      if (isPositivePolarity) {
        ll_res = PropositionalProver::LiteralList();
        ll_res_next = PropositionalProver::LiteralList();
        translate_to_snf_rec_pbl_prop(p_res,
                                      false,
                                      PropositionalProver::step_now_attr,
                                      ll_res);
        translate_to_snf_rec_pbl_prop(p_res,
                                      true,
                                      PropositionalProver::step_next_attr,
                                      ll_res_next);
        translate_to_snf_rec_make_sclause(ll_res,
                                          ll_res_next,
                                          myClausehSet);

        ll_res.clear();
        skip = false;
        translate_to_snf_rec_pbl_prop(p_res,
                                      false,
                                      PropositionalProver::universal_attr,
                                      ll_res);
        Identifier id_1 = translate_to_snf_rec(n->firstChild(),
                                               isPositivePolarity,
                                               false);
        translate_to_snf_rec_pbl_id(id_1,
                                    true,
                                    PropositionalProver::universal_attr,
                                    ll_res,
                                    skip);
        if (!skip) {
          ch = translate_to_snf_rec_make_uclause(ll_res,
                                                 myClausehSet,
                                                 false);
          if (myOptions.isExtractUnsatCore()) {
            translate_to_snf_add_to_msftoch_now(n->firstChild(), ch);
          }
        }
      } else {
        ll_res = PropositionalProver::LiteralList();
        skip = false;
        empty = false;
        translate_to_snf_rec_pbl_prop(p_res,
                                      true,
                                      PropositionalProver::universal_attr,
                                      ll_res);
        Identifier id_1 = translate_to_snf_rec(n->firstChild(),
                                               isPositivePolarity,
                                               false);
        PropositionalProver::Literal l_1 =
          translate_to_snf_rec_l_id(id_1, false, empty, skip);
        if (empty) {
          // add universal clause w/ only ll_res
          ch = translate_to_snf_rec_make_uclause(ll_res,
                                                 myClausehSet,
                                                 false);
          // still need to add false eventuality literal under F operator
          if (myOptions.isExtractUnsatCore()) {
            translate_to_snf_add_to_msftoch_sometime(n->firstChild(), ch);
          }
        } else if (!skip) {
          ch = translate_to_snf_rec_make_eclause_cached(ll_res,
                                                        l_1,
                                                        myClausehSet);
          if (myOptions.isExtractUnsatCore()) {
            translate_to_snf_add_to_msftoch_sometime(n->firstChild(), ch);
          }
        }
      }
    } else if (isSometime(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      if (isPositivePolarity) {
        ll_res = PropositionalProver::LiteralList();
        skip = false;
        empty = false;
        translate_to_snf_rec_pbl_prop(p_res,
                                      false,
                                      PropositionalProver::universal_attr,
                                      ll_res);
        Identifier id_1 = translate_to_snf_rec(n->firstChild(),
                                               isPositivePolarity,
                                               false);
        PropositionalProver::Literal l_1 =
          translate_to_snf_rec_l_id(id_1, true, empty, skip);
        if (empty) {
          // add universal clause w/ only ll_res
          ch = translate_to_snf_rec_make_uclause(ll_res,
                                                 myClausehSet,
                                                 false);
          // still need to add false eventuality literal under F operator
          if (myOptions.isExtractUnsatCore()) {
            translate_to_snf_add_to_msftoch_sometime(n->firstChild(), ch);
          }
        } else if (!skip) {
          ch = translate_to_snf_rec_make_eclause_cached(ll_res,
                                                        l_1,
                                                        myClausehSet);
          if (myOptions.isExtractUnsatCore()) {
            translate_to_snf_add_to_msftoch_sometime(n->firstChild(), ch);
          }
        }
      } else {
        ll_res = PropositionalProver::LiteralList();
        ll_res_next = PropositionalProver::LiteralList();
        translate_to_snf_rec_pbl_prop(p_res,
                                      true,
                                      PropositionalProver::step_now_attr,
                                      ll_res);
        translate_to_snf_rec_pbl_prop(p_res,
                                      false,
                                      PropositionalProver::step_next_attr,
                                      ll_res_next);
        translate_to_snf_rec_make_sclause(ll_res,
                                          ll_res_next,
                                          myClausehSet);
        ll_res.clear();
        skip = false;
        translate_to_snf_rec_pbl_prop(p_res,
                                      true,
                                      PropositionalProver::universal_attr,
                                      ll_res);
        Identifier id_1 = translate_to_snf_rec(n->firstChild(),
                                               isPositivePolarity,
                                               false);
        translate_to_snf_rec_pbl_id(id_1,
                                    false,
                                    PropositionalProver::universal_attr,
                                    ll_res,
                                    skip);
        if (!skip) {
          ch = translate_to_snf_rec_make_uclause(ll_res,
                                                 myClausehSet,
                                                 false);
          if (myOptions.isExtractUnsatCore()) {
            translate_to_snf_add_to_msftoch_now(n->firstChild(), ch);
          }
        }
      }
    } else if (isUntil(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      if (isPositivePolarity) {
        ll_res = PropositionalProver::LiteralList();
        skip = false;
        translate_to_snf_rec_pbl_prop(p_res,
                                      false,
                                      PropositionalProver::universal_attr,
                                      ll_res);
        Identifier id_2 = translate_to_snf_rec(n->secondChild(),
                                               isPositivePolarity,
                                               false);
        translate_to_snf_rec_pbl_id(id_2,
                                    true,
                                    PropositionalProver::universal_attr,
                                    ll_res,
                                    skip);
        Identifier id_1 = translate_to_snf_rec(n->firstChild(),
                                               isPositivePolarity,
                                               false);
        translate_to_snf_rec_pbl_id(id_1,
                                    true,
                                    PropositionalProver::universal_attr,
                                    ll_res,
                                    skip);
        if (!skip) {
          ch = translate_to_snf_rec_make_uclause(ll_res,
                                                 myClausehSet,
                                                 false);
          if (myOptions.isExtractUnsatCore()) {
            translate_to_snf_add_to_msftoch_now(n->firstChild(), ch);
            translate_to_snf_add_to_msftoch_now(n->secondChild(), ch);
          }
        }

        ll_res.clear();
        ll_res_next = PropositionalProver::LiteralList();
        skip = false;
        translate_to_snf_rec_pbl_prop(p_res,
                                      false,
                                      PropositionalProver::step_now_attr,
                                      ll_res);
        translate_to_snf_rec_pbl_id(id_2,
                                    true,
                                    PropositionalProver::step_now_attr,
                                    ll_res,
                                    skip);
        translate_to_snf_rec_pbl_prop(p_res,
                                      true,
                                      PropositionalProver::step_next_attr,
                                      ll_res_next);
        if (!skip) {
          ch = translate_to_snf_rec_make_sclause(ll_res,
                                                 ll_res_next,
                                                 myClausehSet);
          if (myOptions.isExtractUnsatCore()) {
            translate_to_snf_add_to_msftoch_now(n->secondChild(), ch);
          }
        }

        ll_res.clear();
        skip = false;
        empty = false;
        translate_to_snf_rec_pbl_prop(p_res,
                                      false,
                                      PropositionalProver::universal_attr,
                                      ll_res);
        PropositionalProver::Literal l_2 =
          translate_to_snf_rec_l_id(id_2, true, empty, skip);
        if (empty) {
          // add universal clause w/ only ll_res
          ch = translate_to_snf_rec_make_uclause(ll_res,
                                                 myClausehSet,
                                                 false);
          // still need to add false eventuality literal under F operator
          if (myOptions.isExtractUnsatCore()) {
            translate_to_snf_add_to_msftoch_sometime(n->secondChild(), ch);
          }
        } else if (!skip) {
          ch = translate_to_snf_rec_make_eclause_cached(ll_res,
                                                        l_2,
                                                        myClausehSet);
          if (myOptions.isExtractUnsatCore()) {
            translate_to_snf_add_to_msftoch_sometime(n->secondChild(), ch);
          }
        }
      } else {
        ll_res = PropositionalProver::LiteralList();
        skip = false;
        translate_to_snf_rec_pbl_prop(p_res,
                                      true,
                                      PropositionalProver::universal_attr,
                                      ll_res);
        Identifier id_2 = translate_to_snf_rec(n->secondChild(),
                                               isPositivePolarity,
                                               false);
        translate_to_snf_rec_pbl_id(id_2,
                                    false,
                                    PropositionalProver::universal_attr,
                                    ll_res,
                                    skip);
        if (!skip) {
          ch = translate_to_snf_rec_make_uclause(ll_res,
                                                 myClausehSet,
                                                 false);
          if (myOptions.isExtractUnsatCore()) {
            translate_to_snf_add_to_msftoch_now(n->secondChild(), ch);
          }
        }

        ll_res.clear();
        ll_res_next = PropositionalProver::LiteralList();
        skip = false;
        translate_to_snf_rec_pbl_prop(p_res,
                                      true,
                                      PropositionalProver::step_now_attr,
                                      ll_res);
        Identifier id_1 = translate_to_snf_rec(n->firstChild(),
                                               isPositivePolarity,
                                               false);
        translate_to_snf_rec_pbl_id(id_1,
                                    false,
                                    PropositionalProver::step_now_attr,
                                    ll_res,
                                    skip);
        translate_to_snf_rec_pbl_prop(p_res,
                                      false,
                                      PropositionalProver::step_next_attr,
                                      ll_res_next);
        if (!skip) {
          ch = translate_to_snf_rec_make_sclause(ll_res,
                                                 ll_res_next,
                                                 myClausehSet);
          if (myOptions.isExtractUnsatCore()) {
            translate_to_snf_add_to_msftoch_now(n->firstChild(), ch);
          }
        }
      }
    } else if (isUnless(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      if (isPositivePolarity) {
        ll_res = PropositionalProver::LiteralList();
        skip = false;
        translate_to_snf_rec_pbl_prop(p_res,
                                      false,
                                      PropositionalProver::universal_attr,
                                      ll_res);
        Identifier id_2 = translate_to_snf_rec(n->secondChild(),
                                               isPositivePolarity,
                                               false);
        translate_to_snf_rec_pbl_id(id_2,
                                    true,
                                    PropositionalProver::universal_attr,
                                    ll_res,
                                    skip);
        Identifier id_1 = translate_to_snf_rec(n->firstChild(),
                                               isPositivePolarity,
                                               false);
        translate_to_snf_rec_pbl_id(id_1,
                                    true,
                                    PropositionalProver::universal_attr,
                                    ll_res,
                                    skip);
        if (!skip) {
          ch = translate_to_snf_rec_make_uclause(ll_res,
                                                 myClausehSet,
                                                 false);
          if (myOptions.isExtractUnsatCore()) {
            translate_to_snf_add_to_msftoch_now(n->firstChild(), ch);
            translate_to_snf_add_to_msftoch_now(n->secondChild(), ch);
          }
        }

        ll_res.clear();
        ll_res_next = PropositionalProver::LiteralList();
        skip = false;
        translate_to_snf_rec_pbl_prop(p_res,
                                      false,
                                      PropositionalProver::step_now_attr,
                                      ll_res);
        translate_to_snf_rec_pbl_id(id_2,
                                    true,
                                    PropositionalProver::step_now_attr,
                                    ll_res,
                                    skip);
        translate_to_snf_rec_pbl_prop(p_res,
                                      true,
                                      PropositionalProver::step_next_attr,
                                      ll_res_next);
        if (!skip) {
          ch = translate_to_snf_rec_make_sclause(ll_res,
                                                 ll_res_next,
                                                 myClausehSet);
          if (myOptions.isExtractUnsatCore()) {
            translate_to_snf_add_to_msftoch_now(n->secondChild(), ch);
          }
        }
      } else {
        ll_res = PropositionalProver::LiteralList();
        skip = false;
        translate_to_snf_rec_pbl_prop(p_res,
                                      true,
                                      PropositionalProver::universal_attr,
                                      ll_res);
        Identifier id_2 = translate_to_snf_rec(n->secondChild(),
                                               isPositivePolarity,
                                               false);
        translate_to_snf_rec_pbl_id(id_2,
                                    false,
                                    PropositionalProver::universal_attr,
                                    ll_res,
                                    skip);
        if (!skip) {
          ch = translate_to_snf_rec_make_uclause(ll_res,
                                                 myClausehSet,
                                                 false);
          if (myOptions.isExtractUnsatCore()) {
            translate_to_snf_add_to_msftoch_now(n->secondChild(), ch);
          }
        }

        ll_res.clear();
        ll_res_next = PropositionalProver::LiteralList();
        skip = false;
        translate_to_snf_rec_pbl_prop(p_res,
                                      true,
                                      PropositionalProver::step_now_attr,
                                      ll_res);
        Identifier id_1 = translate_to_snf_rec(n->firstChild(),
                                               isPositivePolarity,
                                               false);
        translate_to_snf_rec_pbl_id(id_1,
                                    false,
                                    PropositionalProver::step_now_attr,
                                    ll_res,
                                    skip);
        translate_to_snf_rec_pbl_prop(p_res,
                                      false,
                                      PropositionalProver::step_next_attr,
                                      ll_res_next);
        if (!skip) {
          ch = translate_to_snf_rec_make_sclause(ll_res,
                                                 ll_res_next,
                                                 myClausehSet);
          if (myOptions.isExtractUnsatCore()) {
            translate_to_snf_add_to_msftoch_now(n->firstChild(), ch);
          }
        }

        ll_res.clear();
        skip = false;
        empty = false;
        translate_to_snf_rec_pbl_prop(p_res,
                                      true,
                                      PropositionalProver::universal_attr,
                                      ll_res);
        PropositionalProver::Literal l_1 =
          translate_to_snf_rec_l_id(id_1, false, empty, skip);
        if (empty) {
          // add universal clause w/ only ll_res
          ch = translate_to_snf_rec_make_uclause(ll_res,
                                                 myClausehSet,
                                                 false);
          // still need to add false eventuality literal under F operator
          if (myOptions.isExtractUnsatCore()) {
            translate_to_snf_add_to_msftoch_sometime(n->firstChild(), ch);
          }
        } else if (!skip) {
          ch = translate_to_snf_rec_make_eclause_cached(ll_res,
                                                        l_1,
                                                        myClausehSet);
          if (myOptions.isExtractUnsatCore()) {
            translate_to_snf_add_to_msftoch_sometime(n->firstChild(), ch);
          }
        }
      }
    } else {
      Assert(false, "n has unexpected type.");
    }

    if (!myOptions.isNoSharing()) {
      (*formulaToSnfCache)[k] = std::make_pair(result, n);
    }

    return result;
  } // SnfTranslator::translate_to_snf_rec

  // copy map from subformulas to clausehandles for children of copy
  void
  SnfTranslator::translate_to_snf_copy_msftoch(TreeNode * copy, TreeNode * orig)
  {
    TreeNodeList::const_iterator itcopy = copy->children().begin();
    TreeNodeList::const_iterator itorig = orig->children().begin();
    for (;
         itcopy != copy->children().end();
         ++itcopy, ++itorig)
      {
        Assert(itorig != orig->children().end(), "orig != copy.");
        if (myMapSubformulaToClausehsNow.find(*itorig) !=
            myMapSubformulaToClausehsNow.end()) {
          myMapSubformulaToClausehsNow[*itcopy] =
            myMapSubformulaToClausehsNow[*itorig];
        }
        if (myMapSubformulaToClausehsNext.find(*itorig) !=
            myMapSubformulaToClausehsNext.end()) {
          myMapSubformulaToClausehsNext[*itcopy] =
            myMapSubformulaToClausehsNext[*itorig];
        }
        if (myMapSubformulaToClausehsSometime.find(*itorig) !=
            myMapSubformulaToClausehsSometime.end()) {
          myMapSubformulaToClausehsSometime[*itcopy] =
            myMapSubformulaToClausehsSometime[*itorig];
        }
        translate_to_snf_copy_msftoch(*itcopy, *itorig);
      }
    Assert(itorig == orig->children().end(), "orig != copy.");
  } // SnfTranslator::translate_to_snf_copy_msftoch

  // pbl means push back literal
  void
  SnfTranslator::translate_to_snf_rec_pbl_prop
  (PropositionalProver::Proposition const & p,
   bool pol,
   PropositionalProver::Attribute const & a,
   PropositionalProver::LiteralList & ll)
  {
    PropositionalProver::Literal l = PropositionalProver::Literal(p, pol, a);
    ll.push_back(l);
  } // SnfTranslator::translate_to_snf_rec_pbl_prop

  // pbl means push back literal
  void
  SnfTranslator::translate_to_snf_rec_pbl_id
  (Identifier const & id,
   bool pol,
   PropositionalProver::Attribute const & a,
   PropositionalProver::LiteralList & ll,
   bool & skip)
  {
    if ((id.isTrue() && pol) || (id.isFalse() && !pol)) {
      // the clause would contain True; hence, skip it
      skip = true;
    } else if ((id.isTrue() && !pol) || (id.isFalse() && pol)) {
      // the literal is false; hence, don't add it
      /* nop */
    } else {
      PropositionalProver::Proposition p =
        PropositionalProver::Proposition(id.getIdentifier());
      PropositionalProver::Literal l = PropositionalProver::Literal(p, pol, a);
      ll.push_back(l);
    }
  } // SnfTranslator::translate_to_snf_rec_pbl_id

  PropositionalProver::Literal
  SnfTranslator::translate_to_snf_rec_l_id
  (Identifier const & id,
   bool pol,
   bool & empty,
   bool & skip)
  {
    PropositionalProver::Proposition p =
      PropositionalProver::Proposition(id.getIdentifier());
    PropositionalProver::Literal res = PropositionalProver::Literal(p, pol);
    if ((id.isTrue() && pol) || (id.isFalse() && !pol)) {
      // the clause would contain True; hence, skip it
      skip = true;
    } else if ((id.isTrue() && !pol) || (id.isFalse() && pol)) {
      // the eventuality literal is false; hence, can add universal
      // clause w/o eventuality literal
      empty = true;
    }
    return res;
  } // SnfTranslator::translate_to_snf_rec_l_id

  UnsatCore::ClauseHandle
  SnfTranslator::translate_to_snf_rec_make_uclause
  (PropositionalProver::LiteralList const & ll,
   UnsatCore::ClauseHandleSet & clausehSet,
   bool isInitial)
  {
    if (isInitial) {
      Adaptors::IClause ic = Adaptors::Adaptor::make_iclause(ll);
      (void) clausehSet.insert(UnsatCore::ClauseHandle(ic,
                                                     UnsatCore::mainPartition));
      return UnsatCore::ClauseHandle(ic, UnsatCore::mainPartition);
    } else {
      Adaptors::UClause uc = Adaptors::Adaptor::make_uclause(ll);
      (void) clausehSet.insert(UnsatCore::ClauseHandle(uc,
                                                     UnsatCore::mainPartition));
      return UnsatCore::ClauseHandle(uc, UnsatCore::mainPartition);
    }
  } // SnfTranslator::translate_to_snf_rec_make_uclause

  UnsatCore::ClauseHandle
  SnfTranslator::translate_to_snf_rec_make_sclause
  (PropositionalProver::LiteralList const & ll,
   PropositionalProver::LiteralList const & ll_next,
   UnsatCore::ClauseHandleSet & clausehSet)
  {
    Adaptors::SClause sc = Adaptors::Adaptor::make_sclause(ll, ll_next);
    (void) clausehSet.insert(UnsatCore::ClauseHandle(sc,
                                                     UnsatCore::mainPartition));
    return UnsatCore::ClauseHandle(sc, UnsatCore::mainPartition);
  } // SnfTranslator::translate_to_snf_rec_make_sclause

  UnsatCore::ClauseHandle
  SnfTranslator::translate_to_snf_rec_make_eclause_cached
  (PropositionalProver::LiteralList ll,
   PropositionalProver::Literal const & l,
   UnsatCore::ClauseHandleSet & clausehSet)
  {
    UnsatCore::ClauseHandle ch;

    if (!myOptions.isNoEclauseReduction()) {
      if (myMapEventualityLiteralToLiteral.find(l) ==
          myMapEventualityLiteralToLiteral.end()) {
        Identifier evid = get_fresh_identifier();
        PropositionalProver::Proposition evp =
          PropositionalProver::Proposition(evid.getIdentifier());

        PropositionalProver::Literal evl =
          PropositionalProver::Literal(evp, true);
        myMapEventualityLiteralToLiteral[l] = evl;

        evl = PropositionalProver::Literal(evp, false);
        PropositionalProver::LiteralList ll_ev =
          PropositionalProver::LiteralList();
        ll_ev.push_back(evl);

        (void) translate_to_snf_rec_make_eclause(ll_ev,
                                                 l,
                                                 clausehSet);
      }
      PropositionalProver::Literal evl =
        myMapEventualityLiteralToLiteral[l];
      ll.push_back(evl);
      ch = translate_to_snf_rec_make_uclause(ll,
                                             clausehSet,
                                             false);
    } else {
      ch = translate_to_snf_rec_make_eclause(ll,
                                             l,
                                             clausehSet);
    }

    return ch;
  } // SnfTranslator::translate_to_snf_rec_make_eclause_cached

  UnsatCore::ClauseHandle
  SnfTranslator::translate_to_snf_rec_make_eclause
  (PropositionalProver::LiteralList const & ll,
   PropositionalProver::Literal const & l,
   UnsatCore::ClauseHandleSet & clausehSet)
  {
    TRIMs::EClause ec = TRIMs::TRIM<Adaptors::Adaptor>::make_eclause(ll, l);
    (void) clausehSet.insert(UnsatCore::ClauseHandle(ec,
                                                     UnsatCore::mainPartition));
    return UnsatCore::ClauseHandle(ec, UnsatCore::mainPartition);
  } // SnfTranslator::translate_to_snf_rec_make_eclause

  void
  SnfTranslator::translate_to_snf_add_to_msftoch_now(TreeNode * n,
                                                     UnsatCore::ClauseHandle ch)
  {
    if (myMapSubformulaToClausehsNow.find(n) ==
        myMapSubformulaToClausehsNow.end()) {
      myMapSubformulaToClausehsNow[n] = UnsatCore::ClauseHandleSet();
    }
    (void) myMapSubformulaToClausehsNow[n].insert(ch);
  } // SnfTranslator::translate_to_snf_add_to_msftoch_now

  void
  SnfTranslator::translate_to_snf_add_to_msftoch_next(TreeNode * n,
                                                     UnsatCore::ClauseHandle ch)
  {
    if (myMapSubformulaToClausehsNext.find(n) ==
        myMapSubformulaToClausehsNext.end()) {
      myMapSubformulaToClausehsNext[n] = UnsatCore::ClauseHandleSet();
    }
    (void) myMapSubformulaToClausehsNext[n].insert(ch);
  } // SnfTranslator::translate_to_snf_add_to_msftoch_next

  void
  SnfTranslator::translate_to_snf_add_to_msftoch_sometime(TreeNode * n,
                                                     UnsatCore::ClauseHandle ch)
  {
    if (myMapSubformulaToClausehsSometime.find(n) ==
        myMapSubformulaToClausehsSometime.end()) {
      myMapSubformulaToClausehsSometime[n] = UnsatCore::ClauseHandleSet();
    }
    (void) myMapSubformulaToClausehsSometime[n].insert(ch);
  } // SnfTranslator::translate_to_snf_add_to_msftoch_sometime

  Identifier
  SnfTranslator::get_fresh_identifier()
  {
    std::string fresh_var_prefix = "NVV";

    std::string id = "";
    id += fresh_var_prefix;
    id += boost::lexical_cast<std::string>(++myNvvCounter);

    Identifier result = Identifier(id);
    return result;    
  } // SnfTranslator::get_fresh_identifier
} // namespace SnfTranslator
