/**************************************************************/
/* ********************************************************** */
/* *                                                        * */
/* *  Copyright (C) 2001-2011                               * */
/* *  Boris Konev                                           * */
/* *  The University of Liverpool                           * */
/* *                                                        * */
/* *  Copyright (C) 2012-2013                               * */
/* *  Viktor Schuppan                                       * */
/* *                                                        * */
/* *  This program is free software; you can redistribute   * */
/* *  it and/or modify it under the terms of the GNU        * */
/* *  General Public License as published by the Free       * */
/* *  Software Foundation; either version 2 of the License, * */
/* *  or (at your option) any later version.                * */
/* *                                                        * */
/* *  This program is distributed in the hope that it will  * */
/* *  be useful, but WITHOUT ANY WARRANTY; without even     * */
/* *  the implied warranty of MERCHANTABILITY or FITNESS    * */
/* *  FOR A PARTICULAR PURPOSE.  See the GNU General Public * */
/* *  License for more details.                             * */
/* *                                                        * */
/* *  You should have received a copy of the GNU General    * */
/* *  Public License along with this program; if not, write * */
/* *  to the Free Software Foundation, Inc., 59 Temple      * */
/* *  Place, Suite 330, Boston, MA  02111-1307  USA         * */
/* *                                                        * */
/* *                                                        * */
/* ********************************************************** */
/**************************************************************/
#include "options.h"
#include <boost/tokenizer.hpp>
#include "config.h"
#include "modules.h"
#include "misc/tracer.h"

TRPPPOptions::TRPPPOptions(int argc, char** argv)
{
    if(cmdline_parser(argc, argv, &myArgsInfo) != 0)
    {
        reportFailure("Failed to parse the command line\n");
    }

    if(myArgsInfo.inputs_num != 1)
    {
        reportFailure("");
    }

    myFileName = myArgsInfo.inputs[0];

    if(myArgsInfo.format_input_given)
    {
        if(std::string("snf") == myArgsInfo.format_input_arg)
        {
            myInputFormat = INPUT_FORMAT_SNF;
        }
        else
        {
            if(std::string("ltl") == myArgsInfo.format_input_arg)
            {
                myInputFormat = INPUT_FORMAT_LTL;
            }
            else
            {
                reportFailure(std::string("Illegal value for the \"format-input\" option: ")+ myArgsInfo.format_input_arg);
            }
        }
    }
    else
    {
        myInputFormat = INPUT_FORMAT_SNF; // default value
    }

    myShowInput  = myArgsInfo.show_input_flag;
    myShowResult = myArgsInfo.show_result_flag;
    myQuiet      = myArgsInfo.quiet_flag;

    if(myQuiet)
    {
        if (myShowInput || myShowResult)
        {
            reportFailure("Incompatible options\n");
        }
        else
        {
            myShowInput = false;
            myShowResult = false;
        }
    }

    if(myArgsInfo.select_by_given)
    {
        if(std::string("DFS") == myArgsInfo.select_by_arg)
        {
            myDFS = true;
        }
        else 
            if(std::string("BFS") == myArgsInfo.select_by_arg)
            {
                myDFS = false;
            }
            else
            {
                reportFailure(std::string("Illegal value for the \"select-by\" option: ")+ myArgsInfo.select_by_arg);
            }
    }
    else
    {
        myDFS = true; // default value
    }

    myUsePretest = myArgsInfo.pretest_flag;

    myUseFSR = myArgsInfo.FSR_flag;

    if(myArgsInfo.order_given)
    {
        myOrderFile = myArgsInfo.order_arg;
    }
    else 
    {
        myOrderFile = "";
    }

    if(myArgsInfo.extract_uc_given)
    {
        if(std::string("none") == myArgsInfo.extract_uc_arg)
        {
            myExtractUnsatCoreMode = EXTRACT_UNSAT_CORE_MODE_NONE;
        }
        else
        {
            if(myQuiet || myUseFSR)
            {
                reportFailure("Incompatible options\n");
            }
            if(std::string("simple") == myArgsInfo.extract_uc_arg)
            {
                myExtractUnsatCoreMode = EXTRACT_UNSAT_CORE_MODE_SIMPLE;
            }
            else
            {
                if(std::string("setsoftimepoints") == myArgsInfo.extract_uc_arg)
                {
                    myExtractUnsatCoreMode = EXTRACT_UNSAT_CORE_MODE_SETSOFTIMEPOINTS;
                }
                else
                {
                    reportFailure(std::string("Illegal value for the \"extract-uc\" option: ")+ myArgsInfo.extract_uc_arg);
                }
            }
        }
    }
    else
    {
        myExtractUnsatCoreMode = EXTRACT_UNSAT_CORE_MODE_NONE; // default value
    }

    if(myArgsInfo.write_uc_given)
    {
        if (myExtractUnsatCoreMode == EXTRACT_UNSAT_CORE_MODE_NONE)
        {
            reportFailure(std::string("The \"write-uc\" option requires the \"extract-uc\" option."));
        }
        else
        {
            myExtractUnsatCoreFileName = myArgsInfo.write_uc_arg;
        }
    }
    else 
    {
        myExtractUnsatCoreFileName = ""; // default value
    }

    if(myArgsInfo.format_uc_given)
    {
        if (myExtractUnsatCoreMode == EXTRACT_UNSAT_CORE_MODE_NONE)
        {
            reportFailure(std::string("The \"format-uc\" option requires the \"extract-uc\" option."));
        } 
        else
        {
            if(std::string("internal") == myArgsInfo.format_uc_arg)
            {
                myExtractUnsatCoreFormat = EXTRACT_UNSAT_CORE_FORMAT_INTERNAL;
            }
            else
            {
                if(std::string("snf") == myArgsInfo.format_uc_arg)
                {
                    myExtractUnsatCoreFormat = EXTRACT_UNSAT_CORE_FORMAT_SNF;
                }
                else
                {
                    if(std::string("ltlc") == myArgsInfo.format_uc_arg)
                    {
                        myExtractUnsatCoreFormat = EXTRACT_UNSAT_CORE_FORMAT_LTLC;
                    }
                    else
                    {
                        if(std::string("ltl") == myArgsInfo.format_uc_arg)
                        {
                            if (myInputFormat == INPUT_FORMAT_SNF)
                            {
                                reportFailure(std::string("The \"format-uc ltl\" option requires the \"format-input ltl\" option."));
                            }
                            else
                            {
                                myExtractUnsatCoreFormat = EXTRACT_UNSAT_CORE_FORMAT_LTL;
                            }
                        }
                        else
                        {
                            reportFailure(std::string("Illegal value for the \"format-uc\" option: ")+ myArgsInfo.extract_uc_arg);
                        }
                    }
                }
            }
        }
    }
    else
    {
        myExtractUnsatCoreFormat = EXTRACT_UNSAT_CORE_FORMAT_INTERNAL; // default value
    }

    if(myArgsInfo.algorithm_uc_given)
    {
        if (myExtractUnsatCoreMode == EXTRACT_UNSAT_CORE_MODE_NONE)
        {
            reportFailure(std::string("The \"algorithm-uc\" option requires the \"extract-uc\" option."));
        } 
        else
        {
            if(std::string("proof") == myArgsInfo.algorithm_uc_arg)
            {
                myExtractUnsatCoreAlgorithm = EXTRACT_UNSAT_CORE_ALGORITHM_PROOF;
            }
            else
            {
                if(std::string("deletion") == myArgsInfo.algorithm_uc_arg)
                {
                    myExtractUnsatCoreAlgorithm = EXTRACT_UNSAT_CORE_ALGORITHM_DELETION;
                }
                else
                {
                    if(std::string("proofdeletion") == myArgsInfo.algorithm_uc_arg)
                    {
                        myExtractUnsatCoreAlgorithm = EXTRACT_UNSAT_CORE_ALGORITHM_PROOFDELETION;
                    }
                    else
                    {
                        reportFailure(std::string("Illegal value for the \"algorithm-uc\" option: ")+ myArgsInfo.algorithm_uc_arg);
                    }
                }
            }
        }
    }
    else
    {
        myExtractUnsatCoreAlgorithm = EXTRACT_UNSAT_CORE_ALGORITHM_PROOF; // default value
    }

    if(myArgsInfo.Parikh_algorithm_given)
    {
        if (myExtractUnsatCoreMode != EXTRACT_UNSAT_CORE_MODE_SETSOFTIMEPOINTS)
        {
            reportFailure(std::string("The \"Parikh-algorithm\" option requires the \"extract-uc setsoftimepoints\" option."));
        }
        else
        {
            if(std::string("Gawrychowski") == myArgsInfo.Parikh_algorithm_arg)
            {
                myExtractUnsatCoreParikhAlgorithm = EXTRACT_UNSAT_CORE_PARIKH_ALGORITHM_GAWRYCHOWSKI;
            }
            else
            {
                if(std::string("Sawa") == myArgsInfo.Parikh_algorithm_arg)
                {
                    myExtractUnsatCoreParikhAlgorithm = EXTRACT_UNSAT_CORE_PARIKH_ALGORITHM_SAWA;
                }
                else
                {
                    reportFailure(std::string("Illegal value for the \"Parikh-algorithm\" option: ")+ myArgsInfo.Parikh_algorithm_arg);
                }
            }
        }
    }
    else
    {
      myExtractUnsatCoreParikhAlgorithm = EXTRACT_UNSAT_CORE_PARIKH_ALGORITHM_GAWRYCHOWSKI; // default value
    }

    myNoNnf = myArgsInfo.no_nnf_flag;
    if (myNoNnf && myInputFormat != INPUT_FORMAT_LTL) {
      reportFailure(std::string("The \"no-nnf\" option requires the \"format-input ltl\" option."));
    }

    myNoLTLSimplification = myArgsInfo.no_ltl_simplification_flag;
    if (myNoLTLSimplification && myInputFormat != INPUT_FORMAT_LTL) {
      reportFailure(std::string("The \"no-ltl-simplification\" option requires the \"format-input ltl\" option."));
    }

    myNoDirectClauses = myArgsInfo.no_direct_clauses_flag;
    if (myNoDirectClauses && myInputFormat != INPUT_FORMAT_LTL) {
      reportFailure(std::string("The \"no-direct-clauses\" option requires the \"format-input ltl\" option."));
    }

    myNoEclauseReduction = myArgsInfo.no_eclause_reduction_flag;
    if (myNoEclauseReduction && myInputFormat != INPUT_FORMAT_LTL) {
      reportFailure(std::string("The \"no-eclause-reduction\" option requires the \"format-input ltl\" option."));
    }

    myNoSharing = myArgsInfo.no_sharing_flag;
    if (myNoSharing && myInputFormat != INPUT_FORMAT_LTL) {
      reportFailure(std::string("The \"no-sharing\" option requires the \"format-input ltl\" option."));
    }

    myNoSorting = myArgsInfo.no_sorting_flag;
    if (myNoSorting && myInputFormat != INPUT_FORMAT_LTL) {
      reportFailure(std::string("The \"no-sorting\" option requires the \"format-input ltl\" option."));
    }

    myNoSNFSimplification = myArgsInfo.no_snf_simplification_flag;

    myNoUnsatCoreSimplification = myArgsInfo.no_uc_simplification_flag;
    if (myNoUnsatCoreSimplification && myExtractUnsatCoreMode == EXTRACT_UNSAT_CORE_MODE_NONE) {
      reportFailure(std::string("The \"no-uc-simplification\" option requires the \"extract-uc\" option."));
    } 

    myNoRGNoEdgeAug2 = myArgsInfo.no_rgnoedge_aug2_flag;
    if (myNoRGNoEdgeAug2 && myExtractUnsatCoreMode == EXTRACT_UNSAT_CORE_MODE_NONE) {
      reportFailure(std::string("The \"no-rgnoedge-aug2\" option requires the \"extract-uc\" option."));
    } 

    myNoRGNoEdgeBFSLoopItInitCG = myArgsInfo.no_rgnoedge_bfsloopitinitcg_flag;
    if (myNoRGNoEdgeBFSLoopItInitCG && myExtractUnsatCoreMode == EXTRACT_UNSAT_CORE_MODE_NONE) {
      reportFailure(std::string("The \"no-rgnoedge-bfsloopitinitcg\" option requires the \"extract-uc\" option."));
    } 

    myNoRGNoEdgeBFSLoopItInitCE = myArgsInfo.no_rgnoedge_bfsloopitinitce_flag;
    if (myNoRGNoEdgeBFSLoopItInitCE && myExtractUnsatCoreMode == EXTRACT_UNSAT_CORE_MODE_NONE) {
      reportFailure(std::string("The \"no-rgnoedge-bfsloopitinitce\" option requires the \"extract-uc\" option."));
    } 

    myNoRGNoEdgeBFSLoopConclusion2E = myArgsInfo.no_rgnoedge_bfsloopconclusion2e_flag;
    if (myNoRGNoEdgeBFSLoopConclusion2E && myExtractUnsatCoreMode == EXTRACT_UNSAT_CORE_MODE_NONE) {
      reportFailure(std::string("The \"no-rgnoedge-bfsloopconclusion2e\" option requires the \"extract-uc\" option."));
    } 

    myNoPruneMainFwdReachable = myArgsInfo.no_prune_main_fwd_reachable_flag;
    if (myNoPruneMainFwdReachable && myExtractUnsatCoreMode == EXTRACT_UNSAT_CORE_MODE_NONE) {
      reportFailure(std::string("The \"no-prune-main-fwd-reachable\" option requires the \"extract-uc\" option."));
    } 

    myNoPruneMainBwdReachable = myArgsInfo.no_prune_main_bwd_reachable_flag;
    if (myNoPruneMainBwdReachable && myExtractUnsatCoreMode == EXTRACT_UNSAT_CORE_MODE_NONE) {
      reportFailure(std::string("The \"no-prune-main-bwd-reachable\" option requires the \"extract-uc\" option."));
    } 

    myNoPruneFailedLoopSearchIts = myArgsInfo.no_prune_failed_loop_search_its_flag;
    if (myNoPruneFailedLoopSearchIts && myExtractUnsatCoreMode == EXTRACT_UNSAT_CORE_MODE_NONE) {
      reportFailure(std::string("The \"no-prune-failed-loop-search-its\" option requires the \"extract-uc\" option."));
    }

    if (myNoRGNoEdgeBFSLoopItInitCG && !myNoPruneFailedLoopSearchIts) {
      myNoPruneFailedLoopSearchIts = true;
    }

    if (myArgsInfo.write_rg_before_pruning_given)
    {
        if (myExtractUnsatCoreMode == EXTRACT_UNSAT_CORE_MODE_NONE)
        {
            reportFailure(std::string("The \"write-rg-before-pruning\" option requires the \"extract-uc\" option."));
        }
        else
        {
            myExtractUnsatCoreWriteRGBeforePruningFileName = myArgsInfo.write_rg_before_pruning_arg;
        }
    }
    else 
    {
        myExtractUnsatCoreWriteRGBeforePruningFileName = ""; // default value
    }

    if (myArgsInfo.write_rg_after_pruning_given)
    {
        if (myExtractUnsatCoreMode == EXTRACT_UNSAT_CORE_MODE_NONE)
        {
            reportFailure(std::string("The \"write-rg-after-pruning\" option requires the \"extract-uc\" option."));
        }
        else
        {
            myExtractUnsatCoreWriteRGAfterPruningFileName = myArgsInfo.write_rg_after_pruning_arg;
        }
    }
    else 
    {
        myExtractUnsatCoreWriteRGAfterPruningFileName = ""; // default value
    }

    if(myArgsInfo.format_rg_vertex_label_given)
    {
        if (myExtractUnsatCoreMode == EXTRACT_UNSAT_CORE_MODE_NONE)
        {
            reportFailure(std::string("The \"format-rg-vertex-label\" option requires the \"extract-uc\" option."));
        } 
        else
        {
            if (myExtractUnsatCoreWriteRGBeforePruningFileName == std::string("") &&
                myExtractUnsatCoreWriteRGAfterPruningFileName == std::string(""))
            {
              reportFailure(std::string("The \"format-rg-vertex-label\" option requires one of the \"write-rg-before/after-pruning\" options."));
            }
            if(std::string("internal") == myArgsInfo.format_rg_vertex_label_arg)
            {
                myExtractUnsatCoreFormatRGVertexLabel = EXTRACT_UNSAT_CORE_FORMAT_RG_VERTEX_LABEL_INTERNAL;
            }
            else
            {
                if(std::string("snf") == myArgsInfo.format_rg_vertex_label_arg)
                {
                    myExtractUnsatCoreFormatRGVertexLabel = EXTRACT_UNSAT_CORE_FORMAT_RG_VERTEX_LABEL_SNF;
                }
                else
                {
                    reportFailure(std::string("Illegal value for the \"format-rg-vertex-label\" option: ")+ myArgsInfo.format_rg_vertex_label_arg);
                }
            }
        }
    }
    else
    {
        myExtractUnsatCoreFormatRGVertexLabel = EXTRACT_UNSAT_CORE_FORMAT_RG_VERTEX_LABEL_SNF; // default value
    }

#ifdef USE_TRACING
    if(myArgsInfo.trace_given)
    {
        std::string argString(myArgsInfo.trace_arg);
        boost::tokenizer<> tok(argString);

        for(boost::tokenizer<>::iterator beg=tok.begin(); beg!=tok.end();++beg)
        {
            if(*beg == "index")
            {
                setTrace(indexModule);
            }
            else
                if(*beg == "tempres")
                {
                    setTrace(tempresModule);
                }
                else
                    if(*beg == "loopsearch")
                    {
                        setTrace(loopsearchModule);
                    }
                    else
                        if(*beg == "resolution")
                        {
                            setTrace(resolutionModule);
                        }
                        else
                            if(*beg == "subsumption")
                            {
                                setTrace(subsumptionModule);
                            }
                            else
                                if(*beg == "forward")
                                {
                                    setTrace(forwardsubsumptionModule);
                                }
                                else
                                    if(*beg == "backward")
                                    {
                                        setTrace(backwardsubsumptionModule);
                                    }
                                    else 
                                        if(*beg == "pretest")
                                        {
                                            setTrace(pretestModule);
                                        }
                                        else
                                            if(*beg == "FSR")
                                            {
                                                setTrace(FSRModule);
                                            } 
                                            else 
                                                if(*beg == "unsatcore")
                                                {
                                                    setTrace(unsatcoreModule);
                                                } 
                                                else 
                                                    if(*beg == "builderltl")
                                                    {
                                                        setTrace(builderltlModule);
                                                    } 
                                                    else
                                                    {
                                                        reportFailure(std::string("Tracing of ") + *beg + " unsupported");
                                                    }
        }
    }
#endif
}

// Instead of a destructor this is called explicitly
void
TRPPPOptions::cleanup()
{
    cmdline_parser_free(&myArgsInfo);
}

void
TRPPPOptions::reportFailure(std::string s )
{
    cmdline_parser_print_help();
    throw Exceptions::WrongOption(std::string("\n") + s);
}
