/************************************************************
 *    Copyright (C) 2012                                    *
 *    Viktor Schuppan (Viktor.Schuppan@gmx.de)              *
 *                                                          *
 *    This program is free software; you can redistribute   *
 *    it and/or modify it under the terms of the GNU        *
 *    General Public License as published by the Free       *
 *    Software Foundation, either version 3 of the License, *
 *    or (at your option) any later version.                *
 *                                                          *
 *    This program is distributed in the hope that it will  *
 *    be useful, but WITHOUT ANY WARRANTY; without even     *
 *    the implied warranty of MERCHANTABILITY or FITNESS    *
 *    FOR A PARTICULAR PURPOSE.  See the GNU General Public *
 *    License for more details.                             *
 *                                                          *
 *    You should have received a copy of the GNU General    *
 *    Public License along with this program.  If not, see  *
 *    <http://www.gnu.org/licenses/>.                       *
 ************************************************************/
#include <iostream>
#include <limits>
#include "misc/assert.h"
#include "unsat_core_private.h"
#include "linear_set.h"

namespace UnsatCore
{
  LinearSet::LinearSet()
    : myPeriod(0), myOffset(0)
  { }

  LinearSet::LinearSet(unsigned long const & period, unsigned long const & offset)
    : myPeriod(period), myOffset(offset)
  {
    Assert(period >= 0 &&
           period < std::numeric_limits<unsigned long>::max(),
           "Unexpected value for period.");
    Assert(offset >= 0 &&
           offset < std::numeric_limits<unsigned long>::max(),
           "Unexpected value for offset.");
  }

  unsigned long
  LinearSet::getPeriod() const
  {
    Assert(myPeriod >= 0 &&
           myPeriod < std::numeric_limits<unsigned long>::max(),
           "Unexpected value for period.");
    return myPeriod;
  } // LinearSet::getPeriod

  unsigned long
  LinearSet::getOffset() const
  {
    Assert(myOffset >= 0 &&
           myOffset < std::numeric_limits<unsigned long>::max(),
           "Unexpected value for offset.");
    return myOffset;
  } // LinearSet::getOffset

  void
  LinearSet::setPeriod(unsigned long const & period)
  {
    Assert(period >= 0 &&
           period < std::numeric_limits<unsigned long>::max(),
           "Unexpected value for period.");
    myPeriod = period;
  } // LinearSet::setPeriod

  void
  LinearSet::setOffset(unsigned long const & offset)
  {
    Assert(offset >= 0 &&
           offset < std::numeric_limits<unsigned long>::max(),
           "Unexpected value for offset.");
    myOffset = offset;
  } // LinearSet::setOffset

  bool
  LinearSet::isInvalid() const
  {
    return
      myPeriod == std::numeric_limits<unsigned long>::max() &&
      myOffset == std::numeric_limits<unsigned long>::max();
  } // LinearSet::isInvalid

  void
  LinearSet::makeInvalid()
  {
    myPeriod = std::numeric_limits<unsigned long>::max();
    myOffset = std::numeric_limits<unsigned long>::max();
  } // LinearSet::makeInvalid

  bool
  LinearSet::operator<(LinearSet const & other) const
  {
    if (myPeriod < other.myPeriod)
      return true;
    else if (other.myPeriod < myPeriod)
      return false;
    else
      return myOffset < other.myOffset;
  } // LinearSet::operator<

  std::ostream &
  LinearSet::dump(std::ostream & os) const
  {
    if (isInvalid()) {
      os << "invalid";
    } else {
      os << "("
         << myPeriod
         << " N, "
         << myOffset
         << ")";
    }
    return os;
  } // LinearSet::dump
} // namespace UnsatCore
