/************************************************************
 *    Copyright (C) 2012-2013                               *
 *    Viktor Schuppan (Viktor.Schuppan@gmx.de)              *
 *                                                          *
 *    This program is free software; you can redistribute   *
 *    it and/or modify it under the terms of the GNU        *
 *    General Public License as published by the Free       *
 *    Software Foundation, either version 3 of the License, *
 *    or (at your option) any later version.                *
 *                                                          *
 *    This program is distributed in the hope that it will  *
 *    be useful, but WITHOUT ANY WARRANTY; without even     *
 *    the implied warranty of MERCHANTABILITY or FITNESS    *
 *    FOR A PARTICULAR PURPOSE.  See the GNU General Public *
 *    License for more details.                             *
 *                                                          *
 *    You should have received a copy of the GNU General    *
 *    Public License along with this program.  If not, see  *
 *    <http://www.gnu.org/licenses/>.                       *
 ************************************************************/
#ifndef __RESOLUTION_GRAPH_H__
#define __RESOLUTION_GRAPH_H__

#include <map>
#include <set>
#include <unordered_map>
#include <utility>
#include <vector>
#include <boost/graph/adjacency_list.hpp>
#include <boost/graph/graph_traits.hpp>
#include <boost/graph/properties.hpp>
#include "getopt/options.h"
#include "clause_handle.h"
#include "inference_rule_edge_labeling.h"
#include "pclause_resolution_map.h"
#include "semi_linear_set.h"
#include "unsat_core_private.h"

#undef COMPUTE_REQUIRED_INFERENCE_RULES

namespace UnsatCore
{
  class ResolutionGraph
  {
  private:
    typedef boost::adjacency_list<
      boost::listS,
      boost::listS,
      boost::bidirectionalS,
      boost::property<boost::vertex_index_t, unsigned long,
             boost::property<boost::vertex_color_t, boost::default_color_type> >,
      boost::property<boost::edge_weight_t, long,
                    boost::property<boost::edge_weight2_t, long> > >
      Graph;
    typedef boost::graph_traits<Graph>::vertex_descriptor Vertex;
    typedef boost::graph_traits<Graph>::edge_descriptor Edge;
    typedef std::set<Vertex> VertexSet;
    typedef boost::property_map<Graph, boost::vertex_index_t>::type VertexIndex;
    typedef boost::graph_traits<Graph>::vertex_iterator VertexIterator;
  private:
    Graph myResolutionGraph;
    std::map<Vertex, ClauseHandle> myMapVertexToClauseh;
    std::unordered_map<ClauseHandle, Vertex> myMapClausehToVertex;
    std::map<Edge, InferenceRuleEdgeLabeling>
      myMapEdgeToInferenceRuleEdgeLabeling;
    std::set<ClauseHandle> myStartingClausehs;
    std::set<ClauseHandle> & myCoreStartingClausehs;
    std::map<ClauseHandle, SemiLinearSet> & myMapClausehToSemiLinearSet;
    PClause myLastEmptyClause;
    Partition myLastEmptyClausePartition;
    PClauseResolutionMap myPClauseResolutionMap;
    Partition myCurrentPartition;
    std::vector<std::pair<SClause, UClause> > myLoopInitializationSUClauseStore;
    std::vector<SClause> myLoopInitializationSClauseStore;
    std::map<UClause, SClause> myLoopInitializationCClauseMap;
    boost::property_map<Graph, boost::vertex_index_t>::type myVertexIndex;
    std::map<Vertex, VertexSet> myMapVertexTo0StarReachableVertices;
    VertexSet myLoopItVertexStore;
    bool myLoopItVertexStoreActive;
    VertexSet myMainPartition;
    std::map<VertexSet, VertexSet> myFwdImgCache;
    std::map<VertexSet, VertexSet> myBwdImgCache;
    std::map<VertexSet, VertexSet> myFwdImgRestrictedCache;
    unsigned long myNoVerticesPeak;
    unsigned long myNoVerticesBeforeReachable;
    unsigned long myNoVerticesAfterReachable;
    unsigned long myNoEdgesPeak;
    unsigned long myNoEdgesBeforeReachable;
    unsigned long myNoEdgesAfterReachable;
    unsigned long myNoEdgesAfterUnaryNFA;
    unsigned long myNoVerticesPlusEdgesPeak;
    unsigned long myNoSccs;
    unsigned long myNoFwdImgAccelerated;
    unsigned long myNoFwdImgNonAccelerated;
    unsigned long myNoBwdImgAccelerated;
    unsigned long myNoBwdImgNonAccelerated;
    unsigned long myNoFwdImgCacheHits;
    unsigned long myNoFwdImgCacheMisses;
    unsigned long myNoBwdImgCacheHits;
    unsigned long myNoBwdImgCacheMisses;
    unsigned long myNoFwdImgRestrictedCacheHits;
    unsigned long myNoFwdImgRestrictedCacheMisses;
    TRPPPOptions myOptions;
  public:
    ResolutionGraph(std::set<ClauseHandle> & coreStartingClausehs,
                    std::map<ClauseHandle, SemiLinearSet> &
                    mapClausehToSemiLinearSet,
                    TRPPPOptions const & options);
    void addStartingClause(IClause const & startingClause);
    void addStartingClause(UClause const & startingClause);
    void addStartingClause(SClause const & startingClause);
    void addStartingClause(EClause const & startingClause);
    void addSaturationClause(PClause const & conclusion,
                             PClause const & premise1,
                             PClause const & premise2,
                             bool const loopSearch);
    void addAugmentation1Clause(UClause const & conclusion,
                                EClause const & premise);
    void addAugmentation2Clause(SClause const & conclusion,
                                EClause const & premise);
    void clearLoopInitializationCClauseMap();
    void addLoopItInitializationCClause(SClause const & conclusion,
                                        UClause const & premiseg,
                                        EClause const & premisee);
    PClauseResolutionMap & getPClauseResolutionMap();
    void newPartition();
    Partition const getPreviousPartition(Partition const p);
    void clearLoopInitializationStores();
    void addLoopInitializationSUClauseStore(SClause const & conclusion,
                                            UClause const & premise);
    void addLoopInitializationSClauseStore(SClause const & conclusion);
    void addLoopItInitializationNXClauses();
    void addLoopItSubsumption(UClause const & conclusion,
                              UClause const & premise);
    void addLoopItSubsumptionByEmptyClause(UClause const & conclusion);
    void addLoopConclusion1WEmptyClause(UClause const & conclusion,
                                        EClause const & premisee);
    void addLoopConclusion1WOEmptyClause(UClause const & conclusion,
                                         UClause const & premiseg,
                                         EClause const & premisee);
    void addLoopConclusion2WEmptyClause(SClause const & conclusion,
                                        EClause const & premisee);
    void addLoopConclusion2WOEmptyClause(SClause const & conclusion,
                                         UClause const & premiseg,
                                         EClause const & premisee);
    PClause const & getLastEmptyClause();
    void setLastEmptyClause(PClause const & clause, bool const loopSearch);
    void activateLoopItVertexStore();
    void removeFromLoopItVertexStore();
    void deactivateLoopItVertexStore();
    void computeReachableSubgraphFromEmptyClause();
    void computeReachableSubgraphFromMainPartition();
    void computeCoreStartingClauses();
    void transformToUnaryNFA();
    void computeParikhImages_Sawa();
    void computeParikhImages_Gawrychowski();
    void reduceParikhImages();
    void computeCore();
    std::ostream & dumpStatistics(std::ostream & os);
    void dumpRG(std::string const & fileName);
  private:
    void add_starting_clauseh(ClauseHandle const & clauseh);
    void add_clauseh_no_premise(ClauseHandle const & clauseh);
    void add_clauseh_one_premise(ClauseHandle const & conclusionh,
                                 ClauseHandle const & premiseh,
                                 InferenceRuleEdgeLabeling const plabel);
    void add_clauseh_two_premises(ClauseHandle const & conclusionh,
                                  ClauseHandle const & premise1h,
                                  InferenceRuleEdgeLabeling const p1label,
                                  ClauseHandle const & premise2h,
                                  InferenceRuleEdgeLabeling const p2label);
    Vertex add_vertex(ClauseHandle const & clauseh);
    void remove_vertex(Vertex v);
    Edge add_edge(ClauseHandle const & sourceh,
                  ClauseHandle const & desth,
                  InferenceRuleEdgeLabeling const label);
    void compute_reachable_subgraph(VertexSet const & vs, bool reversed);
    void forward_image(VertexSet const & vs, VertexSet & result);
    void backward_image(VertexSet const & vs, VertexSet & result);
    void forward_image_restricted(VertexSet const & vs,
                                  VertexSet & result,
                                  std::vector<bool> & forbidden);
#ifdef COMPUTE_REQUIRED_INFERENCE_RULES
    void compute_required_inference_rules();
#endif
  }; // class ResolutionGraph
} // namespace UnsatCore
#endif // __RESOLUTION_GRAPH_H__
