/************************************************************
 *    Copyright (C) 2012-2013                               *
 *    Viktor Schuppan (Viktor.Schuppan@gmx.de)              *
 *                                                          *
 *    This program is free software; you can redistribute   *
 *    it and/or modify it under the terms of the GNU        *
 *    General Public License as published by the Free       *
 *    Software Foundation, either version 3 of the License, *
 *    or (at your option) any later version.                *
 *                                                          *
 *    This program is distributed in the hope that it will  *
 *    be useful, but WITHOUT ANY WARRANTY; without even     *
 *    the implied warranty of MERCHANTABILITY or FITNESS    *
 *    FOR A PARTICULAR PURPOSE.  See the GNU General Public *
 *    License for more details.                             *
 *                                                          *
 *    You should have received a copy of the GNU General    *
 *    Public License along with this program.  If not, see  *
 *    <http://www.gnu.org/licenses/>.                       *
 ************************************************************/
#include <fstream>
#include <iostream>
#include <list>
#include <map>
#include <set>
#include <string>
#include <boost/lexical_cast.hpp>
#include "builder_ltl/formula.h"
#include "getopt/options.h"
#include "misc/assert.h"
#include "stl_tim/attribute.h"
#include "stl_tim/clause.h"
#include "stl_tim/literal.h"
#include "stl_tim/pclause.h"
#include "stl_tim/proposition.h"
#include "clause_handle.h"
#include "linear_set.h"
#include "semi_linear_set.h"
#include "unsat_core_private.h"
#include "unsat_core_writer.h"

namespace UnsatCore
{
  void
  UnsatCoreWriter::dumpCore(ClauseHandleSet const & chs,
                            std::map<ClauseHandle, SemiLinearSet> & mChToSls,
                            std::list<TreeNode*> const & formulaList,
                            std::map<TreeNode*, SemiLinearSet> & mSfToSls,
                            TRPPPOptions options)
  {
    if (options.getExtractUnsatCoreFileName() == std::string("")) {
      std::cout << "The following is an unsatisfiable core:"
                << std::endl;
    } else {
      std::cout << "An unsatisfiable core is written to "
                << options.getExtractUnsatCoreFileName()
                << "."
                << std::endl;
    }

    {
      std::ofstream unsatCoreFile;
      if (options.getExtractUnsatCoreFileName() != std::string("")) {
        unsatCoreFile.open(options.getExtractUnsatCoreFileName().c_str());
        if (!unsatCoreFile.is_open()) {
          std::cout << "Could not open file for writing unsatisfiable core "
                    << options.getExtractUnsatCoreFileName()
                    << ". Writing to stdout."
                    << std::endl;
        }
      }
      std::ostream & os =
        (options.getExtractUnsatCoreFileName() != std::string("") &&
         unsatCoreFile.is_open()) ?
        unsatCoreFile :
        std::cout;

      {
        UnsatCoreWriter ucw;
        ucw.dumpClausehs(os, chs, mChToSls, formulaList, mSfToSls, options);
      }

      if (options.getExtractUnsatCoreFileName() != std::string("")) {
        unsatCoreFile.close();
      }
    }

    if (options.getExtractUnsatCoreFileName() == std::string("")) {
      std::cout << "End of unsatisfiable core." << std::endl;
    }            
  } // UnsatCoreWriter::dumpCore

  void
  UnsatCoreWriter::dumpCoreBenchmarking(ClauseHandleSet const & chs,
                               std::map<ClauseHandle, SemiLinearSet> & mChToSls,
                                       std::list<TreeNode*> const & formulaList,
                                  std::map<TreeNode*, SemiLinearSet> & mSfToSls,
                                        TRPPPOptions options)
  {
    std::ofstream unsatCoreFile;
    ExtractUnsatCoreAlgorithm alg = options.getExtractUnsatCoreAlgorithm();
    std::string algs = (alg == EXTRACT_UNSAT_CORE_ALGORITHM_PROOF ? "proof" :
                        (alg == EXTRACT_UNSAT_CORE_ALGORITHM_DELETION ? "deletion" :
                         (alg == EXTRACT_UNSAT_CORE_ALGORITHM_PROOFDELETION ?
                          "proofdeletion" :
                          "unknown")));
    if (options.getExtractUnsatCoreMode() == EXTRACT_UNSAT_CORE_MODE_SIMPLE) {
      // Internal
      {
        std::string fn = options.getFileName();
        fn += ".simple." + algs + ".internal";
        unsatCoreFile.open(fn.c_str());
        if (!unsatCoreFile.is_open()) {
          std::cout << "Could not open file for writing unsatisfiable core "
                    << fn
                    << ". Writing to stdout."
                    << std::endl;
        }
        std::ostream & os = unsatCoreFile.is_open() ? unsatCoreFile : std::cout;
        UnsatCoreWriter ucw;
        ucw.dumpClausehsSimpleInternal(os, chs);
        unsatCoreFile.close();
      }
      // SNF
      {
        std::string fn = options.getFileName();
        fn += ".simple." + algs + ".snf";
        unsatCoreFile.open(fn.c_str());
        if (!unsatCoreFile.is_open()) {
          std::cout << "Could not open file for writing unsatisfiable core "
                    << fn
                    << ". Writing to stdout."
                    << std::endl;
        }
        std::ostream & os = unsatCoreFile.is_open() ? unsatCoreFile : std::cout;
        UnsatCoreWriter ucw;
        ucw.dumpClausehsSimpleSNF(os, chs);
        unsatCoreFile.close();
      }
      // LTLC
      {
        std::string fn = options.getFileName();
        fn += ".simple." + algs + ".ltlc";
        unsatCoreFile.open(fn.c_str());
        if (!unsatCoreFile.is_open()) {
          std::cout << "Could not open file for writing unsatisfiable core "
                    << fn
                    << ". Writing to stdout."
                    << std::endl;
        }
        std::ostream & os = unsatCoreFile.is_open() ? unsatCoreFile : std::cout;
        UnsatCoreWriter ucw;
        ucw.dumpClausehsSimpleLTLC(os, chs);
        unsatCoreFile.close();
      }
      // LTL
      {
        std::string fn = options.getFileName();
        fn += ".simple." + algs + ".ltl";
        unsatCoreFile.open(fn.c_str());
        if (!unsatCoreFile.is_open()) {
          std::cout << "Could not open file for writing unsatisfiable core "
                    << fn
                    << ". Writing to stdout."
                    << std::endl;
        }
        std::ostream & os = unsatCoreFile.is_open() ? unsatCoreFile : std::cout;
        UnsatCoreWriter ucw;
        ucw.dumpClausehsSimpleLTL(os, formulaList);
        unsatCoreFile.close();
      }
    } else {
      // Internal
      {
        std::string fn = options.getFileName();
        fn += ".setsoftimepoints." + algs + ".internal";
        unsatCoreFile.open(fn.c_str());
        if (!unsatCoreFile.is_open()) {
          std::cout << "Could not open file for writing unsatisfiable core "
                    << fn
                    << ". Writing to stdout."
                    << std::endl;
        }
        std::ostream & os = unsatCoreFile.is_open() ? unsatCoreFile : std::cout;
        UnsatCoreWriter ucw;
        ucw.dumpClausehsSetsOfTimePointsInternal(os, chs, mChToSls);
        unsatCoreFile.close();
      }
      // SNF
      {
        std::string fn = options.getFileName();
        fn += ".setsoftimepoints." + algs + ".snf";
        unsatCoreFile.open(fn.c_str());
        if (!unsatCoreFile.is_open()) {
          std::cout << "Could not open file for writing unsatisfiable core "
                    << fn
                    << ". Writing to stdout."
                    << std::endl;
        }
        std::ostream & os = unsatCoreFile.is_open() ? unsatCoreFile : std::cout;
        UnsatCoreWriter ucw;
        ucw.dumpClausehsSetsOfTimePointsSNF(os, chs, mChToSls);
        unsatCoreFile.close();
      }
      // LTLC
      {
        std::string fn = options.getFileName();
        fn += ".setsoftimepoints." + algs + ".ltlc";
        unsatCoreFile.open(fn.c_str());
        if (!unsatCoreFile.is_open()) {
          std::cout << "Could not open file for writing unsatisfiable core "
                    << fn
                    << ". Writing to stdout."
                    << std::endl;
        }
        std::ostream & os = unsatCoreFile.is_open() ? unsatCoreFile : std::cout;
        UnsatCoreWriter ucw;
        ucw.dumpClausehsSetsOfTimePointsLTLC(os, chs, mChToSls);
        unsatCoreFile.close();
      }
      // LTL
      {
        std::string fn = options.getFileName();
        fn += ".setsoftimepoints." + algs + ".ltl";
        unsatCoreFile.open(fn.c_str());
        if (!unsatCoreFile.is_open()) {
          std::cout << "Could not open file for writing unsatisfiable core "
                    << fn
                    << ". Writing to stdout."
                    << std::endl;
        }
        std::ostream & os =
          unsatCoreFile.is_open() ? unsatCoreFile : std::cout;
        UnsatCoreWriter ucw;
        ucw.dumpClausehsSetsOfTimePointsLTL(os,
                                            chs,
                                            mChToSls,
                                            formulaList,
                                            mSfToSls);
        unsatCoreFile.close();
      }
    }
  } // UnsatCoreWriter::dumpCoreBenchmarking

  UnsatCoreWriter::UnsatCoreWriter()
    : myNvvVarPrefix("")
  { } // UnsatCoreWriter::UnsatCoreWriter

  std::ostream &
  UnsatCoreWriter::dumpClausehs(std::ostream & os,
                                ClauseHandleSet const & chs,
                               std::map<ClauseHandle, SemiLinearSet> & mChToSls,
                                std::list<TreeNode*> const & formulaList,
                                std::map<TreeNode*, SemiLinearSet> & mSfToSls,
                                TRPPPOptions options)
  {
    if (options.getExtractUnsatCoreFormat() ==
        EXTRACT_UNSAT_CORE_FORMAT_INTERNAL) {
      if (options.getExtractUnsatCoreMode() ==
          EXTRACT_UNSAT_CORE_MODE_SIMPLE) {
        dumpClausehsSimpleInternal(os, chs);
      } else {
        Assert(options.getExtractUnsatCoreMode() ==
               EXTRACT_UNSAT_CORE_MODE_SETSOFTIMEPOINTS,
               "Unexpected value of options.getExtractUnsatCoreFormat().");
        dumpClausehsSetsOfTimePointsInternal(os, chs, mChToSls);
      }
    } else if (options.getExtractUnsatCoreFormat() ==
               EXTRACT_UNSAT_CORE_FORMAT_SNF) {
      if (options.getExtractUnsatCoreMode() ==
          EXTRACT_UNSAT_CORE_MODE_SIMPLE) {
        dumpClausehsSimpleSNF(os, chs);
      } else {
        Assert(options.getExtractUnsatCoreMode() ==
               EXTRACT_UNSAT_CORE_MODE_SETSOFTIMEPOINTS,
               "Unexpected value of options.getExtractUnsatCoreFormat().");
        dumpClausehsSetsOfTimePointsSNF(os, chs, mChToSls);
      }
    } else if (options.getExtractUnsatCoreFormat() ==
               EXTRACT_UNSAT_CORE_FORMAT_LTLC) {
      if (options.getExtractUnsatCoreMode() ==
          EXTRACT_UNSAT_CORE_MODE_SIMPLE) {
        dumpClausehsSimpleLTLC(os, chs);
      } else {
        Assert(options.getExtractUnsatCoreMode() ==
               EXTRACT_UNSAT_CORE_MODE_SETSOFTIMEPOINTS,
               "Unexpected value of options.getExtractUnsatCoreFormat().");
        dumpClausehsSetsOfTimePointsLTLC(os, chs, mChToSls);
      }
    } else {
      Assert(options.getExtractUnsatCoreFormat() ==
             EXTRACT_UNSAT_CORE_FORMAT_LTL,
             "Unexpected value of options.getExtractUnsatCoreFormat().");
      if (options.getExtractUnsatCoreMode() ==
          EXTRACT_UNSAT_CORE_MODE_SIMPLE) {
        dumpClausehsSimpleLTL(os, formulaList);
      } else {
        Assert(options.getExtractUnsatCoreMode() ==
               EXTRACT_UNSAT_CORE_MODE_SETSOFTIMEPOINTS,
               "Unexpected value of options.getExtractUnsatCoreFormat().");
        dumpClausehsSetsOfTimePointsLTL(os, chs, mChToSls, formulaList, mSfToSls);
      }
    }
    return os;
  } // UnsatCoreWriter::dumpClausehs

  std::ostream &
  UnsatCoreWriter::dumpClausehsSimpleInternal(std::ostream & os,
                                              ClauseHandleSet const & chs)
  {
    ClauseHandleSet::iterator it;
    for (it = chs.begin(); it != chs.end(); ++it)
      {
        it->dump(os);
        os << std::endl;
      }
    return os;
  } // UnsatCoreWriter::dumpClausehsSimpleInternal

  std::ostream &
  UnsatCoreWriter::dumpClausehsSimpleSNF(std::ostream & os,
                                         ClauseHandleSet const & chs)
  {
    os << "and([" << std::endl;
    ClauseHandleSet::iterator it;
    for (it = chs.begin(); it != chs.end();)
      {
        dumpClausehSNF(os, *it);
        ++it;
        if (it != chs.end()) {
          os << ",";
        }
        os << std::endl;
      }
    os << "])." << std::endl;
    return os;
  } // UnsatCoreWriter::dumpClausehsSimpleSNF

  std::ostream &
  UnsatCoreWriter::dumpClausehsSimpleLTLC(std::ostream & os,
                                          ClauseHandleSet const & chs)
  {
    myNvvVarPrefix = get_unused_var_prefix(chs, "nvv");

    ClauseHandleSet::iterator it;
    for (it = chs.begin(); it != chs.end();)
      {
        dump_clauseh_LTLC(os, *it);
        ++it;
        if (it != chs.end()) {
          os << " & ";
        }
        os << std::endl;
      }

    myNvvVarPrefix = "";

    return os;
  } // UnsatCoreWriter::dumpClausehsSimpleLTLC

  std::ostream &
  UnsatCoreWriter::dumpClausehsSimpleLTL(std::ostream & os,
                                       std::list<TreeNode*> const & formulaList)
  {
    for (std::list<TreeNode *>::const_iterator it = formulaList.begin();
         it != formulaList.end();
         )
      {
        os << "(";
        dump_clausehs_simple_LTL_rec(os, *it, true);
        os << ")";
        ++it;
        if (it != formulaList.end()) {
          os << " & ";
        }
        os << std::endl;
      }
    return os;
  } // UnsatCoreWriter::dumpClausehsSimpleLTL

  std::ostream &
  UnsatCoreWriter::dumpClausehsSetsOfTimePointsInternal(std::ostream & os,
                                                    ClauseHandleSet const & chs,
                               std::map<ClauseHandle, SemiLinearSet> & mChToSls)
  {
    ClauseHandleSet::iterator it;
    for (it = chs.begin(); it != chs.end(); ++it)
      {
        Assert(mChToSls.find(*it) != mChToSls.end(),
               "*it not present in mChToSls.");
        it->dump(os);
        os << " @ ";
        mChToSls[*it].dump(os);
        os << std::endl;
      }
    return os;
  } // UnsatCoreWriter::dumpClausehsSetsOfTimePointsInternal

  std::ostream &
  UnsatCoreWriter::dumpClausehsSetsOfTimePointsSNF(std::ostream & os,
                                                   ClauseHandleSet const & chs,
                               std::map<ClauseHandle, SemiLinearSet> & mChToSls)
  {
    os << "and([" << std::endl;
    ClauseHandleSet::iterator it;
    for (it = chs.begin(); it != chs.end();)
      {
        Assert(mChToSls.find(*it) != mChToSls.end(),
               "*it not present in mChToSls.");
        dumpClausehSNF(os, *it);
        os << " @ ";
        mChToSls[*it].dump(os);
        ++it;
        if (it != chs.end()) {
          os << ",";
        }
        os << std::endl;
      }
    os << "])." << std::endl;
    return os;
  } // UnsatCoreWriter::dumpClausehsSetsOfTimePointsSNF

  std::ostream &
  UnsatCoreWriter::dumpClausehsSetsOfTimePointsLTLC(std::ostream & os,
                                                ClauseHandleSet const & chs,
                               std::map<ClauseHandle, SemiLinearSet> & mChToSls)
  {
    std::string fresh_var_prefix = get_unused_var_prefix(chs, "fresh");
    int fresh_var_cnt = 1;

    myNvvVarPrefix = get_unused_var_prefix(chs, "nvv");

    ClauseHandleSet::iterator it;
    for (it = chs.begin(); it != chs.end();)
      {
        Assert(mChToSls.find(*it) != mChToSls.end(),
               "*it not present in mChToSls.");
        SemiLinearSet sls = mChToSls[*it];

        if (it->isIClause()) {
          Assert(sls.size() == 1 && sls.find(LinearSet(0, 0)) != sls.end(),
                 "sls of an initial clause is not {(0 N, 0)}.");
          os << "(";
          dump_clauseh_LTLC(os, *it);
          os << ")"
             << std::endl;
        } else {
          Assert(it->isUClause() || it->isSClause() || it->isEClause(),
                 "*it is not a UClause, SClause, or EClause.");
          Assert(sls.size() > 0,
                 "sls of a universal, step, or eventuality clause is empty.");
          for (SemiLinearSet::iterator it2 = sls.begin(); it2 != sls.end();) {
            os << "(";
            unsigned long p = it2->getPeriod();
            unsigned long o = it2->getOffset();
            os << "(";
            for (unsigned long i = 0; i < o; ++i) {
              os << "next(";
            }
            if (p == 0) {
              dump_clauseh_body_LTLC(os, *it);
            } else {
              std::string fresh_var = "";
              fresh_var += fresh_var_prefix;
              fresh_var += boost::lexical_cast<std::string>(fresh_var_cnt);
              os << fresh_var
                 << " & always("
                 << fresh_var
                 << " -> ";
              for (unsigned long i = 0; i < p; ++i) {
                os << "next(";
              }
              os << fresh_var;
              for (unsigned long i = 0; i < p; ++i) {
                os << ")";
              }
              os << ") & always("
                 << fresh_var
                 << " -> (";
              dump_clauseh_body_LTLC(os, *it);
              os << "))";
              ++fresh_var_cnt;
            }
            for (unsigned long i = 0; i < o; ++i) {
              os << ")";
            }
            os << ")";
            os << ")";
            ++it2;
            if (it2 != sls.end()) {
              os << " & ";
            }
            os << std::endl;
          }
        }
        ++it;
        if (it != chs.end()) {
          os << " & ";
        }
        os << std::endl;
      }

    myNvvVarPrefix = "";

    return os;
  } // UnsatCoreWriter::dumpClausehsSetsOfTimePointsLTLC

  std::ostream &
  UnsatCoreWriter::dumpClausehsSetsOfTimePointsLTL(std::ostream & os,
                                               ClauseHandleSet const & chs,
                               std::map<ClauseHandle, SemiLinearSet> & mChToSls,
                                       std::list<TreeNode*> const & formulaList,
                                  std::map<TreeNode*, SemiLinearSet> & mSfToSls)
  {
    for (std::list<TreeNode *>::const_iterator it = formulaList.begin();
         it != formulaList.end();
         )
      {
        os << "(";
        dump_clausehs_sets_of_time_points_LTL_rec(os, *it, true, mSfToSls);
        os << ")";
        ++it;
        if (it != formulaList.end()) {
          os << " {(0 N, 0)}@_&_@{(0 N, 0)} ";
        }
        os << std::endl;
      }
    return os;
  } // UnsatCoreWriter::dumpClausehsSetsOfTimePointsLTL

  std::ostream &
  UnsatCoreWriter::dumpClausehSNF(std::ostream & os, ClauseHandle const & ch)
  {
    if (!ch.isIClause()) {
      Assert(ch.isUClause() || ch.isSClause() || ch.isEClause(),
             "ch is not a UClause, SClause, or EClause.");
      os << "always(";
    }
    os << "or([";
    if (ch.isIClause()) {
      IClause::const_iterator it;
      for (it = ch.getIClause().begin(); it != ch.getIClause().end();)
        {
          Assert(it->getAttribute().getType() ==
                 PropositionalProver::Attribute::initial,
                 "Attribute not initial");
          dump_literal(os, *it);
          it++;
          if (it != ch.getIClause().end()) {
            os << ",";
          }
        }
    } else if (ch.isUClause()) {
      UClause::const_iterator it;
      for (it = ch.getUClause().begin(); it != ch.getUClause().end();)
        {
          Assert(it->getAttribute().getType() ==
                 PropositionalProver::Attribute::universal ||
                 it->getAttribute().getType() ==
                 PropositionalProver::Attribute::step_now,
                 "Attribute not universal or step_now");
          dump_literal(os, *it);
          it++;
          if (it != ch.getUClause().end()) {
            os << ",";
          }
        }
    } else if (ch.isSClause()) {
      PClause pc = ch.getSClause().getInternalRep();
      PropositionalProver::Clause::iterator it;
      bool first = true;

      // First only step_now
      for (it = pc->begin(); it != pc->end(); ++it)
        {
          Assert(it->getAttribute().getType() ==
                 PropositionalProver::Attribute::step_now ||
                 it->getAttribute().getType() ==
                 PropositionalProver::Attribute::step_next,
                 "Attribute not step_now or step_next.");
          if (it->getAttribute().getType() ==
              PropositionalProver::Attribute::step_now) {
            if (!first) {
              os << ",";
            }
            first = false;
            dump_literal(os, *it);
          }
        }

      // Second only step_next
      for (it = pc->begin(); it != pc->end(); ++it)
        {
          if (it->getAttribute().getType() ==
              PropositionalProver::Attribute::step_next) {
            if (!first) {
              os << ",";
            }
            first = false;
            os << "next(";
            dump_literal(os, *it);
            os << ")";
          }
        }
    } else {
      Assert(ch.isEClause(), "ch is not an EClause.");
      EClause::const_present_iterator it;
      for (it = ch.getEClause().present_begin();
           it != ch.getEClause().present_end();
           ++it)
        {
          Assert(it->getAttribute().getType() ==
                 PropositionalProver::Attribute::universal,
                 "Attribute not universal");
          dump_literal(os, *it);
          os << ",";
        }
      os << "sometime(";
      dump_literal(os, ch.getEClause().getEventuality());
      os << ")";
    }
    os << "])";
    if (!ch.isIClause()) {
      Assert(ch.isUClause() || ch.isSClause() || ch.isEClause(),
             "ch is not a UClause, SClause, or EClause.");
      os << ")";
    }
    return os;
  } // UnsatCoreWriter::dumpClausehSNF

  std::ostream &
  UnsatCoreWriter::dump_literal(std::ostream & os,
                                PropositionalProver::Literal const & l)
  {
    if (l.isNegative()) {
      os << "not ";
    }
    if (l.getProposition().getName().find("NVV") == 0) {
      os << myNvvVarPrefix;
    }
    l.getProposition().dump(os);
    return os;
  } // UnsatCoreWriter::dump_literal

  std::ostream &
  UnsatCoreWriter::dump_clauseh_body_LTLC(std::ostream & os,
                                          ClauseHandle const & ch)
  {
    if (ch.isIClause()) {
      IClause::const_iterator it;
      for (it = ch.getIClause().begin(); it != ch.getIClause().end();)
        {
          Assert(it->getAttribute().getType() ==
                 PropositionalProver::Attribute::initial,
                 "Attribute not initial");
          os << "(";
          dump_literal(os, *it);
          os << ")";
          it++;
          if (it != ch.getIClause().end()) {
            os << " | ";
          }
        }
    } else if (ch.isUClause()) {
      UClause::const_iterator it;
      for (it = ch.getUClause().begin(); it != ch.getUClause().end();)
        {
          Assert(it->getAttribute().getType() ==
                 PropositionalProver::Attribute::universal ||
                 it->getAttribute().getType() ==
                 PropositionalProver::Attribute::step_now,
                 "Attribute not universal or step_now");
          os << "(";
          dump_literal(os, *it);
          os << ")";
          it++;
          if (it != ch.getUClause().end()) {
            os << " | ";
          }
        }
    } else if (ch.isSClause()) {
      PClause pc = ch.getSClause().getInternalRep();
      PropositionalProver::Clause::iterator it;
      bool first = true;

      // First only step_now
      for (it = pc->begin(); it != pc->end(); ++it)
        {
          Assert(it->getAttribute().getType() ==
                 PropositionalProver::Attribute::step_now ||
                 it->getAttribute().getType() ==
                 PropositionalProver::Attribute::step_next,
                 "Attribute not step_now or step_next.");
          if (it->getAttribute().getType() ==
              PropositionalProver::Attribute::step_now) {
            if (!first) {
              os << " | ";
            }
            first = false;
            os << "(";
            dump_literal(os, *it);
            os << ")";
          }
        }

      // Second only step_next
      for (it = pc->begin(); it != pc->end(); ++it)
        {
          if (it->getAttribute().getType() ==
              PropositionalProver::Attribute::step_next) {
            if (!first) {
              os << " | ";
            }
            first = false;
            os << "(next(";
            dump_literal(os, *it);
            os << "))";
          }
        }
    } else {
      Assert(ch.isEClause(), "ch is not an EClause.");
      EClause::const_present_iterator it;
      for (it = ch.getEClause().present_begin();
           it != ch.getEClause().present_end();
           ++it)
        {
          Assert(it->getAttribute().getType() ==
                 PropositionalProver::Attribute::universal,
                 "Attribute not universal");
          os << "(";
          dump_literal(os, *it);
          os << ")";
          os << " | ";
        }
      os << "(sometime(";
      dump_literal(os, ch.getEClause().getEventuality());
      os << "))";
    }
    return os;
  } // UnsatCoreWriter::dump_clauseh_body_LTLC

  std::ostream &
  UnsatCoreWriter::dump_clauseh_LTLC(std::ostream & os, ClauseHandle const & ch)
  {
    if (!ch.isIClause()) {
      Assert(ch.isUClause() || ch.isSClause() || ch.isEClause(),
             "ch is not a UClause, SClause, or EClause.");
      os << "always";
    }
    os << "(";
    dump_clauseh_body_LTLC(os, ch);
    os << ")";
    return os;
  } // UnsatCoreWriter::dump_clauseh_LTLC

  std::ostream &
  UnsatCoreWriter::dump_clausehs_simple_LTL_rec(std::ostream & os,
                                                TreeNode * n,
                                                bool isPositivePolarity)
  {
    if (isIdentifier(n) && ((Identifier *) n)->isTrue()) {
      Assert(n->childrenCount() == 0, "Unexpected number of children.");
      os << "True";
    } else if (isIdentifier(n) && ((Identifier *) n)->isFalse()) {
      Assert(n->childrenCount() == 0, "Unexpected number of children.");
      os << "False";
    } else if (isIdentifier(n)) {
      Assert(n->childrenCount() == 0, "Unexpected number of children.");
      os << ((Identifier *) n)->getIdentifier();
    } else if (isNot(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      os << "~ (";
      dump_clausehs_simple_LTL_rec(os,
                                   n->firstChild(),
                                   !isPositivePolarity);
      os << ")";
    } else if (isAnd(n)) {
      Assert(n->childrenCount() > 0, "Unexpected number of children.");
      for (TreeNodeList::const_iterator it = n->children().begin();
           it != n->children().end();
           )
        {
          TreeNode * child = *it;
          ++it;
          os << "(";
          dump_clausehs_simple_LTL_rec(os,
                                       child,
                                       isPositivePolarity);
          os << ")";
          if (it != n->children().end()) {
            os << " & ";
          }
        }
    } else if (isOr(n)) {
      Assert(n->childrenCount() > 0, "Unexpected number of children.");
      for (TreeNodeList::const_iterator it = n->children().begin();
           it != n->children().end();
           )
        {
          TreeNode * child = *it;
          ++it;
          os << "(";
          dump_clausehs_simple_LTL_rec(os,
                                       child,
                                       isPositivePolarity);
          os << ")";
          if (it != n->children().end()) {
            os << " | ";
          }
        }
    } else if (isImplication(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      os << "(";
      dump_clausehs_simple_LTL_rec(os,
                                   n->firstChild(),
                                   !isPositivePolarity);
      os << ") -> (";
      dump_clausehs_simple_LTL_rec(os,
                                   n->secondChild(),
                                   isPositivePolarity);
      os << ")";
    } else if (isEquivalence(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      os << "(";
      dump_clausehs_simple_LTL_rec(os,
                                   n->firstChild(),
                                   isPositivePolarity);
      os << ") <-> (";
      dump_clausehs_simple_LTL_rec(os,
                                   n->secondChild(),
                                   isPositivePolarity);
      os << ")";
    } else if (isNext(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      os << "next (";
      dump_clausehs_simple_LTL_rec(os,
                                   n->firstChild(),
                                   isPositivePolarity);
      os << ")";
    } else if (isAlways(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      os << "always (";
      dump_clausehs_simple_LTL_rec(os,
                                   n->firstChild(),
                                   isPositivePolarity);
      os << ")";
    } else if (isSometime(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      os << "sometime (";
      dump_clausehs_simple_LTL_rec(os,
                                   n->firstChild(),
                                   isPositivePolarity);
      os << ")";
    } else if (isUntil(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      os << "(";
      dump_clausehs_simple_LTL_rec(os,
                                   n->firstChild(),
                                   isPositivePolarity);
      os << ") until (";
      dump_clausehs_simple_LTL_rec(os,
                                   n->secondChild(),
                                   isPositivePolarity);
      os << ")";
    } else if (isUnless(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      os << "(";
      dump_clausehs_simple_LTL_rec(os,
                                   n->firstChild(),
                                   isPositivePolarity);
      os << ") unless (";
      dump_clausehs_simple_LTL_rec(os,
                                   n->secondChild(),
                                   isPositivePolarity);
      os << ")";
    } else {
      Assert(false, "n has unexpected type.");
    }

    return os;
  } // UnsatCoreWriter::dump_clausehs_simple_LTL_rec

  std::ostream &
  UnsatCoreWriter::dump_clausehs_sets_of_time_points_LTL_rec(std::ostream & os,
                                                             TreeNode * n,
                                                        bool isPositivePolarity,
                                  std::map<TreeNode*, SemiLinearSet> & mSfToSls)
  {
    if (isIdentifier(n) && ((Identifier *) n)->isTrue()) {
      Assert(n->childrenCount() == 0, "Unexpected number of children.");
      os << "True";
    } else if (isIdentifier(n) && ((Identifier *) n)->isFalse()) {
      Assert(n->childrenCount() == 0, "Unexpected number of children.");
      os << "False";
    } else if (isIdentifier(n)) {
      Assert(n->childrenCount() == 0, "Unexpected number of children.");
      os << ((Identifier *) n)->getIdentifier();
    } else if (isNot(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      os << "~_@";
      Assert(mSfToSls.find(n->firstChild()) != mSfToSls.end(),
             "n->firstChild() not present in mSfToSls.");
      mSfToSls[n->firstChild()].dump(os);
      os << " (";
      dump_clausehs_sets_of_time_points_LTL_rec(os,
                                                n->firstChild(),
                                                !isPositivePolarity,
                                                mSfToSls);
      os << ")";
    } else if (isAnd(n)) {
      Assert(n->childrenCount() > 0, "Unexpected number of children.");
      bool isFirst = true;
      for (TreeNodeList::const_iterator it = n->children().begin();
           it != n->children().end();
           )
        {
          TreeNode * child = *it;
          ++it;
          os << "(";
          dump_clausehs_sets_of_time_points_LTL_rec(os,
                                                    child,
                                                    isPositivePolarity,
                                                    mSfToSls);
          os << ")";
          if (it != n->children().end()) {
            if (isFirst) {
              isFirst = false;
              os << " ";
              Assert(mSfToSls.find(child) != mSfToSls.end(),
                     "child not present in mSfToSls.");
              mSfToSls[child].dump(os);
              os << "@_&_@";
              Assert(mSfToSls.find(*it) != mSfToSls.end(),
                     "*it not present in mSfToSls.");
              mSfToSls[*it].dump(os);
              os << " ";
            } else {
              os << " &_@";
              Assert(mSfToSls.find(*it) != mSfToSls.end(),
                     "*it not present in mSfToSls.");
              mSfToSls[*it].dump(os);
              os << " ";
            }
          }
        }
    } else if (isOr(n)) {
      Assert(n->childrenCount() > 0, "Unexpected number of children.");
      bool isFirst = true;
      for (TreeNodeList::const_iterator it = n->children().begin();
           it != n->children().end();
           )
        {
          TreeNode * child = *it;
          ++it;
          os << "(";
          dump_clausehs_sets_of_time_points_LTL_rec(os,
                                                    child,
                                                    isPositivePolarity,
                                                    mSfToSls);
          os << ")";
          if (it != n->children().end()) {
            if (isFirst) {
              isFirst = false;
              os << " ";
              Assert(mSfToSls.find(child) != mSfToSls.end(),
                     "child not present in mSfToSls.");
              mSfToSls[child].dump(os);
              os << "@_|_@";
              Assert(mSfToSls.find(*it) != mSfToSls.end(),
                     "*it not present in mSfToSls.");
              mSfToSls[*it].dump(os);
              os << " ";
            } else {
              os << " |_@";
              Assert(mSfToSls.find(*it) != mSfToSls.end(),
                     "*it not present in mSfToSls.");
              mSfToSls[*it].dump(os);
              os << " ";
            }
          }
        }
    } else if (isImplication(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      os << "(";
      dump_clausehs_sets_of_time_points_LTL_rec(os,
                                                n->firstChild(),
                                                !isPositivePolarity,
                                                mSfToSls);
      os << ") ";
      Assert(mSfToSls.find(n->firstChild()) != mSfToSls.end(),
             "n->firstChild() not present in mSfToSls.");
      mSfToSls[n->firstChild()].dump(os);
      os << "@_->_@";
      Assert(mSfToSls.find(n->secondChild()) != mSfToSls.end(),
             "n->secondChild() not present in mSfToSls.");
      mSfToSls[n->secondChild()].dump(os);
      os << " (";
      dump_clausehs_sets_of_time_points_LTL_rec(os,
                                                n->secondChild(),
                                                isPositivePolarity,
                                                mSfToSls);
      os << ")";
    } else if (isNext(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      os << "next_@";
      Assert(mSfToSls.find(n->firstChild()) != mSfToSls.end(),
             "n->firstChild() not present in mSfToSls.");
      mSfToSls[n->firstChild()].dump(os);
      os << " (";
      dump_clausehs_sets_of_time_points_LTL_rec(os,
                                                n->firstChild(),
                                                isPositivePolarity,
                                                mSfToSls);
      os << ")";
    } else if (isAlways(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      os << "always_@";
      Assert(mSfToSls.find(n->firstChild()) != mSfToSls.end(),
             "n->firstChild() not present in mSfToSls.");
      mSfToSls[n->firstChild()].dump(os);
      os << " (";
      dump_clausehs_sets_of_time_points_LTL_rec(os,
                                                n->firstChild(),
                                                isPositivePolarity,
                                                mSfToSls);
      os << ")";
    } else if (isSometime(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      os << "sometime_@";
      Assert(mSfToSls.find(n->firstChild()) != mSfToSls.end(),
             "n->firstChild() not present in mSfToSls.");
      mSfToSls[n->firstChild()].dump(os);
      os << " (";
      dump_clausehs_sets_of_time_points_LTL_rec(os,
                                                n->firstChild(),
                                                isPositivePolarity,
                                                mSfToSls);
      os << ")";
    } else if (isUntil(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      os << "(";
      dump_clausehs_sets_of_time_points_LTL_rec(os,
                                                n->firstChild(),
                                                isPositivePolarity,
                                                mSfToSls);
      os << ") ";
      Assert(mSfToSls.find(n->firstChild()) != mSfToSls.end(),
             "n->firstChild() not present in mSfToSls.");
      mSfToSls[n->firstChild()].dump(os);
      os << "@_until_@";
      Assert(mSfToSls.find(n->secondChild()) != mSfToSls.end(),
             "n->secondChild() not present in mSfToSls.");
      mSfToSls[n->secondChild()].dump(os);
      os << " (";
      dump_clausehs_sets_of_time_points_LTL_rec(os,
                                                n->secondChild(),
                                                isPositivePolarity,
                                                mSfToSls);
      os << ")";
    } else if (isUnless(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      os << "(";
      dump_clausehs_sets_of_time_points_LTL_rec(os,
                                                n->firstChild(),
                                                isPositivePolarity,
                                                mSfToSls);
      os << ") ";
      Assert(mSfToSls.find(n->firstChild()) != mSfToSls.end(),
             "n->firstChild() not present in mSfToSls.");
      mSfToSls[n->firstChild()].dump(os);
      os << "@_unless_@";
      Assert(mSfToSls.find(n->secondChild()) != mSfToSls.end(),
             "n->secondChild() not present in mSfToSls.");
      mSfToSls[n->secondChild()].dump(os);
      os << " (";
      dump_clausehs_sets_of_time_points_LTL_rec(os,
                                                n->secondChild(),
                                                isPositivePolarity,
                                                mSfToSls);
      os << ")";
    } else {
      Assert(false, "n has unexpected type.");
    }

    return os;
  } // UnsatCoreWriter::dump_clausehs_sets_of_time_points_LTL_rec

  std::string
  UnsatCoreWriter::get_unused_var_prefix(ClauseHandleSet const & chs,
                                         std::string const & startPrefix)
  {
    typedef std::set<std::string> StringSet;
    StringSet aps = StringSet();

    /* Collect atomic propositions */

    for (ClauseHandleSet::iterator it = chs.begin(); it != chs.end(); ++it)
      {
        if (!it->isEClause()) {
          Assert(it->isIClause() || it->isUClause() || it->isSClause(),
                 "*it is not a IClause, UClause, or SClause.");
          PClause pc;
          if (it->isIClause()) {
            pc = it->getIClause().getInternalRep();
          } else if (it->isUClause()) {
            pc = it->getUClause().getInternalRep();
          } else if (it->isSClause()) {
            pc = it->getSClause().getInternalRep();
          }
          for (PropositionalProver::Clause::iterator it2 = pc->begin();
               it2 != pc->end();
               ++it2)
            {
              (void) aps.insert(it2->getProposition().getName());
            }
        } else {
          (void)
       aps.insert(it->getEClause().getEventuality().getProposition().getName());
          for (EClause::const_present_iterator it2 =
                 it->getEClause().present_begin();
               it2 != it->getEClause().present_end();
               ++it2)
            {
              (void) aps.insert(it2->getProposition().getName());
            }
        }
      }

    /* Find unused var prefix */

    std::string unused_var_prefix = startPrefix;
    bool found = true;
    while (found) {
      found = false;
      for (StringSet::iterator it = aps.begin();
           !found && it != aps.end();
           ++it)
        {
          found = it->find(unused_var_prefix) != std::string::npos;
        }
      if (found) {
        unused_var_prefix += "0";
      }
    }

    return unused_var_prefix;
  } // UnsatCoreWriter::get_unused_var_prefix
} // namespace UnsatCore
