/************************************************************
 *    Copyright (C) 2012-2014                               *
 *    Viktor Schuppan (Viktor.Schuppan@gmx.de)              *
 *                                                          *
 *    This program is free software; you can redistribute   *
 *    it and/or modify it under the terms of the GNU        *
 *    General Public License as published by the Free       *
 *    Software Foundation, either version 3 of the License, *
 *    or (at your option) any later version.                *
 *                                                          *
 *    This program is distributed in the hope that it will  *
 *    be useful, but WITHOUT ANY WARRANTY; without even     *
 *    the implied warranty of MERCHANTABILITY or FITNESS    *
 *    FOR A PARTICULAR PURPOSE.  See the GNU General Public *
 *    License for more details.                             *
 *                                                          *
 *    You should have received a copy of the GNU General    *
 *    Public License along with this program.  If not, see  *
 *    <http://www.gnu.org/licenses/>.                       *
 ************************************************************/
#include <iostream>
#include <boost/any.hpp>
#include "misc/assert.h"
#include "stl_tim/pclause.h"
#include "unsat_core_private.h"
#include "clause_handle.h"
#include "misc.h"

namespace UnsatCore
{
  ClauseHandle::ClauseHandle()
    : myClause(boost::any()), myPartition(invalidPartition)
  { }

  ClauseHandle::ClauseHandle(IClause const & iclause, Partition const & partition)
    : myClause(iclause), myPartition(partition)
  { }

  ClauseHandle::ClauseHandle(UClause const & uclause, Partition const & partition)
    : myClause(uclause), myPartition(partition)
  { }

  ClauseHandle::ClauseHandle(SClause const & sclause, Partition const & partition)
    : myClause(sclause), myPartition(partition)
  { }

  ClauseHandle::ClauseHandle(EClause const & eclause, Partition const & partition)
    : myClause(eclause), myPartition(partition)
  { }

  ClauseHandle::ClauseHandle(PClause const & pclause, Partition const & partition)
    : myPartition(partition)
  {
    Assert(isPClauseInitial(pclause) ||
           isPClauseUniversal(pclause) ||
           isPClauseStep(pclause),
           "Not an initial, universal, or step clause.");
    if (isPClauseInitial(pclause))
      myClause = IClause(pclause);
    else if (isPClauseUniversal(pclause))
      myClause = UClause(pclause);
    else
      myClause = SClause(pclause);
  } // ClauseHandle::ClauseHandle

  bool
  ClauseHandle::isIClause() const
  {
    return myClause.type() == typeid(IClause);
  } // ClauseHandle::isIClause

  bool
  ClauseHandle::isUClause() const
  {
    return myClause.type() == typeid(UClause);
  } // ClauseHandle::isUClause

  bool
  ClauseHandle::isSClause() const
  {
    return myClause.type() == typeid(SClause);
  } // ClauseHandle::isSClause

  bool
  ClauseHandle::isEClause() const
  {
    return myClause.type() == typeid(EClause);
  } // ClauseHandle::isEClause

  // Not the logically empty clause but an empty boost::any()
  bool
  ClauseHandle::isEmpty() const
  {
    return myClause.empty();
  } // ClauseHandle::isEmpty

  IClause
  ClauseHandle::getIClause() const
  {
    Assert(isIClause(), "Not an IClause.");
    return boost::any_cast<IClause>(myClause);
  } // ClauseHandle::getIClause

  UClause
  ClauseHandle::getUClause() const
  {
    Assert(isUClause(), "Not a UClause.");
    return boost::any_cast<UClause>(myClause);
  } // ClauseHandle::getUClause

  SClause
  ClauseHandle::getSClause() const
  {
    Assert(isSClause(), "Not an SClause.");
    return boost::any_cast<SClause>(myClause);
  } // ClauseHandle::getSClause

  EClause
  ClauseHandle::getEClause() const
  {
    Assert(isEClause(), "Not an EClause.");
    return boost::any_cast<EClause>(myClause);
  } // ClauseHandle::getEClause

  Partition
  ClauseHandle::getPartition() const
  {
    return myPartition;
  } // ClauseHandle::getPartition

  PClause
  ClauseHandle::getPClause() const
  {
    if (isIClause())
      return getIClause().getInternalRep();
    else if (isUClause())
      return getUClause().getInternalRep();
    else if (isSClause())
      return getSClause().getInternalRep();
    else
      return getEClause().getC().getInternalRep();
  } // ClauseHandle::getPClause

  bool
  ClauseHandle::operator<(ClauseHandle const & other) const
  {
    if (myClause.empty()) {
      if (!other.myClause.empty())
        return true;
      else
        return myPartition < other.myPartition;
    }

    if (other.myClause.empty())
      return false;

    if ((!isEClause()) && (!other.isEClause())) {
      PClause myPClause = getPClause();
      PClause otherPClause = other.getPClause();
      if (myPClause < otherPClause)
        return true;
      else if (otherPClause < myPClause)
        return false;
      else
        return myPartition < other.myPartition;
    }

    if ((!isEClause()) && other.isEClause())
      return true;

    if (isEClause() && (!other.isEClause()))
      return false;

    Assert(isEClause() && other.isEClause(),
           "This or other not an EClause");
    EClause myEClause = getEClause();
    EClause otherEClause = other.getEClause();
    if (myEClause < otherEClause)
      return true;
    else if (otherEClause < myEClause)
      return false;
    else
      return myPartition < other.myPartition;
  } // ClauseHandle::operator<

  bool
  ClauseHandle::operator==(ClauseHandle const & other) const
  {
    if (myPartition != other.myPartition) {
      return false;
    }

    if (myClause.empty() != other.myClause.empty()) {
      return false;
    } else if (myClause.empty()) {
      Assert(myClause.empty() && other.myClause.empty(),
             "This or other not the empty clause");
      return true;
    }

    if (!isEClause() && !other.isEClause()) {
      PClause myPClause = getPClause();
      PClause otherPClause = other.getPClause();
      if (myPClause < otherPClause)
        return false;
      else if (otherPClause < myPClause)
        return false;
      else
        return true;
    }

    if (isEClause() != other.isEClause()) {
      return false;
    } else if (isEClause()) {
      Assert(isEClause() && other.isEClause(),
             "This or other not an EClause");
      EClause myEClause = getEClause();
      EClause otherEClause = other.getEClause();
      if (myEClause < otherEClause)
        return false;
      else if (otherEClause < myEClause)
        return false;
      else
        return true;
    }

    // Silence compiler
    Assert(false, "This point should not be reached.");
    return false;
  } // ClauseHandle::operator==

  std::ostream &
  ClauseHandle::dump(std::ostream & os) const
  {
    os << "(";
    if (isIClause() || isUClause() || isSClause()) {
      getPClause()->dump(os);
    } else {
      Assert(isEClause(), "Not an EClause.");
      getEClause().dump(os);
    }
    os << ", " << myPartition << ")";
    return os;
  } // ClauseHandle::dump
} // namespace UnsatCore
