/************************************************************
 *    Copyright (C) 2014                                    *
 *    Viktor Schuppan (Viktor.Schuppan@gmx.de)              *
 *                                                          *
 *    This program is free software; you can redistribute   *
 *    it and/or modify it under the terms of the GNU        *
 *    General Public License as published by the Free       *
 *    Software Foundation, either version 3 of the License, *
 *    or (at your option) any later version.                *
 *                                                          *
 *    This program is distributed in the hope that it will  *
 *    be useful, but WITHOUT ANY WARRANTY; without even     *
 *    the implied warranty of MERCHANTABILITY or FITNESS    *
 *    FOR A PARTICULAR PURPOSE.  See the GNU General Public *
 *    License for more details.                             *
 *                                                          *
 *    You should have received a copy of the GNU General    *
 *    Public License along with this program.  If not, see  *
 *    <http://www.gnu.org/licenses/>.                       *
 ************************************************************/
#ifndef __LITERAL_CLAUSEH_OCCURRENCE_H__
#define __LITERAL_CLAUSEH_OCCURRENCE_H__

#include <functional>
#include <iostream>
#include <unordered_map>
#include <set>
#include "clause_handle.h"
#include "lcho_partition.h"
#include "unsat_core_private.h"

namespace UnsatCore
{
  class LiteralClausehOccurrence
  {
  private:
    ClauseHandle myClauseh;
    bool myIsEventuality;
    Clause::const_iterator myPos;
  public:
    // for IClauses, UClauses, SClauses, and the non eventuality
    // literals of EClauses
    LiteralClausehOccurrence(ClauseHandle const & clauseh,
                             Clause::const_iterator const & pos);
    // for the eventuality literals of EClauses
    LiteralClausehOccurrence(ClauseHandle const & clauseh);
    ClauseHandle getClauseh() const;
    bool isEventuality() const;
    Clause::const_iterator getPos() const;
    Literal getLiteral() const;
    std::ostream & dump(std::ostream & os) const;
    bool operator<(LiteralClausehOccurrence const & other) const;
    bool operator==(LiteralClausehOccurrence const & other) const;
  private:
    LiteralClausehOccurrence(ClauseHandle const & clauseh,
                             bool isEventuality,
                             Clause::const_iterator const & pos);
  public:
    static LiteralClausehOccurrence makeLiteralClausehOccurrenceInvalidPos(
                                                  ClauseHandle const & clauseh);
  }; // class LiteralClausehOccurrence

  typedef std::set<LiteralClausehOccurrence> LiteralClausehOccurrenceSet;

  // here for now, would be better in lcho_partitioner.h, but need to
  // resolve circular #include with resolution_graph.h.
  typedef std::unordered_map<LiteralClausehOccurrence, LCHOPartition>
      MapLCHOToLCHOPartition;
} // namespace UnsatCore

namespace std
{
  template <>
    class hash<UnsatCore::LiteralClausehOccurrence>
    {
    public :
      size_t operator()(const UnsatCore::LiteralClausehOccurrence & lcho) const
      {
        UnsatCore::ClauseHandle ch = lcho.getClauseh();
        PropositionalProver::PClause pc = ch.getPClause();
        PropositionalProver::Clause::const_iterator it = lcho.getPos();

        if (it == pc->cend()) {
          return hash<UnsatCore::ClauseHandle>()(ch) ^
            hash<bool>()(lcho.isEventuality());
        } else {
          return hash<UnsatCore::ClauseHandle>()(ch) ^
            hash<bool>()(lcho.isEventuality()) ^
            hash<PropositionalProver::Literal>()(*it);
        }
      }
    };
};
#endif // __LITERAL_CLAUSEH_OCCURRENCE_H__
