/************************************************************
 *    Copyright (C) 2012-2014                               *
 *    Viktor Schuppan (Viktor.Schuppan@gmx.de)              *
 *                                                          *
 *    This program is free software; you can redistribute   *
 *    it and/or modify it under the terms of the GNU        *
 *    General Public License as published by the Free       *
 *    Software Foundation, either version 3 of the License, *
 *    or (at your option) any later version.                *
 *                                                          *
 *    This program is distributed in the hope that it will  *
 *    be useful, but WITHOUT ANY WARRANTY; without even     *
 *    the implied warranty of MERCHANTABILITY or FITNESS    *
 *    FOR A PARTICULAR PURPOSE.  See the GNU General Public *
 *    License for more details.                             *
 *                                                          *
 *    You should have received a copy of the GNU General    *
 *    Public License along with this program.  If not, see  *
 *    <http://www.gnu.org/licenses/>.                       *
 ************************************************************/
#include <fstream>
#include <iostream>
#include <list>
#include <map>
#include <set>
#include <sstream>
#include <string>
#include <boost/lexical_cast.hpp>
#include "builder_ltl/formula.h"
#include "getopt/options.h"
#include "misc/assert.h"
#include "stl_tim/attribute.h"
#include "stl_tim/clause.h"
#include "stl_tim/pclause.h"
#include "stl_tim/proposition.h"
#include "clause_handle.h"
#include "lcho_partition.h"
#include "linear_set.h"
#include "literal_clauseh_occurrence.h"
#include "semi_linear_set.h"
#include "unsat_core_private.h"
#include "unsat_core_writer.h"

namespace UnsatCore
{
  void
  UnsatCoreWriter::dumpCore(ClauseHandleSet const & chs,
                         std::map<ClauseHandle, SemiLinearSet> const & mChToSls,
                            MapLCHOToLCHOPartition const & mLCHOToLCHOP,
                       std::map<LCHOPartition, unsigned long> const & mLCHOPToI,
                            std::list<TreeNode*> const & formulaList,
                            std::map<TreeNode*, SemiLinearSet> const & mSfToSls,
                            TRPPPOptions const &options)
  {
    Assert(options.isExtractUnsatCore(), "Unsat core extraction not enabled.");

    if (options.getExtractUnsatCoreFileName() == std::string("")) {
      std::cout << "The following is an unsatisfiable core:"
                << std::endl;
    } else {
      std::cout << "An unsatisfiable core is written to "
                << options.getExtractUnsatCoreFileName()
                << "."
                << std::endl;
    }

    {
      std::ofstream unsatCoreFile;
      if (options.getExtractUnsatCoreFileName() != std::string("")) {
        unsatCoreFile.open(options.getExtractUnsatCoreFileName().c_str());
        if (!unsatCoreFile.is_open()) {
          std::cout << "Could not open file for writing unsatisfiable core "
                    << options.getExtractUnsatCoreFileName()
                    << ". Writing to stdout."
                    << std::endl;
        }
      }
      std::ostream & os =
        (options.getExtractUnsatCoreFileName() != std::string("") &&
         unsatCoreFile.is_open()) ?
        unsatCoreFile :
        std::cout;

      {
        UnsatCoreWriter ucw(os,
                            chs,
                            mChToSls,
                            mLCHOToLCHOP,
                            mLCHOPToI,
                            formulaList,
                            mSfToSls,
                            options);
        ucw.dumpClausehs();
      }

      if (options.getExtractUnsatCoreFileName() != std::string("")) {
        unsatCoreFile.close();
      }
    }

    if (options.getExtractUnsatCoreFileName() == std::string("")) {
      std::cout << "End of unsatisfiable core." << std::endl;
    }            
  } // UnsatCoreWriter::dumpCore

  void
  UnsatCoreWriter::dumpCoreBenchmarking(ClauseHandleSet const & chs,
                         std::map<ClauseHandle, SemiLinearSet> const & mChToSls,
                                    MapLCHOToLCHOPartition const & mLCHOToLCHOP,
                       std::map<LCHOPartition, unsigned long> const & mLCHOPToI,
                                       std::list<TreeNode*> const & formulaList,
                            std::map<TreeNode*, SemiLinearSet> const & mSfToSls,
                                        TRPPPOptions const & options)
  {
    Assert(options.isExtractUnsatCore(), "Unsat core extraction not enabled.");

    Assert(!options.isNuSMVSyntaxUnsatCore(),
           "NuSMV syntax for unsat core extraction enabled.");

    dump_core_benchmarking_aux(chs,
                               mChToSls,
                               mLCHOToLCHOP,
                               mLCHOPToI,
                               formulaList,
                               mSfToSls,
                               options,
                               EXTRACT_UNSAT_CORE_FORMAT_INTERNAL);
    dump_core_benchmarking_aux(chs,
                               mChToSls,
                               mLCHOToLCHOP,
                               mLCHOPToI,
                               formulaList,
                               mSfToSls,
                               options,
                               EXTRACT_UNSAT_CORE_FORMAT_SNF);
    dump_core_benchmarking_aux(chs,
                               mChToSls,
                               mLCHOToLCHOP,
                               mLCHOPToI,
                               formulaList,
                               mSfToSls,
                               options,
                               EXTRACT_UNSAT_CORE_FORMAT_LTLC);
    dump_core_benchmarking_aux(chs,
                               mChToSls,
                               mLCHOToLCHOP,
                               mLCHOPToI,
                               formulaList,
                               mSfToSls,
                               options,
                               EXTRACT_UNSAT_CORE_FORMAT_LTL);
  } // UnsatCoreWriter::dumpCoreBenchmarking

  UnsatCoreWriter::UnsatCoreWriter(std::ostream & os,
                                   ClauseHandleSet const & chs,
                         std::map<ClauseHandle, SemiLinearSet> const & mChToSls,
                                   MapLCHOToLCHOPartition const & mLCHOToLCHOP,
                       std::map<LCHOPartition, unsigned long> const & mLCHOPToI,
                                   std::list<TreeNode*> const & formulaList,
                            std::map<TreeNode*, SemiLinearSet> const & mSfToSls,
                                   TRPPPOptions const & options)
    : myOs(&os),
      myChs(chs),
      myMChToSls(mChToSls),
      myMLCHOToLCHOP(mLCHOToLCHOP),
      myMLCHOPToI(mLCHOPToI),
      myFormulaList(formulaList),
      myMSfToSls(mSfToSls),
      myOptions(options),
      myNvvVarPrefix(""),
      myPartitionVarSuffix(""),
      myApsToDeclare()
  { } // UnsatCoreWriter::UnsatCoreWriter

  void
  UnsatCoreWriter::dumpClausehs()
  {
    Assert(!myOptions.isNuSMVSyntaxUnsatCore() ||
           (myOptions.getExtractUnsatCoreFormat() !=
            EXTRACT_UNSAT_CORE_FORMAT_INTERNAL &&
            (!myOptions.isExtractUnsatCoreSetsOfTimePoints() ||
             myOptions.getExtractUnsatCoreFormat() ==
             EXTRACT_UNSAT_CORE_FORMAT_LTLC)),
           "Unexpected value of myOptions.");
    if (myOptions.isNuSMVSyntaxUnsatCore()) {
      TRUE_STR = "TRUE";
      FALSE_STR = "FALSE";
      NOT_SNF_STR = "!";
      NOT_STR = "!";
      NEXT_STR = "X";
      ALWAYS_STR = "G";
      SOMETIME_STR = "F";
      UNTIL_STR = "U";
      UNLESS_STR = "V";
    }
    if (myOptions.getExtractUnsatCoreFormat() ==
        EXTRACT_UNSAT_CORE_FORMAT_INTERNAL) {
      if (!myOptions.isExtractUnsatCoreSetsOfTimePoints() &&
          !myOptions.isExtractUnsatCorePartitionedPropositions()) {
        dumpClausehsSimplePartitionedInternal(false);
      } else if (myOptions.isExtractUnsatCoreSetsOfTimePoints() &&
                 !myOptions.isExtractUnsatCorePartitionedPropositions()) {
        dumpClausehsSetsOfTimePointsInternal();
      } else if (!myOptions.isExtractUnsatCoreSetsOfTimePoints() &&
                 myOptions.isExtractUnsatCorePartitionedPropositions()) {
        dumpClausehsSimplePartitionedInternal(true);
      } else {
        Assert(false, "False.");
      }
    } else if (myOptions.getExtractUnsatCoreFormat() ==
               EXTRACT_UNSAT_CORE_FORMAT_SNF) {
      if (!myOptions.isExtractUnsatCoreSetsOfTimePoints() &&
          !myOptions.isExtractUnsatCorePartitionedPropositions()) {
        dumpClausehsSimplePartitionedSNF(false);
      } else if (myOptions.isExtractUnsatCoreSetsOfTimePoints() &&
                 !myOptions.isExtractUnsatCorePartitionedPropositions()) {
        dumpClausehsSetsOfTimePointsSNF();
      } else if (!myOptions.isExtractUnsatCoreSetsOfTimePoints() &&
                 myOptions.isExtractUnsatCorePartitionedPropositions()) {
        dumpClausehsSimplePartitionedSNF(true);
      } else {
        Assert(false, "False.");
      }
    } else if (myOptions.getExtractUnsatCoreFormat() ==
               EXTRACT_UNSAT_CORE_FORMAT_LTLC) {
      if (!myOptions.isExtractUnsatCoreSetsOfTimePoints() &&
          !myOptions.isExtractUnsatCorePartitionedPropositions()) {
        dumpClausehsSimplePartitionedLTLC(false);
      } else if (myOptions.isExtractUnsatCoreSetsOfTimePoints() &&
                 !myOptions.isExtractUnsatCorePartitionedPropositions()) {
        dumpClausehsSetsOfTimePointsLTLC();
      } else if (!myOptions.isExtractUnsatCoreSetsOfTimePoints() &&
                 myOptions.isExtractUnsatCorePartitionedPropositions()) {
        dumpClausehsSimplePartitionedLTLC(true);
      } else {
        Assert(false, "False.");
      }
    } else {
      Assert(myOptions.getExtractUnsatCoreFormat() ==
             EXTRACT_UNSAT_CORE_FORMAT_LTL,
             "Unexpected value of myOptions.getExtractUnsatCoreFormat().");
      if (!myOptions.isExtractUnsatCoreSetsOfTimePoints()) {
        dumpClausehsSimplePartitionedLTL();
      } else if (myOptions.isExtractUnsatCoreSetsOfTimePoints() &&
                 !myOptions.isExtractUnsatCorePartitionedPropositions()) {
        dumpClausehsSetsOfTimePointsLTL();
      } else {
        Assert(false, "False.");
      }
    }
  } // UnsatCoreWriter::dumpClausehs

  void
  UnsatCoreWriter::dumpClausehsSimplePartitionedInternal(bool isPartitioned)
  {
    Assert(!myOptions.isNuSMVSyntaxUnsatCore(),
           "NuSMV syntax for unsat core extraction enabled.");

    ClauseHandleSet::iterator it;
    for (it = myChs.begin(); it != myChs.end(); ++it)
      {
        if (!isPartitioned) {
          it->dump(*myOs);
        } else {
          dump_clauseh_partitioned_internal(*it);
        }
        *myOs << std::endl;
      }
  } // UnsatCoreWriter::dumpClausehsSimplePartitionedInternal

  void
  UnsatCoreWriter::dumpClausehsSimplePartitionedSNF(bool isPartitioned)
  {
    std::ostream * myOsSaved = myOs;
    std::ostringstream oss;
    myOs = &oss;
    
    if (isPartitioned) {
      myPartitionVarSuffix = getUnusedVarInfix(myChs, "HASH");
    }

    ClauseHandleSet::iterator it;
    for (it = myChs.begin(); it != myChs.end();)
      {
        dumpClausehSNF(*it, isPartitioned);
        ++it;
        if (it != myChs.end()) {
          if (!myOptions.isNuSMVSyntaxUnsatCore()) {
            *myOs << ",";
          } else {
            *myOs << " &";
          }
        }
        *myOs << std::endl;
      }

    if (isPartitioned) {
      myPartitionVarSuffix = "";
    }

    myOs = myOsSaved;

    if (!myOptions.isNuSMVSyntaxUnsatCore()) {
      *myOs << "and([" << std::endl;
      *myOs << oss.str();
      *myOs << "])." << std::endl;
    } else {
      dump_nusmv_head();
      *myOs << oss.str();
      dump_nusmv_tail();
    }
  } // UnsatCoreWriter::dumpClausehsSimplePartitionedSNF

  void
  UnsatCoreWriter::dumpClausehsSimplePartitionedLTLC(bool isPartitioned)
  {
    std::ostream * myOsSaved = myOs;
    std::ostringstream oss;
    myOs = &oss;
    
    myNvvVarPrefix = getUnusedVarPrefix(myChs, "nvv");
    if (isPartitioned) {
      myPartitionVarSuffix = getUnusedVarInfix(myChs, "HASH");
    }

    ClauseHandleSet::iterator it;
    for (it = myChs.begin(); it != myChs.end();)
      {
        dump_clauseh_LTLC(*it, isPartitioned);
        ++it;
        if (it != myChs.end()) {
          *myOs << " & ";
        }
        *myOs << std::endl;
      }

    myNvvVarPrefix = "";
    if (isPartitioned) {
      myPartitionVarSuffix = "";
    }

    myOs = myOsSaved;

    if (!myOptions.isNuSMVSyntaxUnsatCore()) {
      *myOs << oss.str();
    } else {
      dump_nusmv_head();
      *myOs << oss.str();
      dump_nusmv_tail();
    }
  } // UnsatCoreWriter::dumpClausehsSimplePartitionedLTLC

  void
  UnsatCoreWriter::dumpClausehsSimplePartitionedLTL()
  {
    std::ostream * myOsSaved = myOs;
    std::ostringstream oss;
    myOs = &oss;
    
    for (std::list<TreeNode *>::const_iterator it = myFormulaList.begin();
         it != myFormulaList.end();
         )
      {
        *myOs << "(";
        dump_clausehs_simple_LTL_rec(*it, true);
        *myOs << ")";
        ++it;
        if (it != myFormulaList.end()) {
          *myOs << " & ";
        }
        *myOs << std::endl;
      }

    myOs = myOsSaved;

    if (!myOptions.isNuSMVSyntaxUnsatCore()) {
      *myOs << oss.str();
    } else {
      dump_nusmv_head();
      *myOs << oss.str();
      dump_nusmv_tail();
    }
  } // UnsatCoreWriter::dumpClausehsSimplePartitionedLTL

  void
  UnsatCoreWriter::dumpClausehsSetsOfTimePointsInternal()
  {
    Assert(!myOptions.isNuSMVSyntaxUnsatCore(),
           "NuSMV syntax for unsat core extraction enabled.");

    ClauseHandleSet::iterator it;
    for (it = myChs.begin(); it != myChs.end(); ++it)
      {
        Assert(myMChToSls.find(*it) != myMChToSls.end(),
               "*it not present in myMChToSls.");
        it->dump(*myOs);
        *myOs << " @ ";
        myMChToSls.find(*it)->second.dump(*myOs);
        *myOs << std::endl;
      }
  } // UnsatCoreWriter::dumpClausehsSetsOfTimePointsInternal

  void
  UnsatCoreWriter::dumpClausehsSetsOfTimePointsSNF()
  {
    Assert(!myOptions.isNuSMVSyntaxUnsatCore(),
           "NuSMV syntax for unsat core extraction enabled.");

    *myOs << "and([" << std::endl;
    ClauseHandleSet::iterator it;
    for (it = myChs.begin(); it != myChs.end();)
      {
        Assert(myMChToSls.find(*it) != myMChToSls.end(),
               "*it not present in myMChToSls.");
        dumpClausehSNF(*it, false);
        *myOs << " @ ";
        myMChToSls.find(*it)->second.dump(*myOs);
        ++it;
        if (it != myChs.end()) {
          *myOs << ",";
        }
        *myOs << std::endl;
      }
    *myOs << "])." << std::endl;
  } // UnsatCoreWriter::dumpClausehsSetsOfTimePointsSNF

  void
  UnsatCoreWriter::dumpClausehsSetsOfTimePointsLTLC()
  {
    std::ostream * myOsSaved = myOs;
    std::ostringstream oss;
    myOs = &oss;
    
    std::string fresh_var_prefix = getUnusedVarPrefix(myChs, "fresh");
    int fresh_var_cnt = 1;

    myNvvVarPrefix = getUnusedVarPrefix(myChs, "nvv");

    ClauseHandleSet::iterator it;
    for (it = myChs.begin(); it != myChs.end();)
      {
        Assert(myMChToSls.find(*it) != myMChToSls.end(),
               "*it not present in myMChToSls.");
        SemiLinearSet sls = myMChToSls.find(*it)->second;

        if (it->isIClause()) {
          Assert(sls.size() == 1 && sls.find(LinearSet(0, 0)) != sls.end(),
                 "sls of an initial clause is not {(0 N, 0)}.");
          dump_clauseh_LTLC(*it, false);
          *myOs << std::endl;
        } else {
          Assert(it->isUClause() || it->isSClause() || it->isEClause(),
                 "*it is not a UClause, SClause, or EClause.");
          Assert(sls.size() > 0,
                 "sls of a universal, step, or eventuality clause is empty.");
          for (SemiLinearSet::iterator it2 = sls.begin(); it2 != sls.end();) {
            *myOs << "(";
            unsigned long p = it2->getPeriod();
            unsigned long o = it2->getOffset();
            for (unsigned long i = 0; i < o; ++i) {
              *myOs << NEXT_STR
                    << "(";
            }
            if (p == 0) {
              dump_clauseh_body_LTLC(*it, false);
            } else {
              std::string fresh_var = "";
              fresh_var += fresh_var_prefix;
              fresh_var += boost::lexical_cast<std::string>(fresh_var_cnt);
              (void) myApsToDeclare.insert(fresh_var);
              *myOs << fresh_var
                    << " & ";
              if (myOptions.isNuSMVSyntaxUnsatCore()) {
                *myOs << "(";
              }
              *myOs << ALWAYS_STR
                    << "("
                    << fresh_var
                 << " -> ";
              if (myOptions.isNuSMVSyntaxUnsatCore()) {
                *myOs << "(";
              }
              for (unsigned long i = 0; i < p; ++i) {
                *myOs << NEXT_STR
                      << "(";
              }
              *myOs << fresh_var;
              for (unsigned long i = 0; i < p; ++i) {
                *myOs << ")";
              }
              if (myOptions.isNuSMVSyntaxUnsatCore()) {
                *myOs << ")";
              }
              *myOs << ")";
              if (myOptions.isNuSMVSyntaxUnsatCore()) {
                *myOs << ")";
              }
              *myOs << " & ";
              if (myOptions.isNuSMVSyntaxUnsatCore()) {
                *myOs << "(";
              }
              *myOs << ALWAYS_STR
                    << "("
                    << fresh_var
                    << " -> (";
              dump_clauseh_body_LTLC(*it, false);
              *myOs << "))";
              if (myOptions.isNuSMVSyntaxUnsatCore()) {
                *myOs << ")";
              }
              ++fresh_var_cnt;
            }
            for (unsigned long i = 0; i < o; ++i) {
              *myOs << ")";
            }
            *myOs << ")";
            ++it2;
            if (it2 != sls.end()) {
              *myOs << " & ";
            }
            *myOs << std::endl;
          }
        }
        ++it;
        if (it != myChs.end()) {
          *myOs << " & ";
        }
        *myOs << std::endl;
      }

    myNvvVarPrefix = "";

    myOs = myOsSaved;

    if (!myOptions.isNuSMVSyntaxUnsatCore()) {
      *myOs << oss.str();
    } else {
      dump_nusmv_head();
      *myOs << oss.str();
      dump_nusmv_tail();
    }
  } // UnsatCoreWriter::dumpClausehsSetsOfTimePointsLTLC

  void
  UnsatCoreWriter::dumpClausehsSetsOfTimePointsLTL()
  {
    Assert(!myOptions.isNuSMVSyntaxUnsatCore(),
           "NuSMV syntax for unsat core extraction enabled.");

    for (std::list<TreeNode *>::const_iterator it = myFormulaList.begin();
         it != myFormulaList.end();
         )
      {
        *myOs << "(";
        dump_clausehs_sets_of_time_points_LTL_rec(*it, true);
        *myOs << ")";
        ++it;
        if (it != myFormulaList.end()) {
          *myOs << " {(0 N, 0)}@_&_@{(0 N, 0)} ";
        }
        *myOs << std::endl;
      }
  } // UnsatCoreWriter::dumpClausehsSetsOfTimePointsLTL

  void
  UnsatCoreWriter::dumpClausehSNF(ClauseHandle const & ch, bool isPartitioned)
  {
    if (myOptions.isNuSMVSyntaxUnsatCore()) {
      *myOs << "(";
    }
    if (!ch.isIClause()) {
      Assert(ch.isUClause() || ch.isSClause() || ch.isEClause(),
             "ch is not a UClause, SClause, or EClause.");
      *myOs << ALWAYS_STR
            << "(";
    }
    if (!myOptions.isNuSMVSyntaxUnsatCore()) {
      *myOs << "or([";
    }
    if (ch.isIClause()) {
      IClause::const_iterator it;
      for (it = ch.getIClause().begin(); it != ch.getIClause().end();)
        {
          Assert(it->getAttribute().getType() == Attribute::initial,
                 "Attribute not initial");
          if (!isPartitioned) {
            dump_literal(*it);
          } else {
            LiteralClausehOccurrence lcho = LiteralClausehOccurrence(ch, it);
            dump_lcho_partitioned_SNF(lcho);
          }
          it++;
          if (it != ch.getIClause().end()) {
            if (!myOptions.isNuSMVSyntaxUnsatCore()) {
              *myOs << ",";
            } else {
              *myOs << " | ";
            }
          }
        }
    } else if (ch.isUClause()) {
      UClause::const_iterator it;
      for (it = ch.getUClause().begin(); it != ch.getUClause().end();)
        {
          Assert(it->getAttribute().getType() == Attribute::universal ||
                 it->getAttribute().getType() == Attribute::step_now,
                 "Attribute not universal or step_now");
          if (!isPartitioned) {
            dump_literal(*it);
          } else {
            LiteralClausehOccurrence lcho = LiteralClausehOccurrence(ch, it);
            dump_lcho_partitioned_SNF(lcho);
          }
          it++;
          if (it != ch.getUClause().end()) {
            if (!myOptions.isNuSMVSyntaxUnsatCore()) {
              *myOs << ",";
            } else {
              *myOs << " | ";
            }
          }
        }
    } else if (ch.isSClause()) {
      PClause pc = ch.getSClause().getInternalRep();
      Clause::iterator it;
      bool first = true;

      // First only step_now
      for (it = pc->begin(); it != pc->end(); ++it)
        {
          Assert(it->getAttribute().getType() == Attribute::step_now ||
                 it->getAttribute().getType() == Attribute::step_next,
                 "Attribute not step_now or step_next.");
          if (it->getAttribute().getType() == Attribute::step_now) {
            if (!first) {
              if (!myOptions.isNuSMVSyntaxUnsatCore()) {
                *myOs << ",";
              } else {
                *myOs << " | ";
              }
            }
            first = false;
            if (!isPartitioned) {
              dump_literal(*it);
            } else {
              LiteralClausehOccurrence lcho = LiteralClausehOccurrence(ch, it);
              dump_lcho_partitioned_SNF(lcho);
            }
          }
        }

      // Second only step_next
      for (it = pc->begin(); it != pc->end(); ++it)
        {
          if (it->getAttribute().getType() == Attribute::step_next) {
            if (!first) {
              if (!myOptions.isNuSMVSyntaxUnsatCore()) {
                *myOs << ",";
              } else {
                *myOs << " | ";
              }
            }
            first = false;
            if (myOptions.isNuSMVSyntaxUnsatCore()) {
              *myOs << "(";
            }
            *myOs << NEXT_STR;
            if (!myOptions.isNuSMVSyntaxUnsatCore()) {
              *myOs << "(";
            }
            if (!isPartitioned) {
              dump_literal(*it);
            } else {
              LiteralClausehOccurrence lcho = LiteralClausehOccurrence(ch, it);
              dump_lcho_partitioned_SNF(lcho);
            }
            *myOs << ")";
          }
        }
    } else {
      Assert(ch.isEClause(), "ch is not an EClause.");
      EClause::const_present_iterator it;
      for (it = ch.getEClause().present_begin();
           it != ch.getEClause().present_end();
           ++it)
        {
          Assert(it->getAttribute().getType() == Attribute::universal,
                 "Attribute not universal");
          if (!isPartitioned) {
            dump_literal(*it);
          } else {
            LiteralClausehOccurrence lcho = LiteralClausehOccurrence(ch, it);
            dump_lcho_partitioned_SNF(lcho);
          }
          if (!myOptions.isNuSMVSyntaxUnsatCore()) {
            *myOs << ",";
          } else {
            *myOs << " | ";
          }
        }
      if (myOptions.isNuSMVSyntaxUnsatCore()) {
        *myOs << "(";
      }
      *myOs << SOMETIME_STR;
      if (!myOptions.isNuSMVSyntaxUnsatCore()) {
        *myOs << "(";
      }
      if (!isPartitioned) {
        dump_literal(ch.getEClause().getEventuality());
      } else {
        LiteralClausehOccurrence lcho = LiteralClausehOccurrence(ch);
        dump_lcho_partitioned_SNF(lcho);
      }
      *myOs << ")";
    }
    if (!myOptions.isNuSMVSyntaxUnsatCore()) {
      *myOs << "])";
    }
    if (!ch.isIClause()) {
      Assert(ch.isUClause() || ch.isSClause() || ch.isEClause(),
             "ch is not a UClause, SClause, or EClause.");
      *myOs << ")";
    }
    if (myOptions.isNuSMVSyntaxUnsatCore()) {
      *myOs << ")";
    }
  } // UnsatCoreWriter::dumpClausehSNF

  void
  UnsatCoreWriter::dump_core_benchmarking_aux(ClauseHandleSet const & chs,
                         std::map<ClauseHandle, SemiLinearSet> const & mChToSls,
                                    MapLCHOToLCHOPartition const & mLCHOToLCHOP,
                       std::map<LCHOPartition, unsigned long> const & mLCHOPToI,
                                       std::list<TreeNode*> const & formulaList,
                            std::map<TreeNode*, SemiLinearSet> const & mSfToSls,
                                              TRPPPOptions const & options,
                                              ExtractUnsatCoreFormat format)
  {
    std::ofstream unsatCoreFile;
    ExtractUnsatCoreAlgorithm alg = options.getExtractUnsatCoreAlgorithm();
    std::string algs = (alg == EXTRACT_UNSAT_CORE_ALGORITHM_PROOF ? "proof" :
                        (alg == EXTRACT_UNSAT_CORE_ALGORITHM_DELETION ? "deletion" :
                         (alg == EXTRACT_UNSAT_CORE_ALGORITHM_PROOFDELETION ?
                          "proofdeletion" :
                          "unknown")));
    std::string fn = options.getFileName();
    std::string s;

    if (!options.isExtractUnsatCoreSetsOfTimePoints() &&
        !options.isExtractUnsatCorePartitionedPropositions()) {
      s = ".simple.";
    } else if (options.isExtractUnsatCoreSetsOfTimePoints() &&
               !options.isExtractUnsatCorePartitionedPropositions()) {
      s = ".setsoftimepoints.";
    } else if (!options.isExtractUnsatCoreSetsOfTimePoints() &&
               options.isExtractUnsatCorePartitionedPropositions()) {
      s = ".partitioned.";
    } else {
      Assert(false, "False.");
    }
    switch (format) {
    case EXTRACT_UNSAT_CORE_FORMAT_INTERNAL:
      fn += s + algs + ".internal";
      break;
    case EXTRACT_UNSAT_CORE_FORMAT_SNF:
      fn += s + algs + ".snf";
      break;
    case EXTRACT_UNSAT_CORE_FORMAT_LTLC:
      fn += s + algs + ".ltlc";
      break;
    default:
      Assert(format == EXTRACT_UNSAT_CORE_FORMAT_LTL,
             "Unexpected value of format.");
      fn += s + algs + ".ltl";
      break;
    }
    unsatCoreFile.open(fn.c_str());
    if (!unsatCoreFile.is_open()) {
      std::cout << "Could not open file for writing unsatisfiable core "
                << fn
                << ". Writing to stdout."
                << std::endl;
    }
    std::ostream & os = unsatCoreFile.is_open() ? unsatCoreFile : std::cout;
    UnsatCoreWriter ucw(os,
                        chs,
                        mChToSls,
                        mLCHOToLCHOP,
                        mLCHOPToI,
                        formulaList,
                        mSfToSls,
                        options);
    switch (format) {
    case EXTRACT_UNSAT_CORE_FORMAT_INTERNAL:
      if (!options.isExtractUnsatCoreSetsOfTimePoints() &&
          !options.isExtractUnsatCorePartitionedPropositions()) {
        ucw.dumpClausehsSimplePartitionedInternal(false);
      } else if (options.isExtractUnsatCoreSetsOfTimePoints() &&
                 !options.isExtractUnsatCorePartitionedPropositions()) {
        ucw.dumpClausehsSetsOfTimePointsInternal();
      } else if (!options.isExtractUnsatCoreSetsOfTimePoints() &&
                 options.isExtractUnsatCorePartitionedPropositions()) {
        ucw.dumpClausehsSimplePartitionedInternal(true);
      } else {
        Assert(false, "False.");
      }
      break;
    case EXTRACT_UNSAT_CORE_FORMAT_SNF:
      if (!options.isExtractUnsatCoreSetsOfTimePoints() &&
          !options.isExtractUnsatCorePartitionedPropositions()) {
        ucw.dumpClausehsSimplePartitionedSNF(false);
      } else if (options.isExtractUnsatCoreSetsOfTimePoints() &&
                 !options.isExtractUnsatCorePartitionedPropositions()) {
        ucw.dumpClausehsSetsOfTimePointsSNF();
      } else if (!options.isExtractUnsatCoreSetsOfTimePoints() &&
                 options.isExtractUnsatCorePartitionedPropositions()) {
        ucw.dumpClausehsSimplePartitionedSNF(true);
      } else {
        Assert(false, "False.");
      }
      break;
    case EXTRACT_UNSAT_CORE_FORMAT_LTLC:
      if (!options.isExtractUnsatCoreSetsOfTimePoints() &&
          !options.isExtractUnsatCorePartitionedPropositions()) {
        ucw.dumpClausehsSimplePartitionedLTLC(false);
      } else if (options.isExtractUnsatCoreSetsOfTimePoints() &&
                 !options.isExtractUnsatCorePartitionedPropositions()) {
        ucw.dumpClausehsSetsOfTimePointsLTLC();
      } else if (!options.isExtractUnsatCoreSetsOfTimePoints() &&
                 options.isExtractUnsatCorePartitionedPropositions()) {
        ucw.dumpClausehsSimplePartitionedLTLC(true);
      } else {
        Assert(false, "False.");
      }
      break;
    default:
      Assert(format == EXTRACT_UNSAT_CORE_FORMAT_LTL,
             "Unexpected value of format.");
      if (!options.isExtractUnsatCoreSetsOfTimePoints()) {
        ucw.dumpClausehsSimplePartitionedLTL();
      } else if (options.isExtractUnsatCoreSetsOfTimePoints() &&
                 !options.isExtractUnsatCorePartitionedPropositions()) {
        ucw.dumpClausehsSetsOfTimePointsLTL();
      } else {
        Assert(false, "False.");
      }
      break;
    }
    unsatCoreFile.close();
  } // UnsatCoreWriter::dump_core_benchmarking_aux

  void
  UnsatCoreWriter::dump_literal(Literal const & l)
  {
    std::ostringstream oss;
    if (myOptions.isNuSMVSyntaxUnsatCore()) {
      *myOs << "(";
    }
    if (l.isNegative()) {
      *myOs << NOT_SNF_STR
            << " ";
    }
    if (l.getProposition().getName().find("NVV") == 0) {
      oss << myNvvVarPrefix;
    }
    l.getProposition().dump(oss);
    (void) myApsToDeclare.insert(oss.str());
    *myOs << oss.str();
    if (myOptions.isNuSMVSyntaxUnsatCore()) {
      *myOs << ")";
    }
  } // UnsatCoreWriter::dump_literal

  void
  UnsatCoreWriter::dump_clauseh_body_LTLC(ClauseHandle const & ch,
                                          bool isPartitioned)
  {
    if (ch.isIClause()) {
      IClause::const_iterator it;
      for (it = ch.getIClause().begin(); it != ch.getIClause().end();)
        {
          Assert(it->getAttribute().getType() == Attribute::initial,
                 "Attribute not initial");
          if (!myOptions.isNuSMVSyntaxUnsatCore()) {
            *myOs << "(";
          }
          if (!isPartitioned) {
            dump_literal(*it);
          } else {
            LiteralClausehOccurrence lcho = LiteralClausehOccurrence(ch, it);
            dump_lcho_partitioned_SNF(lcho);
          }
          if (!myOptions.isNuSMVSyntaxUnsatCore()) {
            *myOs << ")";
          }
          it++;
          if (it != ch.getIClause().end()) {
            *myOs << " | ";
          }
        }
    } else if (ch.isUClause()) {
      UClause::const_iterator it;
      for (it = ch.getUClause().begin(); it != ch.getUClause().end();)
        {
          Assert(it->getAttribute().getType() == Attribute::universal ||
                 it->getAttribute().getType() == Attribute::step_now,
                 "Attribute not universal or step_now");
          if (!myOptions.isNuSMVSyntaxUnsatCore()) {
            *myOs << "(";
          }
          if (!isPartitioned) {
            dump_literal(*it);
          } else {
            LiteralClausehOccurrence lcho = LiteralClausehOccurrence(ch, it);
            dump_lcho_partitioned_SNF(lcho);
          }
          if (!myOptions.isNuSMVSyntaxUnsatCore()) {
            *myOs << ")";
          }
          it++;
          if (it != ch.getUClause().end()) {
            *myOs << " | ";
          }
        }
    } else if (ch.isSClause()) {
      PClause pc = ch.getSClause().getInternalRep();
      Clause::iterator it;
      bool first = true;

      // First only step_now
      for (it = pc->begin(); it != pc->end(); ++it)
        {
          Assert(it->getAttribute().getType() == Attribute::step_now ||
                 it->getAttribute().getType() == Attribute::step_next,
                 "Attribute not step_now or step_next.");
          if (it->getAttribute().getType() == Attribute::step_now) {
            if (!first) {
              *myOs << " | ";
            }
            first = false;
            if (!myOptions.isNuSMVSyntaxUnsatCore()) {
              *myOs << "(";
            }
            if (!isPartitioned) {
              dump_literal(*it);
            } else {
              LiteralClausehOccurrence lcho = LiteralClausehOccurrence(ch, it);
              dump_lcho_partitioned_SNF(lcho);
            }
            if (!myOptions.isNuSMVSyntaxUnsatCore()) {
              *myOs << ")";
            }
          }
        }

      // Second only step_next
      for (it = pc->begin(); it != pc->end(); ++it)
        {
          if (it->getAttribute().getType() == Attribute::step_next) {
            if (!first) {
              *myOs << " | ";
            }
            first = false;
            *myOs << "("
                  << NEXT_STR;
            if (!myOptions.isNuSMVSyntaxUnsatCore()) {
              *myOs << "(";
            }
            if (!isPartitioned) {
              dump_literal(*it);
            } else {
              LiteralClausehOccurrence lcho = LiteralClausehOccurrence(ch, it);
              dump_lcho_partitioned_SNF(lcho);
            }
            if (!myOptions.isNuSMVSyntaxUnsatCore()) {
              *myOs << ")";
            }
            *myOs << ")";
          }
        }
    } else {
      Assert(ch.isEClause(), "ch is not an EClause.");
      EClause::const_present_iterator it;
      for (it = ch.getEClause().present_begin();
           it != ch.getEClause().present_end();
           ++it)
        {
          Assert(it->getAttribute().getType() == Attribute::universal,
                 "Attribute not universal");
          if (!myOptions.isNuSMVSyntaxUnsatCore()) {
            *myOs << "(";
          }
          if (!isPartitioned) {
            dump_literal(*it);
          } else {
            LiteralClausehOccurrence lcho = LiteralClausehOccurrence(ch, it);
            dump_lcho_partitioned_SNF(lcho);
          }
          if (!myOptions.isNuSMVSyntaxUnsatCore()) {
            *myOs << ")";
          }
          *myOs << " | ";
        }
      *myOs << "("
            << SOMETIME_STR;
      if (!myOptions.isNuSMVSyntaxUnsatCore()) {
        *myOs << "(";
      }
      if (!isPartitioned) {
        dump_literal(ch.getEClause().getEventuality());
      } else {
        LiteralClausehOccurrence lcho = LiteralClausehOccurrence(ch);
        dump_lcho_partitioned_SNF(lcho);
      }
      if (!myOptions.isNuSMVSyntaxUnsatCore()) {
        *myOs << ")";
      }
      *myOs << ")";
    }
  } // UnsatCoreWriter::dump_clauseh_body_LTLC

  void
  UnsatCoreWriter::dump_clauseh_LTLC(ClauseHandle const & ch,
                                     bool isPartitioned)
  {
    if (!ch.isIClause()) {
      Assert(ch.isUClause() || ch.isSClause() || ch.isEClause(),
             "ch is not a UClause, SClause, or EClause.");
      if (myOptions.isNuSMVSyntaxUnsatCore()) {
        *myOs << "(";
      }
      *myOs << ALWAYS_STR;
    }
    *myOs << "(";
    dump_clauseh_body_LTLC(ch, isPartitioned);
    *myOs << ")";
    if (!ch.isIClause() && myOptions.isNuSMVSyntaxUnsatCore()) {
      *myOs << ")";
    }
  } // UnsatCoreWriter::dump_clauseh_LTLC

  void
  UnsatCoreWriter::dump_clausehs_simple_LTL_rec(TreeNode * n,
                                                bool isPositivePolarity)
  {
    if (isIdentifier(n) && ((Identifier *) n)->isTrue()) {
      Assert(n->childrenCount() == 0, "Unexpected number of children.");
      *myOs << TRUE_STR;
    } else if (isIdentifier(n) && ((Identifier *) n)->isFalse()) {
      Assert(n->childrenCount() == 0, "Unexpected number of children.");
      *myOs << FALSE_STR;
    } else if (isIdentifier(n)) {
      Assert(n->childrenCount() == 0, "Unexpected number of children.");
      (void) myApsToDeclare.insert(((Identifier *) n)->getIdentifier());
      *myOs << ((Identifier *) n)->getIdentifier();
    } else if (isNot(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      *myOs << NOT_STR
            << " (";
      dump_clausehs_simple_LTL_rec(n->firstChild(), !isPositivePolarity);
      *myOs << ")";
    } else if (isAnd(n)) {
      Assert(n->childrenCount() > 0, "Unexpected number of children.");
      for (TreeNodeList::const_iterator it = n->children().begin();
           it != n->children().end();
           )
        {
          TreeNode * child = *it;
          ++it;
          *myOs << "(";
          dump_clausehs_simple_LTL_rec(child, isPositivePolarity);
          *myOs << ")";
          if (it != n->children().end()) {
            *myOs << " & ";
          }
        }
    } else if (isOr(n)) {
      Assert(n->childrenCount() > 0, "Unexpected number of children.");
      for (TreeNodeList::const_iterator it = n->children().begin();
           it != n->children().end();
           )
        {
          TreeNode * child = *it;
          ++it;
          *myOs << "(";
          dump_clausehs_simple_LTL_rec(child, isPositivePolarity);
          *myOs << ")";
          if (it != n->children().end()) {
            *myOs << " | ";
          }
        }
    } else if (isImplication(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      *myOs << "(";
      dump_clausehs_simple_LTL_rec(n->firstChild(), !isPositivePolarity);
      *myOs << ") -> (";
      dump_clausehs_simple_LTL_rec(n->secondChild(), isPositivePolarity);
      *myOs << ")";
    } else if (isEquivalence(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      *myOs << "(";
      dump_clausehs_simple_LTL_rec(n->firstChild(), isPositivePolarity);
      *myOs << ") <-> (";
      dump_clausehs_simple_LTL_rec(n->secondChild(), isPositivePolarity);
      *myOs << ")";
    } else if (isNext(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      *myOs << NEXT_STR
            << " (";
      dump_clausehs_simple_LTL_rec(n->firstChild(), isPositivePolarity);
      *myOs << ")";
    } else if (isAlways(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      *myOs << ALWAYS_STR
            << " (";
      dump_clausehs_simple_LTL_rec(n->firstChild(), isPositivePolarity);
      *myOs << ")";
    } else if (isSometime(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      *myOs << SOMETIME_STR
            << " (";
      dump_clausehs_simple_LTL_rec(n->firstChild(), isPositivePolarity);
      *myOs << ")";
    } else if (isUntil(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      *myOs << "(";
      dump_clausehs_simple_LTL_rec(n->firstChild(), isPositivePolarity);
      *myOs << ") "
            << UNTIL_STR
            << " (";
      dump_clausehs_simple_LTL_rec(n->secondChild(), isPositivePolarity);
      *myOs << ")";
    } else if (isUnless(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      *myOs << "(";
      if (!myOptions.isNuSMVSyntaxUnsatCore()) {
        dump_clausehs_simple_LTL_rec(n->firstChild(), isPositivePolarity);
      } else {
        dump_clausehs_simple_LTL_rec(n->secondChild(), isPositivePolarity);
      }
      *myOs << ") "
            << UNLESS_STR
            << " (";
      if (!myOptions.isNuSMVSyntaxUnsatCore()) {
        dump_clausehs_simple_LTL_rec(n->secondChild(), isPositivePolarity);
      } else {
        dump_clausehs_simple_LTL_rec(n->firstChild(), isPositivePolarity);
      }
      *myOs << ")";
    } else {
      Assert(false, "n has unexpected type.");
    }
  } // UnsatCoreWriter::dump_clausehs_simple_LTL_rec

  void
  UnsatCoreWriter::dump_clausehs_sets_of_time_points_LTL_rec(TreeNode * n,
                                                        bool isPositivePolarity)
  {
    if (isIdentifier(n) && ((Identifier *) n)->isTrue()) {
      Assert(n->childrenCount() == 0, "Unexpected number of children.");
      *myOs << "True";
    } else if (isIdentifier(n) && ((Identifier *) n)->isFalse()) {
      Assert(n->childrenCount() == 0, "Unexpected number of children.");
      *myOs << "False";
    } else if (isIdentifier(n)) {
      Assert(n->childrenCount() == 0, "Unexpected number of children.");
      *myOs << ((Identifier *) n)->getIdentifier();
    } else if (isNot(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      *myOs << "~_@";
      Assert(myMSfToSls.find(n->firstChild()) != myMSfToSls.end(),
             "n->firstChild() not present in myMSfToSls.");
      myMSfToSls.find(n->firstChild())->second.dump(*myOs);
      *myOs << " (";
      dump_clausehs_sets_of_time_points_LTL_rec(n->firstChild(),
                                                !isPositivePolarity);
      *myOs << ")";
    } else if (isAnd(n)) {
      Assert(n->childrenCount() > 0, "Unexpected number of children.");
      bool isFirst = true;
      for (TreeNodeList::const_iterator it = n->children().begin();
           it != n->children().end();
           )
        {
          TreeNode * child = *it;
          ++it;
          *myOs << "(";
          dump_clausehs_sets_of_time_points_LTL_rec(child, isPositivePolarity);
          *myOs << ")";
          if (it != n->children().end()) {
            if (isFirst) {
              isFirst = false;
              *myOs << " ";
              Assert(myMSfToSls.find(child) != myMSfToSls.end(),
                     "child not present in myMSfToSls.");
              myMSfToSls.find(child)->second.dump(*myOs);
              *myOs << "@_&_@";
              Assert(myMSfToSls.find(*it) != myMSfToSls.end(),
                     "*it not present in myMSfToSls.");
              myMSfToSls.find(*it)->second.dump(*myOs);
              *myOs << " ";
            } else {
              *myOs << " &_@";
              Assert(myMSfToSls.find(*it) != myMSfToSls.end(),
                     "*it not present in myMSfToSls.");
              myMSfToSls.find(*it)->second.dump(*myOs);
              *myOs << " ";
            }
          }
        }
    } else if (isOr(n)) {
      Assert(n->childrenCount() > 0, "Unexpected number of children.");
      bool isFirst = true;
      for (TreeNodeList::const_iterator it = n->children().begin();
           it != n->children().end();
           )
        {
          TreeNode * child = *it;
          ++it;
          *myOs << "(";
          dump_clausehs_sets_of_time_points_LTL_rec(child, isPositivePolarity);
          *myOs << ")";
          if (it != n->children().end()) {
            if (isFirst) {
              isFirst = false;
              *myOs << " ";
              Assert(myMSfToSls.find(child) != myMSfToSls.end(),
                     "child not present in myMSfToSls.");
              myMSfToSls.find(child)->second.dump(*myOs);
              *myOs << "@_|_@";
              Assert(myMSfToSls.find(*it) != myMSfToSls.end(),
                     "*it not present in myMSfToSls.");
              myMSfToSls.find(*it)->second.dump(*myOs);
              *myOs << " ";
            } else {
              *myOs << " |_@";
              Assert(myMSfToSls.find(*it) != myMSfToSls.end(),
                     "*it not present in myMSfToSls.");
              myMSfToSls.find(*it)->second.dump(*myOs);
              *myOs << " ";
            }
          }
        }
    } else if (isImplication(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      *myOs << "(";
      dump_clausehs_sets_of_time_points_LTL_rec(n->firstChild(),
                                                !isPositivePolarity);
      *myOs << ") ";
      Assert(myMSfToSls.find(n->firstChild()) != myMSfToSls.end(),
             "n->firstChild() not present in myMSfToSls.");
      myMSfToSls.find(n->firstChild())->second.dump(*myOs);
      *myOs << "@_->_@";
      Assert(myMSfToSls.find(n->secondChild()) != myMSfToSls.end(),
             "n->secondChild() not present in myMSfToSls.");
      myMSfToSls.find(n->secondChild())->second.dump(*myOs);
      *myOs << " (";
      dump_clausehs_sets_of_time_points_LTL_rec(n->secondChild(),
                                                isPositivePolarity);
      *myOs << ")";
    } else if (isNext(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      *myOs << "next_@";
      Assert(myMSfToSls.find(n->firstChild()) != myMSfToSls.end(),
             "n->firstChild() not present in myMSfToSls.");
      myMSfToSls.find(n->firstChild())->second.dump(*myOs);
      *myOs << " (";
      dump_clausehs_sets_of_time_points_LTL_rec(n->firstChild(),
                                                isPositivePolarity);
      *myOs << ")";
    } else if (isAlways(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      *myOs << "always_@";
      Assert(myMSfToSls.find(n->firstChild()) != myMSfToSls.end(),
             "n->firstChild() not present in myMSfToSls.");
      myMSfToSls.find(n->firstChild())->second.dump(*myOs);
      *myOs << " (";
      dump_clausehs_sets_of_time_points_LTL_rec(n->firstChild(),
                                                isPositivePolarity);
      *myOs << ")";
    } else if (isSometime(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
      *myOs << "sometime_@";
      Assert(myMSfToSls.find(n->firstChild()) != myMSfToSls.end(),
             "n->firstChild() not present in myMSfToSls.");
      myMSfToSls.find(n->firstChild())->second.dump(*myOs);
      *myOs << " (";
      dump_clausehs_sets_of_time_points_LTL_rec(n->firstChild(),
                                                isPositivePolarity);
      *myOs << ")";
    } else if (isUntil(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      *myOs << "(";
      dump_clausehs_sets_of_time_points_LTL_rec(n->firstChild(),
                                                isPositivePolarity);
      *myOs << ") ";
      Assert(myMSfToSls.find(n->firstChild()) != myMSfToSls.end(),
             "n->firstChild() not present in myMSfToSls.");
      myMSfToSls.find(n->firstChild())->second.dump(*myOs);
      *myOs << "@_until_@";
      Assert(myMSfToSls.find(n->secondChild()) != myMSfToSls.end(),
             "n->secondChild() not present in myMSfToSls.");
      myMSfToSls.find(n->secondChild())->second.dump(*myOs);
      *myOs << " (";
      dump_clausehs_sets_of_time_points_LTL_rec(n->secondChild(),
                                                isPositivePolarity);
      *myOs << ")";
    } else if (isUnless(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      *myOs << "(";
      dump_clausehs_sets_of_time_points_LTL_rec(n->firstChild(),
                                                isPositivePolarity);
      *myOs << ") ";
      Assert(myMSfToSls.find(n->firstChild()) != myMSfToSls.end(),
             "n->firstChild() not present in myMSfToSls.");
      myMSfToSls.find(n->firstChild())->second.dump(*myOs);
      *myOs << "@_unless_@";
      Assert(myMSfToSls.find(n->secondChild()) != myMSfToSls.end(),
             "n->secondChild() not present in myMSfToSls.");
      myMSfToSls.find(n->secondChild())->second.dump(*myOs);
      *myOs << " (";
      dump_clausehs_sets_of_time_points_LTL_rec(n->secondChild(),
                                                isPositivePolarity);
      *myOs << ")";
    } else {
      Assert(false, "n has unexpected type.");
    }
  } // UnsatCoreWriter::dump_clausehs_sets_of_time_points_LTL_rec

  // based on clause_handle.cc::dump, clause.h::dump, eclause.h::dump
  void
  UnsatCoreWriter::dump_clauseh_partitioned_internal(ClauseHandle const & ch)
  {
    *myOs << "(";
    if (ch.isIClause() || ch.isUClause() || ch.isSClause()) {
      PClause pc = ch.getPClause();
      *myOs << " {" << pc->getId() << ": " << pc->getInfo();
      for (Clause::const_iterator it = pc->begin(); it != pc->end(); ++it) {
        LiteralClausehOccurrence lcho = LiteralClausehOccurrence(ch, it);
        dump_lcho_partitioned_internal(lcho);
      }
      *myOs << "} ";
    } else {
      Assert(ch.isEClause(), "Not an EClause.");
      EClause ec = ch.getEClause();
      *myOs << "E:(";
      for (EClause::const_present_iterator it = ec.present_begin();
           it != ec.present_end();
           ++it) {
        LiteralClausehOccurrence lcho = LiteralClausehOccurrence(ch, it);
        dump_lcho_partitioned_internal(lcho);
        *myOs << " ";
      }
      *myOs << ", Sometime(";
      LiteralClausehOccurrence lchoe = LiteralClausehOccurrence(ch);
      dump_lcho_partitioned_internal(lchoe);
      *myOs << ") )" << std::endl;
    }
    *myOs << ", " << ch.getPartition() << ")";
  } // UnsatCoreWriter::dump_clauseh_partitioned_internal

  // based on literal.h::dump
  void
  UnsatCoreWriter::dump_lcho_partitioned_internal(
                                                LiteralClausehOccurrence & lcho)
  {
    Literal l;
    unsigned long i;

    l = lcho.getLiteral();
    *myOs << "( " << (l.getSign() ? "+" : "-") ;
    l.getProposition().dump(*myOs);
    Assert(myMLCHOToLCHOP.find(lcho) != myMLCHOToLCHOP.end(),
           "lcho not contained in myMLCHOToLCHOP.");
    LCHOPartition lchop = myMLCHOToLCHOP.find(lcho)->second;
    Assert(myMLCHOPToI.find(lchop) != myMLCHOPToI.end(),
           "lchop not contained in myMLCHOPToI.");
    i = myMLCHOPToI.find(lchop)->second;
    *myOs << "#" << i;
    *myOs << " : " << l.getAttribute() << " )";
  } // UnsatCoreWriter::dump_lcho_partitioned_internal

  void
  UnsatCoreWriter::dump_lcho_partitioned_SNF(LiteralClausehOccurrence & lcho)
  {
    std::ostringstream oss;
    Literal l;
    unsigned long i;

    if (myOptions.isNuSMVSyntaxUnsatCore()) {
      *myOs << "(";
    }
    l = lcho.getLiteral();
    if (l.isNegative()) {
      *myOs << NOT_SNF_STR
            << " ";
    }
    if (l.getProposition().getName().find("NVV") == 0) {
      oss << myNvvVarPrefix;
    }
    l.getProposition().dump(oss);
    oss << myPartitionVarSuffix;
    Assert(myMLCHOToLCHOP.find(lcho) != myMLCHOToLCHOP.end(),
           "lcho not contained in myMLCHOToLCHOP.");
    LCHOPartition lchop = myMLCHOToLCHOP.find(lcho)->second;
    Assert(myMLCHOPToI.find(lchop) != myMLCHOPToI.end(),
           "lchop not contained in myMLCHOPToI.");
    i = myMLCHOPToI.find(lchop)->second;
    oss << i;
    (void) myApsToDeclare.insert(oss.str());
    *myOs << oss.str();
    if (myOptions.isNuSMVSyntaxUnsatCore()) {
      *myOs << ")";
    }
  } // UnsatCoreWriter::dump_lcho_partitioned_SNF

  void
  UnsatCoreWriter::dump_nusmv_head()
  {
    Assert(myOptions.isNuSMVSyntaxUnsatCore(),
           "NuSMV syntax for unsat core extraction not enabled.");
    *myOs << "MODULE main" << std::endl;
    *myOs << "VAR" << std::endl;
    for (StringSet::iterator it = myApsToDeclare.begin();
         it != myApsToDeclare.end();
         ++it)
      {
        *myOs << "  "
              << *it
              << ": boolean;"
              << std::endl;
      }
    *myOs << "LTLSPEC" << std::endl;
    *myOs << "!(" << std::endl;
  } // UnsatCoreWriter::dump_nusmv_head

  void
  UnsatCoreWriter::dump_nusmv_tail()
  {
    Assert(myOptions.isNuSMVSyntaxUnsatCore(),
           "NuSMV syntax for unsat core extraction not enabled.");
    *myOs << ")" << std::endl;
  } // UnsatCoreWriter::dump_nusmv_tail

  std::string
  UnsatCoreWriter::getUnusedVarPrefix(ClauseHandleSet const & chs,
                                      std::string const & startPrefix)
  {
    StringSet aps = get_aps(chs);
    std::string unused_var_prefix = startPrefix;
    bool found = true;
    while (found) {
      found = false;
      for (StringSet::iterator it = aps.begin();
           !found && it != aps.end();
           ++it)
        {
          found = it->find(unused_var_prefix) != std::string::npos;
        }
      if (found) {
        unused_var_prefix += "0";
      }
    }
    return unused_var_prefix;
  } // UnsatCoreWriter::getUnusedVarPrefix

  std::string
  UnsatCoreWriter::getUnusedVarInfix(ClauseHandleSet const & chs,
                                     std::string const & startInfix)
  {
    StringSet aps = get_aps(chs);
    std::string unused_var_infix = startInfix;
    bool found = true;
    while (found) {
      found = false;
      for (StringSet::iterator it = aps.begin();
           !found && it != aps.end();
           ++it) {
        found = it->find(unused_var_infix) != std::string::npos;
      }
      if (found) {
        unused_var_infix += "0";
      }
    }
    return unused_var_infix;
  } // UnsatCoreWriter::getUnusedVarInfix

  UnsatCoreWriter::StringSet
  UnsatCoreWriter::get_aps(ClauseHandleSet const & chs)
  {
    StringSet aps = StringSet();

    for (ClauseHandleSet::iterator it = chs.begin(); it != chs.end(); ++it) {
      if (it->isEClause()) {
        (void)
          aps.insert(it->getEClause().getEventuality().getProposition().getName());
      }
      PClause pc = it->getPClause();
      for (Clause::iterator it2 = pc->begin();
           it2 != pc->end();
           ++it2) {
        (void) aps.insert(it2->getProposition().getName());
      }
    }

    return aps;
  } // UnsatCoreWriter::get_aps
} // namespace UnsatCore
