/************************************************************
 *    Copyright (C) 2012-2013                               *
 *    Viktor Schuppan (Viktor.Schuppan@gmx.de)              *
 *                                                          *
 *    This program is free software; you can redistribute   *
 *    it and/or modify it under the terms of the GNU        *
 *    General Public License as published by the Free       *
 *    Software Foundation, either version 3 of the License, *
 *    or (at your option) any later version.                *
 *                                                          *
 *    This program is distributed in the hope that it will  *
 *    be useful, but WITHOUT ANY WARRANTY; without even     *
 *    the implied warranty of MERCHANTABILITY or FITNESS    *
 *    FOR A PARTICULAR PURPOSE.  See the GNU General Public *
 *    License for more details.                             *
 *                                                          *
 *    You should have received a copy of the GNU General    *
 *    Public License along with this program.  If not, see  *
 *    <http://www.gnu.org/licenses/>.                       *
 ************************************************************/
#include <iostream>
#include <map>
#include "builder/builder.h"
#include "getopt/options.h"
#include "misc/assert.h"
#include "misc/tracer.h"
#include "modules.h"
#include "stl_tim/pclause.h"
#include "unsat_core/clause_handle.h"
#include "formula.h"
#include "simplifier.h"
#include "snf_translator.h"
#include "trp_ltl_parser.h"
#include "builder_ltl.h"

namespace BuilderLtl
{
  TreeNodeList
  BuilderLtl::parseLtl(TRPPPOptions const & options)
  {
    std::cout << "BuilderLtl: Parsing LTL formula begin."
              << std::endl;

    TreeNodeList formulaList;

    TrpLtlParser trpLtlParser(options);
    trpLtlParser.parseLtl();
    formulaList = trpLtlParser.getLtlFormulaList();

    TRACE(builderltlModule, {
        std::cout << "BuilderLtl: parsed LTL formula:"
                  << std::endl;
        for (TreeNodeList::const_iterator it = formulaList.begin();
             it != formulaList.end();
             ++it)
          {
            std::cout << *it << std::endl;
          }
      });

    std::cout << "BuilderLtl: Parsing LTL formula end."
              << std::endl;

    return formulaList;
  } // BuilderLtl::parseLtl

  void
  BuilderLtl::replaceEquivalencesLtl(TreeNodeList & formulaList)
  {
    std::cout << "BuilderLtl: Replacing equivalences in LTL formula begin."
              << std::endl;

    TRACE(builderltlModule, {
        std::cout << "BuilderLtl: LTL formula before replacing equivalences:"
                  << std::endl;
        for (TreeNodeList::const_iterator it = formulaList.begin();
             it != formulaList.end();
             ++it)
          {
            std::cout << *it << std::endl;
          }
      });

    replaceEquivalencesFormulasLtl(formulaList);

    TRACE(builderltlModule, {
        std::cout << "BuilderLtl: LTL formula after replacing equivalences:"
                  << std::endl;
        for (TreeNodeList::const_iterator it = formulaList.begin();
             it != formulaList.end();
             ++it)
          {
            std::cout << *it << std::endl;
          }
      });

    std::cout << "BuilderLtl: Replacing equivalences in LTL formula end."
              << std::endl;
  } // BuilderLtl::reqplaceEquivalencesLtl

  void
  BuilderLtl::nnfLtl(TreeNodeList & formulaList)
  {
    std::cout << "BuilderLtl: Transforming LTL formula to NNF begin."
              << std::endl;

    TRACE(builderltlModule, {
        std::cout << "BuilderLtl: LTL formula before transforming to NNF:"
                  << std::endl;
        for (TreeNodeList::const_iterator it = formulaList.begin();
             it != formulaList.end();
             ++it)
          {
            std::cout << *it << std::endl;
          }
      });

    nnfFormulasLtl(formulaList);

    TRACE(builderltlModule, {
        std::cout << "BuilderLtl: LTL formula after transforming to NNF:"
                  << std::endl;
        for (TreeNodeList::const_iterator it = formulaList.begin();
             it != formulaList.end();
             ++it)
          {
            std::cout << *it << std::endl;
          }
      });

    std::cout << "BuilderLtl: Transforming LTL formula to NNF end."
              << std::endl;
  } // BuilderLtl::nnfLtl

  void
  BuilderLtl::simplifyLtl(TreeNodeList & formulaList,
                          TRPPPOptions const & options)
  {
    std::cout << "BuilderLtl: Simplifying LTL formula begin."
              << std::endl;

    TRACE(builderltlModule, {
        std::cout << "BuilderLtl: LTL formula before simplification:"
                  << std::endl;
        for (TreeNodeList::const_iterator it = formulaList.begin();
             it != formulaList.end();
             ++it)
          {
            std::cout << *it << std::endl;
          }
      });

    unsigned long noVerticesSyntaxTreeBeforeLtlSimplification =
      sizeTreeNodeList(formulaList) + formulaList.size() - 1;
    simplifyFormulasLtl(formulaList, options);
    unsigned long noVerticesSyntaxTreeAfterLtlSimplification =
      sizeTreeNodeList(formulaList) + formulaList.size() - 1;

    TRACE(builderltlModule, {
        std::cout << "BuilderLtl: LTL formula after simplification:"
                  << std::endl;
        for (TreeNodeList::const_iterator it = formulaList.begin();
             it != formulaList.end();
             ++it)
          {
            std::cout << *it << std::endl;
          }
      });

    std::cout << "BuilderLtl: number of nodes in syntax tree before LTL "
                 "simplification: "
              << noVerticesSyntaxTreeBeforeLtlSimplification
              << std::endl;
    std::cout << "BuilderLtl: number of nodes in syntax tree after LTL "
                 "simplification: "
              << noVerticesSyntaxTreeAfterLtlSimplification
              << std::endl;

    std::cout << "BuilderLtl: Simplifying LTL formula end."
              << std::endl;
  } // BuilderLtl::simplifyLtl

  void
  BuilderLtl::translateLtlToSnf(TreeNodeList & formulaList,
                                UnsatCore::ClauseHandleSet & clausehSet,
                               std::map<TreeNode*, UnsatCore::ClauseHandleSet> &
                                mapSubformulaToClausehsNow,
                               std::map<TreeNode*, UnsatCore::ClauseHandleSet> &
                                mapSubformulaToClausehsNext,
                               std::map<TreeNode*, UnsatCore::ClauseHandleSet> &
                                mapSubformulaToClausehsSometime,
                                TRPPPOptions const & options)
  {
    std::cout << "BuilderLtl: Translating LTL formula to SNF begin."
              << std::endl;

    SnfTranslator snfTranslator = SnfTranslator(formulaList,
                                                clausehSet,
                                                mapSubformulaToClausehsNow,
                                                mapSubformulaToClausehsNext,
                                                mapSubformulaToClausehsSometime,
                                                options);
    snfTranslator.translate();

    std::cout << "BuilderLtl: number of top level LTL subformulas directly "
                 "turned into SNF clauses: "
              << snfTranslator.myNoTopLevelSubformulaClauses
              << std::endl;
    std::cout << "BuilderLtl: number of formula to SNF cache hits: "
              << snfTranslator.myNoFormulaToSnfCacheHits
              << std::endl;
    std::cout << "BuilderLtl: number of formula to SNF cache misses: "
              << snfTranslator.myNoFormulaToSnfCacheMisses
              << std::endl;

    std::cout << "BuilderLtl: Translating LTL formula to SNF end."
              << std::endl;
  } // BuilderLtl::translateLtlToSnf

  void
  BuilderLtl::simplifySnf(UnsatCore::ClauseHandleSet & clausehSet)
  {
    std::cout << "BuilderLtl: Simplifying SNF begin."
              << std::endl;

    unsigned long noClausehsBeforePureLiteral = clausehSet.size();
    simplifyClausehsPureLiteral(clausehSet);
    unsigned long noClausehsAfterPureLiteral = clausehSet.size();

    std::cout << "BuilderLtl: number of clauses before pure literal simplification: "
              << noClausehsBeforePureLiteral
              << std::endl;
    std::cout << "BuilderLtl: number of clauses after pure literal simplification: "
              << noClausehsAfterPureLiteral
              << std::endl;

    std::cout << "BuilderLtl: Simplifying SNF end."
              << std::endl;
  } // BuilderLtl::simplifySnf

  void
  BuilderLtl::checkSnfForEmptyClause(UnsatCore::ClauseHandleSet const &
                                     clausehSet)
  {
    for (UnsatCore::ClauseHandleSet::iterator it = clausehSet.begin();
         it != clausehSet.end();
         ++it)
      {
        PropositionalProver::PClause pc;
        if (it->isIClause()) {
          pc = it->getIClause().getInternalRep();
        } else if (it->isUClause()) {
          pc = it->getUClause().getInternalRep();
        } else if (it->isSClause()) {
          pc = it->getSClause().getInternalRep();
        } else {
          Assert(it->isEClause(), "*it is not an EClause.");
          // EClause can't be empty
          continue;
        }
        if (pc->size() == 0) {
          throw Exceptions::EmptyClauseInInput();
        }
      }
  } // BuilderLtl::checkSnfForEmptyClause
} // namespace BuilderLtl
