/************************************************************
 *    Copyright (C) 2012-2013                               *
 *    Viktor Schuppan (Viktor.Schuppan@gmx.de)              *
 *                                                          *
 *    This program is free software; you can redistribute   *
 *    it and/or modify it under the terms of the GNU        *
 *    General Public License as published by the Free       *
 *    Software Foundation, either version 3 of the License, *
 *    or (at your option) any later version.                *
 *                                                          *
 *    This program is distributed in the hope that it will  *
 *    be useful, but WITHOUT ANY WARRANTY; without even     *
 *    the implied warranty of MERCHANTABILITY or FITNESS    *
 *    FOR A PARTICULAR PURPOSE.  See the GNU General Public *
 *    License for more details.                             *
 *                                                          *
 *    You should have received a copy of the GNU General    *
 *    Public License along with this program.  If not, see  *
 *    <http://www.gnu.org/licenses/>.                       *
 ************************************************************/
#include <iostream>
#include <list>
#include <map>
#include <set>
#include <string>
#include "builder_ltl/formula.h"
#include "getopt/options.h"
#include "misc/assert.h"
#include "clause_handle.h"
#include "semi_linear_set.h"
#include "unsat_core_private.h"
#include "mapper_snf_to_ltl.h"

namespace UnsatCore
{
  void
  MapperSnfToLtl::computeCoreLtl(ClauseHandleSet const & coreStartingClausehs,
                                 std::list<TreeNode*> & formulaList,
                                 std::map<TreeNode*, ClauseHandleSet> &
                                 mapSubformulaToClausehsNow,
                                 std::map<TreeNode*, ClauseHandleSet> &
                                 mapSubformulaToClausehsNext,
                                 std::map<TreeNode*, ClauseHandleSet> &
                                 mapSubformulaToClausehsSometime,
                                 std::map<ClauseHandle, SemiLinearSet> &
                                 mapClausehToSemiLinearSet,
                                 std::map<TreeNode*, SemiLinearSet> &
                                 mapSubformulaToSemiLinearSet,
                                 TRPPPOptions const & options)
  {
    std::set<TreeNode*> coreSimplifiedSubformulas;
    unsigned long noVerticesSyntaxTreeInputProblem = 0;
    unsigned long noVerticesSyntaxTreeUnsatCore = 0;

    // Take implicit conjunctions into account
    noVerticesSyntaxTreeInputProblem = formulaList.size() - 1;

    for (std::list<TreeNode *>::iterator it = formulaList.begin();
         it != formulaList.end();
         )
      {
        std::list<TreeNode *>::iterator itcur = it;
        ++it;

        // Compute size before core computation
        noVerticesSyntaxTreeInputProblem += (*itcur)->size();
        
        if (options.getExtractUnsatCoreMode() ==
            EXTRACT_UNSAT_CORE_MODE_SIMPLE) {
          TreeNode * tmp = *itcur;
          *itcur = compute_core_simple_ltl_rec(*itcur,
                                               true,
                                               coreStartingClausehs,
                                               mapSubformulaToClausehsNow,
                                               mapSubformulaToClausehsNext,
                                               mapSubformulaToClausehsSometime,
                                               coreSimplifiedSubformulas,
                                               options);
          if (tmp != *itcur) {
            delete(tmp);
          }
          if (!options.isNoUnsatCoreSimplification()) {
            tmp = *itcur;
            *itcur = simplify_core_ltl_rec(*itcur,
                                           true,
                                           coreSimplifiedSubformulas);
            if (tmp != *itcur) {
              delete(tmp);
            }
          }
        } else {
          Assert(options.getExtractUnsatCoreMode() ==
                 EXTRACT_UNSAT_CORE_MODE_SETSOFTIMEPOINTS,
                 "Unexpected core extraction mode.");
          TreeNode * tmp = *itcur;
          *itcur = compute_core_sets_of_time_points_ltl_rec(*itcur,
                                                            true,
                                                           coreStartingClausehs,
                                                     mapSubformulaToClausehsNow,
                                                    mapSubformulaToClausehsNext,
                                                mapSubformulaToClausehsSometime,
                                                      mapClausehToSemiLinearSet,
                                                   mapSubformulaToSemiLinearSet,
                                                      coreSimplifiedSubformulas,
                                                            options);
          if (tmp != *itcur) {
            delete(tmp);
          }
        }

        if (!options.isNoUnsatCoreSimplification() &&
            isIdentifier(*itcur) &&
            ((Identifier *) *itcur)->isTrue() &&
            coreSimplifiedSubformulas.find(*itcur) !=
            coreSimplifiedSubformulas.end()) {
          TreeNode * tmp = *itcur;
          formulaList.erase(itcur);
          delete(tmp);
        } else {
          // Compute size after core computation
          noVerticesSyntaxTreeUnsatCore += (*itcur)->size();
        }
      }

    // Take implicit conjunctions into account
    noVerticesSyntaxTreeUnsatCore += formulaList.size() - 1;

    // Statistics
    std::cout << "MapperSnfToLtl: number of nodes in syntax tree of input "
              << "problem: "
              << noVerticesSyntaxTreeInputProblem
              << std::endl;
    std::cout << "MapperSnfToLtl: number of nodes in syntax tree of "
              << "unsatisfiable core: "
              << noVerticesSyntaxTreeUnsatCore
              << std::endl;
  } // MapperSnfToLtl::computeCoreLtl

  void
  MapperSnfToLtl::simplifyCoreLtl(std::list<TreeNode*> & formulaList,
                                std::set<TreeNode*> & coreSimplifiedSubformulas,
                                  TRPPPOptions const & options)
  {
    Assert(options.getExtractUnsatCoreMode() == EXTRACT_UNSAT_CORE_MODE_SIMPLE,
           "Unsat core mode not simple.");
    Assert(!options.isNoUnsatCoreSimplification(),
           "options.isNoUnsatCoreSimplification is true.");

    unsigned long noVerticesSyntaxTreeCoreInput = 0;
    unsigned long noVerticesSyntaxTreeCoreSimplified = 0;

    // Take implicit conjunctions into account
    noVerticesSyntaxTreeCoreInput = formulaList.size() - 1;

    for (std::list<TreeNode *>::iterator it = formulaList.begin();
         it != formulaList.end();
         )
      {
        std::list<TreeNode *>::iterator itcur = it;
        ++it;

        // Compute size before simplification
        noVerticesSyntaxTreeCoreInput += (*itcur)->size();
        
        TreeNode * tmp = *itcur;
        *itcur = simplify_core_ltl_rec(*itcur,
                                       true,
                                       coreSimplifiedSubformulas);
        if (*itcur != tmp) {
          delete(tmp);
        }

        if (!options.isNoUnsatCoreSimplification() &&
            isIdentifier(*itcur) &&
            ((Identifier *) *itcur)->isTrue() &&
            coreSimplifiedSubformulas.find(*itcur) !=
            coreSimplifiedSubformulas.end()) {
          tmp = *itcur;
          formulaList.erase(itcur);
          delete(tmp);
        } else {
          // Compute size after simplification
          noVerticesSyntaxTreeCoreSimplified += (*itcur)->size();
        }
      }

    // Take implicit conjunctions into account
    noVerticesSyntaxTreeCoreSimplified += formulaList.size() - 1;

    // Statistics
    std::cout << "MapperSnfToLtl: number of nodes in syntax tree of "
              << "input unsatisfiable core: "
              << noVerticesSyntaxTreeCoreInput
              << std::endl;
    std::cout << "MapperSnfToLtl: number of nodes in syntax tree of "
              << "simplified unsatisfiable core: "
              << noVerticesSyntaxTreeCoreSimplified
              << std::endl;
  } // MapperSnfToLtl::simplifyCoreLtl

  TreeNode *
  MapperSnfToLtl::compute_core_simple_ltl_rec(TreeNode * n,
                                              bool isPositivePolarity,
                                   ClauseHandleSet const & coreStartingClausehs,
                                          std::map<TreeNode*, ClauseHandleSet> &
                                              mapSubformulaToClausehsNow,
                                          std::map<TreeNode*, ClauseHandleSet> &
                                              mapSubformulaToClausehsNext,
                                          std::map<TreeNode*, ClauseHandleSet> &
                                              mapSubformulaToClausehsSometime,
                                std::set<TreeNode*> & coreSimplifiedSubformulas,
                                              TRPPPOptions const & options)
  {
    TreeNode * res;

    // Determine whether n is part of the unsat core
    bool found = false;
    if (mapSubformulaToClausehsNow.find(n) !=
        mapSubformulaToClausehsNow.end()) {
      ClauseHandleSet chs = mapSubformulaToClausehsNow[n];
      for (ClauseHandleSet::iterator it = chs.begin();
           !found && it != chs.end();
           ++it)
        {
          if (coreStartingClausehs.find(*it) != coreStartingClausehs.end()) {
            found = true;
          }
        }
    }
    if (mapSubformulaToClausehsNext.find(n) !=
        mapSubformulaToClausehsNext.end()) {
      ClauseHandleSet chs = mapSubformulaToClausehsNext[n];
      for (ClauseHandleSet::iterator it = chs.begin();
           !found && it != chs.end();
           ++it)
        {
          if (coreStartingClausehs.find(*it) != coreStartingClausehs.end()) {
            found = true;
          }
        }
    }
    if (mapSubformulaToClausehsSometime.find(n) !=
        mapSubformulaToClausehsSometime.end()) {
      ClauseHandleSet chs = mapSubformulaToClausehsSometime[n];
      for (ClauseHandleSet::iterator it = chs.begin();
           !found && it != chs.end();
           ++it)
        {
          if (coreStartingClausehs.find(*it) !=
              coreStartingClausehs.end()) {
            found = true;
          }
        }
    }

    if (!found) {
      if (isPositivePolarity) {
        res = new Identifier(true);
      } else {
        res = new Identifier(false);
      }
      if (!options.isNoUnsatCoreSimplification()) {
        (void) coreSimplifiedSubformulas.insert(res);
      }
    } else {
      for (TreeNodeList::const_iterator it = n->children().begin();
           it != n->children().end();
           )
        {
          TreeNodeList::const_iterator itcur = it;
          ++it;
          if (isNot(n)) {
            TreeNode * tmp = *itcur;
            n->replaceChild(*itcur,
                            compute_core_simple_ltl_rec(*itcur,
                                                        false,
                                                        coreStartingClausehs,
                                                     mapSubformulaToClausehsNow,
                                                    mapSubformulaToClausehsNext,
                                                mapSubformulaToClausehsSometime,
                                                      coreSimplifiedSubformulas,
                                                        options));
            if (tmp != *itcur) {
              delete(tmp);
            }
          } else if (isImplication(n) && *itcur == n->firstChild()) {
            TreeNode * tmp = *itcur;
            n->replaceChild(*itcur,
                            compute_core_simple_ltl_rec(*itcur,
                                                        false,
                                                        coreStartingClausehs,
                                                     mapSubformulaToClausehsNow,
                                                    mapSubformulaToClausehsNext,
                                                mapSubformulaToClausehsSometime,
                                                      coreSimplifiedSubformulas,
                                                        options));
            if (tmp != *itcur) {
              delete(tmp);
            }
          } else {
            Assert(!isEquivalence(n), "Unexpected equivalence operator.");
            TreeNode * tmp = *itcur;
            n->replaceChild(*itcur,
                            compute_core_simple_ltl_rec(*itcur,
                                                        true,
                                                        coreStartingClausehs,
                                                     mapSubformulaToClausehsNow,
                                                    mapSubformulaToClausehsNext,
                                                mapSubformulaToClausehsSometime,
                                                      coreSimplifiedSubformulas,
                                                        options));
            if (tmp != *itcur) {
              delete(tmp);
            }
          }
        }
      res = n;
    }

    return res;
  } // MapperSnfToLtl::compute_core_simple_ltl_rec

  TreeNode *
  MapperSnfToLtl::compute_core_sets_of_time_points_ltl_rec(TreeNode * n,
                                                        bool isPositivePolarity,
                                   ClauseHandleSet const & coreStartingClausehs,
                                          std::map<TreeNode*, ClauseHandleSet> &
                                                     mapSubformulaToClausehsNow,
                                          std::map<TreeNode*, ClauseHandleSet> &
                                                    mapSubformulaToClausehsNext,
                                          std::map<TreeNode*, ClauseHandleSet> &
                                                mapSubformulaToClausehsSometime,
                                         std::map<ClauseHandle, SemiLinearSet> &
                                                      mapClausehToSemiLinearSet,
                                            std::map<TreeNode*, SemiLinearSet> &
                                                   mapSubformulaToSemiLinearSet,
                                std::set<TreeNode*> & coreSimplifiedSubformulas,
                                                   TRPPPOptions const & options)
  {
    TreeNode * res;

    // Form union of all semi-linear sets associated to n
    SemiLinearSet sls = SemiLinearSet();
    if (mapSubformulaToClausehsNow.find(n) !=
        mapSubformulaToClausehsNow.end()) {
      ClauseHandleSet chs = mapSubformulaToClausehsNow[n];
      for (ClauseHandleSet::iterator it = chs.begin();
           it != chs.end();
           ++it)
        {
          if (mapClausehToSemiLinearSet.find(*it) !=
              mapClausehToSemiLinearSet.end()) {
            SemiLinearSet sls2 = mapClausehToSemiLinearSet[*it];
            (void) sls.insert(sls2.begin(), sls2.end());
          }
        }
    }
    if (mapSubformulaToClausehsNext.find(n) !=
        mapSubformulaToClausehsNext.end()) {
      ClauseHandleSet chs = mapSubformulaToClausehsNext[n];
      for (ClauseHandleSet::iterator it = chs.begin();
           it != chs.end();
           ++it)
        {
          if (mapClausehToSemiLinearSet.find(*it) !=
              mapClausehToSemiLinearSet.end()) {
            SemiLinearSet sls2 = mapClausehToSemiLinearSet[*it];
            sls2.toNext();
            (void) sls.insert(sls2.begin(), sls2.end());
          }
        }
    }
    if (mapSubformulaToClausehsSometime.find(n) !=
        mapSubformulaToClausehsSometime.end()) {
      ClauseHandleSet chs = mapSubformulaToClausehsSometime[n];
      for (ClauseHandleSet::iterator it = chs.begin();
           it != chs.end();
           ++it)
        {
          if (mapClausehToSemiLinearSet.find(*it) !=
              mapClausehToSemiLinearSet.end()) {
            SemiLinearSet sls2 = mapClausehToSemiLinearSet[*it];
            sls2.toSometime();
            (void) sls.insert(sls2.begin(), sls2.end());
          }
        }
    }
    sls.reduce();

    if (sls.empty()) {
      if (isPositivePolarity) {
        res = new Identifier(true);
      } else {
        res = new Identifier(false);
      }
      if (!options.isNoUnsatCoreSimplification()) {
        (void) coreSimplifiedSubformulas.insert(res);
      }
    } else {
      for (TreeNodeList::const_iterator it = n->children().begin();
           it != n->children().end();
           )
        {
          TreeNodeList::const_iterator itcur = it;
          ++it;
          if (isNot(n)) {
            TreeNode * tmp = *itcur;
            n->replaceChild(*itcur,
                            compute_core_sets_of_time_points_ltl_rec(*itcur,
                                                                     false,
                                                           coreStartingClausehs,
                                                     mapSubformulaToClausehsNow,
                                                    mapSubformulaToClausehsNext,
                                                mapSubformulaToClausehsSometime,
                                                      mapClausehToSemiLinearSet,
                                                   mapSubformulaToSemiLinearSet,
                                                      coreSimplifiedSubformulas,
                                                                     options));
            if (tmp != *itcur) {
              delete(tmp);
            }
          } else if (isImplication(n) && *itcur == n->firstChild()) {
            TreeNode * tmp = *itcur;
            n->replaceChild(*itcur,
                            compute_core_sets_of_time_points_ltl_rec(*itcur,
                                                                     false,
                                                           coreStartingClausehs,
                                                     mapSubformulaToClausehsNow,
                                                    mapSubformulaToClausehsNext,
                                                mapSubformulaToClausehsSometime,
                                                      mapClausehToSemiLinearSet,
                                                   mapSubformulaToSemiLinearSet,
                                                      coreSimplifiedSubformulas,
                                                                     options));
            if (tmp != *itcur) {
              delete(tmp);
            }
          } else {
            Assert(!isEquivalence(n), "Unexpected equivalence operator.");
            TreeNode * tmp = *itcur;
            n->replaceChild(*itcur,
                            compute_core_sets_of_time_points_ltl_rec(*itcur,
                                                                     true,
                                                           coreStartingClausehs,
                                                     mapSubformulaToClausehsNow,
                                                    mapSubformulaToClausehsNext,
                                                mapSubformulaToClausehsSometime,
                                                      mapClausehToSemiLinearSet,
                                                   mapSubformulaToSemiLinearSet,
                                                      coreSimplifiedSubformulas,
                                                                     options));
            if (tmp != *itcur) {
              delete(tmp);
            }
          }
        }
      res = n;
    }

    mapSubformulaToSemiLinearSet[res] = sls;

    return res;
  } // MapperSnfToLtl::compute_core_sets_of_time_points_ltl_rec

  TreeNode * MapperSnfToLtl::simplify_core_ltl_rec(TreeNode * n,
                                                   bool isPositivePolarity,
                                std::set<TreeNode*> & coreSimplifiedSubformulas)
  {
    TreeNode * res = NULL;
    
    // simplify children
    {
      for (TreeNodeList::const_iterator it = n->children().begin();
           it != n->children().end();
           )
        {
          TreeNodeList::const_iterator itcur = it;
          ++it;
          TreeNode * childSimplified;
          if (isNot(n)) {
            childSimplified = simplify_core_ltl_rec(*itcur,
                                                    !isPositivePolarity,
                                                    coreSimplifiedSubformulas);
            if (*itcur != childSimplified) {
              TreeNode * tmp = *itcur;
              n->replaceChild(*itcur, childSimplified);
              erase_tn_from_set_rec(tmp, coreSimplifiedSubformulas);
              delete(tmp);
            }
          } else if (isImplication(n) && *itcur == n->firstChild()) {
            childSimplified = simplify_core_ltl_rec(*itcur,
                                                    !isPositivePolarity,
                                                    coreSimplifiedSubformulas);
            if (*itcur != childSimplified) {
              TreeNode * tmp = *itcur;
              n->replaceChild(*itcur, childSimplified);
              erase_tn_from_set_rec(tmp, coreSimplifiedSubformulas);
              delete(tmp);
            }
          } else {
            Assert(!isEquivalence(n), "Unexpected equivalence operator.");
            childSimplified = simplify_core_ltl_rec(*itcur,
                                                    isPositivePolarity,
                                                    coreSimplifiedSubformulas);
            if (*itcur != childSimplified) {
              TreeNode * tmp = *itcur;
              n->replaceChild(*itcur, childSimplified);
              erase_tn_from_set_rec(tmp, coreSimplifiedSubformulas);
              delete(tmp);
            }
          }
        }
    }

    // simplify self
    if (isIdentifier(n)) {
      Assert(n->childrenCount() == 0, "Unexpected number of children.");
      res = n;
    } else if (isNot(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
#ifdef DEBUG
      TreeNode * child; child = n->firstChild(); // silence compiler
#endif
      Assert(coreSimplifiedSubformulas.find(child) ==
             coreSimplifiedSubformulas.end(),
             "child present in coreSimplifiedSubformulas.");
      res = n;
    } else if (isAnd(n)) {
      Assert(n->childrenCount() > 0, "Unexpected number of children.");

      // remove True
      {
        for (TreeNodeList::const_iterator it = n->children().begin();
             it != n->children().end();
             )
          {
            TreeNodeList::const_iterator itcur = it;
            ++it;
            if (isIdentifier(*itcur) &&
                ((Identifier *) *itcur)->isTrue() &&
                coreSimplifiedSubformulas.find(*itcur) !=
                coreSimplifiedSubformulas.end()) {
              TreeNode * tmp = *itcur;
              n->removeChild(*itcur);
              delete(tmp);
            }
          }
        Assert(n->childrenCount() > 0, "Unexpected number of children.");
      }

      // single child
      if (n->childrenCount() == 1) {
        Assert(coreSimplifiedSubformulas.find(n->firstChild()) ==
               coreSimplifiedSubformulas.end(),
               "n->firstChild() present in coreSimplifiedSubformulas.");
        res = n->firstChild()->clone();
        TreeNode * tmp = n->firstChild();
        n->removeChild(n->firstChild());
        delete(tmp);
        return res;
      }
      
      res = n;
    } else if (isOr(n)) {
      Assert(n->childrenCount() > 0, "Unexpected number of children.");

      // remove False
      {
        for (TreeNodeList::const_iterator it = n->children().begin();
             it != n->children().end();
             )
          {
            TreeNodeList::const_iterator itcur = it;
            ++it;
            if (isIdentifier(*itcur) &&
                ((Identifier *) *itcur)->isFalse() &&
                coreSimplifiedSubformulas.find(*itcur) !=
                coreSimplifiedSubformulas.end()) {
              TreeNode * tmp = *itcur;
              n->removeChild(*itcur);
              delete(tmp);
            }
          }
        Assert(n->childrenCount() > 0, "Unexpected number of children.");
      }

      // single child
      if (n->childrenCount() == 1) {
        Assert(coreSimplifiedSubformulas.find(n->firstChild()) ==
               coreSimplifiedSubformulas.end(),
               "n->firstChild() present in coreSimplifiedSubformulas.");
        res = n->firstChild()->clone();
        TreeNode * tmp = n->firstChild();
        n->removeChild(n->firstChild());
        delete(tmp);
        return res;
      }
      
      res = n;
    } else if (isImplication(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      TreeNode * fst = n->firstChild();
      TreeNode * snd = n->secondChild();
      if (isIdentifier(fst) &&
          ((Identifier *) fst)->isTrue() &&
          coreSimplifiedSubformulas.find(fst) !=
          coreSimplifiedSubformulas.end()) {
        Assert(coreSimplifiedSubformulas.find(snd) ==
               coreSimplifiedSubformulas.end(),
               "snd present in coreSimplifiedSubformulas.");
        res = snd->clone();
        n->removeChild(snd);
        delete(snd);
        return res;
      } else if (isIdentifier(snd) &&
                 ((Identifier *) snd)->isFalse() &&
                 coreSimplifiedSubformulas.find(snd) !=
                 coreSimplifiedSubformulas.end()) {
        Assert(coreSimplifiedSubformulas.find(fst) ==
               coreSimplifiedSubformulas.end(),
               "fst present in coreSimplifiedSubformulas.");
        res = Not(fst->clone());
        n->removeChild(fst);
        delete(fst);
        return res;
      }
      res = n;
    } else if (isNext(n) || isAlways(n) || isSometime(n)) {
      Assert(n->childrenCount() == 1, "Unexpected number of children.");
#ifdef DEBUG
      TreeNode * child; child = n->firstChild(); // silence compiler
#endif
      Assert(coreSimplifiedSubformulas.find(child) ==
             coreSimplifiedSubformulas.end(),
             "child present in coreSimplifiedSubformulas.");
      res = n;
    } else if (isUntil(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      TreeNode * fst = n->firstChild();
      TreeNode * snd = n->secondChild();
      if (isIdentifier(fst) &&
          ((Identifier *) fst)->isTrue() &&
          coreSimplifiedSubformulas.find(fst) !=
          coreSimplifiedSubformulas.end()) {
        Assert(coreSimplifiedSubformulas.find(snd) ==
               coreSimplifiedSubformulas.end(),
               "snd present in coreSimplifiedSubformulas.");
        ((Operator *) n)->setOperator(SOMETIME);
        n->removeChild(fst);
        delete(fst);
        res = n;
        return res;
      } else if (isIdentifier(fst) &&
                 ((Identifier *) fst)->isFalse() &&
                 coreSimplifiedSubformulas.find(fst) !=
                 coreSimplifiedSubformulas.end()) {
        Assert(coreSimplifiedSubformulas.find(snd) ==
               coreSimplifiedSubformulas.end(),
               "snd present in coreSimplifiedSubformulas.");
        res = snd->clone();
        n->removeChild(snd);
        delete(snd);
        return res;
      }
      res = n;
    } else if (isUnless(n)) {
      Assert(n->childrenCount() == 2, "Unexpected number of children.");
      TreeNode * fst = n->firstChild();
      TreeNode * snd = n->secondChild();
      if (isIdentifier(snd) &&
          ((Identifier *) snd)->isFalse() &&
          coreSimplifiedSubformulas.find(snd) !=
          coreSimplifiedSubformulas.end()) {
        Assert(coreSimplifiedSubformulas.find(fst) ==
               coreSimplifiedSubformulas.end(),
               "fst present in coreSimplifiedSubformulas.");
        ((Operator *) n)->setOperator(ALWAYS);
        n->removeChild(snd);
        delete(snd);
        res = n;
        return res;
      } else if (isIdentifier(fst) &&
                 ((Identifier *) fst)->isFalse() &&
                 coreSimplifiedSubformulas.find(fst) !=
                 coreSimplifiedSubformulas.end()) {
        Assert(coreSimplifiedSubformulas.find(snd) ==
               coreSimplifiedSubformulas.end(),
               "snd present in coreSimplifiedSubformulas.");
        res = snd->clone();
        n->removeChild(snd);
        delete(snd);
        return res;
      }
      res = n;
    } else {
      Assert(false, "n has unexpected type.");
    }

    return res;
  } // MapperSnfToLtl::simplify_core_ltl_rec

  void MapperSnfToLtl::erase_tn_from_set_rec(TreeNode * n,
                                             std::set<TreeNode*> & s)
  {
    for (TreeNodeList::const_iterator it = n->children().begin();
         it != n->children().end();
         ++it) {
      erase_tn_from_set_rec(*it, s);
    }
    if (s.find(n) != s.end()) {
      (void) s.erase(n);
    }
  } // MapperSnfToLtl::erase_tn_from_set_rec
} // namespace UnsatCore
