/************************************************************
 *    Copyright (C) 2012                                    *
 *    Viktor Schuppan (Viktor.Schuppan@gmx.de)              *
 *                                                          *
 *    This program is free software; you can redistribute   *
 *    it and/or modify it under the terms of the GNU        *
 *    General Public License as published by the Free       *
 *    Software Foundation, either version 3 of the License, *
 *    or (at your option) any later version.                *
 *                                                          *
 *    This program is distributed in the hope that it will  *
 *    be useful, but WITHOUT ANY WARRANTY; without even     *
 *    the implied warranty of MERCHANTABILITY or FITNESS    *
 *    FOR A PARTICULAR PURPOSE.  See the GNU General Public *
 *    License for more details.                             *
 *                                                          *
 *    You should have received a copy of the GNU General    *
 *    Public License along with this program.  If not, see  *
 *    <http://www.gnu.org/licenses/>.                       *
 ************************************************************/
#ifndef __LINEAR_SET_H__
#define __LINEAR_SET_H__

#include <iostream>
#include "misc/assert.h"
#include "unsat_core_private.h"

namespace UnsatCore
{
  class LinearSet
  {
  private:
    unsigned long myPeriod;
    unsigned long myOffset;
  public:
    LinearSet();
    LinearSet(unsigned long const & period, unsigned long const & offset);
    unsigned long getPeriod() const;
    unsigned long getOffset() const;
    void setPeriod(unsigned long const & period);
    void setOffset(unsigned long const & offset);
    bool isInvalid() const;
    void makeInvalid();
    bool operator<(LinearSet const & other) const;
    std::ostream & dump(std::ostream & os) const;

    /*
- p1 > 0 && p2 > 0
  - p1 == p2
    - o1 <= o2 && o1 mod p1 == o2 mod p1
      - {s1}
    - o1 > o2 && o1 mod p1 == o2 mod p1
      - {s2}
  - p1 > p2
    - p1 mod p2 == 0
      - o1 >= o2 && o1 mod p2 == o2 mod p2
        - {s2}
      - o1 == o2 - p2
        - {(p2, o2 - p2)}
      - o1 == o2 - p1
        - {(0, o1), s2}
    - p1 mod p2 != 0
      - E x >= 0 . o2 - p2 == x * p1 + o1
        E x >= 0 . o2 - p2 - o1 == x * p1
        o2 - p2 - o1 >= 0 && (o2 - p2 - o1) mod p1 == 0
        - {s1, (p2, o2 - p2)}
      - (symmetric)
        E x >= 0 . o1 - p1 == x * p2 + o2
        E x >= 0 . o1 - p1 - o2 == x * p2
        o1 - p1 - o2 >= 0 && (o1 - p1 - o2) mod p2 == 0
        - {(p1, o1 - p1), s2}
  - p1 < p2
    (symmetric)
    - p2 mod p1 == 0
      - o1 <= o2 && o2 mod p1 == o1 mod p1
        - {s1}
      - o2 == o1 - p1
        - {(p1, o1 - p1)}
      - o2 == o1 - p2
        - {s1, (0, o2)}
    - p2 mod p1 != 0
      - E x >= 0 . o1 - p1 == x * p2 + o2
        E x >= 0 . o1 - p1 - o2 == x * p2
        o1 - p1 - o2 >= 0 && (o1 - p1 - o2) mod p2 == 0
        - {(p1, o1 - p1), s2}
      - (symmetric)
        E x >= 0 . o2 - p2 == x * p1 + o1
        E x >= 0 . o2 - p2 - o1 == x * p1
        o2 - p2 - o1 >= 0 && (o2 - p2 - o1) mod p1 == 0
        - {s1, (p2, o2 - p2)}
- p1 > 0 && p2 == 0
  - o2 - o1 >= 0 && (o2 - o1) mod p1 == 0
    - {s1}
  - o2 == o1 - p1
    - {(p1, o1 - p1)}
- p2 > 0 && p1 == 0
  (symmetric)
  - o1 - o2 >= 0 && (o1 - o2) mod p2 == 0
    - {s2}
  - o1 == o2 - p2
    - {(p2, o2 - p2)}
    */
    static inline bool reduce(LinearSet & ls1, LinearSet & ls2)
    {
      Assert(!ls1.isInvalid(), "ls1 is invalid.");
      Assert(!ls2.isInvalid(), "ls2 is invalid.");

      bool result = false;
      unsigned long p1 = ls1.getPeriod();
      unsigned long o1 = ls1.getOffset();
      unsigned long p2 = ls2.getPeriod();
      unsigned long o2 = ls2.getOffset();

      if (p1 > 0 && p2 > 0) {
        if (p1 == p2 && o1 % p1 == o2 % p1) {
          if (o1 <= o2) {
            ls2.makeInvalid();
            result = true;
          } else {
            ls1.makeInvalid();
            result = true;
          }
        } else if (p1 > p2) {
          if (p1 % p2 == 0) {
            if (o1 >= o2 && o1 % p2 == o2 % p2) {
              ls1.makeInvalid();
              result = true;
            } else if (o2 >= p2 && o1 == o2 - p2) {
              ls1.makeInvalid();
              ls2.setOffset(o2 - p2);
              result = true;
            } else if (o2 >= p1 && o1 == o2 - p1) {
              ls1.setPeriod(0);
              result = true;
            }
          } else {
            if (o2 >= p2 &&
                o2 - p2 >= o1 &&
                (o2 - p2 - o1) % p1 == 0) {
              ls2.setOffset(o2 - p2);
              result = true;
            } else if (o1 >= p1 &&
                       o1 - p1 >= o2 &&
                       (o1 - p1 - o2) % p2 == 0) {
              ls1.setOffset(o1 - p1);
              result = true;
            }
          }
        } else if (p1 < p2) {
          if (p2 % p1 == 0) {
            if (o2 >= o1 && o2 % p1 == o1 % p1) {
              ls2.makeInvalid();
              result = true;
            } else if (o1 >= p1 && o2 == o1 - p1) {
              ls1.setOffset(o1 - p1);
              ls2.makeInvalid();
              result = true;
            } else if (o1 >= p2 && o2 == o1 - p2) {
            ls2.setPeriod(0);
            result = true;
            }
          } else {
            if (o1 >= p1 &&
                o1 - p1 >= o2 &&
                (o1 - p1 - o2) % p2 == 0) {
              ls1.setOffset(o1 - p1);
              result = true;
            } else if (o2 >= p2 &&
                       o2 - p2 >= o1 &&
                       (o2 - p2 - o1) % p1 == 0) {
              ls2.setOffset(o2 - p2);
              result = true;
            }
          }
        }
      } else if (p1 > 0 && p2 == 0) {
        if (o2 >= o1 && (o2 - o1) % p1 == 0) {
          ls2.makeInvalid();
          result = true;
        } else if (o1 >= p1 && o2 == o1 - p1) {
          ls1.setOffset(o1 - p1);
          ls2.makeInvalid();
          result = true;
        }
      } else if (p2 > 0 && p1 == 0) {
        if (o1 >= o2 && (o1 - o2) % p2 == 0) {
          ls1.makeInvalid();
          result = true;
        } else if (o2 >= p2 && o1 == o2 - p2) {
          ls1.makeInvalid();
          ls2.setOffset(o2 - p2);
          result = true;
        }
      }

      return result;
    } // LinearSet::reduce
  }; // class LinearSet
} // namespace UnsatCore
#endif // __LINEAR_SET_H__
