/************************************************************
 *    Copyright (C) 2014                                    *
 *    Viktor Schuppan (Viktor.Schuppan@gmx.de)              *
 *                                                          *
 *    This program is free software; you can redistribute   *
 *    it and/or modify it under the terms of the GNU        *
 *    General Public License as published by the Free       *
 *    Software Foundation, either version 3 of the License, *
 *    or (at your option) any later version.                *
 *                                                          *
 *    This program is distributed in the hope that it will  *
 *    be useful, but WITHOUT ANY WARRANTY; without even     *
 *    the implied warranty of MERCHANTABILITY or FITNESS    *
 *    FOR A PARTICULAR PURPOSE.  See the GNU General Public *
 *    License for more details.                             *
 *                                                          *
 *    You should have received a copy of the GNU General    *
 *    Public License along with this program.  If not, see  *
 *    <http://www.gnu.org/licenses/>.                       *
 ************************************************************/
#include <iostream>
#include "misc/assert.h"
#include "clause_handle.h"
#include "unsat_core_private.h"
#include "literal_clauseh_occurrence.h"

namespace UnsatCore
{
  LiteralClausehOccurrence::LiteralClausehOccurrence(
                                                   ClauseHandle const & clauseh,
                                             Clause::const_iterator const & pos)
    : myClauseh(clauseh), myIsEventuality(false), myPos(pos)
  {
#ifdef DEBUG
    PClause pc = myClauseh.getPClause();
    Assert(pc->cbegin() <= myPos && myPos < pc->cend(),
           "myPos not a position in pc.");
#endif
  } // LiteralClausehOccurrence::LiteralClausehOccurrence

  LiteralClausehOccurrence::LiteralClausehOccurrence(
                                                   ClauseHandle const & clauseh)
    : myClauseh(clauseh), myIsEventuality(true)
  {
    Assert(clauseh.isEClause(), "Not an eventuality clause.");
    myPos = clauseh.getEClause().getC().getInternalRep()->cend();
  } // LiteralClausehOccurrence::LiteralClausehOccurrence

  LiteralClausehOccurrence::LiteralClausehOccurrence(
                                                   ClauseHandle const & clauseh,
                                                     bool is_eventuality,
                                             Clause::const_iterator const & pos)
    : myClauseh(clauseh), myIsEventuality(is_eventuality), myPos(pos)
  { } // LiteralClausehOccurrence::LiteralClausehOccurrence

  ClauseHandle
  LiteralClausehOccurrence::getClauseh() const
  {
    return myClauseh;
  } // LiteralClausehOccurrence::getClauseh

  bool
  LiteralClausehOccurrence::isEventuality() const
  {
    return myIsEventuality;
  } // LiteralClausehOccurrence::isEventuality

  Clause::const_iterator
  LiteralClausehOccurrence::getPos() const
  {
    return myPos;
  } // LiteralClausehOccurrence::getPos

  Literal
  LiteralClausehOccurrence::getLiteral() const
  {
    Literal l;

    if (myIsEventuality) {
      Assert(myClauseh.isEClause(), "Not an eventuality clause.");
      l = myClauseh.getEClause().getEventuality();
    } else {
#ifdef DEBUG
      PClause pc= myClauseh.getPClause();
      Assert(pc->cbegin() <= myPos && myPos < pc->cend(),
             "myPos not a position in pc.");
#endif
      l = *myPos;
    }
    return l;
  } // LiteralClausehOccurrence::getLiteral

  std::ostream &
  LiteralClausehOccurrence::dump(std::ostream & os) const
  {
    os << "(";
    if (myIsEventuality ||
        (myClauseh.getPClause()->cbegin() <= myPos &&
         myPos < myClauseh.getPClause()->cend())) {
      getLiteral().dump(os);
    } else {
      os << "invalid";
    }
    os << " in ";
    myClauseh.dump(os);
    os << ")";
    return os;
  } // LiteralClausehOccurrence::dump

  bool
  LiteralClausehOccurrence::operator<
  (LiteralClausehOccurrence const & other) const
  {
    if (myClauseh < other.myClauseh) {
      return true;
    } else if (other.myClauseh < myClauseh) {
      return false;
    } else if (myIsEventuality < other.myIsEventuality) {
      return true;
    } else if (other.myIsEventuality < myIsEventuality) {
      return false;
    } else {
      return myPos < other.myPos;
    }
  } // LiteralClausehOccurrence::operator<

  bool
  LiteralClausehOccurrence::operator==
  (LiteralClausehOccurrence const & other) const
  {
    return ((myClauseh == other.myClauseh) &&
            (myIsEventuality == other.myIsEventuality) &&
            (myPos == other.myPos));
  } // LiteralClausehOccurrence::operator==

  LiteralClausehOccurrence
  LiteralClausehOccurrence::makeLiteralClausehOccurrenceInvalidPos
  (ClauseHandle const & clauseh)
  {
    PClause pc = clauseh.getPClause();
    return LiteralClausehOccurrence(clauseh, false, pc->cend());
  } // LiteralClausehOccurrence::makeLiteralClausehOccurrenceInvalidPos
} // namespace UnsatCore
